/**
* An ordinary JavaScript as obtained by "new Object()" or via 
*  ObjectLiteral syntax.
*  
*  Methods that return Objects or take Objects as parameters make use of the ability of a
*  JavaScript Object to contain an arbitrary set of named properties, without requiring
*  declaration in advance. This capability makes it possible to use a JavaScript Object much
*  like a HashMap in Java or .NET, but without the need to call get() or set() to create and
*  retrieve properties.
*  
*  For example if you created an Object using ObjectLiteral syntax
*  like so:
*  
*   var request = {
*     actionURL : "/foo.do",
*     showPrompt:false
*   };
*  
*  You could then access it's properties like so:
*  
*   var myActionURL = request.actionURL;
*   var myShowPrompt = request.showPrompt;
*  
*  .. and you could assign new values to those properties like so:
*  
*   request.actionURL = "newActionURL";
*   request.showPrompt = newShowPromptSetting;
*  
*  Note that while JavaScript allows you to get and set properties in this way on any Object,
*  SmartClient components require that if a setter or getter exists, it must be called, or no
*  action will occur. For example, if you had a ListGrid and you wanted to change the
*  ListGrid.showHeader property:
*  
*    myListGrid.setShowHeader(false); // correct
*    myListGrid.showHeader = false; // incorrect (nothing happens)
*  
*  All documented attributes have flags (eg IRW) that indicate when direct
*  property access is allowed or not.
*/
interface Object {

}
interface ObjectConstructor {

}/**
* Generic extensions to JavaScript Arrays. You can call these on any Array.
*  
*  JavaScript's native Array is retrofitted to support the List API.
*/
interface Array {


/**
*  Return the item at a particular position
* 
* @param {number} position of the element to get
*/
get(pos:number): object;

/**
*  Return the number of items in this list
* 
* 
*/
getLength(): number;

/**
*  Return whether or not this array is empty
* 
* 
*/
isEmpty(): boolean;

/**
*  Return the first item in this list
* 
* 
*/
first(): any;

/**
*  Return the last item in this list
* 
* 
*/
last(): any;

/**
*  Return the position in the list of the first instance of the specified object.
*  
*  If pos is specified, starts looking after that position.
*  
*  Returns -1 if not found.
* 
* 
* @param {any} object to look for
* @param {number=} earliest index to consider
* @param {number=} last index to consider
*/
indexOf(obj:any, pos?:number, endPos?:number): number;

/**
*  Return the position in the list of the last instance of the specified object.
*  
*  If pos is specified, starts looking before that position.
*  
*  Returns -1 if not found.
* 
* 
* @param {any} object to look for
* @param {number=} last index to consider
* @param {number=} earliest index to consider
*/
lastIndexOf(obj:any, pos?:number, endPos?:number): number;

/**
*  Return if this list contains the specified object.
*  
*  If pos is specified, starts looking after that position.
* 
* 
* @param {any} item to look for
* @param {number=} optional position in the list to look after
*/
contains(obj:any, pos?:number): boolean;

/**
*  Return whether this list contains all the item in the specified list.
* 
* 
* @param {List} items to look for
*/
containsAll(list:isc.List): boolean;

/**
*  Return the list of items that are in both this list and the passed-in list(s).
* 
* 
* @param {any} lists to intersect with
*/
intersect(lists:any): isc.List;

/**
*  Return whether this list is equal to another list.
*  
*  Two lists are equal only if they have the same length and all contained items are in the same
*  order and are also equal.
* 
* 
* @param {List} list to check for equality
*/
equals(list:isc.List): boolean;

/**
*  Return the items at a list of specified positions.
* 
* 
* @param {Array<Partial<number>>} array of positions
*/
getItems(itemList:Array<Partial<number>>): Array<any>;

/**
*  Return the items between position start and end, non-inclusive at the end.
* 
* 
* @param {number} start position
* @param {number} end position
*/
getRange(start:number, end:number): Array<any>;

/**
*  Return an Array that is a shallow copy of the list, that is, containing the same items.
* 
* 
*/
duplicate(): Array<any>;

/**
*  Change the array element at a particular position.
*  
*  set() can be used to expand the length of the list.
* 
* 
* @param {number} position in the list to change
* @param {object} new value for that position
*/
set(pos:number, obj:object): object;

/**
*  Add a single item to this array at a specific position in the list, sliding other items over
*  to fit.
* 
* 
* @param {object} object to add
* @param {number} position in the list to add at
*/
addAt(obj:object, pos:number): object;

/**
*  Remove the item at the specified position, rearranging all subsequent items to fill the gap
* 
* 
* @param {number} position to remove
*/
removeAt(pos:number): any;

/**
*  Add an object to this list, at the end
* 
* 
* @param {any} object to add
*/
add(object:any): any;

/**
*  Add a list of items to this array.
*  
*  Note: you can specify that a subset range be added by passing start and end indices
* 
* 
* @param {Array<any>} list of items to add
* @param {number=} optional start index in list
* @param {number=} optional end index in list (non-inclusive)
*/
addList(list:Array<any>, listStartRow?:number, listEndRow?:number): isc.List;

/**
*  Set the length of this list.
*  
*  If the length of the list is shortened, any elements past the new length of the list are removed.
*  If the length is increased, all positions past the old length have the value
*  undefined.
* 
* 
* @param {number} new length
*/
setLength(length:number): void;

/**
*  Add list of items list to this array at item pos. All items after array[pos] will slide down to
*  fit new items.
* 
* 
* @param {Array<any>} new array of items
* @param {number} position in this list to put the new items
*/
addListAt(list:Array<any>, pos:number): Array<any>;

/**
*  Remove first instance of the passed object from this array, sliding other items around to
*  fill gaps.
* 
* 
* @param {any} item to remove
*/
remove(obj:any): boolean;

/**
*  Remove all instances of objects in the specified list from this list, sliding the remaining
*  objects around to fill gaps.
* 
* 
* @param {Array<any>} list of items to remove
*/
removeList(list:Array<any>): isc.List;

/**
*  Method called when this array changes in some way. Observe the method to react to changes in
*  this list.
*  
*  Note: dataChanged() will only fire when items are added, removed or rearranged. If a list
*  contains objects, dataChanged() will not fire if changes are made to objects within the list
*  without changing their position within the list. If an observer of dataChanged() needs to react
*  to such a change, you can manually fire dataChanged() by simply calling it.
*  
*  Note: may be called multiple times as the result of a multi-item add or remove, etc.
* 
* 
*/
dataChanged(): void;

/**
*  Return a new Array where the value of item i is the value of "property" of item i in this
*  array. If an item doesn't have that property or is null, return item will be null.
* 
* 
* @param {string} name of the property to look for
*/
getProperty(property:string): Array<any>;

/**
*  Get a map of the form { item[idField] -&gt; item[displayField] }, for all 
*  items in the list. Note that if more than one item has the same idProperty, 
*  the value for the later item in the list will clobber the value for the earlier item.
* 
* 
* @param {string} Property to use as ID (data value) in the valueMap
* @param {string} Property to use a display value in the valueMap
*/
getValueMap(idField:string, displayField:string): object;

/**
*  Calls a function for each member of an array, passing in the member, its index and
*  the array itself as arguments. Returns a new array containing the
*  resulting values.
*  
*  This behavior is part of the
*  ECMA-262 specification.
*  
*  Backwards compatibility note: Historically SmartClient provided a version of
*  array.map() which differed from the native behavior in a couple of ways:
*  
*  
* - If passed a string as the function argument, it would invoke a same-named method on
*   each member of the array. This is now deprecated in favor of 
*   calling Array.callMethod directly
*  
* - If additional arguments other than the function were passed to this
*   method, when the function was invoked for each member, these additional arguments
*   would be passed in when the function was invoked. This is also deprecated as it
*   conflicts with the default native implementation
*  
* 
* @param {Function} function to execute for each item
*/
map(method:Function): Array<any>;

/**
*  Calls a method on each member of an array and returns a new array containing the
*  resulting values.
*  
*  The method argument should be the String name of a
*  method present on each item, which will be invoked. If any item is null
*  or lacks the named method, null will be returned for that item.
*  
*  Examples:
*   // line up widgets at 20 pixels from page edge
*   [widget1, widget2].callMethod("setPageLeft", 20);
* 
*   // find furthest right widget
*   [widget1, widget2].callMethod("getPageRight").max();
*  
*  
* 
* @param {string} Name of method to execute for on item
* @param {any=} arguments to pass to the method                      invoked on each item
*/
callMethod(method:string, ...args:any[]): Array<any>;

/**
* 	Set item[property] = value for each item in this array.
* 
* @param {string} name of the property to set
* @param {any} value to set to
*/
setProperty(property:string, value:any): void;

/**
*  Delete property in each item in this array.
* 
* @param {string} name of the property to clear
*/
clearProperty(property:string): boolean;

/**
*  Return a list of each unique item in this list exactly once.
*  
*  Returns in the same order they were found in the list.
*  
*  Usage example:
*  &nbsp;&nbsp;&nbsp;&nbsp;uniqueList = myArray.getProperty("foo").getUniqueItems();
* 
* 
*/
getUniqueItems(): Array<any>;

/**
*  Find the index of the first Object where property == value in the object.
*  
*  Pass an Object instead to match multiple properties.
*  
*  Note: for string values, matches are case sensitive.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
findIndex(propertyName:string | object | isc.AdvancedCriteria, value?:any): isc.Integer;

/**
*  Like Array.findIndex, but inspects a range from startIndex to endIndex.
*  
*  
*  For convenience, findNextIndex() may also be called with a function (called the predicate
*  function) for the propertyName parameter. In this usage pattern, the predicate
*  function is invoked for each value of the list until the predicate returns a true value.
*  The predicate function is passed three parameters: the current value, the current index, and
*  the list. The value of this when the predicate function is called is the
*  value parameter. For example:
*  var currentUserRecord = recordList.findNextIndex(0, function (record, i, recordList) {
*   if (record.username == currentUsername && !record.accountDisabled) {
*     return true;
*   }
* });
*  
* 
* 
* @param {Integer} first index to consider.
* @param {string | Function | object | AdvancedCriteria} property to match; or, if a function is passed, the predicate function to call; or, if an object is passed, set of properties and values to match.
* @param {any=} value to compare against (if propertyName is a string) or the value of this when the predicate function is invoked (if propertyName is a function)
* @param {Integer=} last index to consider (inclusive).
*/
findNextIndex(startIndex:isc.Integer, propertyName:string | Function | object | isc.AdvancedCriteria, value?:any, endIndex?:isc.Integer): isc.Integer;

/**
*  Like Array.findIndex, but returns the object itself instead of its index.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
find(propertyName:string | object | isc.AdvancedCriteria, value?:any): object;

/**
*  Determine whether this array contains any members where the property passed in matches the value
*  passed in.
* 
* 
* @param {string} property to look for   	(object)	key:value pairs to look for
* @param {any=} value to compare against (if property is a string)
*/
containsProperty(property:string, value?:any): boolean;

/**
*  Find all objects where property == value in the object.
*  
*  Pass an Object as the propertyName argument to match multiple properties.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
findAll(propertyName:string | object | isc.AdvancedCriteria, value?:any): Array<any>;

/**
*  Slide element at position start to position destination, moving all the other elements to cover
*  the gap.
* 
* 
* @param {number} start position
* @param {number} destination position for the value at start
*/
slide(start:number, destination:number): void;

/**
*  Slide a range of elements from start to end to position destination, moving all the other
*  elements to cover the gap.
* 
* 
* @param {number} start position
* @param {number} end position (exclusive, like substring() and slice())
* @param {number} destination position for the range
*/
slideRange(start:number, end:number, destination:number): void;

/**
*  Make an index for the items in this Array by a particular property of each item.
*  
*  Returns an Object with keys for each distinct listItem[property] value. Each key will point
*  to an array of items that share that property value. The sub-array will be in the same order
*  that they are in this list.
* 
* 
* @param {string} names of the property to index by
* @param {boolean} [false] if true, we always make an array for every index. if false, we make an Array only        when more than one item has the same value for the index property
*/
makeIndex(property:string, alwaysMakeArray:boolean): object;

/**
*  Sort a list of objects by a given property of each item.
*  
*  The optional normalizer, if passed as a function, is called for each item in the List, and
*  should return whatever value should be used for sorting, which does not have to agree with
*  the property value. By passing a normalizer function you can achieve any kind of sorting
*  you'd like, including sorting by multiple properties.
*  
*  NOTE: string sort is case INsensitive by default
* 
* 
* @param {string} name of the property to sort by
* @param {boolean} true == sort ascending, false == sort descending
* @param {Function | ValueMap=} May be specified as a function, with signature        normalize(item, propertyName, context), where item is        a pointer to the item in the array, propertyName is the        property by which the array is being sorted, and context is the        arbitrary context passed into this method. Normalizer function should return        the value normalized for sorting.        May also be specified as a ValueMap which maps property values to sortable values.
* @param {any=} Callers may pass an arbitrary context into the sort method, which             will then be made available to the normalizer function
*/
sortByProperty(property:string, up:boolean, normalizer?:Function | isc.ValueMap, context?:any): isc.List;

/**
*  Sort this Array by a list of SortSpecifiers. 
* 
* @param {Array<Partial<SortSpecifier>>} the list of SortSpecifiers to sort by
*/
setSort(sortSpecifiers:Array<Partial<isc.SortSpecifier>>): Array<any>;

/**
*  Turn sorting off for this array, indicating that the current sort
*  order should be preserved. Return true if this is supported in this List.
* 
*  Some implementations may not support this -- they should return false
*  to indicate to the caller that sort order must be maintained (eg: in
*  the case where sort order is derived from the server, etc).
* 
* 
*/
unsort(): boolean;

/**
* 
*   Returns the largest number in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, searches the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
max(start?:number, end?:number): number;

/**
*      
*   Returns the smallest number in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, searches the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
min(start?:number, end?:number): number;

/**
*   Returns the sum of the numbers in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, uses only the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
sum(start?:number, end?:number): number;

/**
*  Returns true if all values between the start and end indices are true.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
and(start?:number, end?:number): boolean;

/**
*  Returns true if at least one value between the start and end indices is true.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
or(start?:number, end?:number): boolean;




}
interface ArrayConstructor {




/**
*  Is the object passed in a loading marker value? For use with Lists that manage remote
*  datasets, to indicate that a record has not yet been retrieved from the server. A typical
*  use case might be to check if a row has been loaded in a ListGrid - for example:
*  
*  
*  if (Array.isLoading(myList.getRecord(0))) isc.warn("Please wait for the data to load.");
*  
* 
* @param {any} data to test.
*/
isLoading(value:any): void;

/**
*  Compare two values for an ascending order sort, using locale-sensitive comparison.
* 
* @param {any} first value to compare
* @param {any} second value to compare
*/
compareAscending(a:any, b:any): number;

/**
*  Compare two values for a descending order sort, using locale-sensitive comparison.
* 
* @param {any} first value to compare
* @param {any} second value to compare
*/
compareDescending(first:any, second:any): number;




}/**
* Boolean object. Attributes, parameters, or return values declared as Boolean
*  may be null. Contrast with boolean.
*/
interface Boolean {





}
interface BooleanConstructor {







}/**
* Extra methods added to the Number object, available on all number variables. Attributes,
*  parameters, or return values declared as Number may be null.
*/
interface Number {


/**
* 
*  Return this number as a string padded out to digits length.
* 
* 
* @param {number=} [2] Number of digits to pad to. (Default is 2)
* @deprecated * Moved to a static method on NumberUtil to avoid the possibility of collision        with other libraries on the native Number object

*/
stringify(digits?:number): string;

/**
*  Returns true if the number parameter falls between the 'first' and 'second' paramters.
* 
* 
* @param {number} Number object to be evaluated
* @param {number=} Number at the lower boundary
* @param {number=} Number at the upper boundary
* @param {number=} Whether or not the numbers at either end of the boundary should be included in the comparison
*/
isBetween(number:number, first?:number, second?:number, inclusive?:number): boolean;

/**
*  Return this number as a currency-formatted string.
* 
* 
* @param {string=} Currency symbol, can be set to an empty string.                 If unset "$" will be used.
* @param {string=} Decimal separator symbol. If unset "." will be used.
* @param {boolean=} Should decimal portion be padded out to two digits? True                by default.
* @param {boolean=} Should currency symbol come at the end of the string?                   If unspecified, currency symbol will be shown at the                    beginning of the string.
* @deprecated * Moved to a static method on NumberUtil to avoid the possibility of collision        with other libraries on the native Number object

*/
toCurrencyString(currencyChar?:string, decimalChar?:string, padDecimal?:boolean, currencyCharLast?:boolean): string;




}
interface NumberConstructor {







}/**
* Generic extensions to JavaScript Strings. You can call these on any String.
*/
interface String {


/**
*  	Returns true if this string contains the specified substring.
* 
* @param {string} string to look for
*/
contains(substring:string): boolean;

/**
*  Returns true if this string starts with another string, or if the other string
*  occurs at the given position within this string.
* 
* 
* @param {string} other string to check
* @param {Integer=} optional position in this string. Defaults to 0.
*/
startsWith(substring:string, position?:isc.Integer): boolean;

/**
*  Returns true if this string ends with another string, or if the other string
*  occurs in this string beginning at position - substring.length.
* 
* 
* @param {string} other string to check
* @param {Integer=} optional position in this string. Defaults to the length of this string.
*/
endsWith(substring:string, position?:isc.Integer): boolean;




}
interface StringConstructor {


/**
*  Tests whether the given string is a valid JavaScript identifier.
* 
* 
* @param {string} the string to test.
*/
isValidID(string:string): boolean;






}
declare namespace isc {
interface Map<T> { [key: string]: T }
/**
*  Enables full screen reader mode. Must be called before any components are created. See
*  accessibility.
* 
* @param {boolean} new setting
*/
function setScreenReaderMode(newState:boolean):void;
/**
*  Displays a new message that's automatically dismissed after a configurable amount of time,
*  as an alternative to isc.confirm dialogs that can lower end user
*  productivity.
*  
*  This method is simply a shorthand way to call Notify.addMessage. For further study,
*  see the Notify class overview, and the class methods Notify.dismissMessage and Notify.configureMessages.
* 
* @param {HTMLString} message to be displayed
* @param {Array<Partial<NotifyAction>>=} actions (if any) for this message
* @param {NotifyType=} category of message
* @param {NotifySettings=} display and behavior settings for this message,                        overriding any stored settings for this                        NotifyType
*/
function notify(contents:HTMLString, actions?:Array<Partial<NotifyAction>>, notifyType?:NotifyType, settings?:NotifySettings):MessageID;
/**
* 
* 	Method available on the isc object to show a modal prompt to the user.
*  This method will display the message using the Dialog.Prompt singleton object.
*  Note: if this prompt is to be shown to the user during some slow JavaScript logic, we
*  advise calling this method, then using Class.delayCall or Timer.setTimeout
*  to kick off the slow logic in a separate thread. This ensures that the prompt is showing
*  before the lengthy execution begins.
*  Use "&#36;{loadingImage}" to include Canvas.loadingImageSrc.
* 
* 
* 
* @param {string} message to display
* @param {Partial<Dialog>=} additional properties for the Dialog, applied before                    the Dialog is shown
*/
function showPrompt(message:string, properties?:Partial<Dialog>):void;
/**
* 
* 	Clear the modal prompt being shown to the user.
* 
* 
*/
function clearPrompt():void;
/**
* 
* 	Method available on the isc object to show a temporary modal prompt to the user.
*  This method will display the message using the Dialog.Prompt singleton object, then hide it
*  using a fade animation effect.
*  Note: if this prompt is to be shown to the user during some slow JavaScript logic, we
*  advise calling this method, then using Class.delayCall or Timer.setTimeout
*  to kick off the slow logic in a separate thread. This ensures that the prompt is showing
*  before the lengthy execution begins.
*  
*  The prompt may be cleared before the duration has elapsed via a call to isc.clearPrompt
*  and any callback specified will still be fired even if the prompt is dismissed early.
* 
* 
* @param {string} message to display
* @param {number} how long the message should appear for in milliseconds before  fading from view.
* @param {Callback=} When the prompt is hidden, callback will be fired.
* @param {Partial<Dialog>=} additional properties for the Dialog, applied before                    the Dialog is shown
*/
function showFadingPrompt(message:string, duration:number, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" button. See Dialog.warnIcon.
*  
*  The callback will receive boolean true for an OK button click, or null if the Dialog is
*  dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Optional Callback to fire when the user                   dismisses the dialog. This has the single parameter                    'value', indicating the value returned by the Warn                    dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to                    an array of buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function warn(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" button. Intended for notifications which
*  are not really warnings (default icon is less severe). See Dialog.sayIcon.
*  
*  The callback will receive boolean true for an OK button click, or null if the Dialog is
*  dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Optional Callback to fire when the user                   dismisses the dialog. This has the single parameter                    'value', indicating the value returned by the Warn                    dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function say(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "Yes" and "No" buttons. See Dialog.askIcon.
*  
*  The callback will receive boolean true for a Yes button click, boolean false for a No button
*  click, or null if the Dialog is dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    value returned by the Warn dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array                    of buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function ask(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" and "Cancel" buttons. See Dialog.confirmIcon.
*  
*  The callback will receive boolean true for an OK button click, or null for a Cancel click or
*  if the Dialog is dismissed via the close button.
*  
*  Note: this does not override the native window.confirm() method.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    value returned by the Warn dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function confirm(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a text entry box, asking the user to enter a value.
*  
*  As with other convenience methods that show Dialogs, such as isc.warn,
*  the dialog is shown and the function immediately returns. When the user responds, the
*  provided callback is called.
*  
*  If the user clicks OK, the value typed in is passed to the callback (including the empty
*  string ("") if nothing was entered. If the user clicks cancel, the value passed to the
*  callback is null.
*  
*  A default value for the text field can be passed via properties.defaultValue.
*  
*  Keyboard focus is automatically placed in the text entry field, and hitting the enter key is
*  the equivalent of pressing OK.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    user entry, or null if cancel was pressed or the window                    closed
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function askForValue(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  If a dialog triggered via isc.say, isc.ask,
*  isc.warn, isc.confirm or
*  isc.askForValue is currently visible, it will be dismissed. The
*  callback passed to the relevant method will never fire.
*  
*  Note this is a rarely used API with very few valid use cases. As an example, perhaps some kind of
*  periodic (non-user triggered) event would cause an entire area of the UI to be removed (such as a tab)
*  and the system wants to ensure that no modal dialogs are currently showing from that part of the UI.
*  In this case, while dismissCurrentDialog could be used to ensure the part of the UI being
*  removed didn't leave behind a modal dialog.
*  
*  To clear a modal prompt shown by isc.showPrompt, use isc.clearPrompt instead.
* 
* 
*/
function dismissCurrentDialog():void;
/**
*  Handle a complete login interaction with a typical login dialog asking for username and
*  password credentials using the LoginDialog class.
*  
*  As with other convenience methods that show Dialogs, such as isc.warn,
*  the dialog is shown and the function immediately returns. When the user responds, the
*  provided callback function is called.
*  
*  If the user clicks the "Log in" button, the credentials entered by the user are passed to
*  the provided "loginFunc" as an Object with properties "username" and "password" (NOTE: both
*  property names are all lowercase), as the variable "credentials". For example:
*  { username: "barney", password: "rUbbL3" }
*  
*  The "loginFunc" should then attempt to log in by whatever means is necessary. The second
*  parameter to the loginFunc, "dialogCallback", is a function, which must be called whether
*  login succeeds or fails with a true/false value indicating whether login succeeded.
*  
*  If the login dialog is dismissable (settable as properties.dismissable, default false) and
*  the user dismisses it, the loginFunc will be fired with null for the credentials.
*  
*  The following code shows typical usage. This code assumes you have created a global
*  function sendCredentials() that send credentials to some authentication system and fires a
*  callback function with the result:
*  
*  isc.showLoginDialog(function (credentials, dialogCallback) {
*    if (credentials == null) return; // dismissed
* 
*    // send credentials
*    sendCredentials(credentials, function (loginSucceeded) {
*      // report success or failure
*      dialogCallback(loginSucceeded);
*    })
*  })
*  
*  The login dialog has several built-in behaviors:
*  
*  
* - keyboard focus is automatically placed in the username field
*  
* - hitting enter in the username field proceeds to the password field
*  
* - hitting enter in the password field submits (fires the provided callback)
*  
*  In addition to normal properties supported by Dialog/Window, the following special
*  properties can be passed:
*  
*  
* - username: initial value for the username field
*  
* - password: initial value for the password field
*  
* - usernameTitle: title for the username field
*  
* - passwordTitle: title for the password field
*  
* - errorMessage: default error message on login failure
*  
* - loginButtonTitle: title for the login button
*  
* - dismissable: whether the dialog can be dismissed, default false
*  
* - errorStyle: CSS style for the error message, if shown
*  
*  See below for links to the default values for these properties.
* 
* 
* @param {Callback} Function to call to attempt login. Receives parameters                   "credentials" and "dialogCallback", described above
* @param {Partial<LoginDialog>=} additional properties for the Dialog
*/
function showLoginDialog(loginFunc:Callback, properties?:Partial<LoginDialog>):void;
/**
*  Same as Log.logWarn.
* 
* 
* @param {string} message to log
* @param {string=} category to log in, defaults to "Log"
*/
function logWarn(message:string, category?:string):void;
/**
*  Same as Log.echo.
* 
* 
* @param {any} object to echo
*/
function echo(value:any):string;
/**
*  Same as Log.echoAll.
* 
* 
* @param {any} object to echo
*/
function echoAll(value:any):string;
/**
*  Same as Log.echoLeaf.
* 
* 
* @param {any} object to echo
*/
function echoLeaf(value:any):string;
/**
*  Logs the echoed object (using isc.echo) as a warning, prefixed with an
*  optional message.
* 
* 
* @param {any} object to echo
* @param {string} message to log
*/
function logEcho(value:any, message:string):void;
/**
*  Logs the echoed object (using isc.echoAll) as a warning, prefixed with an
*  optional message.
* 
* 
* @param {any} object to echo
* @param {string} message to log
*/
function logEchoAll(value:any, message:string):void;
/**
* 
*  Add all properties and methods from any number of objects to a destination object, 
*  overwriting properties in the destination object.
*  
*  Common uses of addProperties include creating a shallow copy of an object:
* 
*    isc.addProperties({}, someObject);
* 
*  Combining settings in order of precedence:
* 
*    isc.addProperties({}, defaults, overrides, skinOverrides);
* 
*  
*  
*  NOTES:
*  
* - Do not use addProperties to add defaults to an ISC class.
*  Use Class.addProperties, as in: 
*  MyClassName.addProperties().
*  
* - You may find it more convenient to use the instance method Class.addProperties,
*  as in: myClassInstance.addProperties(), but there's no functional
*  difference from using this method.
*  
* 
* 
* @param {object} object to add properties to
* @param {object=} objects to obtain properties from. Properties of all      arguments other than destination are applied in turn.
*/
function addProperties(destination:object, ...args:any[]):object;
/**
* 
* 	Return all keys (property names) of a given object
* 
* 
* @param {object} object to get properties from
*/
function getKeys(object:object):Array<any>;
/**
*  Return the first property name in a given Object, according to for..in iteration order.
* 
* 
* @param {object} Object to get properties from
*/
function firstKey(object:object):string;
/**
* 
* 	Return all values of a given object
* 
* 
* @param {object} object to get properties from
*/
function getValues(object:object):Array<any>;
/**
*  Given a simple javascript object, return that object sorted by keys, such that when iterating
*  through the properties of the object, they will show up in sorted order.
*  Usage example - may be used to sort a FormItem.valueMap defined
*  as an object.
* 
* @param {object} Object to sort
* @param {Function=} Comparator function to use when sorting the objects keys
*/
function sortObject(object:object, comparator?:Function):object;
/**
*  Given a simple javascript object, return that object sorted by properties, such that when 
*  iterating through the properties of the object, values will show up in sorted order.
*  Usage example - may be used to sort a FormItem.valueMap defined
*  as an object by display value.
* 
* @param {object} Object to sort
* @param {Function=} Comparator function to use when sorting the object properties
*/
function sortObjectByProperties(object:object, comparator?:Function):object;
/**
* 
*  Copy any properties that do not already have a value in destination. Null and zero values
*  are not overwritten, but 'undef' values will be.
* 
* 
* @param {object} Object to which properties will be added.
* @param {object} Object from which properties will be added.
*/
function addDefaults(destination:object, source:object):object;
/**
* 
* 	Is some property specified on the object passed in? This will return true if 
*  object[propertyName] has ever been set to any value, and not deleted.
*  May return true even if object[propertyName] === undefined if the property 
*  is present on the object and has been explicitly set to undefined.
* 
* 
* @param {object} Object to test
* @param {string} Which property is being tested for?
*/
function propertyDefined(object:object, propertyName:string):boolean;
/**
*  Shortcut for isc.ClassFactory.defineClass().
* 
* @param {string} Name for the new class.
* @param {Class=} Optional SuperClass Class object or name
*/
function defineClass(className:string, superClass?:Class):Class;
/**
*  Shortcut for isc.ClassFactory.overwriteClass().
* 
*/
function overwriteClass():void;
/**
*  Evaluate a string of script and return the result. Falls through to
*  Class.evaluate
* 
* 
* @param {string} Expression to evaluate
*/
function eval(expression:string):any;
/**
*  Shorthand for new Date().getTime();, this returns a timeStamp - a large number
*  which is incremented by 1 every millisecond. Can be used to generate unique identifiers,
*  or perform timing tasks.
* 
* 
*/
function timeStamp():Integer;
/**
* 
* 	Method available on the isc object to open the Developer Console.
* 
* 
*/
function showConsole():void;
/**
*  Given a key and an object of key:value pairs, return the value that corresponds to
*  that key.
*  
*  If the key is not found, defaultValue will be returned if provided, otherwise the
*  key will be returned.
* 
* 
* @param {string | number} key to look for
* @param {object} object of key:value pairs
* @param {any=} default value to return if key not found
*/
function getValueForKey(key:string | number, valueMap:object, defaultValue?:any):any;
/**
*  Given a value and an object of key:value pairs, return a key that corresponds
*  to that value.
*  
*  If the key is not found, defaultKey will be returned if provided, otherwise the
*  value will be returned.
* 
* 
* @param {string | number} value to look for
* @param {object} object of key:value pairs
* @param {any=} default key to return if value not found
*/
function getKeyForValue(value:string | number, valueMap:object, defaultKey?:any):any;
/**
*  Given a key:value map, return a new map as value:key.
*  
*  If the same value appears more than once, the key will correspond to the last instance of that
*  value.
* 
* 
* @param {object} object of key:value pairs
*/
function makeReverseMap(valueMap:object):object;
/**
*  Method to set up a static Class.observe on some target object. 
*  This allows developers to perform some action every time a particular method is invoked 
*  on a target object.
*  
*  This method returns a unique observation identifier string. To cancel the observation, 
*  pass this identifier to Page.ignore.
*  
*  If multiple observations are set up for the same target object and method, the 
*  notification actions will be fired in the order in which they were registered.
*  
*  This method is available as isc.Page.observe() or just isc.observe()
*  
*  Note [potential memory leak]: If the target object is a simple JavaScript object
*  (not an instance of a SmartClient class), developers should always call 
*  Page.ignore to stop observing the object if an observation is no longer necessary.
*  
*  This ensures that if the object is subsequently allowed to go out of scope by application 
*  code, the Page level observation system will not retain a reference to it (so the browser
*  can reclaim the allocated memory).
*  While cleaning up observations that are no longer required is always good practice, this
*  memory leak concern is not an issue if the target object is an instance of a SmartClient
*  class. In that case the observation is automatically released when the 
*  target is Class.destroy.
* 
* 
* @param {object} Object to observe. This may be any JavaScript object with the specified  target method, including native arrays, and instances of SmartClient classes such as   Canvas.
* @param {string} Name of the method to observe. Every time this method is invoked  on the target object the specified action will fire   (after the default implementation completes).
* @param {Function | string} Action to take when the observed method is invoked.  If action is a string to execute, certain keywords are available for context:    
- observed is the target object being observed (on which the method was invoked).  
- returnVal is the return value from the observed method (if there is one)  
- For functions defined with explicit parameters, these will also be available as keywords within    the action string    If action is a function, the arguments for the original method will also be   passed to this action function as arguments. If developers need to access the target   object being observed from the action function they may use native javascript techniques  such as   javascript closure  to do so. The return value from the observed method is not available to the action function.
*/
function observe(object:object, methodName:string, action:Function | string):string;
/**
*  Clear an observation set up by Page.observe.
*  
*  This method is available as isc.Page.ignore() or just isc.ignore()
* 
* 
* @param {string} ID returned from Page.observe call we want to clear
*/
function ignore(observerID:string):void;
/**
*  Set the global default setting for Canvas.autoDraw.
*  
*  After calling isc.setAutoDraw(), any newly created Canvas which is not given an
*  explicit setting for Canvas.autoDraw will follow the new default setting.
*  
*  autoDraw:false is the recommended default setting for most applications since it ensures
*  that extra draws will not occur when developers inadvertently omit the autoDraw:false
*  setting on child components.
* 
* 
* @param {boolean=} whether autoDraw should be enabled or disabled. Defaults to true.
*/
function setAutoDraw(enable?:boolean):void;
/**
*  Create a deep clone of an object that can be edited without affecting the original
*  
*  All mutable types, including Objects, Arrays and Dates, are copied. All immutable types
*  (Number, String, etc) are just preserved by reference.
*  
*  Only JavaScript built-in types may be cloned. SmartClient UI widgets do not support
*  cloning but must be created explicitly via Class.create.
*  Note that you also can't duplicate a live canvas by passing into create() as an
*  argument. If you need to create multiple components with similar configuration, some common
*  patterns inclulde:
*  
* - Create a new SmartClient class with the desired default configuration, and
*    create instances of this class as needed.
*  
* - For components created by some specific instance, the AutoChild system may be used.
*    Developers can specify a standard configuration in 
*    autoChildNameDefaults and 
*    autoChildNameProperties, and use Class.createAutoChild
*    to create a number of standard auto child components.
*  
* - A less formal approach might be to have a simple getter type method which
*    created and returned a new component each time it was called, passing in a standard
*    configuration block.
*  
*  Does not handle looping references (will infinite loop).
* 
* 
* @param {object} object to clone
*/
function clone(object:object):object;
/**
*  Creates a shallow copy of the passed-in Object or Array of Objects, that is, copies all
*  properties of an Object to a new Object, so that the clone now has exactly the same property
*  values as the original Object.
*  
*  If shallowClone() is passed an immutable type such as String and Number, it is returned
*  unchanged. Dates are copied via new Date(originalDate.getTime()).
*  
*  Note that if an Array is passed, all members of the Array will be cloned. For a copy of an
*  Array that contains exactly the same members (not copies), use Array.duplicate().
*  
*  Only an Array directly passed to shallowClone() is copied. Arrays contained
*  within Arrays will not be copied.
* 
* 
* @param {object | Array<any> | object} object to be cloned
*/
function shallowClone(object:object | Array<any> | object):object | Array<Partial<object>>;
/**
*  Returns the (offset) left-coordinate of an icon within its containing widget.
* 
* @param {object} icon definition
*/
function getIconLeft(icon:object):number;
/**
*  Returns the (offset) top-coordinate of an icon within its containing widget.
* 
* @param {object} icon definition
*/
function getIconTop(icon:object):number;
/**
*  Returns the size / position of an icon with respect to the widget rendering out the
*  form item as an array of coordinates.
* 
* @param {object} icon definition for the icon you want to determine the                position of (defaults to first icon in this.icons).
*/
function getIconRect(icon:object):Array<any>;
/**
*  Returns the size / position of an icon on the page as an array of coordinates.
* 
* @param {object} icon definition for the icon you want to determine the                position of (defaults to first icon in this.icons).
*/
function getIconPageRect(icon:object):Array<any>;
/**
* A whole number, for example, 5. Decimal numbers, for example 5.5, are not allowed. May not
*  be null.
*/
type int = Integer;




/**
* A subclass of EventCanvas, used to render 
*  Calendar.indicators at important points in 
*  CalendarView.
*  
*  An IndicatorCanvas is a non-interactive, semi-transparent canvas that highlights a portion of a 
*  calendar view, by rendering across all lanes and behind normal Calendar.data.
*  
*  By default, the canvas shows no label but does show a hover.
*  
*  Default styling is specified at the Calendar.indicatorStyleName 
*  and can be overridden for CalendarEvent.styleName.
*/
class IndicatorCanvas extends EventCanvas {
/**
* Creates a new IndicatorCanvas
* 
* @param typeCheckedProperties {Partial<IndicatorCanvas>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {IndicatorCanvas} The new instance
*/
static create(typeCheckedProperties? : Partial<IndicatorCanvas>, uncheckedProperties?: Object): IndicatorCanvas;











}

/**
* Start editing a new record.
*/
class FormEditNewRecordTask extends ComponentTask {
/**
* Creates a new FormEditNewRecordTask
* 
* @param typeCheckedProperties {Partial<FormEditNewRecordTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FormEditNewRecordTask} The new instance
*/
static create(typeCheckedProperties? : Partial<FormEditNewRecordTask>, uncheckedProperties?: Object): FormEditNewRecordTask;


/**
* Initial values for new edit record.
*  
*  Data values prefixed with "$" will be treated as a taskInputExpression
*  excluding "$input" and "$inputRecord" references.
* @type {Record}
* @default null
*/
initialValues?: Record;










}

/**
* Maintains a representation of selection over a 2-dimensional grid of objects.
*  Automatically created to manage cell-selection on CubeGrid widgets.
*/
class CellSelection extends Class {
/**
* Creates a new CellSelection
* 
* @param typeCheckedProperties {Partial<CellSelection>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CellSelection} The new instance
*/
static create(typeCheckedProperties? : Partial<CellSelection>, uncheckedProperties?: Object): CellSelection;








/**
*  Return true if a particular item is selected
* 
* @param {Integer} row index of the cell to check
* @param {Integer} column index of the cell to check
*/
cellIsSelected(rowNum:Integer, colNum:Integer): boolean;

/**
*  Is anything in the list selected?
* 
*/
anySelected(): boolean;

/**
*  Returns an array of the currently selected cells. Each cell is returned as a 2 element
*  array in the form [rowNum, colNum].
* 
*/
getSelectedCells(): Array<any>;

/**
*  Returns the first record that has any cells selected.
* 
*/
getSelectedRecord(): ListGridRecord;

/**
*  Observable handler fired
*  whenever the cell selection is modified
* 
* 
*/
selectionChanged(): void;

/**
*  Select a particular cell
* 
* 
* @param {Integer} row index of cell to select
* @param {Integer} column index of cell to select
*/
selectCell(rowNum:Integer, colNum:Integer): boolean;

/**
*  Deselect a particular cell
* 
* 
* @param {Integer} row index of the cell to select
* @param {Integer} column index of the cell to select
*/
deselectCell(rowNum:Integer, colNum:Integer): boolean;

/**
*   select an array of cells
* 
* @param {Array<Partial<any>>} Array of cells to select. Each cell can be specified                    as a 2 element array [rowNum, colNum]
*/
selectCellList(list:Array<Partial<any>>): boolean;

/**
*   deselect an array of cells
* 
* 
* @param {Array<Partial<any>>} Array of cells to deselect. Each cell can be specified                    as a 2 element array [rowNum, colNum]
*/
deselectCellList(list:Array<Partial<any>>): boolean;

/**
*   select a single cell and deselect everything else
* 
* @param {Integer} row index of cell to select
* @param {Integer} column index of cell to select
*/
selectSingleCell(rowNum:Integer, colNum:Integer): boolean;




}

/**
* EditProxy that handles FacetChart when editMode is enabled.
*/
class FacetChartEditProxy extends DrawPaneEditProxy {
/**
* Creates a new FacetChartEditProxy
* 
* @param typeCheckedProperties {Partial<FacetChartEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FacetChartEditProxy} The new instance
*/
static create(typeCheckedProperties? : Partial<FacetChartEditProxy>, uncheckedProperties?: Object): FacetChartEditProxy;


/**
* If InlineEditEvent for this chart edits the
*  FacetChart.data, character that should be used as a separator between
*  values, or between pairs of label vs values.
*  
*  The FacetChartEditProxy.dataEscapeChar can be used to enter the separator
*  char as part of a field name or value.
* @type {string}
* @default ","
*/
dataSeparatorChar?: string;

/**
* If InlineEditEvent for this chart edits the
*  FacetChart.data, character that should be used as a separator for
*  entering label vs value entries.
*  
*  With the default of ":", the following input defines four values with titles:
*  
*    North:10, South:20, East:30, West:40
*  
*  
*  The FacetChartEditProxy.dataEscapeChar can be used to enter literal colon characters.
* @type {string}
* @default ":"
*/
dataDisplaySeparatorChar?: string;

/**
* If InlineEditEvent for this chart edits the
*  FacetChart.data, character that can be used to enter literal separator
*  chars (such as the FacetChartEditProxy.dataSeparatorChar).
*  
*  Repeat this character twice to enter it literally. For example, with the default
*  of "\", inputting "\\" would result in a literal backslash in the value.
* @type {string}
* @default "\"
*/
dataEscapeChar?: string;







/**
*  Returns the text based on the current component state to be edited inline.
*  Called by the EditProxy.inlineEditForm to obtain the starting edit value.
* 
* 
*/
getInlineEditText(): void;

/**
*  Save the new value into the component's state. Called by the
*  EditProxy.inlineEditForm to commit the change.
*  
*  Updates the component's facets and data.
*  
*  Lines starting with "--" or "==" are considered titles. A single title
*  is used as the chart title. Titles are matched to the next series of
*  data. If titles are provided for each series, a legend will be shown.
*  
*  Series data can be entered as a list of values separated by commas
*  (see FacetChartEditProxy.dataSeparatorChar) or as a valueMap-style
*  list of label:value pairs. The first data series defines the
*  number of chart values and the titles, if provided.
* 
* 
* @param {string} the new component data
*/
setInlineEditText(newValue:string): void;




}

/**
* Dialogs are a specialized version of Window used for small windows that contain just
*  a text message or a text mesage with some standard buttons.
*  
*  Many typical modal dialogs such as alerts and confirmations are built into the system with
*  convenience APIs - see isc.say, isc.warn 
*  and isc.askForValue
*  .
*  
*  Dialogs can be modal or non-modal according to Window.isModal.
*  
*  NOTE: If you are building a dialog that will involve more than just buttons and a message,
*  consider starting from the Window class instead, where arbitrary components can be
*  added to the body area via Window.addItem.
*  
*  This is an example of creating a custom dialog:
*  
*  
*  isc.Dialog.create({
*    message : "Please choose whether to proceed",
*    icon:"[SKIN]ask.png",
*    buttons : [
*      isc.Button.create({ title:"OK" }),
*      isc.Button.create({ title:"Cancel" })
*    ],
*    buttonClick : function (button, index) {
*      this.hide();
*    }
*  });
*  
*  
*  
*/
class Dialog extends Window {
/**
* Creates a new Dialog
* 
* @param typeCheckedProperties {Partial<Dialog>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Dialog} The new instance
*/
static create(typeCheckedProperties? : Partial<Dialog>, uncheckedProperties?: Object): Dialog;
/**
* Title for the "OK" button.
* @type {HTMLString}
* @default "OK"
*/
static OK_BUTTON_TITLE?: HTMLString;

/**
* Title for the "Apply" button.
* @type {HTMLString}
* @default "Apply"
*/
static APPLY_BUTTON_TITLE?: HTMLString;

/**
* Title for the "Yes" button.
* @type {HTMLString}
* @default "Yes"
*/
static YES_BUTTON_TITLE?: HTMLString;

/**
* Title for the "No" button.
* @type {HTMLString}
* @default "No"
*/
static NO_BUTTON_TITLE?: HTMLString;

/**
* Title for the "Cancel" button.
* @type {HTMLString}
* @default "Cancel"
*/
static CANCEL_BUTTON_TITLE?: HTMLString;

/**
* Title for the "Done" button.
* @type {HTMLString}
* @default "Done"
*/
static DONE_BUTTON_TITLE?: HTMLString;

/**
* Default title for the dialog displayed in response to the isc.confirm method.
*  Note that a custom title can be specified as the title attribute of the
*  properties parameter passed to that method.
* @type {HTMLString}
* @default "Confirm"
*/
static CONFIRM_TITLE?: HTMLString;

/**
* Default title for the dialog displayed in response to the isc.say method.
*  Note that a custom title can be specified as the title attribute of the
*  properties parameter passed to that method.
* @type {HTMLString}
* @default "Note"
*/
static SAY_TITLE?: HTMLString;

/**
* Default title for the dialog displayed in response to the isc.warn method.
*  Note that a custom title can be specified as the title attribute of the
*  properties parameter passed to that method.
* @type {HTMLString}
* @default "Warning"
*/
static WARN_TITLE?: HTMLString;

/**
* Default title for the dialog displayed in response to the isc.ask method.
*  Note that a custom title can be specified as the title attribute of the
*  properties parameter passed to that method.
* @type {HTMLString}
* @default "Question"
*/
static ASK_TITLE?: HTMLString;

/**
* Default title for the dialog displayed by isc.askForValue.
*  A custom title can alternatively be specified as the title attribute of the
*  properties parameter passed to that method.
* @type {HTMLString}
* @default "Please enter a value"
*/
static ASK_FOR_VALUE_TITLE?: HTMLString;

/**
* The "Prompt" object on the dialog class is a singleton Dialog instance.
*  The Prompt is used to show text to the user in a modal fashion - it will expand to show
*  all the text that you put into it.
*  By default this Dialog has no end-user controls and is expected to be programmatically
*  dismissed.
*  Common use-case: During server-interactions, the Prompt will be used to display a suitable
*  wait message, and suppress user input.
* 
*  Notes:
*  Because this is a singleton object, properties set on the Prompt directly will persist each
*  time it is shown.
*  Developers should use the showPrompt() and clearPrompt() methods
*  to show and hide the prompt rather than manipulating the prompt directly.
* @type {Partial<Dialog>}
* @default "dialog instance properties"
*/
static Prompt?: Partial<Dialog>;

/**
* A singleton Dialog instance that will show text to the user and provide buttons for their
*  response. The Dialog will expand to show all the text that you put into it. This dialog
*  is shown in response to calls to isc.say,
*  isc.warn, isc.ask and
*  isc.confirm.
*  
*  This can be used in cases where a developer would alternatively make use of the native
*  JavaScript alert() and confirm() methods. The main differences
*  between those methods and using the Warn object are:
*  - The Warn object can be customized by modifying which buttons are visible, the style
*   applied to it, etc.
*  - The isc.ask() and isc.warn() methods are asynchronous - rather
*   than returning a value indicating the user's response, a callback method will be fired
*   when the user interacts with the dialog.
*  
*  Notes:
*  Because this is a singleton object, properties set on the Warn object directly will persist
*  each time it is shown.
*  Developers should use the warn() or ask() methods to show and
*  hide this object rather than manipulating the Dialog directly.
* @type {Partial<Dialog>}
* @default "dialog instance properties"
*/
static Warn?: Partial<Dialog>;

/**
* A singleton Dialog instance that will be shown in response to a isc.askForValue call.
* 
*  Notes:
*  Because this is a singleton object, properties set on the Ask object directly will persist
*  each time it is shown.
*  Developers should use the askForValue() method to show this object rather than
*  manipulating the Dialog directly.
* @type {Partial<Dialog>}
* @default "dialog instance properties"
*/
static Ask?: Partial<Dialog>;



/**
* @type {Integer}
* @default "360"
*/
defaultWidth?: Integer;

/**
* Style of the Dialog background
* @type {CSSStyleName}
* @default "dialogBackground"
*/
styleName?: CSSStyleName;

/**
* Message to show in this dialog.
*  
*  If a message is set the primary purpose of the dialog will be assumed to be to show a message with
*  buttons - auto-sizing to the message text will be enabled, and, if Dialog.icon has also
*  been set, the Dialog.messageLabel and Dialog.messageIcon
*  AutoChildren will be created and placed together in the Dialog.messageStack
*  AutoChild, with the toolbar underneath as usual. If any of these behaviors are inconvenient or
*  you want more precise control over a message and some custom widgets, start from the superclass
*  Window instead, and add controls via Window.addItem.
*  
*  The message string may contain "${loadingImage}", if so, the standard loading spinner will appear at
*  that location in the text (see Canvas.loadingImageSrc).
*  
*  The message will be styled with the Dialog.messageStyle.
* @type {HTMLString}
* @default null
*/
message?: HTMLString;

/**
* Style to apply to the message text shown in the center of the dialog
* @type {CSSStyleName}
* @default "normal"
*/
messageStyle?: CSSStyleName;

/**
* AutoChild that shows Dialog.message.
* @type {Label}
* @default null
*/
messageLabel?: Label;

/**
* AutoChild that shows Dialog.icon.
* @type {Img}
* @default null
*/
messageIcon?: Img;

/**
* AutoChild that combines Dialog.message and Dialog.icon.
* @type {Layout}
* @default null
*/
messageStack?: Layout;

/**
* Icon to show in this dialog - see Dialog.message.
* @type {SCImgURL}
* @default null
*/
icon?: SCImgURL;

/**
* Size of the icon to show in this dialog.
* @type {Integer}
* @default "32"
*/
iconSize?: Integer;

/**
* Whether to show a toolbar of buttons at the bottom of the Dialog.
*  Default of null will cause the value to be resolved automatically to true or
*  false when the Dialog is first drawn according as Dialog.toolbarButtons
*  contains buttons or not.
* @type {boolean}
* @default null
*/
showToolbar?: boolean;

/**
* If a toolbar is showing, automatically place keyboard focus in the first button.
*  
*  An alternative button can be specified by Dialog.autoFocusButton.
* @type {boolean}
* @default true
*/
autoFocus?: boolean;

/**
* If a toolbar is showing and Dialog.autoFocus is enabled, which button should
*  receive initial focus.
* @type {Canvas | string | Integer}
* @default true
*/
autoFocusButton?: Canvas | string | Integer;

/**
* AutoChild of type Toolbar used to create the Dialog.toolbarButtons.
* @type {Toolbar}
* @default null
*/
toolbar?: Toolbar;

/**
* Array of Buttons to show in the Dialog.showToolbar, if shown.
*  
*  The set of buttons to use is typically set by calling one of the shortcuts such as
*  isc.say or isc.confirm
*  .
*  A custom set of
*  buttons can be passed to these shortcuts methods via the "properties" argument, or to a
*  directly created Dialog.
*  
*  In both cases, a mixture of DialogButtons, custom buttons,
*  and other components (such as a LayoutSpacer) can be passed. 
*  
*  Built-in buttons can be referred to as isc.Dialog.OK, for example:
*  
*  isc.Dialog.create({
*   buttons:[
*     isc.Dialog.OK,
*     isc.Dialog.CANCEL,
*     isc.LayoutSpacer.create({width:50}),
*     { title:"Not now", click:"doSomething()" }
*   ]
*  })
*  
*  Built-in buttons will call standard methods on the Dialog itself, such as
*  Dialog.cancelClick, as explained in the
*  DialogButtons.
*  
*  
* @type {Array<Partial<Button>> | Array<Partial<Partial<Button>>>}
* @default null
*/
buttons?: Array<Partial<Button>> | Array<Partial<Partial<Button>>>;

/**
* This is a synonym for Dialog.buttons
* @type {Array<Partial<Button>> | Array<Partial<Partial<Button>>>}
* @default null
*/
toolbarButtons?: Array<Partial<Button>> | Array<Partial<Partial<Button>>>;

/**
* Icon to show in the isc.ask
*  dialog.
* @type {SCImgURL}
* @default "[SKIN]ask.png"
*/
askIcon?: SCImgURL;

/**
* Icon to show in the isc.say
*  dialog.
* @type {SCImgURL}
* @default "[SKIN]say.png"
*/
sayIcon?: SCImgURL;

/**
* Icon to show in the isc.warn
*  dialog.
* @type {SCImgURL}
* @default "[SKIN]warn.png"
*/
warnIcon?: SCImgURL;

/**
* Icon to show in the isc.confirm
*  dialog.
* @type {SCImgURL}
* @default "[SKIN]confirm.png"
*/
confirmIcon?: SCImgURL;







/**
*  Set the buttons for the toolbar displayed in this dialog.
*  Synonym for Dialog.setToolbarButtons
* 
* @param {Array<Partial<Button>> | Array<Partial<Partial<Button>>>} [null] buttons for the toolbar
*/
setButtons(newButtons:Array<Partial<Button>> | Array<Partial<Partial<Button>>>): void;

/**
*  Set the Dialog.toolbarButtons for this dialog.
*  Synonym for Dialog.setButtons.
* 
* @param {Array<Partial<Button>> | Array<Partial<Partial<Button>>>} [null] buttons for the toolbar
*/
setToolbarButtons(newButtons:Array<Partial<Button>> | Array<Partial<Partial<Button>>>): void;

/**
*  Method to save this Dialog's data. Called from okClick(),
*  applyClick().
*  No default implementation - override to perform some action if required.
* 
* 
*/
saveData(): void;

/**
*  Handles a click on the close button of this window. The default implementation
*  calls Window.close and returns false to prevent bubbling of the click event.
*  
*  Override this method if you want
*  other actions to be taken.
*  
*  Custom implementations may call close() to trigger the default behavior.
* 
*/
closeClick(): boolean;

/**
*  Handle a click on the 'cancel' button of this Dialog.
*  Default implementation is to return null and hide the Dialog.
*  Override to do something else.
* 
*/
cancelClick(): void;

/**
*  Handle a click on the 'ok' button of this Dialog.
*  Default implementation is to call saveData(), hide the Dialog, then return
*  true.
*  Override to do something else.
* 
*/
okClick(): void;

/**
*  Handle a click on the 'apply' button of this Dialog.
*  Default implementation is to call saveData(), but NOT close the Dialog.
* 
*/
applyClick(): void;

/**
*  Handle a click on the 'yes' button of this Dialog.
*  Default implementation is to return true.
*  Override to do something else
* 
*/
yesClick(): void;

/**
*  Handle a click on the 'no' button of this Dialog.
*  Default implementation is to return false.
*  Override to do something else.
* 
*/
noClick(): void;

/**
*  Handle a click on the 'done' button of this Dialog.
*  Default implementation is to hide the dialog then return true.
*  Override to do something else.
* 
*/
doneClick(): void;

/**
*  Fires when any button in this Dialog's toolbar is clicked. Default implementation does nothing.
* 
* 
* @param {StatefulCanvas} button that was clicked
* @param {Integer} index of the button that was clicked
*/
buttonClick(button:StatefulCanvas, index:Integer): void;




}

/**
* Possible values for Canvas.backgroundRepeat.
*/
type BackgroundRepeat = 
/**
* Tile the background image horizontally and vertically.
*/
"repeat" |

/**
* Don't tile the background image at all.
*/
"no-repeat" |

/**
* Repeat the background image horizontally but not vertically.
*/
"repeat-x" |

/**
* Repeat the background image vertically but not horizontally.
*/
"repeat-y";


/**
* Clear form values and errors.
*/
class FormClearValuesTask extends ComponentTask {
/**
* Creates a new FormClearValuesTask
* 
* @param typeCheckedProperties {Partial<FormClearValuesTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FormClearValuesTask} The new instance
*/
static create(typeCheckedProperties? : Partial<FormClearValuesTask>, uncheckedProperties?: Object): FormClearValuesTask;











}

/**
* An object returned from AutoTest.getLocatorWithIndicators that includes the
*  locator and properties that describe the quality of the locator.
*/
class QualityIndicatedLocator {


/**
* The AutoTestLocator associated with some DOM element in a SmartClient
*  application page.
* @type {AutoTestLocator}
* @default null
*/
locator?: AutoTestLocator;

/**
* True if the returned QualityIndicatedLocator.locator includes
*  a reference using an auto-generated global ID.
* @type {boolean}
* @default null
*/
containsGlobalId?: boolean;

/**
* The ID of the component within the locator segments that has an auto-generated global
*  ID.
* @type {string}
* @default null
*/
globalID?: string;

/**
* True if the returned QualityIndicatedLocator.locator includes
*  references using index positions.
* @type {boolean}
* @default null
*/
containsIndices?: boolean;

/**
* The ID of the first parent within the locator segments that has a child referenced
*  by index. Note that a child component with an explicit
*  Canvas.locatorName will be excluded since the name is a reliable
*  means to locate the component.
* @type {string}
* @default null
*/
firstParentOfIndex?: string;










}

/**
* An interface that provides the ability to create components from a PaletteNode.
*/
interface Palette {
/**
* Default EditContext that this palette should use. Palettes generally create components via
*  drag and drop, but may also support creation via double-click or other UI idioms when a
*  defaultEditContext is set.
* @type {EditContext | EditTree | EditPane}
* @default null
*/
defaultEditContext?: EditContext | EditTree | EditPane;

/**
* Whether created components should have their "ID" or "name" property automatically set
*  to a unique value based on the component's type, eg, "ListGrid0".
* @type {boolean}
* @default true
*/
generateNames?: boolean;



/**
*  Sets the default EditContext that this palette should use. Palettes generally create components via
*  drag and drop, but may also support creation via double-click or other UI idioms when a
*  defaultEditContext is set.
* 
* @param {EditContext | EditTree | EditPane} the default EditContext used by this Palette
*/
setDefaultEditContext?(defaultEditContext:EditContext | EditTree | EditPane): void;

/**
*  Given a PaletteNode, make an EditNode from it by creating a 
*  EditNode.liveObject from the PaletteNode.defaults
*  and copying presentation properties (eg PaletteNode.title
*  to the editNode.
*  
*  If editNodeProperties is specified as an object on
*  on the paletteNode, each property in this object will also be copied across to
*  the editNode.
* 
* 
* @param {PaletteNode} paletteNode to create from
*/
makeEditNode?(paletteNode:PaletteNode): EditNode;




}



/**
* Component for editing a single record.
*  RecordEditors are implemented as a subclass of ListGrid, showing no header and a single 
*  row always drawn in the editable state, allowing the user to modify the values at any time.
*  The RecordEditor.actionButton is automatically shown as a way for the user to act 
*  upon the edited values.
*  
*  The RecordEditor class exists as a helper class for ListGrids, used to provide
*  an interface for editing criteria when ListGrid.showFilterEditor 
*  is set to true.
*/
class RecordEditor extends ListGrid {
/**
* Creates a new RecordEditor
* 
* @param typeCheckedProperties {Partial<RecordEditor>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {RecordEditor} The new instance
*/
static create(typeCheckedProperties? : Partial<RecordEditor>, uncheckedProperties?: Object): RecordEditor;


/**
* GridRenderer.baseStyle for this listGrid.
*  If this property is unset, base style may be derived from ListGrid.normalBaseStyle
*  or ListGrid.tallBaseStyle as described in
*  ListGrid.getBaseStyle.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined
*  with the base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "recordEditorCell"
*/
baseStyle?: CSSStyleName;

/**
* If showing any record summary fields (IE: fields of ListGridFieldType),
*  this attribute specifies a custom base style to apply to cells in the summary field
* @type {CSSStyleName}
* @default "recordEditorCell"
*/
recordSummaryBaseStyle?: CSSStyleName;

/**
* Where do 'skin' images (those provided with the class) live?
* @type {SCImgURL}
* @default "images/RecordEditor/"
*/
skinImgDir?: SCImgURL;

/**
* Button.icon to show on the RecordEditor.actionButton if this 
*  component is being used as a ListGrid.filterEditor.
*  
*  Note that this SCImgURL will be resolved using the RecordEditor.skinImgDir defined
*  for the RecordEditor.
* @type {SCImgURL}
* @default "[SKIN]filter.png"
*/
filterImg?: SCImgURL;

/**
* Automatically created Button auto-child shown at the edge of the recordEditor.
*  For a recordEditor acting as a ListGrid.filterEditor, this button will
*  show the RecordEditor.filterImg as an Button.icon by default.
*  
*  Clicking this button will call RecordEditor.performAction on the editor.
*  
*  May be customized using the standard AutoChild pattern, by overriding
*  RecordEditor.actionButtonProperties.
* @type {Button}
* @default null
*/
actionButton?: Button;

/**
* Properties to apply to the automatically generated RecordEditor.actionButton.
*  
*  Note that for a recordEditor being used as a ListGrid.filterEditor, the
*  ListGrid.filterButtonProperties can be used to specify actionButton properties
*  directly at the grid level.
* @type {Partial<Button>}
* @default null
*/
actionButtonProperties?: Partial<Button>;

/**
* Button.baseStyle for the RecordEditor.actionButton
* @type {CSSStyleName}
* @default "normal"
*/
actionButtonStyle?: CSSStyleName;







/**
*  Fired when the user clicks the RecordEditor.actionButton for this RecordEditor.
*  May also be triggered from other user interaction with edit values (for
*  example filter-editor change - see ListGrid.filterOnKeypress).
*  
*  This is the method which initiates a filter in a listGrid
*  ListGrid.showFilterEditor. Note that for custom
*  filtering behavior, developers can use the ListGrid.filterEditorSubmit
*  notification method rather than overriding this method directly.
* 
* 
*/
performAction(): void;




}



/**
* EditProxy that handles SectionStackSection objects when editMode is enabled.
*/
class SectionStackSectionEditProxy extends LabelEditProxy {
/**
* Creates a new SectionStackSectionEditProxy
* 
* @param typeCheckedProperties {Partial<SectionStackSectionEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SectionStackSectionEditProxy} The new instance
*/
static create(typeCheckedProperties? : Partial<SectionStackSectionEditProxy>, uncheckedProperties?: Object): SectionStackSectionEditProxy;











}

/**
* EditProxy that handles DrawLabel when editMode is enabled.
*/
class DrawLabelEditProxy extends DrawItemEditProxy {
/**
* Creates a new DrawLabelEditProxy
* 
* @param typeCheckedProperties {Partial<DrawLabelEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DrawLabelEditProxy} The new instance
*/
static create(typeCheckedProperties? : Partial<DrawLabelEditProxy>, uncheckedProperties?: Object): DrawLabelEditProxy;








/**
*  Returns the text based on the current component state to be edited inline.
*  Called by the EditProxy.inlineEditForm to obtain the starting edit value.
*  
*  Returns the component's contents.
* 
* 
*/
getInlineEditText(): void;

/**
*  Save the new value into the component's state. Called by the
*  EditProxy.inlineEditForm to commit the change.
*  
*  Updates the component's contents.
* 
* 
* @param {string} the new component contents
*/
setInlineEditText(newValue:string): void;




}











/**
* The Gauge widget class implements a graphical speedometer-style gauge for displaying
*  a measurement by means of a needle on a dial. The dial is divided into sectors, each having
*  its own color and value.
*  
*  NOTE: you must load the Drawing loadingOptionalModules
*  before you can use Gauge.
*/
class Gauge extends DrawPane {
/**
* Creates a new Gauge
* 
* @param typeCheckedProperties {Partial<Gauge>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Gauge} The new instance
*/
static create(typeCheckedProperties? : Partial<Gauge>, uncheckedProperties?: Object): Gauge;


/**
* Default height of the Gauge.pivotPoint if no specific pivotPoint is specified.
*  
*  Can be specified as a numeric pixel value, or a String percentage value.
* @type {number | string}
* @default "70%"
*/
pivotPointHeight?: number | string;

/**
* AutoChild representing the shape drawn at the Gauge.pivotPoint (where all sectors of the
*  gauge meet).
* @type {DrawItem}
* @default null
*/
pivotShape?: DrawItem;

/**
* AutoChild representing the needle shape that points to the gauge's current value. Default is
*  to use a DrawTriangle.
* @type {DrawItem}
* @default null
*/
needle?: DrawItem;

/**
* MultiAutoChild representing the labels used to different data points on the gauge.
* @type {DrawLabel}
* @default null
*/
valueLabel?: DrawLabel;

/**
* MultiAutoChild representing the tick marks drawn along the circumference of the gauge.
*  Default is to use DrawLine.
* @type {DrawItem}
* @default null
*/
tickLine?: DrawItem;

/**
* MultiAutoChild representing the sectors drawn to show different segments of the gauge.
* @type {DrawSector}
* @default null
*/
sectorShape?: DrawSector;

/**
* Radius in pixels of the dial.
* @type {Float}
* @default "150"
*/
dialRadius?: Float;

/**
* Font size of sector labels. Must be at least 3.
* @type {Integer}
* @default "11"
*/
fontSize?: Integer;

/**
* Pixel width for gauge sector borders.
* @type {Integer}
* @default "1"
*/
borderWidth?: Integer;

/**
* Color for gauge sector borders.
* @type {CSSColor}
* @default "#333333"
*/
borderColor?: CSSColor;

/**
* Array of preset fill colors used by the default implementation of Gauge.getDefaultFillColor
*  to initialize the fill color of new sectors.
* 
*  The default array of colors is:
*  
*   
*    #AFFFFF
*    #008080
*    #AAAFFF
*    #FF0000
*    #FFCC99
*    #800080
*   
*  
* @type {Array<Partial<CSSColor>>}
* @default "[ "#AFFFFF", "#008080", "#AAAFFF", "#FF0000", "#FFCC99", "#800080" ]"
*/
sectorColors?: Array<Partial<CSSColor>>;

/**
* The minimum dial value.
* @type {Float}
* @default "0"
*/
minValue?: Float;

/**
* The maximum dial value.
* @type {Float}
* @default "100"
*/
maxValue?: Float;

/**
* The current value on the dial.
* @type {Float}
* @default "0"
*/
value?: Float;

/**
* The number of major tick lines.
* @type {Integer}
* @default "0"
*/
numMajorTicks?: Integer;

/**
* The number of minor tick lines.
* @type {Integer}
* @default "0"
*/
numMinorTicks?: Integer;

/**
* The label prefix.
* @type {string}
* @default ""
*/
labelPrefix?: string;

/**
* The label suffix.
* @type {string}
* @default "%"
*/
labelSuffix?: string;

/**
* Whether the sectors are drawn clockwise, and increasing the value causes the
*  needle to move clockwise.
* @type {boolean}
* @default true
*/
drawnClockwise?: boolean;

/**
* The GaugeSectors contained in this Gauge. 
* 
*  If this this property is not specified, the gauge will
*  be created with a default sector filling the gauge.
* @type {Array<Partial<GaugeSector>>}
* @default null
*/
sectors?: Array<Partial<GaugeSector>>;

/**
* The pivot point of the needle.
* @type {Point}
* @default null
*/
pivotPoint?: Point;







/**
*  Sets the minimum dial value, rescaling all sectors and the dial value.
* 
* 
* @param {Float} the new minimum dial value. Must be at least 1 less than the maximum dial value. If minValue is not at least 1 less than the maximum value, then it is set to maxValue - 1.
*/
setMinValue(minValue:Float): void;

/**
*  All DrawItems currently associated with this Gauge are destroyed and 
*  new DrawItems are created instead.
* 
* 
* @param {Float} Radius in pixels of the dial
*/
setDialRadius(dialRadius:Float): void;

/**
*  All DrawItems currently associated with this Gauge are destroyed and 
*  new DrawItems are created instead.
*  
*  The pivot point is set by default by choosing 1/2 of width and 70% of height of the Gauge. See
*  Gauge.pivotPointHeight
* 
* 
* @param {Point} The pivot point of the needle
*/
setPivotPoint(point:Point): void;

/**
*  Sets the maximum dial value, rescaling all sectors and the dial value.
* 
* 
* @param {Float} the new maximum dial value. Must be at least 1 greater than the minimum dial value. If maxValue is not at least 1 greater than the minimum value, then it is set to 1 + minValue.
*/
setMaxValue(maxValue:Float): void;

/**
*  Sets the value on the dial that the needle is displaying.
* 
* 
* @param {Float} the new dial value. Must be between Gauge.minValue and Gauge.maxValue.
*/
setValue(value:Float): void;

/**
*  Sets the number of major tick lines.
* 
*  NOTE: To divide the dial into n regions, you will need n + 1 ticks.
*  For example, if the minimum value is 0 and the maximum value is 100, then to place major
*  tick lines at 0, 10, 20, 30, ..., 90, 100, you need 11 (10 + 1) major ticks.
* 
* 
* @param {Integer} the number of major tick lines to draw. Must be either 0 or an integer greater than or equal to 2.
*/
setNumMajorTicks(numMajorTicks:Integer): void;

/**
*  Sets the number of minor tick lines.
* 
*  NOTE: To divide the dial into n regions, you will need n + 1 ticks.
*  For example, if the minimum value is 0 and the maximum value is 100, then to place minor
*  tick lines at 0, 1, 2, 3, 4, 5, ..., 99, 100, you need 101 (100 + 1) minor ticks.
* 
* 
* @param {Integer} the number of minor tick lines to draw. Must be either 0 or an integer greater than or equal to 2.
*/
setNumMinorTicks(numMinorTicks:Integer): void;

/**
*  Sets the Gauge.labelPrefix property and re-creates all sector labels.
* 
* 
* @param {string} the new label prefix.
*/
setLabelPrefix(labelPrefix:string): void;

/**
*  Sets the Gauge.labelSuffix property and re-creates all sector labels.
* 
* 
* @param {string} the new label suffix.
*/
setLabelSuffix(labelSuffix:string): void;

/**
*  Sets the Gauge.drawnClockwise property and redraws the gauge.
* 
* 
* @param {boolean} whether the sectors are drawn clockwise.
*/
setDrawnClockwise(drawnClockwise:boolean): void;

/**
*  Formats a value as a string to be used as the contents of a DrawLabel. The default
*  implementation prepends Gauge.labelPrefix and appends
*  Gauge.labelSuffix to value.
*  
*  NOTE: If a subclass overrides this, then whenever it changes the way that values are
*  formatted, it must call Gauge.reformatLabelContents.
* 
* 
* @param {Float} the value to format.
*/
formatLabelContents(value:Float): string;

/**
*  Resets the contents of all labels. This involves calling Gauge.formatLabelContents
*  to get the label contents for each corresponding value and repositioning the label.
* 
* 
*/
reformatLabelContents(): void;

/**
*  Gets the number of sectors.
* 
* 
*/
getNumSectors(): Integer;

/**
*  Gets the value of the sector at sectorIndex.
* 
* 
* @param {Integer} index of the target sector.
*/
getSectorValue(sectorIndex:Integer): Float;

/**
*  Gets the default fill color for the sector at index sectorIndex. The default
*  implementation cycles through Gauge.sectorColors
*  using modular arithmetic.
* 
* 
* @param {Integer} index of the target sector.
*/
getDefaultFillColor(sectorIndex:Integer): CSSColor;

/**
*  Gets the fill color of the sector at index sectorIndex.
* 
* 
* @param {Integer} index of the target sector.
*/
getSectorFillColor(sectorIndex:Integer): CSSColor;

/**
*  Sets the fill color of the sector at sectorIndex.
* 
* 
* @param {Integer} index of the target sector.
* @param {CSSColor} the new fill color.
*/
setSectorFillColor(sectorIndex:Integer, fillColor:CSSColor): void;

/**
*  Gets the label contents of the label for the sector at sectorIndex.
* 
* 
* @param {Integer} index of the target sector.
*/
getSectorLabelContents(sectorIndex:Integer): string;

/**
*  Sets the sectors for this gauge.
* 
* @param {Array<Partial<GaugeSector>>} the sectors to show on the gauge.
*/
setSectors(sectors:Array<Partial<GaugeSector>>): void;

/**
*  Adds a new sector.
* 
* 
* @param {GaugeSector | Double} the new GaugeSector or the new sector's value. This is formatted with Gauge.formatLabelContents to get its label.
*/
addSector(newSector:GaugeSector | Double): Integer;

/**
*  Removes the sector at sectorIndex.
*  
*  NOTE: There must always be one sector and it is not possible to remove the sole remaining
*  sector. Calling this method to attempt to remove the sole remaining sector is a no-op.
* 
* 
* @param {Integer} the index of the sector to remove.
*/
removeSector(sectorIndex:Integer): void;




}

/**
* FormItem intended for inserting blurbs of instructional HTML into DynamicForms.
*  
*  Set the defaultValue of this item to the HTML you want to embed in the form.
*/
class BlurbItem extends FormItem {
/**
* Creates a new BlurbItem
* 
* @param typeCheckedProperties {Partial<BlurbItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BlurbItem} The new instance
*/
static create(typeCheckedProperties? : Partial<BlurbItem>, uncheckedProperties?: Object): BlurbItem;


/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "TextItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* Blurb items show no title by default.
* @type {boolean}
* @default false
*/
showTitle?: boolean;

/**
* By default, texts span all remaining columns
* @type {Integer | string}
* @default "*"
*/
colSpan?: Integer | string;

/**
* Base css style for this item.
* @type {CSSStyleName}
* @default "staticTextItem"
*/
textBoxStyle?: CSSStyleName;

/**
* Should the user be able to select the text in this item?
* @type {boolean}
* @default true
*/
canSelectText?: boolean;

/**
* If true, item contents can wrap. If false, all the contents should appear on a single line.
* @type {boolean}
* @default null
*/
wrap?: boolean;

/**
* If true, text that exceeds the specified size of the form item will be clipped.
* @type {boolean}
* @default false
*/
clipValue?: boolean;










}

/**
* The different ways in which the DrawItem.titleLabel of a DrawItem
*  can be rotated with the item.
*  
*  NOTE: The effect of the
*  "withItemAlwaysUp" and "withLineAlwaysUp"
*  
*  settings is not affected by the global rotation, if any (see DrawPane.rotation).
*/
type TitleRotationMode = 
/**
* the titleLabel is never rotated with the item.
*/
"neverRotate" |

/**
* the titleLabel is rotated exactly to match the item's rotation (see DrawItem.rotation).
*/
"withItem" |

/**
* the titleLabel is rotated exactly to match the item's rotation, except that at certain rotations, the titleLabel is flipped
*  by 180&deg; so that the title text is never upside down.
*/
"withItemAlwaysUp" |

/**
* (applies only to DrawLine and DrawLinePath) the titleLabel is rotated to match the line or center segment.
*  
*  If used on a DrawItem that is not a DrawLine or DrawLinePath,
*  then the effect is the same as
*  "withItem".
*  
*/
"withLine" |

/**
* (applies only to DrawLine and DrawLinePath) the titleLabel is rotated to match the line or center segment, except that
*  at certain rotations, the titleLabel is flipped by 180&deg; so that the
*  title text is never upside down.
*  
*  If used on a DrawItem that is not a DrawLine or DrawLinePath,
*  then the effect is the same as
*  "withItemAlwaysUp".
*  
*/
"withLineAlwaysUp";


/**
* A general purpose Window class for implementing dialogs, portlets, alerts, prompts, wizards
*  and desktop-like windowing interfaces.
*  
*  Windows can contain arbitrary SmartClient components, configured via the Window.items
*  property. Windows may be Window.isModal or non-modal.
*  
*  Windows provide a series of highly configurable and skinnable AutoChild
*  including a header, various header controls, footer, and corner resizer.
*  
*  The more specialized Dialog subclass of Window has additional functionality targetted
*  at simple prompts and confirmations, such as buttons with default actions, and single-method
*  isc.warn for common application dialogs.
*/
class Window extends Layout {
/**
* Creates a new Window
* 
* @param typeCheckedProperties {Partial<Window>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Window} The new instance
*/
static create(typeCheckedProperties? : Partial<Window>, uncheckedProperties?: Object): Window;
/**
* A declared value of the enum type 
*  ContentLayoutPolicy.
* @type {Constant}
* @default "none"
*/
static NONE?: Constant;

/**
* A declared value of the enum type 
*  ContentLayoutPolicy.
* @type {Constant}
* @default "vertical"
*/
static VERTICAL?: Constant;

/**
* A declared value of the enum type 
*  ContentLayoutPolicy.
* @type {Constant}
* @default "horizontal"
*/
static HORIZONTAL?: Constant;



/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "WindowEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* If true, this Window may be moved around by the user by dragging on the Window header. 
*  Note that if the header is not showing, the Window can't be drag-repositioned regardless
*  of this setting.
* @type {boolean}
* @default true
*/
canDragReposition?: boolean;

/**
* Constrains drag-resizing and drag-repositioning of this canvas to either the rect of its
*  parent (if set to true) or an arbitrary rect based on its parent (if set to a
*  [Left,Top,Width,Height] rect array). In the latter mode you may use negative offsets
*  for left/top and a width/height greater than the visible or scroll width of the parent
*  to allow positioning beyond the confines of the parent.
*  
*  If this canvas has no parent, constrains dragging to within the browser window.
*  
*  Affects target and outline dragAppearance, not tracker.
*  
*  Note: keepInParentRect affects only user drag interactions, not programmatic moves.
*  
*  Example use cases:
*  keepInParentRect: true - confine to parent
*  keepInParentRect: [0, 0, 500, 500] - confine to top left 500x500 region within parent
*  keepInParentRect: [0, 0, 10000, 10000] - in combination with
*  overflow: "auto", confine to parent, but allow moving off the right and bottom of the
*  parent to force scrolling (and hence enlarge the scrollWidth of the parent).
* @type {boolean | Array<Partial<Integer>>}
* @default true
*/
keepInParentRect?: boolean | Array<Partial<Integer>>;

/**
* Can the window be drag-resized? If true the window may be drag resized from its edges,
*  and if showing, via the resizer icon in the footer.
* @type {boolean}
* @default false
*/
canDragResize?: boolean;

/**
* When drag resizing is enabled via Window.canDragResize, restricts resizes to only certain
*  edges or corners.
*  
*  This property on Window overrides the default defined by Canvas.resizeFrom.
* @type {Array<Partial<EdgeName>>}
* @default "["R","B","BR"]"
*/
resizeFrom?: Array<Partial<EdgeName>>;

/**
* By default Windows show a Canvas.useBackMask in Internet Explorer
*  versions predating Internet Explorer 9. This is a workaround for a native browser
*  issue whereby certain DOM elements such as IFRAMEs (whether rendered
*  within SmartClient components via features such as HTMLFlow.contentsURL or 
*  explicitly written into the HTML of the page) will not be properly occluded
*  by DOM elements which overlap them but have a higher z-index.
*  
*  A side-effect of this is that the Canvas.opacity can not be modified
*  for the entire window. Developers may disable the backmask in order to support
*  opacity in IE versions less than 9 by setting this property to false, however you
*  should be aware that in doing this there is a potential for the "burn through"
*  problem described above.
* @type {boolean}
* @default "varies"
*/
useBackMask?: boolean;

/**
* Renders the widget to be partly transparent. A widget's opacity property may
*  be set to any number between 0 (transparent) to 100 (opaque).
*  Null means don't specify opacity directly, 100 is fully opaque.
*  Note that heavy use of opacity may have a performance impact on some older
*  browsers.
*  
*  In older versions of Internet Explorer (Pre IE9 / HTML5), opacity is achieved
*  through proprietary filters. If 
*  Canvas.neverUseFilters within this application
*  developers must set Canvas.useOpacityFilter to true for specific components
*  on which opacity support is required.
*  
*  Also note that opacity is incompatible 
*  with Canvas.useBackMask, and that this property is enabled
*  by default for Window instances.
* @type {Integer}
* @default null
*/
opacity?: Integer;

/**
* If true, when shown this Window will intercept and block events to all other existing
*  components on the page.
*  
*  Use Window.showModalMask to darken all other elements on the screen when a modal dialog
*  is showing.
*  
*  Chained modal windows - that is, modal windows that launch other modal windows - are
*  allowed. You can accomplish this by simply creating a second modal Window while a modal
*  Window is showing.
*  
*  Note only top-level Windows (Windows without parents) can be modal.
* @type {boolean}
* @default false
*/
isModal?: boolean;

/**
* A ScreenSpan instance used to darken the rest of a page when a modal window is
*  active. To use, set Window.showModalMask to true, add a CSS style 
*  "modalMask" to the active skin (generally with background-color set), 
*  and adjust Window.modalMaskOpacity.
* @type {Canvas}
* @default null
*/
modalMask?: Canvas;

/**
* If true, displays a translucent mask over the rest of the page when a modal window
*  is displayed.
* @type {boolean}
* @default null
*/
showModalMask?: boolean;

/**
* Controls the opacity of the modal mask displayed behind modal windows.
* @type {number}
* @default "50"
*/
modalMaskOpacity?: number;

/**
* Specifies the CSS style for the modal mask.
* @type {string}
* @default "modalMask"
*/
modalMaskStyle?: string;

/**
* If true, this Window widget will automatically be centered on the page when shown.
*    If false, it will show up in the last position it was placed (either programmatically,
*    or by user interaction).
*    
*    Note: If an auto-centering Window is either programmatically moved or dragged 
*    by an end user, auto-centering behavior is automatically turned off. To manually center 
*    a Window, you can use Window.centerInPage.
* @type {boolean}
* @default false
*/
autoCenter?: boolean;

/**
* If true, a click outside the bounds of the Window will have the same effect as
*    pressing its cancel button.
*    Note: Applies only to modal windows.
* @type {boolean}
* @default false
*/
dismissOnOutsideClick?: boolean;

/**
* Should this window be dismissed (same effect as pressing the "Cancel" button) when the 
*  user presses the "Escape" key?
*  Windows with this setting will dismiss on Escape keypresses in any of the following
*  cases:
*  
*  
* - The window or one of its descendants has focus (and does not cancel the Escape 
*    keypress)
*  
* - The window is Window.isModal, and not itself masked. This ensures
*    that focus is not on some unrelated element on the page.
*  
*  
*  If unset default behavior depends on whether a close / cancel button is visible for
*  this item.
* @type {boolean}
* @default null
*/
dismissOnEscape?: boolean;

/**
* Body of the Window, where Items or Window.src
*  is shown.
*  
*  The following autoChildUsage apply:
*  
*  
* - Window.bodyStyle for the Canvas.styleName
*  
* - Window.printBodyStyle for the styleName to use when
*    printing
*  
* - Window.bodyColor / Window.hiliteBodyColor for
*    the Canvas.backgroundColor
*  
* @type {Canvas}
* @default null
*/
body?: Canvas;

/**
* If true, draw the body contents when this Window is drawn.
* @type {boolean}
* @default true
*/
showBody?: boolean;

/**
* Style of the Window body.
* @type {CSSStyleName}
* @default "windowBody"
*/
bodyStyle?: CSSStyleName;

/**
* Style for the Window body in printed output.
* @type {CSSStyleName}
* @default "printHeader"
*/
printBodyStyle?: CSSStyleName;

/**
* Color of the Window body. Overrides the background color specified in the style.
* @type {CSSColor}
* @default "#FFFFFF"
*/
bodyColor?: CSSColor;

/**
* Highlight color for the Window body (shown when the body is flashed).
* @type {CSSColor}
* @default "#EEEEEE"
*/
hiliteBodyColor?: CSSColor;

/**
* The contents of the Window body. Can be specified three different ways:
*    
* - an Array of Canvases that will become the children of the Window's body when it
*    is initialized; the canvases in this array should be created, but not drawn (autodraw:
*    false).
*    
* - a single canvas that will become a child of the Window body.
*    
* - a string that will be set as the body's contents.
* @type {Array<Partial<Canvas>> | Canvas | string}
* @default null
*/
items?: Array<Partial<Canvas>> | Canvas | string;

/**
* A URL to load as content for the Window's body. If specified, this
*  attribute will take precedence over the items attribute.
*  
*  Note that setting window.src is essentially a shortcut for setting Window.items
*  to a single HTMLflow with a specified HTMLFlow.contentsURL.
* @type {string}
* @default null
*/
src?: string;

/**
* If this window has Window.src specified, this property can be used to indicate
*  whether the source is a standalone HTML page or an HTML fragment.
*  
*  This is similar to the HTMLFlow.contentsType property - be sure to read the
*  HTMLFlow documentation to understand circumstances where contentsType:"page" is
*  unsafe and not recommended.
* @type {string}
* @default "page"
*/
contentsType?: string;

/**
* The name of the widget class (as a string) to use for the body. If unset the appropriate
*  constructor type will be determined as follows:
*  - if Window.items is defined as an array of widgets, and Window.contentLayout 
*   is not set to "none", bodyConstructor defaults to a VLayout
*  - if Window.src is set, bodyConstructor defaults to an HTMLFlow
*  - otherwise bodyConstructor will default to a simple Canvas
*  Note that if this property is overridden for some window, the specified constructor 
*  should be a subclass of one of these defaults to ensure the window renders out as 
*  expected.
* @type {string}
* @default null
*/
bodyConstructor?: string;

/**
* Default properties for the body of the Window
*  You can change the class-level bodyDefaults for all Windows by changing this item
*  or set instance.body to be another object of properties to override for your instance only
* @type {object}
* @default "..."
*/
bodyDefaults?: object;

/**
* The layout policy that should be used for widgets within the Window body.
*  
*  See ContentLayoutPolicy and Window.bodyConstructor for details.
* @type {ContentLayoutPolicy}
* @default "vertical"
*/
contentLayout?: ContentLayoutPolicy;

/**
* If true, the window is resized automatically to accommodate the contents
*  	of the body, if they would otherwise require scrolling.
* @type {boolean}
* @default false
*/
autoSize?: boolean;

/**
* Header for the Window, based on an HLayout. The header contains the title and some standard
*  controls for the window, which may be configured via Window.headerControls.
*  
*  The following autoChildUsage apply:
*  
*  
* - Window.headerStyle for Canvas.styleName
*  
* - Window.printHeaderStyle for the styleName to use
*    when printing.
*  
* @type {HLayout}
* @default null
*/
header?: HLayout;

/**
* If true, show a Window.header for this Window. 
*  
*  Note that in certain Smartclient skins Window.showHeaderBackground may be set to
*  false and the header's appearance implemented as part of the
*  window's Canvas.showEdges. In this case suppressing the header can be achieved
*  by overriding the edge media as well as setting this property to false. For example, to
*  create a headerless window with a similar appearance to a Menu in the
*  TreeFrog skin, the following attributes could be used:
*  
*    showHeader:false,
*    edgeImage:"[SKIN]/Menu/m.png",
*    edgeSize:10, edgeTop:17, edgeBottom:17,
*    edgeCenterBackgroundColor:"#F7F7F7"
*  
* @type {boolean}
* @default true
*/
showHeader?: boolean;

/**
* Img background component for the header, for gradient or image-based display
* @type {Img}
* @default null
*/
headerBackground?: Img;

/**
* Should the window header show a background image? Default value is true for all browsers
*  except for Internet Explorer.
*  If set to true the image source is derived from Window.headerSrc and 
*  Window.hiliteHeaderSrc, otherwise the background will be styled according to 
*  Window.headerStyle / Window.hiliteHeaderStyle.
* @type {boolean}
* @default "varies"
*/
showHeaderBackground?: boolean;

/**
* Style for the Window header.
* @type {CSSStyleName}
* @default "WindowHeader"
*/
headerStyle?: CSSStyleName;

/**
* CSS Style for header in printed output
* @type {CSSStyleName}
* @default "printHeader"
*/
printHeaderStyle?: CSSStyleName;

/**
* If Window.showHeaderBackground is true, this property provides
*  the URL of the background image for the header.
* @type {SCImgURL}
* @default ""[SKIN]Window/headerGradient.gif" | null"
*/
headerSrc?: SCImgURL;

/**
* Array of members to show in the Window header. 
*  
*  The default value of headerControls is an Array of Strings listing the
*  standard header controls in their default order:
*  
*   headerControls : ["headerIcon", "headerLabel", 
*            "minimizeButton", "maximizeButton", "closeButton"]
*  
*  You can override headerControls to change the order of standard controls in
*  the header. You can also omit standard controls this way, although it more efficient to
*  use the related "show" property if available (eg Window.showMinimizeButton). 
*  
*  By embedding a Canvas directly in this list you can add arbitrary additional controls to
*  the header, for example, an additional button (eg return to dock) or a DynamicForm with
*  various kinds of input controls. 
*  
*  Note that having added controls to headerControls, you can still call APIs directly on
*  those controls to change their appearance, and you can also show() and hide() them if
*  they should not be shown in some circumstances.
*  
*  Tip: custom controls need to set layoutAlign:"center" to appear vertically centered.
*  
*  Component XML:
*  
*  To define headerControls in Component XML a special set of components
*  are used as markers. The standard header controls can be explicitly specified as: 
*  
*  &lt;headerControls&gt;
*    &lt;WindowHeaderIcon/&gt;
*    &lt;WindowHeaderLabel/&gt;
*    &lt;WindowMinimizeButton/&gt;
*    &lt;WindowMaximizeButton/&gt;
*    &lt;WindowCloseButton/&gt;
*  &lt;/headerControls&gt;
*  
* @type {Array<Partial<string>>}
* @default "(see below)"
*/
headerControls?: Array<Partial<string>>;

/**
* Highlight style for the Window header. Displayed when a window 
*  is Window.flash
* @type {CSSStyleName}
* @default "WindowHeader"
*/
hiliteHeaderStyle?: CSSStyleName;

/**
* If Window.showHeaderBackground is true, this governs the URL of the image to 
*  use in the header's highlighted state when the window is Window.flash
* @type {SCImgURL}
* @default ""[SKIN]Window/headerGradient_hilite.gif" | null"
*/
hiliteHeaderSrc?: SCImgURL;

/**
* Label that shows Window title in header.
*  
*  The following autoChildUsage applies:
*  Window.title for Label.contents.
* @type {Label}
* @default null
*/
headerLabel?: Label;

/**
* Show a title (typically just text) on the header for this window.
* @type {boolean}
* @default true
*/
showTitle?: boolean;

/**
* Title for this Window, shown if Window.showTitle is true in the Window.header
*  (if drawn).
* @type {HTMLString}
* @default "Untitled Window"
*/
title?: HTMLString;

/**
* This is an object literal property block specifying various properties of the header
*  label that displays the Window.title. Overrideable defaults are as follows:
*  
*  
* - styleName- defaults to "windowHeaderText" and specifies the css style
*  that is used to render the Window.title text.
*  
*  You can override the the above properties by calling Class.changeDefaults.
* @type {object}
* @default "see below"
*/
headerLabelDefaults?: object;

/**
* Header icon shown at left end of header by default.
* @type {Img}
* @default null
*/
headerIcon?: Img;

/**
* If true, we show an icon on the left in the header.
* @type {boolean}
* @default true
*/
showHeaderIcon?: boolean;

/**
* This is an object literal property block specifying the various properties of the
*  headerIcon - the icon that appears at the top left of the window and is by default the
*  Isomorphic logo. Overrideable defaults are as follows:
*  
*  
* - width - default to 16 and specifies the width of the headerIcon.
*  
* - height - default to 14 and specifies the height of the headerIcon.
*  
* - src - defaults to "[SKIN]/Window/minimize.gif" and specifies the image
*  for the headerIcon.
*  
*  You can override the the above properties by calling Class.changeDefaults.
* @type {object}
* @default "..."
*/
headerIconDefaults?: object;

/**
* If true, the user can give the header buttons focus (see Window.minimizeButton, 
*  Window.maximizeButton, Window.restoreButton and Window.closeButton).
* @type {boolean}
* @default false
*/
canFocusInHeaderButtons?: boolean;

/**
* Button show in the header that will close this Window by calling Window.closeClick.
* @type {ImgButton}
* @default null
*/
closeButton?: ImgButton;

/**
* If true, show a close button in the header, which will dismiss this window by 
*  calling Window.closeClick.
* @type {boolean}
* @default true
*/
showCloseButton?: boolean;

/**
* ImgButton shown in the header that will minimize this Window by calling Window.minimize.
* @type {ImgButton}
* @default null
*/
minimizeButton?: ImgButton;

/**
* If true, show a minimize button in the header--clicking it minimizes the Window.
* @type {boolean}
* @default true
*/
showMinimizeButton?: boolean;

/**
* Is this window minimized. If true at init time, the window will be drawn minimized.
*  To set this property at runtime use Window.minimize or Window.restore.
* @type {boolean}
* @default false
*/
minimized?: boolean;

/**
* If Window.minimizeHeight is unset, by the window will shrink to the height of the
*  header when minimized.
*  
*  If there is no header, the defaultMinimizeHeight will be used instead.
* @type {number}
* @default "16"
*/
defaultMinimizeHeight?: number;

/**
* Height for the window when minimized.
*  If unset the window will shrink to the height of the header, if present, otherwise
*  Window.defaultMinimizeHeight
* @type {number}
* @default null
*/
minimizeHeight?: number;

/**
* Should this window minimize, maximize, and restore as an animation, or as a 
*  simple 1-step transition?
* @type {boolean}
* @default null
*/
animateMinimize?: boolean;

/**
* If this window is minimizeable, and animateMinimize is true, what should the duration of 
*  the minimize / maximize be (in ms)? If unset defaults to canvas.animationTime.
* @type {number}
* @default null
*/
minimizeTime?: number;

/**
* Default acceleration function for performing an animated minimize / maximize. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
minimizeAcceleration?: AnimationAcceleration;

/**
* ImgButton that restores the Window via Window.restore.
* @type {ImgButton}
* @default null
*/
restoreButton?: ImgButton;

/**
* Is this window maximized. If true at init time, the window will be drawn maximized.
*  To set this property at runtime use Window.maximize or Window.restore.
* @type {boolean}
* @default false
*/
maximized?: boolean;

/**
* Button that will make this Window fill the browser via Window.maximize.
* @type {ImgButton}
* @default null
*/
maximizeButton?: ImgButton;

/**
* If true, show a maximize button in the header - clicking it maximizes the Window
* @type {boolean}
* @default false
*/
showMaximizeButton?: boolean;

/**
* Optional footer for the window, providing space for controls such as the resizer and 
*  status bar.
* @type {HLayout}
* @default null
*/
footer?: HLayout;

/**
* If true, show a footer for this Window, including resizer, statusBar, etc.
*  This setting is commonly overridden for skinning purposes.
* @type {boolean}
* @default true
*/
showFooter?: boolean;

/**
* The height of the footer, in pixels.
* @type {number}
* @default "18"
*/
footerHeight?: number;

/**
* Array of members to show in the Window footer. 
*  
*  The default value of footerControls is an Array of Strings listing the
*  standard footer controls in their default order:
*  
*   footerControls : ["spacer", "resizer"]
*  
*  As with Window.headerControls, you can override footerControls
*  to change the order of standard controls in the footer. "spacer" is a special
*  value which will create a LayoutSpacer in the footer bar. "resizer"
*  will show the Window.resizer in the footer.
*  
*  By embedding a Canvas directly in this list you can add arbitrary additional controls to
*  the footer. 
*  
*  Note that the Window.statusBar is not part of the set of footer controls - it is a
*  separate canvas rendered behind all footer controls. If you include some custom status bar
*  directly in the footerControls you may want to set Window.showFooter to false.
*  
*  Tip: custom controls need to set layoutAlign:"center" to appear vertically centered.
*  
*  Component XML:
*  
*  To define footerControls in Component XML a special set of components
*  are used as markers. The standard footer controls can be explicitly specified as: 
*  
*  &lt;footerControls&gt;
*    &lt;WindowFooterSpacer/&gt;
*    &lt;WindowResizer/&gt;
*  &lt;/footerControls&gt;
*  
* @type {Array<Partial<string>>}
* @default "(see below)"
*/
footerControls?: Array<Partial<string>>;

/**
* Simple Canvas-based status bar, shown in the footer. Window.setStatus can be used to
*  show text here.
* @type {Canvas}
* @default null
*/
statusBar?: Canvas;

/**
* If true, show a statusBar for this Window, including resizer.
*  Note that the status bar will only be displayed if the footer
*  is showing for the window (Window.showFooter).
* @type {boolean}
* @default true
*/
showStatusBar?: boolean;

/**
* ImgButton-based resizer, shown in the footer.
* @type {ImgButton}
* @default null
*/
resizer?: ImgButton;

/**
* If true, show a button in the lower right corner that allows users to resize the Window.
*  Note that the resizer will only be displayed if the footer is showing for the window
*  (Window.showFooter) and Window.canDragResize is true.
* @type {boolean}
* @default true
*/
showResizer?: boolean;

/**
* Where should the window be placed on the screen? Valid settings include
*  "fillScreen", "fillPanel", "halfScreen"
*  and "none"
*  
*  If not explicitly specified, default is to use PanelPlacement "fillScreen" 
*  if Browser.isHandset, and "none" for non-handset devices.
*  
*  If window.placement is something other than "none",
*  sizing and positioning settings (either explicit left, top, width, height settings or
*  the Window.autoCenter and Window.autoSize features) will have no effect.
* @type {PanelPlacement}
* @default null
*/
placement?: PanelPlacement;

/**
* showEdges dynamically defaults to false when the Window.placement setting
*  indicates the Window will be filling a portion of the screen or a panel.
* @type {boolean}
* @default null
*/
showEdges?: boolean;

/**
* Whether to show a drop shadow for this Canvas. 
*  
*  Developers should be aware that the drop shadow
*  is drawn outside the specified width and height of the widget meaning a widget with shadows 
*  takes up a little more space than it otherwise would. A full screen canvas with showShadow set 
*  to true as this would be likely to cause browser scrollbars to appear - developers can handle
*  this by either setting this property to false on full-screen widgets, or by setting
*  overflow to "hidden" on the &lt;body&gt; element browser-level scrolling is never intended to occur.
*  
*  showShadow dynamically defaults to false when the Window.placement setting
*  indicates the Window will be filling a portion of the screen or a panel.
* @type {boolean}
* @default null
*/
showShadow?: boolean;

/**
* Alternative style for the window used whenever Window.placement settings indicate the
*  menu will be filling a portion of the screen or a panel. Generally this alternative
*  style should not have rounded or excessively large edges.
* @type {CSSStyleName}
* @default "windowBackgroundFill"
*/
fillSpaceStyleName?: CSSStyleName;

/**
* Should this window automatically be shown at the top of the page's z-order and
*  be brought to front via Canvas.bringToFront whenever the user clicks it?
*  
*  If Window.isModal is true for this window, this setting will have no effect - we
*  always bring the window to the front on initial display and on mouseDown.
*  By default we also do this for non-modal windows (which matches user expectation for
*  most standard interfaces - think of switching between OS-level application windows), but 
*  this may be disabled for cases where it is not appropriate by setting this attribute to
*  false
* @type {boolean}
* @default true
*/
bringToFrontOnMouseUp?: boolean;

/**
* Text to show in the status bar of the window (if one is visible)
* @type {string}
* @default null
*/
status?: string;







/**
*  Setter for Window.headerStyle.
* 
* @param {CSSStyleName} new Canvas.styleName for the Window.header.
*/
setHeaderStyle(newHeaderStyle:CSSStyleName): void;

/**
*  Dynamically update Window.showCloseButton to show / hide the closeButton
* 
*/
setShowCloseButton(): void;

/**
*  Dynamically update Window.showMinimizeButton to show / hide the minimizeButton
* 
*/
setShowMinimizeButton(): void;

/**
*  Dynamically update Window.showMaximizeButton to show / hide the maximizeButton
* 
*/
setShowMaximizeButton(): void;

/**
*  Dynamically update Window.showHeaderIcon to show / hide the headerIcon
* 
*/
setShowHeaderIcon(): void;

/**
*  Sets the Window.title that appears in the window Window.header.
*  The header will be redrawn if necessary.
* 
* @param {HTMLString} new title.
*/
setTitle(newTitle:HTMLString): void;

/**
*  Updates whether the Window.title is shown in the Window.header.
*  No impact unless the header is being shown. The header will be redrawn if necessary.
* 
* @param {boolean} new showTitle
*/
setShowTitle(newShowTitle:boolean): void;

/**
*  	Sets the text in the status bar of the window, redrawing if necessary.
* 
* @param {string} new text for the status bar
*/
setStatus(statusString:string): void;

/**
*  Sets the URL of the contents to display in the body of the window, redrawing if
*  necessary.
* 
* @param {string} URL of new contents to be displayed in the window body
*/
setSrc(url:string): void;

/**
*  Setter for Window.bodyStyle.
* 
* @param {CSSStyleName} new Canvas.styleName for the Window.body.
*/
setBodyStyle(newBodyStyle:CSSStyleName): void;

/**
*  Setter for Window.bodyColor.
* 
* @param {CSSColor} new Canvas.backgroundColor for the Window.body.
*/
setBodyColor(newBodyColor:CSSColor): void;

/**
*  Setter for Window.hiliteBodyColor.
* 
* @param {CSSColor} new hiliteBodyColor.
*/
setHiliteBodyColor(newHiliteBodyColor:CSSColor): void;

/**
*  Adds a widget to the body area of the window.
* 
* @param {Canvas} the widget to be added
*/
addItem(item:Canvas): Array<any>;

/**
*  Removes a widget from the body area of the window. 
* 
* @param {Canvas} the widget to be removed
*/
removeItem(item:Canvas): Array<any>;

/**
*  	Adds an array of widgets to the window.
* 
* @param {Array<Partial<Canvas>>} an array of widgets to be added
*/
addItems(items:Array<Partial<Canvas>>): Array<any>;

/**
*  	Removes an array of widgets from the window.
* 
* @param {Array<Partial<Canvas>>} an array of widgets to be removed
*/
removeItems(items:Array<Partial<Canvas>>): Array<any>;

/**
*  Same as Layout.addMember. Note that in order to add items to Window.body,
*  you use Window.addItem rather than addMember. Adding a member to 
*  a Window adds the member as a sibling to the header, body and other built-in Window
*  subcomponents.
* 
* @param {Canvas} the canvas object to be added to the layout
* @param {Integer=} the position in the layout to place newMember (starts with 0);              if omitted, it will be added at the last position
*/
addMember(newMember:Canvas, position?:Integer): void;

/**
*  Same as Layout.addMembers. Note that in order to add items to Window.body,
*  you use Window.addItem rather than addMembers. Adding a member to 
*  a Window adds the member as a sibling to the header, body and other built-in Window
*  subcomponents.
* 
* @param {Array<Partial<Canvas>> | Canvas} array of canvases to be added or single Canvas
* @param {number=} position to add newMembers; if omitted newMembers will be added              at the last position
*/
addMembers(newMembers:Array<Partial<Canvas>> | Canvas, position?:number): void;

/**
*  Reveals the child Canvas passed in by showing it if it is currently hidden. Note, in the 
*  case of Window, "child Canvas" means widgets in the Window's "items" collection as well as 
*  real children (the children of a Window - ie, the elements of its "children" array - are 
*  its component parts like header and body)
* 
* 
* @param {GlobalId | Canvas} the child Canvas to reveal, or its global ID
*/
revealChild(child:GlobalId | Canvas): void;

/**
*  Setter for Window.autoSize
* 
* @param {boolean} true if the window should auto-size to its content
*/
setAutoSize(autoSize:boolean): void;

/**
*  Should this window be dismissed (same effect as pressing the "Cancel" button) when the 
*  user presses the "Escape" key?
*  Default behavior: if Window.dismissOnEscape is set, just return it. Otherwise return
*  true if this window is showing a "close" control in the header 
*  (see Window.headerControls).
* 
*/
shouldDismissOnEscape(): boolean;

/**
*  Centers the Window in the page. This is called automatically in window.show() if
*  Window.autoCenter is true.
*  Note - if the Window is a child of another widget, we center in the parent widget
*  rather than centering in the page.
* 
*/
centerInPage(): void;

/**
*      Makes the window header flash if it's visible; if there's no header, or the header
*      is hidden, makes the window body flash instead.
*      
*  	This method is executed when users click outside the bounds of a modal window
*  	so they'll notice that they have to do something with the window.
* 
*/
flash(): void;

/**
*  Minimize the window. Fired when the user clicks the minimize button if 
*  Window.showMinimizeButton is true.
*  Default implementation shrinks the window to just the height of the header bar, hiding the
*  body. If Window.animateMinimize is true, the resize will be animated.
*  A restore button will be displayed in place of the minimize button when the window is 
*  minimized.
* 
*/
minimize(): void;

/**
*  Restores the window to its specified height and width after a call to Window.minimize or 
*  Window.maximize. Called from a click on the restore button shown in place of the 
*  minimize or maximize button when the window is minimized or maximized.
*  Resizing will occur as an animation if Window.animateMinimize is true.
* 
*/
restore(): void;

/**
*  Maximize the window. Fired when the user clicks the maximize button if 
*  Window.showMaximizeButton is true.
*  Default implementation moves the window to 0, 0 and resizes the window to 
*  "100%" on both axes, so it will fill the browser window (or the parent
*  of the Window instance, if appropriate).
*  If Window.animateMinimize is true, the maximize will be animated.
*  A restore button will be displayed in place of the maximize button when the window is 
*  maximized.
*  
* 
*/
maximize(): void;

/**
*  Handles a click on the close button of this window. The default implementation
*  calls Window.close and returns false to prevent bubbling of the click event.
*  
*  Override this method if you want
*  other actions to be taken.
*  
*  Custom implementations may call close() to trigger the default behavior.
* 
*/
closeClick(): boolean;

/**
*  Close this window. 
*  This method is fired by the default Window.closeClick implementation.
*  Default implementation will hide the window.
* 
*/
close(): void;




}

/**
*/
type ProcessValueType = 
/**
* values that are not already strings are converted via toString()
*/
"string" |

/**
* the strings "true" and "false" become boolean true and false. All other         Strings non-empty String values are true, all numbers are true except 0, and
*          all other non-null values are true
*/
"boolean" |

/**
* values are converted via toString() and parsing as a decimal number.         Invalid values trigger a transition to the StateTask.failureElement
*/
"decimal" |

/**
* values are converted via toString(), parsing as a number, and rounding to         nearest integer. Invalid values trigger a transition to the
*          StateTask.failureElement
*/
"integer" |

/**
* any input which is not already a Record or Map becomes null
*/
"record" |

/**
* generic array type - will convert value to an array of the same type as the        existing value
*/
"array";


/**
* Function to produce a summary value based on field values within a record.
*  Example usage is record-level summaries in 
*  ListGridFieldType.
*  
*  SummaryFunctions may be specified in one of 2 ways:
*  
* - as an explicit function or executable
*  stringMethods. Takes 3 parameters:
*  record the record for which the value is being calculated
*  fields Array of listGridFields from which summaries should be calculated
*  summaryField pointer to the summary type field on which the summary function
*  is being run.
*  
* - As a registered RecordSummaryFunction identifier
*  Note that DataSource.registerRecordSummaryFunction may be used to expand the
*  set of registered RecordSummaryFunctions.
*/
type RecordSummaryFunction = 
/**
* Iterates through each field, picking up the numeric field values from the record and summing them.
*/
"sum" |

/**
* Iterates through each field, picking up the numeric field values from the record and calculating the mean value.
*/
"avg" |

/**
* Iterates through each field, picking up the numeric field values from the record and calculating the maximum value.
*/
"max" |

/**
* Iterates through each field, picking up the numeric field values from the record and calculating the minimum value.
*/
"min" |

/**
* Iterates through each field, picking up the numeric field values from the record and multiplying them together.
*/
"multiplier";




/**
* Appearance for a SelectItem that allows multiple selection
*/
type MultipleAppearance = 
/**
* a drop-down picklist that allows multiple choices by       clicking on a checkbox next to each item
*/
"picklist" |

/**
* a grid that displays all items in-place. Multiple selection is        accomplished by ctrl-click or shift-click.
*/
"grid";


/**
* A WSRequest (or "web service request") is an extended RPCRequest with additional properties
*  applicable to WSDL/SOAP web services.
*  
*  All properties which are legal on RPCRequest are legal on a WSRequest, in
*  addition to the properties listed here.
*/
class WSRequest {


/**
* Name of the web service operation to invoke.
* @type {string}
* @default null
*/
wsOperation?: string;

/**
* Data to be serialized to XML to form the SOAP body.
* @type {any}
* @default null
*/
data?: any;

/**
* When useFlatFields is set for a request to be sent to a WSDL web service, when
*  creating the input XML message to send to the web service, properties in
*  WSRequest.data will be used as the values for XML elements of the same name, at
*  any level of nesting.
*  
*  useFlatFields allows you to ignore gratuitous XML message structure, such as
*  extra levels of nested elements, and provides some insulation against changes in the
*  required structure of the input message.
*  
*  For example, given this input message:
*  
*  &lt;FindServices&gt;
*    &lt;searchFor&gt;search text&lt;/searchFor&gt;
*    &lt;Options&gt;
*      &lt;caseSensitive&gt;false&lt;/caseSensitive&gt;
*    &lt;/Options&gt;
*    &lt;IncludeInSearch&gt;
*      &lt;serviceName&gt;true&lt;/serviceName&gt;
*      &lt;documentation&gt;true&lt;/documentation&gt;
*      &lt;keywords&gt;true&lt;/keywords&gt;
*    &lt;/IncludeInSearch&gt;
*  &lt;/FindServices&gt;
*  
*  If useFlatFields were not set, in order to fill out this message
*  correctly, request.data would need to be:
*  {
*   searchFor: "search text",
*   Options : {
*     caseSensitive: false,
*   },
*   IncludeInSearch : {
*     serviceName: true,
*     documentation : true,
*     keywords : true
*   }
*  }
*  However if useFlatFields were set, request.data could be just:
*  {
*   searchFor: "search text",
*   caseSensitive: false,
*   serviceName: true,
*   documentation : true,
*   keywords : true
*  }
*  useFlatFields is often set when the input data comes from a DynamicForm
*  to avoid the cumbersome and fragile process of mapping input fields to an XML structure.
*  
*  OperationBinding.useFlatFields can also be set to cause all dsRequests of a
*  particular type to useFlatFields automatically.
*  
*  For DataBoundComponent, 
*  DataBoundComponent.useFlatFields can be set use "flattened"
*  binding to fields of a WSDL message or XML Schema.
*  
*  Note that useFlatFields is not generally recommended for use with XML input
*  messages where multiple simple type fields exist with the same name, however if used in this
*  way, the first field to use a given name wins. "first" means the first field encountered in a
*  depth first search. "wins" means only the first field will be populated in the generated
*  XML message.
* @type {boolean}
* @default null
*/
useFlatFields?: boolean;

/**
* Optional object declaring namespace prefixes for use in evaluating the
*  resultType parameter of WebService.callOperation, if resultType is an
*  XPath.
*  
*  Format is identical to OperationBinding.xmlNamespaces, and default namespaces
*  bindings are also identical.
* @type {object}
* @default null
*/
xmlNamespaces?: object;

/**
* Valid only with WebService.callOperation. If set, do not transform XML results to
*  JavaScript. Instead just return the XML nodes selected by the passed XPath or recordName,
*  or all nodes within the SOAP body if no XPath was passed.
* @type {boolean}
* @default false
*/
xmlResult?: boolean;

/**
* Data to be serialized to form the SOAP headers, as a map from the header part name to the
*  data. For example, given WSDL like this:
*  
*    &lt;soap:header part="SessionHeader" message="tns:HeaderMessage"/&gt;
*    &lt;soap:header part="CallOptions" message="tns:HeaderMessage/&gt;
*  
*  headerData like this might be provided:
*  
*    dsRequest.headerData = 
*      { SessionHeader : data
*       CallOptions : data };
*  
*  The provided data will be serialized to XML by the
*  WebService.getInputHeaderSchema via
*  DataSource.xmlSerialize
* @type {any}
* @default null
*/
headerData?: any;










}



/**
* Subclass of PickTreeItem which shows an IMenuButton rather than a
*  simple MenuButton as it's main button.
*/
class IPickTreeItem extends PickTreeItem {
/**
* Creates a new IPickTreeItem
* 
* @param typeCheckedProperties {Partial<IPickTreeItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {IPickTreeItem} The new instance
*/
static create(typeCheckedProperties? : Partial<IPickTreeItem>, uncheckedProperties?: Object): IPickTreeItem;


/**
* This item is an AutoChild generated Canvas displayed by
*  the IPickTreeItem and is an instance of ITreeMenuButton by default.
* @type {Canvas}
* @default null
*/
button?: Canvas;










}

/**
* TextForm item for managing a text field that displays a decimal value.
*/
class DoubleItem extends FloatItem {
/**
* Creates a new DoubleItem
* 
* @param typeCheckedProperties {Partial<DoubleItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DoubleItem} The new instance
*/
static create(typeCheckedProperties? : Partial<DoubleItem>, uncheckedProperties?: Object): DoubleItem;








/**
*  Return the value tracked by this form item as a Double. If the value cannot
*  be parsed to a valid double, null will be returned.
* 
* 
*/
getValueAsDouble(): Double;




}

/**
* Resize bar for use in Layout.resizeBarClass, based on the
*  Img class. As with the Splitbar class, 
*  widgets of this class can be displayed as a resize-bar for widgets 
*  in Layouts where showResizeBar is set to true. Provides a different appearance from
*  the Splitbar class.
*  
*  To specify the resizeBar class for some layout, use the Layout.resizeBarClass
*  property.
*/
class ImgSplitbar extends Img {
/**
* Creates a new ImgSplitbar
* 
* @param typeCheckedProperties {Partial<ImgSplitbar>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ImgSplitbar} The new instance
*/
static create(typeCheckedProperties? : Partial<ImgSplitbar>, uncheckedProperties?: Object): ImgSplitbar;


/**
* When a Splitbar is created by a layout, the target property
*  of the Splitbar will be a pointer to the member for which it is acting as a resizeBar.
*  The Splitbar will be positioned next to its target, and will resize it on drag completion.
*  
*  See Layout.resizeBarClass, Canvas.showResizeBar and 
*  Canvas.resizeBarTarget for details on configuring the resize bars shown in Layouts.
* @type {Canvas}
* @default null
*/
target?: Canvas;

/**
* Is this split bar vertically orientated?
*  When a Splitbar is created by a layout to be the resizeBar for some
*  member of the layout, the vertical property will be set to true
*  if the layout is horizontal, meaning this resizeBar will be taller than it is wide, 
*  and will allow horizontal resizing of the member.
* @type {boolean}
* @default null
*/
vertical?: boolean;

/**
* canDrag set to true to allow dragging of the split bar. Dragging the
*  Splitbar will resize it's Splitbar.target
* @type {boolean}
* @default true
*/
canDrag?: boolean;

/**
* If this property is true, a click on the Splitbar will collapse its 
*  Splitbar.target, hiding it and shifting the Splitbar and other members
*  of the layout across to fill the newly available space. If the target is already hidden
*  a click will expand it again (showing it at its normal size).
*  
*  Note that on touch devices, to enable collapsing/uncollapsing the target
*  in response to a tap, Splitbar.canCollapseOnTap must be set to
*  true.
* @type {boolean}
* @default true
*/
canCollapse?: boolean;

/**
* Default directory for skin images (those defined by the class), relative to the
*  Page-wide Page.getSkinDir.
* @type {SCImgURL}
* @default "images/SplitBar/"
*/
skinImgDir?: SCImgURL;

/**
* The base filename or stateful image configuration for the image. 
*  Note that as the StatefulCanvas.state 
*  of the component changes, the image displayed will be updated as described in
*  statefulImages.
* @type {string}
* @default null
*/
src?: string;

/**
* Default src to display when ImgSplitbar.vertical is false, 
*  and ImgSplitbar.src is unset.
* @type {string}
* @default "[SKIN]hgrip.png"
*/
hSrc?: string;

/**
* Default src to display when ImgSplitbar.vertical is true, 
*  and ImgSplitbar.src is unset.
* @type {SCImgURL}
* @default "[SKIN]vgrip.png"
*/
vSrc?: SCImgURL;










}

/**
* A subclass of Layout that applies a sizing policy along the horizontal axis, interpreting
*  percent and "*" sizes as proportions of the width of the layout. HLayouts will set any members
*  that do not have explicit heights to match the layout.
*/
class HLayout extends Layout {
/**
* Creates a new HLayout
* 
* @param typeCheckedProperties {Partial<HLayout>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {HLayout} The new instance
*/
static create(typeCheckedProperties? : Partial<HLayout>, uncheckedProperties?: Object): HLayout;











}



/**
* Represents the position of a Portlet within a PortalLayout, indicating the
*  column, row, and position within the row.
*/
class PortalPosition {


/**
* The column number occupied by a Portlet within a PortalLayout.
* @type {Integer}
* @default "0"
*/
colNum?: Integer;

/**
* The row number occupied by a Portlet within a PortalLayout column.
* @type {Integer}
* @default "0"
*/
rowNum?: Integer;

/**
* The position occupied by a Portlet within a PortalLayout row
*  (generally 0, unless there is more than one Portlet in the row).
* @type {Integer}
* @default "0"
*/
position?: Integer;










}

/**
* Definition of a linear gradient between two points, (LinearGradient.x1, LinearGradient.y1)
*  and (LinearGradient.x2, LinearGradient.y2).
*/
class LinearGradient extends Gradient {


/**
* X coordinate of the start point. This can be a number or a percentage of the width of the
*  bounding box of the DrawItem to which it is applied.
* @type {string}
* @default null
*/
x1?: string;

/**
* Y coordinate of the start point. This can be a number or a percentage of the height of the
*  bounding box of the DrawItem to which it is applied.
* @type {string}
* @default null
*/
y1?: string;

/**
* X coordinate of the end point. This can be a number or a percentage of the width of the
*  bounding box of the DrawItem to which it is applied.
* @type {string}
* @default null
*/
x2?: string;

/**
* Y coordinate of the end point. This can be a number or a percentage of the height of the
*  bounding box of the DrawItem to which it is applied.
* @type {string}
* @default null
*/
y2?: string;










}

/**
* What should drive the automatic horizontal expansion of the chart?
*/
type AutoScrollDataApproach = 
/**
* Expand the chart to make room for data label facet value.
*/
"labels" |

/**
* Expand the chart to accommodate FacetChart.barMargin, FacetChart.minBarThickness,          and FacetChart.getMinClusterSize.
*/
"clusters" |

/**
* Expand the chart to make room for both labels and clusters (whichever requires        more space).
*/
"both";


/**
* A form that allows the user to input advanced search criteria, including operators on
*  field values such as "less than", and sub-clauses using "AND" and "OR" operators.
*  
*  A FilterBuilder produces an AdvancedCriteria object, which the DataSource
*  subsystem can use to filter datasets, including the ability to perform such filtering within
*  the browser for datasets that are completely loaded.
*  
*  The operators available for each field can be customized at the DataSource level via
*  DataSourceField.validOperators, DataSource.setTypeOperators and related
*  APIs.
*/
class FilterBuilder extends Layout {
/**
* Creates a new FilterBuilder
* 
* @param typeCheckedProperties {Partial<FilterBuilder>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FilterBuilder} The new instance
*/
static create(typeCheckedProperties? : Partial<FilterBuilder>, uncheckedProperties?: Object): FilterBuilder;


/**
* The message to display next to fieldNames that do not exist in the available dataSource.
* @type {string}
* @default "[missing field definition]"
*/
missingFieldPrompt?: string;

/**
* If specified, the FilterBuilder will dynamically fetch DataSourceField definitions from 
*  this DataSource rather than using FilterBuilder.dataSource. The FieldPicker 
*  will default to being a ComboBoxItem rather than a SelectItem so that the user 
*  will have type-ahead auto-completion.
*  
*  The records returned from the fieldDataSource must have properties 
*  corresponding to a DataSourceField definition, at a minimum, 
*  DataSourceField.name and DataSourceField.type. Any property 
*  legal on a DataSourceField is legal on the returned records, including 
*  DataSourceField.valueMap.
*  
*  Even when a fieldDataSource is specified, FilterBuilder.dataSource may
*  still be specified in order to control the list of 
*  DataSource.setTypeOperators for each field.
* @type {DataSource}
* @default null
*/
fieldDataSource?: DataSource;

/**
* Should the FieldPicker items be sorted alphabetically in the drop down list.
* @type {boolean}
* @default true
*/
sortFields?: boolean;

/**
* AutoChild for the FormItem that allows a user to pick a DataSource field when 
*  creating filter clauses.
*  
*  This will be a SelectItem by default, or a ComboBoxItem if
*  FilterBuilder.fieldDataSource has been specified.
* @type {PickList}
* @default null
*/
fieldPicker?: PickList;

/**
* The title for the FilterBuilder.fieldPicker select-item.
* @type {string}
* @default "Field Name"
*/
fieldPickerTitle?: string;

/**
* Properties to combine with the FieldPicker autoChild FormItem.
* @type {Partial<FormItem>}
* @default null
*/
fieldPickerProperties?: Partial<FormItem>;

/**
* AutoChild for the FormItem that allows a user to select the operator
*  when creating filter clauses. Each clause will create an operatorPicker automatically.
*  To customize this item, use FilterBuilder.operatorPickerProperties
* @type {SelectItem}
* @default null
*/
operatorPicker?: SelectItem;

/**
* Properties to combine with the FilterBuilder.operatorPicker autoChild FormItem.
* @type {Partial<FormItem>}
* @default null
*/
operatorPickerProperties?: Partial<FormItem>;

/**
* The title for the operator-picker select-item.
* @type {string}
* @default "Operator"
*/
operatorPickerTitle?: string;

/**
* Width for the field picker formItem displayed in clauses within this FilterBuilder.
* @type {Integer | string}
* @default "*"
*/
fieldPickerWidth?: Integer | string;

/**
* Width for the operator picker formItem displayed in clauses within this FilterBuilder.
* @type {Integer | string}
* @default "150"
*/
operatorPickerWidth?: Integer | string;

/**
* Width for the value-chooser formItem displayed in clauses within this FilterBuilder.
*  Note that depending on the selected operator type, this item may not be displayed, or
*  may have different characteristics. See FilterBuilder.getValueFieldProperties for information
*  on customizing the value item.
* @type {Integer | string}
* @default "150"
*/
valueItemWidth?: Integer | string;

/**
* DataSource this filter should use for field definitions and available Operators.
* @type {DataSource | GlobalId}
* @default null
*/
dataSource?: DataSource | GlobalId;

/**
* Initial criteria.
*  
*  When initialized with criteria, appropriate clauses for editing the provided criteria will
*  be automatically generated.
*  
*  Note that empty or partial criteria are allowed, for example, criteria that specify
*  Criterion.fieldName only will generate an expression with the operator not chosen
*  yet, and a Criterion with a logical operator ("and" or "or") but not
*  Criterion.criteria defined will generate an empty subclause.
* @type {AdvancedCriteria}
* @default null
*/
criteria?: AdvancedCriteria;

/**
* If true, when the user hits the Enter key while focused in a text-item in this 
*  FilterBuilder, we automatically invoke the user-supplied FilterBuilder.search method.
* @type {boolean}
* @default null
*/
saveOnEnter?: boolean;

/**
* If true (the default), show field titles in the drop-down box used to select a field for querying.
*  If false, show actual field names instead.
* @type {boolean}
* @default true
*/
showFieldTitles?: boolean;

/**
* If true (the default), validates each entered value when it changes, to make sure it is a 
*  a valid value of its type (valid string, number, and so on). No other validation is 
*  carried out. If you switch this property off, it is still possible to validate the 
*  FilterBuilder by calling FilterBuilder.validate from your own code.
* @type {boolean}
* @default true
*/
validateOnChange?: boolean;

/**
* If set, a button will be shown for each clause allowing it to be removed.
* @type {boolean}
* @default true
*/
showRemoveButton?: boolean;

/**
* The hover prompt text for the remove button.
* @type {string}
* @default "Remove"
*/
removeButtonPrompt?: string;

/**
* The removal ImgButton that appears before each clause if
*  FilterBuilder.showRemoveButton is set.
* @type {ImgButton}
* @default null
*/
removeButton?: ImgButton;

/**
* If set, a button will be shown underneath all current clauses allowing a new clause to be
*  added.
* @type {boolean}
* @default true
*/
showAddButton?: boolean;

/**
* The hover prompt text for the add button.
* @type {string}
* @default "Add"
*/
addButtonPrompt?: string;

/**
* An ImgButton that allows new clauses to be added if FilterBuilder.showAddButton
*  is set.
* @type {ImgButton}
* @default null
*/
addButton?: ImgButton;

/**
* If set to false, the last clause cannot be removed.
* @type {boolean}
* @default false
*/
allowEmpty?: boolean;

/**
* Dictates whether values entered by a user should be retained in the value fields when a 
*  different field is selected. Default value is true.
*  
*  Note that, when switching between fields that have an optionDataSource or valueMap, this
*  property is ignored and the values are never retained.
* @type {boolean}
* @default true
*/
retainValuesAcrossFields?: boolean;

/**
* Logical operators to allow for TopOperatorAppearances of "radio" and "bracket".
*  Note that this list may be further limited according to the 
*  DataSource.getTypeOperatorMap returned by the 
*  DataSource.
* @type {Array<Partial<OperatorId>>}
* @default "["and", "or", "not"]"
*/
topOperatorOptions?: Array<Partial<OperatorId>>;

/**
* Default logical operator for all top-level clauses in the FilterBuilder.
*  
*  May be able to be changed by the user via the UI, according to TopOperatorAppearance.
* @type {LogicalOperator}
* @default "and"
*/
topOperator?: LogicalOperator;

/**
* Logical operators to allow if we have a TopOperatorAppearance of "radio".
* @deprecated * in favor of FilterBuilder.topOperatorOptions.

* @type {Array<Partial<OperatorId>>}
* @default "["and", "or", "not"]"
*/
radioOptions?: Array<Partial<OperatorId>>;

/**
* How to display and edit the FilterBuilder.topOperator for this FilterBuilder.
*  
*  See TopOperatorAppearance for a list of options.
* @type {TopOperatorAppearance}
* @default "bracket"
*/
topOperatorAppearance?: TopOperatorAppearance;

/**
* HLayout of radioOperationForm and optional modeSwitcher.
* @type {HLayout}
* @default null
*/
radioOperatorLayout?: HLayout;

/**
* With TopOperatorAppearance:"radio", form that appears above the stack of clauses
*  and allows picking the LogicalOperator for the overall FilterBuilder.
*  
*  By default, consists of a simple RadioGroupItem.
* @type {DynamicForm}
* @default null
*/
radioOperatorForm?: DynamicForm;

/**
* Label to change between simple and advanced mode. When clicked the filter mode
*  is switched to the other mode. This label is only shown if
*  FilterBuilder.showModeSwitcher is true.
*  
*  Shows either FilterBuilder.modeSwitcherSimpleMessage or
*  FilterBuilder.modeSwitcherAdvancedMessage
*  depending on the current state of the filter.
* @type {Label}
* @default null
*/
modeSwitcher?: Label;

/**
* The title for the Operator RadioGroupItem displayed in the FilterBuilder.radioOperatorForm.
* @type {string}
* @default "Overall Operator"
*/
radioOperatorTitle?: string;

/**
* If true, causes a CheckboxItem to appear to the left of each clause in "inline" 
*  TopOperatorAppearance. This checkbox allows the user to select 
*  individual clauses so that, for example, clauses can be removed from the filterBuilder 
*  by application code. This property is ignored for appearances other than "inline".
* @type {boolean}
* @default false
*/
showSelectionCheckbox?: boolean;

/**
* With TopOperatorAppearance "bracket" and "inline", a form that appears to the left
*  of the stack of clauses and allows picking the LogicalOperator for the overall 
*  FilterBuilder (or for that specific FilterClause, in the case of "inline")
*  
*  By default, consists of a CheckboxItem if FilterBuilder.showSelectionCheckbox is true, and a 
*  simple SelectItem containing the available logical operators.
*  
*  If this FilterBuilder shows nested sub-clauses, the same defaults will be applied to the
*  top-operator item for each sub-clause.
* @type {DynamicForm}
* @default null
*/
topOperatorForm?: DynamicForm;

/**
* Automatically generated SelectItem autoChild shown in the FilterBuilder.topOperatorForm.
*  Developers may customize this item using the standard autoChild pattern (by
*  modifying topOperatorItemDefaults and 
*  topOperatorItemProperties).
*  
*  If this FilterBuilder shows nested sub-clauses, the same defaults will be applied to the
*  top-operator item for each sub-clause.
* @type {SelectItem}
* @default null
*/
topOperatorItem?: SelectItem;

/**
* Width for the FilterBuilder.topOperatorItem autoChild.
* @type {number | string}
* @default "*"
*/
topOperatorItemWidth?: number | string;

/**
* The title for the left-aligned Operator selectItem in the FilterBuilder.topOperatorForm.
* @type {string}
* @default "Clause Operator"
*/
topOperatorTitle?: string;

/**
* Default operator for subclauses added via the FilterBuilder.subClauseButton.
* @type {LogicalOperator}
* @default "or"
*/
defaultSubClauseOperator?: LogicalOperator;

/**
* Title for the "Match All" (and) operator 
*  when using TopOperatorAppearance:"radio".
* @type {string}
* @default "Match All"
*/
matchAllTitle?: string;

/**
* Title for the "Match None" (not) operator
*  when using TopOperatorAppearance:"radio".
* @type {string}
* @default "Match None"
*/
matchNoneTitle?: string;

/**
* Title for the "Match Any" (or) operator
*  when using TopOperatorAppearance:"radio".
* @type {string}
* @default "Match Any"
*/
matchAnyTitle?: string;

/**
* Title for the "And" operator (only applicable to the "inline" appearance)
* @type {string}
* @default "and"
*/
inlineAndTitle?: string;

/**
* Title for the "Or" operator (only applicable to the "inline" appearance)
* @type {string}
* @default "or"
*/
inlineOrTitle?: string;

/**
* Title for the "And Not" operator (only applicable to the "inline" appearance)
* @type {string}
* @default "and not"
*/
inlineAndNotTitle?: string;

/**
* Title for the "Advanced.." mode switcher label (only applicable to the "radio" appearance).
* @type {string}
* @default "Advanced.."
*/
modeSwitcherAdvancedMessage?: string;

/**
* Title for the "Simple Mode.." mode switcher label (only applicable to the "bracket" appearance).
* @type {string}
* @default "Simple Mode.."
*/
modeSwitcherSimpleMessage?: string;

/**
* Message displayed when switching to "radio" mode if the criteria will be logically changed.
* @type {string}
* @default "Criteria will be modified to fit in simpler editing interface"
*/
modeSwitcherFlattenWarningMessage?: string;

/**
* When enabled allows FilterBuilder in topOperatorAppearance:"radio" or 
*  topOperatorAppearance:"bracket" mode to be switch to the other view by the user.
*  "radio" mode is considered simple where "bracket" mode is advanced mode.
*  
*  Note that when switching from "bracket" to "radio" mode any entered criteria will be
*  flattened by calling DataSource.flattenCriteria. If the criteria cannot be
*  flattened without losing symantics (see DataSource.canFlattenCriteria) the user is
*  prompted to confirm.
*  
*  If showModeSwitcher is set and topOperatorAppearance is unset:
*  
*  
* - when first drawn, the filterBuilder will choose which mode to use based on the
*    provided FilterBuilder.criteria if any: advanced mode ("bracket") will be used if
*    AdvancedCriteria are provided which cannot be flattened without loss of data (see
*    DataSource.canFlattenCriteria), otherwise simple mode ("radio") will be used. 
*  
* - for any calls to FilterBuilder.setCriteria after draw, the FilterBuilder will
*    switch to advanced mode if the criteria cannot be shown in simple mode without losing
*    information, but will never automatically switch to simple mode, but an explicit call
*    FilterBuilder.setTopOperatorAppearance can be used to do so.
*  
* @type {boolean}
* @default null
*/
showModeSwitcher?: boolean;

/**
* VStack of all clauses that are part of this FilterBuilder
* @type {VStack}
* @default null
*/
clauseStack?: VStack;

/**
* For operators that check that a value is within a range, text to show between the start and
*  end input fields for specifying the limits of the range.
* @type {string}
* @default "and"
*/
rangeSeparator?: string;

/**
* Whether to show a button that allows the user to add subclauses. Defaults to false if 
*  the TopOperatorAppearance is "radio" or "inline", true in all other cases.
* @type {boolean}
* @default "See Description"
*/
showSubClauseButton?: boolean;

/**
* The title of the subClauseButton
* @type {string}
* @default "+()"
*/
subClauseButtonTitle?: string;

/**
* The hover prompt text for the subClauseButton.
* @type {string}
* @default "Add Subclause"
*/
subClauseButtonPrompt?: string;

/**
* Button allowing the user to add subclauses grouped by a LogicalOperator.
* @type {IButton}
* @default null
*/
subClauseButton?: IButton;

/**
* Widget used as a "bracket" to hint to the user that a subclause groups several
*  field-by-field filter criteria under one logical operator.
*  
*  By default, a simple CSS-style Canvas with borders on three sides. A vertical StretchImg
*  could provide a more elaborate appearance.
* @type {Canvas}
* @default null
*/
bracket?: Canvas;





/**
*  Returns a human-readable string describing the clauses in this advanced criteria or criterion.
*  
* 
* @param {AdvancedCriteria | Criterion} Criteria to convert to a readable string
* @param {DataSource} DataSource to provide definitions of operators
*/
static getFilterDescription(criteria:AdvancedCriteria | Criterion, dataSource:DataSource): string;



/**
*  Remove a clause this FilterBuilder is currently showing.
* 
* @param {FilterClause} clause as retrieved from filterBuilder.clauses
*/
removeClause(clause:FilterClause): void;

/**
*  Programmatically change the FilterBuilder.topOperator for this FilterBuilder.
* 
* @param {LogicalOperator} new top-level operator
*/
setTopOperator(operator:LogicalOperator): void;

/**
*  Modify TopOperatorAppearance at runtime.
*  
*  Note that when changing from "bracket" to "radio" mode the criteria
*  will be flattened by calling DataSource.flattenCriteria which could
*  result in a logical change to the criteria.
* 
* 
* @param {TopOperatorAppearance} new topOperatorAppearance
*/
setTopOperatorAppearance(appearance:TopOperatorAppearance): void;

/**
*  Add a new FilterClause to this FilterBuilder.
*  
*  This API is intended for the rare use case of adding a highly customized FilterClause
*  component that does not include the standard field/operator/value picking interface, instead
*  providing a custom interface and returning a criterion via FilterClause.getCriterion.
*  
*  If you just want to programmatically add a new FilterClause showing a specific Criterion use
*  FilterBuilder.addCriterion.
*  
*  If you want to use the standard field/operator/value interface but provide a custom control
*  for editing the value, see DataSource.addSearchOperator and Operator.editorType.
*  
* 
* @param {FilterClause} A FilterClause instance
*/
addClause(filterClause:FilterClause): void;

/**
*  Returns an array of child FilterBuilders, representing the list of complex 
*  clauses, or an empty array if there aren't any.
*  
* 
*/
getChildFilters(): Array<Partial<FilterBuilder>>;

/**
*  Returns a human-readable string describing the clauses in this filterBuilder.
*  
* 
*/
getFilterDescription(): string;

/**
*  Validate the clauses of this FilterBuilder.
* 
*/
validate(): boolean;

/**
*  Get the list of OperatorId that are valid for the passed field. By 
*  default, all operators returned by DataSource.getFieldOperators are used.
*  
*  Called automatically by the default implementation of the same method on each
*  FilterClause.getFieldOperators, whenever its fieldName is changed.
*  
* 
* @param {string} the name of the field for which to return the set of available operators
*/
getFieldOperators(fieldName:string): Array<Partial<OperatorId>>;

/**
*  
*  Override to return properties for the FormItem(s) used for the "value" field displayed within
*  clauses within this filterBuilder.
*  
*  Note that the Operator.valueType impacts when this method is called. For operators with
*  valueType "fieldType" or "custom", a single value field is displayed.
*  For operators with valueType "valueRange" two value-field items are displayed
*  (one for the start and one for the end position). The valueItemType parameter may
*  be used to determine which form item is being generated.
*  
* 
* @param {FieldType} type of the DataSource field for this filter row
* @param {string} name of the DataSource field for this filter row
* @param {OperatorId} OperatorId for the chosen operator
* @param {ValueItemType} What valueItem is being generated.
*/
getValueFieldProperties(type:FieldType, fieldName:string, operatorId:OperatorId, itemType:ValueItemType): Partial<FormItem>;

/**
*  Get the criteria entered by the user.
*  
* 
* @param {boolean=} By default if a user has selected a field and operator  type, but has failed to enter a value for the field it will be skipped. This optional parameter  allows you to retrieve all criteria, including those with an empty value attribute.
*/
getCriteria(includeEmptyValues?:boolean): AdvancedCriteria;

/**
*  Set new criteria for editing. 
*  
*  An interface for editing the provided criteria will be generated identically to what happens
*  when initialized with Criteria.
*  
*  Any existing criteria entered by the user will be discarded. 
*  
* 
* @param {AdvancedCriteria} new criteria. Pass null or {} to effectively reset the                  filterBuilder to it's initial state when no criteria are                   specified
*/
setCriteria(criteria:AdvancedCriteria): void;

/**
*  Clear all current criteria.
* 
*/
clearCriteria(): void;

/**
*  Add a new criterion, including recursively adding sub-criteria for a criterion that
*  contains other criteria.
*  
* 
* @param {Criterion} new criterion to be added
*/
addCriterion(criterion:Criterion): void;

/**
*  Returns the type of editor to use for the field.
*  
*  Default behavior is to use the Operator.editorType for a custom operator, otherwise, 
*  use RelativeDateItem for before/after/between operators on date fields, otherwise, 
*  use the same editor as would be chosen by a SearchForm.
* 
* 
* @param {DataSourceField} DataSourceField definition
* @param {OperatorId} OperatorId for the chosen operator
*/
getEditorType(field:DataSourceField, operatorId:OperatorId): SCClassName;

/**
*  Returns the list of this FilterBuilder's FilterClauses that are currently selected. A 
*  clause is "selected" if the user has checked the checkbox next to it; therefore, this 
*  method always returns an empty list unless the 
*  FilterBuilder.showSelectionCheckbox property is set. This method is only 
*  applicable where TopOperatorAppearance is "inline" (because that is the only 
*  appearance that supports showSelectionCheckbox)
* 
* 
*/
getSelectedClauses(): Array<Partial<FilterClause>>;

/**
*  A StringMethod that is automatically invoked if FilterBuilder.saveOnEnter is set 
*  and the user presses Enter whilst in a text-item in any clause or subclause.
* 
* 
* @param {AdvancedCriteria} The criteria represented by the filterBuilder
*/
search(criteria:AdvancedCriteria): void;

/**
*  Handler fired when there is a change() event fired on any FormItem within the 
*  filterBuilder. 
* 
* 
*/
filterChanged(): void;




}

/**
* EditProxy that handles FormItems when editMode is enabled.
*/
class FormItemEditProxy extends EditProxy {
/**
* Creates a new FormItemEditProxy
* 
* @param typeCheckedProperties {Partial<FormItemEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FormItemEditProxy} The new instance
*/
static create(typeCheckedProperties? : Partial<FormItemEditProxy>, uncheckedProperties?: Object): FormItemEditProxy;


/**
* If InlineEditEvent for this FormItem edits the
*  FormItem.valueMap, character that should be used as a separator between
*  values, or between pairs of stored vs display values if the user is entering such
*  a ValueMap using the FormItemEditProxy.valueMapDisplaySeparatorChar.
*  
*  If EditProxy.inlineEditMultiline is enabled, newlines will be used as value
*  separators and the valueMapSeparatorChar
*  
*  The FormItemEditProxy.valueMapEscapeChar can be used to enter the separator
*  char as part of a valueMap value.
* @type {string}
* @default ","
*/
valueMapSeparatorChar?: string;

/**
* If InlineEditEvent for this FormItem edits the
*  FormItem.valueMap, character that should be used as a separator for
*  entering ValueMaps that map from a stored value to a user-displayed value.
*  
*  With the default of ":", the following input:
*  
*    1:Fixed, 2:Won't Fix, 3:Resolved
*  
*  Would be assumed to be a mapping like this (expressed in JSON):
*  
*   {
*    "1" : "Fixed",
*    "2" : "Won't Fix",
*    "3" : "Resolved"
*   }
*  
*  
*  If the input has inconsistent use of the separator char, the input will be assumed
*  to be stored-to-displayed mapping if the separator char is present in a majority
*  of values, and any values that lack a separator will use the same value for
*  storage and display. For example, for this input:
*  
*     Fixed:Reported Fixed, WontFix:Won't Fix, Resolved
*  
*  The resulting valueMap would be:
*  
*   {
*    "Fixed" : "Reported Fixed",
*    "WontFix" : "Won't Fix",
*    "Resolved" : "Resolved"
*   }
*  
*  
*  The FormItemEditProxy.valueMapEscapeChar can be used to enter literal colon characters.
*  
*  Set valueMapDisplaySeparatorChar to null to prevent entry of stored
*  vs displayed values - user input will always be treated as just a list of legal
*  values.
* @type {string}
* @default ":"
*/
valueMapDisplaySeparatorChar?: string;

/**
* If InlineEditEvent for this FormItem edits the
*  FormItem.valueMap, character that can be used to mark the default selected
*  option. Can appear before or after a value, for example, with this input:
*  
*    Fixed,Won't Fix,Resolved*
*  
*  "Resolved" would be the default selected value.
*  
*  If multiple values are marked selected for a SelectItem,
*  SelectItem.multiple will automatically be set.
*  
*  The FormItemEditProxy.valueMapEscapeChar can be used to allow the
*  valueMapSelectedChar to appear at the beginning or end of a 
*  valueMap value.
* @type {string}
* @default "*"
*/
valueMapSelectedChar?: string;

/**
* If InlineEditEvent for this FormItem edits the
*  FormItem.valueMap, character that can be used to enter literal separator
*  chars (such as the FormItemEditProxy.valueMapSeparatorChar) or literal
*  leading or trailing whitespace.
*  
*  Repeat this character twice to enter it literally. For example, with the default
*  of "\", inputting "\\" would result in a literal backslash in the value.
* @type {string}
* @default "\"
*/
valueMapEscapeChar?: string;










}





/**
* Simple subclass of ImgSplitbar with appearance appropriate for a ToolStrip resizer.
*/
class ToolStripResizer extends ImgSplitbar {
/**
* Creates a new ToolStripResizer
* 
* @param typeCheckedProperties {Partial<ToolStripResizer>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ToolStripResizer} The new instance
*/
static create(typeCheckedProperties? : Partial<ToolStripResizer>, uncheckedProperties?: Object): ToolStripResizer;


/**
* Path to resizer image.
* @type {SCImgURL}
* @default "images/ToolStrip/"
*/
skinImgDir?: SCImgURL;

/**
* Image for resizer
* @type {SCImgURL}
* @default "[SKIN]resizer.png"
*/
vSrc?: SCImgURL;

/**
* Image for horizontal resizer for a vertical Toolstrip
* @type {SCImgURL}
* @default "[SKIN]hresizer.png"
*/
hSrc?: SCImgURL;










}



/**
* A widget for editing a single, advanced HiliteRule for use by 
*  DataBoundComponent. Where a simple hilite provides 
*  configuration of a single criterion and either foreground or background color for 
*  application to a single field, an advanced hilite can specify more complex criteria which can 
*  both test and affect multiple fields and allow both background and foreground colors to 
*  be specified in a single rule. 
*  
*  Important Note: this class should not be used directly - it is exposed purely for
*  i18nMessages
*/
class AdvancedHiliteEditor extends VStack {
/**
* Creates a new AdvancedHiliteEditor
* 
* @param typeCheckedProperties {Partial<AdvancedHiliteEditor>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AdvancedHiliteEditor} The new instance
*/
static create(typeCheckedProperties? : Partial<AdvancedHiliteEditor>, uncheckedProperties?: Object): AdvancedHiliteEditor;


/**
* AutoChild FilterBuilder for configuring the criteria for this Hilite.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.filterBuilderProperties.
* @type {FilterBuilder}
* @default null
*/
filterBuilder?: FilterBuilder;

/**
* The title for the Filter group.
* @type {string}
* @default "Filter"
*/
filterGroupTitle?: string;

/**
* Specifies a list of icons that can be used in hilites.
*  
*  hiliteIcons should be specified as an Array of SCImgURL.
*  When present, HiliteRule
*  will offer the user a drop down for picking one of these icons.
*  
*  If the user picks an icon, the created hiliting rule will have Hilite.icon set to 
*  the chosen icon. ListGridField.hiliteIconPosition controls where the icon will 
*  appear for that field -- the default is that it appears in front of the normal cell content.
* @type {Array<Partial<SCImgURL>>}
* @default null
*/
hiliteIcons?: Array<Partial<SCImgURL>>;

/**
* AutoChild DynamicForm for configuring the details of this Hilite.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.hiliteFormProperties.
* @type {DynamicForm}
* @default null
*/
hiliteForm?: DynamicForm;

/**
* The title for the Appearance group.
* @type {string}
* @default "Appearance"
*/
appearanceGroupTitle?: string;

/**
* The title for the Target Field(s) picker.
* @type {string}
* @default "Target Field(s)"
*/
targetFieldsItemTitle?: string;

/**
* AutoChild ImgButton that accepts this Hilite and fires the 
*  AdvancedHiliteEditor.callback.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.saveButtonProperties.
* @type {StatefulCanvas}
* @default null
*/
saveButton?: StatefulCanvas;

/**
* The title text for the AdvancedHiliteEditor.saveButton.
* @type {string}
* @default "Save"
*/
saveButtonTitle?: string;

/**
* AutoChild ImgButton that cancels this AdvancedHiliteEditor without saving 
*  any changes, firing the AdvancedHiliteEditor.callback with a null
*  parameter.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.cancelButtonProperties.
* @type {StatefulCanvas}
* @default null
*/
cancelButton?: StatefulCanvas;

/**
* The title text for the AdvancedHiliteEditor.cancelButton.
* @type {string}
* @default "Cancel"
*/
cancelButtonTitle?: string;

/**
* The message to show when the user clicks "Save" without entering any criteria.
* @type {string}
* @default "Enter at least one rule, a color or icon, and a target field, or press 'Cancel' to abandon changes."
*/
invalidHilitePrompt?: string;

/**
* The title text shown in the header bar of this editor's dialog.
* @type {string}
* @default "Advanced Highlight Editor"
*/
title?: string;

/**
* The callback to fire when the AdvancedHiliteEditor.saveButton is clicked.
* @type {Callback}
* @default null
*/
callback?: Callback;







/**
*  Save changes and fire the AdvancedHiliteEditor.callback.
* 
* 
*/
saveHilite(): void;

/**
*  Discard changes and fire the AdvancedHiliteEditor.callback with a null 
*  parameter.
* 
* 
*/
cancelEditing(): void;




}



/**
* Events registerable via Page.setEvent
*/
type PageEvent = 
/**
* Fires repeatedly (every 10 ms by default) when the system is idle (i.e.,
*     not busy running other scripts) after the page is loaded.
*/
"idle" |

/**
* Fires when the page has finished loading. It corresponds to the
*     browser 'load' event normally handled by window.onload.
*/
"load" |

/**
* Fires when the page is exited or unloaded. It corresponds to the
*     browser 'unload' event normally handled by window.onunload.
*/
"unload" |

/**
* Fires when the browser window is resized by the user. It corresponds
*     to the browser 'resize' event normally handled by window.onresize.
*/
"resize" |

/**
* Fires when the left mouse button is pressed on the Page.
*/
"mouseDown" |

/**
* Fires when the right mouse button is pressed on the Page.
*/
"rightMouseDown" |

/**
* Fires when the mouse moves on the Page.
*/
"mouseMove" |

/**
* Fires when the left mouse button released on the Page.
*/
"mouseUp" |

/**
* Fires when the user clicks the mouse on the Page.
*/
"click" |

/**
* Fires when the uesr double-clicks on the Page.
*/
"doubleClick" |

/**
* Fires when the right mouse button is clicked on the page. If your event handler
*     for this event returns false, the native browser context menu will be suppressed.
*     Note: On the Macintosh platform, Command+Click may be used instead
*     of right-button click to trigger a context menu event.
*     On the Opera browser, Ctrl+Shift+Click should be used instead of
*     right-button click.
*/
"showContextMenu" |

/**
* Fires when a user presses a key on the keyboard.
*/
"keyPress" |

/**
* Fires when the Page.getOrientation changes due    to browser-window resize or rotation of a mobile device.
*/
"orientationChange" |

/**
* Fires when the FontLoader completes loading custom fonts.
*/
"fontsLoaded" |

/**
* Fires after a timeout if the FontLoader fails to load all custom
*     fonts.
* 
*  see classMethod:Page.setEvent()
*  see classMethod:Page.clearEvent()
*/
"fontLoadingFailed";






/**
* Supported styles of drawing lines.
*/
type LinePattern = 
/**
* Solid line
*/
"solid" |

/**
* Dotted line
*/
"dot" |

/**
* Dashed line
*/
"dash" |

/**
* Dotted line, with more tightly spaced dots
*/
"shortdot" |

/**
* Dashed line, with shorter, more tightly spaced dashes
*/
"shortdash" |

/**
* Dashed line, with longer, more widely spaced dashes
*/
"longdash";


/**
* Identifier for a key pressed by the user, optionally specifying whether the Shift, Control,
*  and/or Alt keys should be held or not held when the key is pressed, used by various methods.
*/
class KeyIdentifier {


/**
* Name of the key.
* @type {KeyName}
* @default null
*/
keyName?: KeyName;

/**
* If set, whether the Shift key must be held down.
* @type {boolean}
* @default null
*/
shiftKey?: boolean;

/**
* If set, whether the Control key must be held down.
* @type {boolean}
* @default null
*/
ctrlKey?: boolean;

/**
* If set, whether the Alt key must be held down.
* @type {boolean}
* @default null
*/
altKey?: boolean;










}

/**
* A ListGrid that implements the Palette behavior, so it can be used as the source for 
*  drag and drop instantiation of components when combined with an EditContext as 
*  the drop target.
*  
*  Each ListGridRecord can be a PaletteNode.
*/
class ListPalette extends ListGrid implements Palette {
/**
* Creates a new ListPalette
* 
* @param typeCheckedProperties {Partial<ListPalette>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ListPalette} The new instance
*/
static create(typeCheckedProperties? : Partial<ListPalette>, uncheckedProperties?: Object): ListPalette;










// stubs from interface: Palette
// -----------------------------------------------------------------------------------------------------------


/**
* Default EditContext that this palette should use. Palettes generally create components via
*  drag and drop, but may also support creation via double-click or other UI idioms when a
*  defaultEditContext is set.
* @type {EditContext | EditTree | EditPane}
* @default null
*/
defaultEditContext?: EditContext | EditTree | EditPane;

/**
* Whether created components should have their "ID" or "name" property automatically set
*  to a unique value based on the component's type, eg, "ListGrid0".
* @type {boolean}
* @default true
*/
generateNames?: boolean;







/**
*  Sets the default EditContext that this palette should use. Palettes generally create components via
*  drag and drop, but may also support creation via double-click or other UI idioms when a
*  defaultEditContext is set.
* 
* @param {EditContext | EditTree | EditPane} the default EditContext used by this Palette
*/
setDefaultEditContext(defaultEditContext:EditContext | EditTree | EditPane): void;

/**
*  Given a PaletteNode, make an EditNode from it by creating a 
*  EditNode.liveObject from the PaletteNode.defaults
*  and copying presentation properties (eg PaletteNode.title
*  to the editNode.
*  
*  If editNodeProperties is specified as an object on
*  on the paletteNode, each property in this object will also be copied across to
*  the editNode.
* 
* 
* @param {PaletteNode} paletteNode to create from
*/
makeEditNode(paletteNode:PaletteNode): EditNode;




}



/**
* An edge or corner of a rectange, or it's center. Used in APIs such as
*  Canvas.resizeFrom and Canvas.getEventEdge.
*/
type EdgeName = 
/**
* top edge
*/
"T" |

/**
* bottom edge
*/
"B" |

/**
* left edge
*/
"L" |

/**
* right edge
*/
"R" |

/**
* top-left corner
*/
"TL" |

/**
* top-right corner
*/
"TR" |

/**
* bottom-left corner
*/
"BL" |

/**
* bottom-right corner
*/
"BR" |

/**
* center
*/
"C";


/**
* DrawItem subclass to render a connector between two points. If the points are aligned
*  on one axis, this connector will draw as a straight line. If the points are offset on
*  both axes and there is enough space, the connector will by default draw short horizontal 
*  segments from the start and end points, and a diagonal segment connecting the ends of these
*  'tail' segments. Connector style and orientation defaults may be changed through
*  configuration.
*/
class DrawLinePath extends DrawItem {
/**
* Creates a new DrawLinePath
* 
* @param typeCheckedProperties {Partial<DrawLinePath>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DrawLinePath} The new instance
*/
static create(typeCheckedProperties? : Partial<DrawLinePath>, uncheckedProperties?: Object): DrawLinePath;


/**
* The ConnectorStyle controlling the presentation and behavior of this line's 
*  tail segments.
* @type {ConnectorStyle}
* @default "diagonal"
*/
connectorStyle?: ConnectorStyle;

/**
* The ConnectorOrientation controlling the orientation and behavior of this line's 
*  tail segments.
* @type {ConnectorOrientation}
* @default "auto"
*/
connectorOrientation?: ConnectorOrientation;

/**
* Array of control knobs to display for this item. Each KnobType specified in this
*  will turn on UI element(s) allowing the user to manipulate this DrawLinePath. To update the
*  set of knobs at runtime use DrawItem.showKnobs and DrawItem.hideKnobs.
*  
*  DrawLinePath supports the
*  "startPoint", "endPoint", "controlPoint1", and "controlPoint2"
*  
*  knob types.
* @type {Array<Partial<KnobType>>}
* @default null
*/
knobs?: Array<Partial<KnobType>>;

/**
* The mode in which the DrawItem.titleLabel (if shown) is rotated with
*  this draw item.
* @type {TitleRotationMode}
* @default "withLineAlwaysUp"
*/
titleRotationMode?: TitleRotationMode;

/**
* If the DrawItem.titleLabel is showing, should the DrawItem.titleLabelBackground
*  be created and placed behind the titleLabel?
*  
*  This defaults to true for DrawSectors and shapes that are not commonly filled
*  (e.g. DrawLines).
* @type {boolean}
* @default true
*/
showTitleLabelBackground?: boolean;

/**
* Start point of the line
* @type {Point}
* @default "[0,0]"
*/
startPoint?: Point;

/**
* End point of the line
* @type {Point}
* @default "[100,100]"
*/
endPoint?: Point;

/**
* The point at which the leading tail segment joins the connecting center segment.
* @type {Point}
* @default null
*/
controlPoint1?: Point;

/**
* The point at which the trailing tail segment joins the connecting center segment.
*  Has no effect on lines with right angle ConnectorStyles.
* @type {Point}
* @default null
*/
controlPoint2?: Point;

/**
* Length of the horizontal/vertical "tail segments" between the start and end points of 
*  this DrawLinePath and the connecting center segment.
* @type {Distance}
* @default "30"
*/
tailSize?: Distance;

/**
* Starting left coordinate of the line. Overrides left coordinate of DrawLinePath.startPoint if
*  both are set.
* @type {Coordinate}
* @default "0 , IRW"
*/
startLeft?: Coordinate;

/**
* Starting top coordinate of the line. Overrides top coordinate of DrawLinePath.startPoint if
*  both are set.
* @type {Coordinate}
* @default "0 , IRW"
*/
startTop?: Coordinate;

/**
* Ending left coordinate of the line. Overrides left coordinate of DrawLinePath.endPoint if
*  both are set.
* @type {Coordinate}
* @default "0 , IRW"
*/
endLeft?: Coordinate;

/**
* Ending top coordinate of the line. Overrides top coordinate of DrawLinePath.endPoint if
*  both are set.
* @type {Coordinate}
* @default "0 , IRW"
*/
endTop?: Coordinate;

/**
* Style of arrow head to draw at the end of the line or path.
* @type {ArrowStyle}
* @default ""open", IRW"
*/
endArrow?: ArrowStyle;







/**
*  Get the center point of the line path.
* 
*/
getCenter(): Point;

/**
*  Update the startPoint
* 
* 
* @param {Coordinate} left coordinate for start point, in pixels
* @param {Coordinate} top coordinate for start point, in pixels
*/
setStartPoint(left:Coordinate, top:Coordinate): void;

/**
*  Update the endPoint
* 
* 
* @param {Coordinate} left coordinate for end point, in pixels
* @param {Coordinate} top coordinate for end point, in pixels
*/
setEndPoint(left:Coordinate, top:Coordinate): void;

/**
*  Sets the coordinates of the controlPoint1 knob and by extension the coordinates of this
*  DrawLinePath's leading tail segment.
* 
* @param {Coordinate} left coordinate for start point, in pixels
* @param {Coordinate} top coordinate for start point, in pixels
*/
setControlPoint1(left:Coordinate, top:Coordinate): void;

/**
*  Sets the coordinates of the controlPoint2 knob and by extension the coordinates of this
*  DrawLinePath's trailing tail segment.
* 
* @param {Coordinate} left coordinate for start point, in pixels
* @param {Coordinate} top coordinate for start point, in pixels
*/
setControlPoint2(left:Coordinate, top:Coordinate): void;

/**
*  Returns the startPoint, endPoint
* 
*/
getBoundingBox(): Array<Partial<Double>>;

/**
*  Move both the start and end points of the line by a relative amount.
* 
* 
* @param {Distance} change to left coordinate in pixels
* @param {Distance} change to top coordinate in pixels
*/
moveBy(left:Distance, top:Distance): void;

/**
*  Moves the line path such that the DrawLinePath.startPoint ends up at the specified point.
* 
* 
* @param {Integer} new left coordinate in pixels
* @param {Integer} new top coordinate in pixels
*/
moveStartPointTo(left:Integer, top:Integer): void;




}

/**
* The EventCanvas component is a lightweight VLayout subclass for 
*  displaying a CalendarEvent in a CalendarView.
*  
*  Each instance can be CalendarEvent.styleName, and can render a single area,
*  or separate EventCanvas.showHeader and EventCanvas.showBody
*  areas, for the look of a Window.
*  
*  The component's close and context buttons, and any necessary resizers, are
*  shown on EventCanvas.showRolloverControls.
*/
class EventCanvas extends VLayout {
/**
* Creates a new EventCanvas
* 
* @param typeCheckedProperties {Partial<EventCanvas>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {EventCanvas} The new instance
*/
static create(typeCheckedProperties? : Partial<EventCanvas>, uncheckedProperties?: Object): EventCanvas;


/**
* Readonly property indicating whether this is a special ZoneCanvas subclass.
* @type {boolean}
* @default false
*/
isZoneCanvas?: boolean;

/**
* Readonly property dictating whether this is a special IndicatorCanvas 
*  subclass.
* @type {boolean}
* @default false
*/
isIndicatorCanvas?: boolean;

/**
* Renders a header DIV above the main body of the event, an area of limited
*  height, styled to stand out from the main EventCanvas.showBody of the 
*  event, and typically showing a CalendarEvent.name or title - like a Window.
*  This header area can be styled via EventCanvas.headerStyle and the HTML it shows 
*  is retrieved from a call to EventCanvas.getHeaderHTML. 
*  The default is taken from Calendar.showEventHeaders.
* @type {boolean}
* @default null
*/
showHeader?: boolean;

/**
* Renders a body DIV that fills the main area of the canvas, or all of it if no 
*  EventCanvas.showHeader is shown. This area typically displays an 
*  CalendarEvent.description. This area can be styled via 
*  EventCanvas.bodyStyle and the HTML it shows is retrieved
*  from a call to EventCanvas.getBodyHTML. The default is taken
*  from Calendar.showEventDescriptions.
* @type {boolean}
* @default null
*/
showBody?: boolean;

/**
* Indicates the orientation of the event in its containing view. Affects drag and resize
*  orientation and which edges of the canvas are available for resizing.
* @type {boolean}
* @default true
*/
vertical?: boolean;

/**
* The CSS class for this EventCanvas. Defaults to the style on 
*  CalendarEvent.styleName, if specified, or on the
*  Calendar.eventStyleName otherwise.
*  
*  Also see EventCanvas.headerStyle and EventCanvas.bodyStyle.
* @type {CSSStyleName}
* @default null
*/
styleName?: CSSStyleName;

/**
* The CalendarEvent associated with this EventCanvas.
* @type {CalendarEvent}
* @default null
*/
event?: CalendarEvent;

/**
* The Calendar in which this EventCanvas is being rendered.
* @type {Calendar}
* @default null
*/
calendar?: Calendar;

/**
* The CalendarView in which this EventCanvas is being rendered.
* @type {CalendarView}
* @default null
*/
calendarView?: CalendarView;

/**
* When set to true, the EventCanvas.getHeaderHTML for the
*  associated event is not rendered inside the eventCanvas itself.
*  
*  Instead, it is rendered in it's own EventCanvas.label and shown 
*  as a peer of this eventCanvas, immediately outside of it.
* @type {boolean}
* @default null
*/
showLabel?: boolean;

/**
* When EventCanvas.showLabel is true, this autoChild is 
*  used to display the EventCanvas.getHeaderHTML, adjacent to
*  this eventCanvas.
* @type {Label}
* @default null
*/
label?: Label;

/**
* When set to true, shows the EventCanvas.gripper component, which snaps,
*  centered, to the top edge of the eventCanvas and can be used to move it with the mouse.
* @type {boolean}
* @default null
*/
showGripper?: boolean;

/**
* The source for the icon displayed as the "gripper" that snaps to the top of an event 
*  canvas and allows an event to be dragged with the mouse.
* @type {SCImgURL}
* @default null
*/
gripperIcon?: SCImgURL;

/**
* When EventCanvas.showGripper is true, this is the component that will
*  be rendered adjacent to the canvas and allow the canvas to be moved with the mouse.
* @type {Img}
* @default null
*/
gripper?: Img;

/**
* Whether the EventCanvas.showHeader should autosize vertically to 
*  display all contents. If true, the header will wrap to multiple lines. If false, the
*  header will be sized according to the specified EventCanvas.headerHeight,
*  or to the full height of the canvas is EventCanvas.showBody is false.
* @type {boolean}
* @default null
*/
headerWrap?: boolean;

/**
* The height for the EventCanvas.showHeader, when 
*  EventCanvas.headerWrap is false and 
*  EventCanvas.showBody is true. If showBody is false, the
*  header area fills the canvas.
* @type {Integer}
* @default null
*/
headerHeight?: Integer;

/**
* CSS class for the EventCanvas.showHeader of the EventCanvas.
*  If unset, defaults to the EventCanvas.styleName with the suffix
*  "Header".
* @type {CSSStyleName}
* @default null
*/
headerStyle?: CSSStyleName;

/**
* CSS class for the EventCanvas.showBody of the EventCanvas.
*  If unset, defaults to the EventCanvas.styleName with the suffix
*  "Body".
* @type {CSSStyleName}
* @default null
*/
bodyStyle?: CSSStyleName;

/**
* When set to the default value of true, this attribute causes a set of components to be
*  shown when the mouse rolls over this EventCanvas. These components include the
*  Calendar.eventCanvasCloseButton and 
*  Calendar.eventCanvasContextButton buttons, the latter's 
*  Calendar.eventCanvasContextMenu and the images used for 
*  drag-resizing.
*  
*  Using rollover controls is more efficient that showing static buttons in each 
*  eventCanvas, so this is the default behavior. See 
*  Calendar.useEventCanvasRolloverControls for the alternative.
* @type {boolean}
* @default true
*/
showRolloverControls?: boolean;







/**
*  Assigns a new CalendarEvent to this EventCanvas, including updates to 
*  drag, style and EventCanvas.showRolloverControls properties.
* 
* 
* @param {CalendarEvent} the new event to apply to this EventCanvas
* @param {CSSStyleName=} optional CSS class to apply to this EventCanvas
* @param {CSSStyleName=} optional separate CSS class to apply to the                   EventCanvas.showHeader.
* @param {CSSStyleName=} optional separate CSS class to apply to the                   EventCanvas.showBody.
*/
setEvent(event:CalendarEvent, styleName?:CSSStyleName, headerStyle?:CSSStyleName, bodyStyle?:CSSStyleName): void;

/**
*  Returns the HTML to show in the header of this EventCanvas. The default implementation 
*  returns the Calendar.nameField of the current 
*  EventCanvas.event.
* 
* 
*/
getHeaderHTML(): HTMLString;

/**
*  Return the HTML to show in the body of this EventCanvas. The default implementation 
*  calls Calendar.getEventBodyHTML, which returns the value of the 
*  Calendar.descriptionField for the current 
*  CalendarEvent.
* 
* 
*/
getBodyHTML(): HTMLString;

/**
*  Returns the HTML to show in the EventCanvas as a whole. By default, this method 
*  generates one or two styled DIVs, depending on the values of 
*  EventCanvas.showHeader and EventCanvas.showBody.
* 
* 
*/
getInnerHTML(): HTMLString;




}



/**
* A number representing a horizontal or vertical offset from the origin of a
*  DrawPane in the DrawPane. DrawRect.left is
*  an example of a DrawItem attribute that's a Coordinate.
* 
*  A Coordinate is not limited to integers except for
*  DrawingType "vml".
*/
type Coordinate = Double;


/**
* Button used to display a hierarchical Menu group for representing / selecting tree data.
*  
*  Important Note: this class should not be used directly - it is exposed purely for
*  i18nMessages
*/
class TreeMenuButton extends MenuButton {
/**
* Creates a new TreeMenuButton
* 
* @param typeCheckedProperties {Partial<TreeMenuButton>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {TreeMenuButton} The new instance
*/
static create(typeCheckedProperties? : Partial<TreeMenuButton>, uncheckedProperties?: Object): TreeMenuButton;


/**
* Title for this button. If not specified, the selected value from the tree will
*  be displayed instead.
* @type {string}
* @default null
*/
title?: string;

/**
* If TreeMenuButton.title is null, this value will be displayed as a title 
*  when the user has not selected any value from the hierachichal menu.
* @type {HTMLString}
* @default "Choose a value"
*/
unselectedTitle?: HTMLString;

/**
* If this button's menu (or any of its submenus) are empty, this property can be used
*  to specify the message to display (as a disabled item) in the empty menu.
* @type {HTMLString}
* @default null
*/
emptyMenuMessage?: HTMLString;

/**
* If TreeMenuButton.title is null, when the user selects an item, should we 
*  show the full path to the item, or just the item's title as the button's title?
* @type {boolean}
* @default false
*/
showPath?: boolean;

/**
* If TreeMenuButton.showPath is true, this property specifies what will 
*  appear between the folders in the selected value's path.
* @type {HTMLString}
* @default "&amp;nbsp;&amp;gt;&amp;nbsp;"
*/
pathSeparatorString?: HTMLString;

/**
* Base style to apply to the selected path within the menu. (The "over" version of this
*  style should also be defined in the stylesheet applied to this widget).
* @type {CSSStyleName}
* @default "treeMenuSelected"
*/
selectedBaseStyle?: CSSStyleName;

/**
* For a TreeMenuButton that uses a DataSource, these properties will be passed to
*  the automatically-created ResultTree. This can be used for various customizations such as
*  modifying the automatically-chosen Tree.parentIdField.
* @type {Tree}
* @default null
*/
dataProperties?: Tree;

/**
* Widget class for the menu created by this button. The default is 
*  SelectionTreeMenu.
* @type {SCClassName}
* @default "SelectionTreeMenu"
*/
treeMenuConstructor?: SCClassName;

/**
* AutoChild menu displayed when the button is clicked.
* @type {Menu}
* @default null
*/
treeMenu?: Menu;










}

/**
* Algorithms to use when hashing field values (see DataSourceField.storeWithHash)
*/
type HashAlgorithm = 
/**
* Message Digest algorithm 5
*/
"MD5" |

/**
* Secure Hashing Algorithm (SHA-1)
*/
"SHA" |

/**
* bcrypt is a password hashing function designed by Niels Provos and David MaziÃ¨res, based on the Blowfish cipher
*/
"bcrypt";


/**
* Supported regression algorithms for fitting the data points of a scatter plot.
*/
type RegressionLineType = 
/**
* linear regression
*/
"line" |

/**
* polynomial regression
*/
"polynomial";










/**
* Different styles of selection that a list, etc. might support
*/
type SelectionStyle = 
/**
* don't select at all
*/
"none" |

/**
* select only one item at a time
*/
"single" |

/**
* select one or more items
*/
"multiple" |

/**
* select one or more items as a toggle      so you don't need to hold down control keys to select
*                  more than one object
*/
"simple";




/**
* A flexible, high-speed table that offers consistent cross-platform sizing, clipping, and events.
*/
class GridRenderer extends Canvas {
/**
* Creates a new GridRenderer
* 
* @param typeCheckedProperties {Partial<GridRenderer>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {GridRenderer} The new instance
*/
static create(typeCheckedProperties? : Partial<GridRenderer>, uncheckedProperties?: Object): GridRenderer;


/**
* Total number of rows in the grid.
* 
*  NOTE: in order to create a valid grid, you must either provide a totalRows value or implement
*  getTotalRows()
* @type {number}
* @default "0"
*/
totalRows?: number;

/**
* Whether all rows should be drawn all at once, or only rows visible in the viewport.
*  
*  Drawing all rows causes longer initial rendering time, but allows smoother vertical scrolling.
*  With a very large number of rows, showAllRows will become too slow.
*  
*  See also GridRenderer.drawAheadRatio and GridRenderer.drawAllMaxCells.
* @type {boolean}
* @default false
*/
showAllRows?: boolean;

/**
* When incremental rendering is switched on and there are variable record heights, the virtual
*  scrolling mechanism manages the differences in scroll height calculations due to the
*  unknown sizes of unrendered rows to make the scrollbar and viewport appear correctly.
*  
*  virtualScrolling is switched on automatically when fixedRowHeights is false but you should 
*  switch it on any time there are variable record heights.
* @type {boolean}
* @default null
*/
virtualScrolling?: boolean;

/**
* Whether all columns should be drawn all at once, or only columns visible in the viewport.
*  
*  Drawing all columns causes longer initial rendering time, but allows smoother horizontal
*  scrolling. With a very large number of columns, showAllColumns will become too slow.
* @type {boolean}
* @default false
*/
showAllColumns?: boolean;

/**
* If drawing all rows would cause less than drawAllMaxCells cells to be rendered,
*  the full dataset will instead be drawn even if ListGrid.showAllRecords
*  is false and the viewport size and GridRenderer.drawAheadRatio setting would normally have caused
*  incremental rendering to be used.
*  
*  The drawAllMaxCells setting prevents incremental rendering from being used in
*  situations where it's really unnecessary, such as a 40 row, 5 column dataset (only 200
*  cells) which happens to be in a grid with a viewport showing only 20 or so rows.
*  Incremental rendering causes a brief "flash" during scrolling as the visible portion of the
*  dataset is redrawn, and a better scrolling experience can be obtained in this situation by
*  drawing the entire dataset up front, which in this example would have negligible effect on
*  initial draw time.
*  
*  drawAllMaxCells:0 disables this features. You may want to disable this feature
*  if performance is an issue and:
*  
*  
* - you are very frequently redraw a grid
*  
* - you do a lot of computation when rendering each cell (eg formulas)
*  
* - you are showing many grids on one screen and the user won't scroll most of them
*  
* @type {Integer}
* @default "250"
*/
drawAllMaxCells?: Integer;

/**
* If true, if the user clicks on the scroll buttons at the end of the track or clicks once on
*  the scroll track, there will be an instant redraw of the grid content so that the user
*  doesn't see any blank space. For drag scrolling or other types of scrolling, the
*  GridRenderer.scrollRedrawDelay applies.
* @type {boolean}
* @default true
*/
instantScrollTrackRedraw?: boolean;

/**
* While drag scrolling in an incrementally rendered grid, time in milliseconds to wait
*  before redrawing, after the last mouse movement by the user. This delay may be 
*  separately customized for mouse-wheel scrolling via GridRenderer.scrollWheelRedrawDelay.
*  
*  See also
*  GridRenderer.instantScrollTrackRedraw for cases where this delay is skipped.
*  
*  NOTE: In Browser.isTouch,
*  GridRenderer.touchScrollRedrawDelay is used instead.
* @type {Integer}
* @default "0"
*/
scrollRedrawDelay?: Integer;

/**
* Like GridRenderer.scrollRedrawDelay, but applies when the component is being
*  drag-scrolled (via a scrollbar). This value is typically set higher than GridRenderer.scrollRedrawDelay
*  to avoid too many concurrent fetches to the server for ResultSet-backed components
*  since it's quite easy to induce such a case with a scrollbar and a grid bound to a large databaset.
* @type {Integer}
* @default "75"
*/
dragScrollRedrawDelay?: Integer;

/**
* In Browser.isTouch, the time in milliseconds to wait after a scroll
*  before redrawing. In non-touch browsers, the GridRenderer.scrollRedrawDelay
*  or GridRenderer.scrollWheelRedrawDelay is used instead.
* @type {Integer}
* @default "0"
*/
touchScrollRedrawDelay?: Integer;

/**
* While scrolling an incrementally rendered grid, using the mouseWheel, time in 
*  milliseconds to wait before redrawing, after the last mouseWheel movement by the user. 
*  If not specified GridRenderer.scrollRedrawDelay will be used as a default for both
*  drag scrolling and mouseWheel scrolling.
*  
*  See also
*  GridRenderer.instantScrollTrackRedraw for cases where this delay is skipped.
* @type {Integer}
* @default null
*/
scrollWheelRedrawDelay?: Integer;

/**
* How far should we render rows ahead of the currently visible area? This is expressed as a
*  ratio from viewport size to rendered area size.
*  
*  Tweaking drawAheadRatio allows you to make tradeoffs between continuous scrolling speed vs
*  initial render time and render time when scrolling by large amounts.
*  
*  NOTE: Only applies when showAllRows is false.
* @type {Float}
* @default "2.0"
*/
drawAheadRatio?: Float;

/**
* Alternative to GridRenderer.drawAheadRatio, to be used when the user
*  is rapidly changing the grids viewport (for example drag scrolling through the grid).
*  If unspecified GridRenderer.drawAheadRatio will be used in all cases
* @type {Float}
* @default "2.0"
*/
quickDrawAheadRatio?: Float;

/**
* The default height of each row in pixels.
* @type {number}
* @default "20"
*/
cellHeight?: number;

/**
* Should we vertically clip cell contents, or allow rows to expand vertically to show all
*  contents?
*  
*  If we allow rows to expand, the row height as derived from
*  GridRenderer.getRowHeight or the
*  default GridRenderer.cellHeight is treated as a minimum.
* @type {boolean}
* @default true
*/
fixedRowHeights?: boolean;

/**
* Should we horizontally clip cell contents, or allow columns to expand horizontally to
*  show all contents?
* 
*  If we allow columns to expand, the column width is treated as a minimum.
* @type {boolean}
* @default true
*/
fixedColumnWidths?: boolean;

/**
* If true, make columns only wide enough to fit content, ignoring any widths specified.
*  Overrides fixedFieldWidths.
* @type {boolean}
* @default false
*/
autoFit?: boolean;

/**
* Should content within cells be allowed to wrap?
* @type {boolean}
* @default false
*/
wrapCells?: boolean;

/**
* The amount of empty space, in pixels, surrounding each value in its cell.
* @type {number}
* @default "2"
*/
cellPadding?: number;

/**
* If true, rightMouseDown events will fire 'selectOnRightMouseDown()' for the appropriate cells.
* @type {boolean}
* @default true
*/
canSelectOnRightMouse?: boolean;

/**
* If true, cellHover and rowHover events will fire and then a hover will be shown (if not
*  canceled) when the user leaves the mouse over a row / cell unless the corresponding field has
*  ListGridField.showHover set to false. If unset or null, the hover will be
*  shown if the corresponding field has showHover:true. If false, then hovers are disabled.
*  
*  Note that standard hovers override GridRenderer.showClippedValuesOnHover. Thus,
*  to enable clipped value hovers, canHover must be unset or null and the corresponding field
*  must have showHover unset or null as well.
* @type {boolean}
* @default null
*/
canHover?: boolean;

/**
* If true, and canHover is also true, when the user hovers over a cell, hover text will pop up
*  next to the mouse. The contents of the hover is determined by GridRenderer.cellHoverHTML.
* @type {boolean}
* @default null
*/
showHover?: boolean;

/**
* If true and a cell's value is clipped, then a hover containing the full cell value is
*  enabled.
*  
*  Note that standard cell hovers override clipped value hovers. Thus, to enable clipped value
*  hovers, GridRenderer.canHover must be unset or null and the corresponding field must have
*  ListGridField.showHover unset or null as well.
* @type {boolean}
* @default null
*/
showClippedValuesOnHover?: boolean;

/**
* The base name for the CSS class applied to cells. This style will have "Dark", 
*  "Over", "Selected", or "Disabled" appended to it according to the state of the cell.
* @type {CSSStyleName}
* @default "cell"
*/
baseStyle?: CSSStyleName;

/**
* Whether alternating rows (or blocks of rows, depending
*  on GridRenderer.alternateRowFrequency) should be drawn in alternating styles,
*  in order to create a "ledger" effect for easier reading.
*  
*  If enabled, the cell style for alternate rows will have the
*  GridRenderer.alternateRowSuffix appended to it.
*  See also GridRenderer.alternateColumnStyles.
* @type {boolean}
* @default false
*/
alternateRowStyles?: boolean;

/**
* Suffix to append to GridRenderer.alternateRowStyles.
*  Note that if GridRenderer.alternateColumnStyles is enabled, cells which fall
*  into both an alternate row and column will have both suffixes appended - for
*  example "cellDarkAltCol".
* @type {string}
* @default "Dark"
*/
alternateRowSuffix?: string;

/**
* The number of consecutive rows to draw in the same style before alternating, when
*  GridRenderer.alternateRowStyles is true.
* @type {number}
* @default "1"
*/
alternateRowFrequency?: number;

/**
* Whether alternating columns (or blocks of columns, depending
*  on GridRenderer.alternateColumnFrequency) should be drawn in alternating styles,
*  in order to create a vertical "ledger" effect for easier reading.
*  
*  If enabled, the cell style for alternate rows will have the
*  GridRenderer.alternateColumnSuffix appended to it.
*  See also GridRenderer.alternateRowStyles.
* @type {boolean}
* @default false
*/
alternateColumnStyles?: boolean;

/**
* Suffix to append to GridRenderer.alternateColumnStyles.
*  Note that if GridRenderer.alternateRowStyles is enabled, cells which fall
*  into both an alternate row and column will have both suffixes appended - for
*  example "cellDarkAltCol".
* @type {string}
* @default "AltCol"
*/
alternateColumnSuffix?: string;

/**
* The number of consecutive columns to draw in the same style before alternating, when
*  GridRenderer.alternateColumnStyles is true.
* @type {number}
* @default "1"
*/
alternateColumnFrequency?: number;

/**
* Value to show in empty cells (when getCellValue returns null).
* @type {string}
* @default "&amp;nbsp;"
*/
emptyCellValue?: string;

/**
* Indicates whether the text of the emptyMessage property should be displayed if no data is
*  available.
* @type {boolean}
* @default true
*/
showEmptyMessage?: boolean;

/**
* The string to display in the body of a listGrid with an empty data array, if
*  showEmptyMessage is true.
* @type {string}
* @default "No data to display"
*/
emptyMessage?: string;

/**
* CSS styleName for the table as a whole if we're showing the empty message
* @type {CSSStyleName}
* @default null
*/
emptyMessageTableStyle?: CSSStyleName;

/**
* The CSS style name applied to the emptyMessage string if displayed.
* @type {CSSStyleName}
* @default null
*/
emptyMessageStyle?: CSSStyleName;

/**
* Indicates whether the text of the offlineMessage property should be displayed if no data
*  is available because we are offline and there is no suitable cached response
* @type {boolean}
* @default true
*/
showOfflineMessage?: boolean;

/**
* The CSS style name applied to the offlineMessage string if displayed.
* @type {CSSStyleName}
* @default null
*/
offlineMessageStyle?: CSSStyleName;

/**
* The string to display in the body of a listGrid with an empty data array, if
*  showOfflineMessage is true and the data array is empty because we are offline and there
*  is no suitable cached response
* @type {string}
* @default "No data available while offline"
*/
offlineMessage?: string;

/**
* Note: This property only has an effect in Internet Explorer
*  
*  Advanced property to improve performance for dynamic styling of gridRenderer cells in
*  Internet Explorer, at the expense of slightly slower initial drawing, and some 
*  limitations on supported styling options.
*  
*  fastCellUpdates speeds up the dynamic styling system used by rollovers,
*  selections, and custom styling that calls GridRenderer.refreshCellStyle, at
*  the cost of slightly slower draw() and redraw() times.
*  
*  Notes:
*  
*  
* - When this property is set, ListGrid cells may be styled using the 
*    ListGrid.tallBaseStyle. See ListGrid.getBaseStyle for 
*    more information.
*  
* - If any cell styles specify a a background image URL, the URL will be resolved relative
*    to the page location rather than the location of the CSS stylesheet. This means cell
*    styles with a background URL should either supply a fully qualified path, or the
*    background image media should be made available at a second location for IE.
*  
* - fastCellUpdates will not work if the styles involved are in an external stylesheet loaded
*    from a remote host. Either the stylesheet containing cell styles needs to be loaded
*    from the same host as the main page, or the cell styles need to be inlined in the html 
*    of the bootstrap page.
*  
* - fastCellUpdates will not work if the css styles for cells are defined in
*    a .css file loaded via @import. Instead the .css
*    file should be loaded via a &lt;link ...&gt; tag.
*  
* @type {boolean}
* @default true
*/
fastCellUpdates?: boolean;

/**
* Should drag-and-drop operations snap the dragged object into line with the nearest cell?
* @type {boolean}
* @default false
*/
snapToCells?: boolean;

/**
* If true, snap-to-cell drops will snap the dropped object inside the selected cell's border.
*  If false, snap-to-cell drops will snap the dropped object to the edge of the selected cell, 
*  regardless of borders
* @type {boolean}
* @default false
*/
snapInsideBorder?: boolean;

/**
* Denotes the name of a property that can be set on records to display a custom style.
*  For example if this property is set to "customStyle", setting 
*  record.customStyle to a css styleName will cause the record in question to
*  render out with that styling applied to it. Note that this will be a static 
*  style - it will not be modified as the state of the record (selected / over etc) changes.
* @type {"CustomStyle"}
* @default "string"
*/
recordCustomStyleProperty?: "CustomStyle";

/**
* Should the "Selected" style be applied to selected records?
* @type {boolean}
* @default true
*/
showSelectedStyle?: boolean;

/**
* If GridRenderer.showHover is true, should cell hover HTML be displayed on disabled cells?
* @type {boolean}
* @default false
*/
showHoverOnDisabledCells?: boolean;







/**
*  Setter for GridRenderer.fastCellUpdates. Has no effect in browsers other than
*  Internet Explorer.
* 
* @param {boolean} whether to enable fastCellUpdates.
*/
setFastCellUpdates(fastCellUpdates:boolean): void;

/**
*  Is the value in a given cell clipped?
* 
* @param {number} row number of the cell
* @param {number} column number of the cell
*/
cellValueIsClipped(rowNum:number, colNum:number): boolean;

/**
*  Return the total number of rows in the grid.
* 
*  NOTE: in order to create a valid grid, you must either provide a totalRows value or implement
*  getTotalRows()
* 
* 
*/
getTotalRows(): number;

/**
*  Sets the width of a single column.
* 
* 
* @param {number} the number of the column to resize
* @param {number} the new width
*/
setColumnWidth(colNum:number, newWidth:number): void;

/**
* 
*  Sets the width of all columns in the grid.
* 
* 
* @param {Array<any>} array of new widths - one for each column.
*/
setColumnWidths(newWidths:Array<any>): void;

/**
*  When using GridRenderer.getRowSpan, returns the row number where a row-spanning cell
*  starts.
*  
*  For example, if row 2 col 0 spans 3 cells, getCellStartRow() for row 2 col 0,
*  row 3 col 0, row 4 col 0 will all return 2, because that's the row when spanning starts.
* 
* 
* @param {Integer} row number of cell for which the start row should be returned
* @param {Integer} column number of cell for which the start row should be returned
*/
getCellStartRow(rowNum:Integer, colNum:Integer): Integer;

/**
*  When using GridRenderer.getRowSpan, returns the number of cells spanned by the cell
*  at the given coordinates.
*  
*  If the passed coordinates are in the middle of a series of spanned cells, the row span of
*  the spanning cell is returned. For example, if row 2 col 0 spans 3 cells, calls to
*  getCellRowSpan() for row 2 col 0, row 3 col 0, row 4 col 0 will all return 3.
*  
*  This method returns row span information for the current rendered cells. In contrast, if
*  the grid is about to be redrawn, a call to getRowSpan() may return row span
*  values for how the grid is about to be drawn. Also, user-provided getRowSpan() functions
*  are not required to operate properly when called outside of the grid rendering loop.
*  
*  Note: This method is a utility method for developers - it is not called
*  directly by the grid rendering path and therefore is not intended for override. To
*  set up custom row-spanning behavior, override GridRenderer.getRowSpan instead.
* 
* 
* @param {Integer} row number of cell to return the row span for
* @param {Integer} column number of cell to return the row span for
*/
getCellRowSpan(rowNum:Integer, colNum:Integer): Integer;

/**
*  Return the CSS class for a cell. By default this method has the following implementation:
*  - return any custom style for the record (GridRenderer.recordCustomStyleProperty) 
*   if defined.
*  - create a style name based on the result of GridRenderer.getBaseStyle and the 
*   state of the record using the rules described in cellStyleSuffixes.
*  
*  Cell Styles are customizable by:
*  
*  
* - attaching a custom style to a record by setting 
*   record[this.recordCustomStyleProperty] to some valid CSS style name. 
*  
* - modifying the base style returned by getBaseStyle() [see that method for further 
*    documentation on this]
*  
* - overriding this function
*  
*  
*  In addition to this, GridRenderer.getCellCSSText may be overriden to
*  provide custom cssText to apply on top of the styling attributes derived from the named
*  style.
*  
* 
* 
* @param {ListGridRecord} record object for this row and column
* @param {number} number of the row
* @param {number} number of the column
*/
getCellStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Whether this cell should be considered enabled. Affects whether events will fire for the
*  cell, and the default styling behavior in getCellStyle. 
* 
* 
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellIsEnabled(rowNum:number, colNum:number): boolean;

/**
*  Refresh the styling of an individual cell without redrawing the grid.
*  
*  The cell's CSS class and CSS text will be refreshed, to the current values returned by
*  getCellStyle() and getCellCSSText() respectively.
*  
*  The cell's contents (as returned by getCellValue()) will not be refreshed. To
*  refresh both styling and contents, call refreshCell() instead.
* 
* 
* @param {number} row number of cell to refresh
* @param {number} column number of cell to refresh
*/
refreshCellStyle(rowNum:number, colNum:number): void;

/**
*  Refresh an individual cell without redrawing the grid.
*  
*  The cell's value, CSS class, and CSS text will be refreshed, to the current values returned
*  by getCellValue(), getCellStyle() and getCellCSSText() respectively.
* 
* 
* @param {number} row number of cell to refresh
* @param {number} column number of cell to refresh
*/
refreshCell(rowNum:number, colNum:number): void;

/**
*  Refresh an entire row of cells without redrawing the grid.
*  
*  The cells' values, CSS classes, and CSS text will be refreshed, to the current values
*  returned by getCellValue(), getCellStyle() and getCellCSSText() respectively.
* 
* 
* @param {number} row number of cell to refresh
*/
refreshRow(rowNum:number): void;

/**
*  Returns the page offsets and size of the cell at the passed row and column. If auto-sizing 
*  is enabled, sizes are not definitive until the grid has finished drawing, so calling this 
*  method before drawing completes will return the configured column sizes.
* 
* @param {number} row index of the cell
* @param {number} column index of the cell
*/
getCellPageRect(rowNum:number, colNum:number): Array<Partial<Integer>>;

/**
*  Return the left coordinate (in local coordinate space) of a particular column.
* 
* @param {Integer} number of the column
*/
getColumnLeft(colNum:Integer): Integer;

/**
*  Return the left coordinate for a given column number as a GLOBAL coordinate
* 
* @param {Integer} number of the column
*/
getColumnPageLeft(colNum:Integer): Integer;

/**
*  Return the width of a particular column.
* 
* @param {number} number of the column.
*/
getColumnWidth(colNum:number): number;

/**
*  Returns the top coordinate for a given row number, relative to the top of body content. Use
*  GridRenderer.getRowPageTop for a page-relative coordinate.
*  
*  This method is reliable only for rows that are currently drawn, which is generally only rows
*  that are visible in the viewport. If row heights vary (see fixedRowHeights),
*  coordinates for rows that are not currently shown are rough approximations.
* 
* 
* @param {Integer} */
getRowTop(rowNum:Integer): Integer;

/**
*  Returns the Y-coordinate for a given row number as a page-relative coordinate. See
*  GridRenderer.getRowTop.
* 
* 
* @param {Integer} */
getRowPageTop(rowNum:Integer): Integer;

/**
*  Returns the row number of the most recent mouse event.
* 
* @param {Integer=} optional y-coordinate to obtain row number, in lieu of the y            coordinate of the last mouse event
*/
getEventRow(y?:Integer): Integer;

/**
*  Returns the column number of the most recent mouse event.
* 
* @param {Integer=} optional x-coordinate to obtain column number for, in lieu of the x            coordinate of the last mouse event
*/
getEventColumn(x?:Integer): Integer;

/**
*  	Returns the nearest row to the event coordinates
* 
*/
getNearestRowToEvent(): void;

/**
*  	Returns the nearest column to the event coordinates
* 
*/
getNearestColToEvent(): void;

/**
*  Get the rows that are currently visible in the viewport, as an array of 
*  [firstRowNum, lastRowNum]. 
*  
*  If the grid contains no records, will return [-1,-1]. Will also return [-1,-1] if called at
*  an invalid time (for example, data is in the process of being fetched - see
*  ResultSet.lengthIsKnown).
*  
* 
*/
getVisibleRows(): Array<Partial<Integer>>;

/**
*  Get the rows that are currently drawn (exist in the DOM), as an array of [firstRowNum,
*  lastRowNum]. 
*  
*  The drawn rows differ from the GridRenderer.getVisibleRows because of
*  GridRenderer.drawAheadRatio. The drawn rows are the appropriate range to consider if
*  you need to, eg, using GridRenderer.refreshCell to update all the cells in a column.
*  
*  If the grid is undrawn or the GridRenderer.emptyMessage is currently shown, returns
*  [null,null];
* 
* 
*/
getDrawnRows(): Array<any>;

/**
*  Return the record that holds the value for this cell.
*  
*  Implementing getCellRecord is optional: the actual HTML placed into each
*  grid cell comes from getCellValue, and a valid grid can be created without any
*  notion of "records" at all.
*  
*  If you do implement getCellRecord, the value you return is passed to you as the
*  "record" parameter in other methods.
* 
* 
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getCellRecord(rowNum:number, colNum:number): object;

/**
*  Return the HTML to display in this cell. Implementing this is required to get a non-empty
*  grid.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getCellValue(record:ListGridRecord, rowNum:number, colNum:number): string;

/**
*  Given a record displayed in this grid, find and return the rowNum in which the record
*  appears.
*  
*  As with GridRenderer.getCellRecord implementing this method is optional as a valid
*  grid may be created without any notion of records.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
*/
findRowNum(record:ListGridRecord): number;

/**
*  Given a record displayed in this grid, find and return the colNum in which the record
*  appears.
*  
*  As with GridRenderer.getCellRecord implementing this method is optional as a valid
*  grid may be created without any notion of records, or records may not be displayed in a
*  single column (as with the ListGrid class where each record is 
*  displayed in an entire row. 
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
*/
findColNum(record:ListGridRecord): number;

/**
*  Return the base stylename for this cell. Default implementation just returns this.baseStyle.
*  See ListGrid.getCellStyle for a general discussion of how to style cells.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getBaseStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Return CSS text for styling this cell, which will be applied in addition to the CSS class
*  for the cell, as overrides.
*  
*  "CSS text" means semicolon-separated style settings, suitable for inclusion in a CSS
*  stylesheet or in a STYLE attribute of an HTML element.
*  
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getCellCSSText(record:ListGridRecord, rowNum:number, colNum:number): string;

/**
*  Return the height this row should be. Default is this.cellHeight. If
*  GridRenderer.fixedRowHeights is false, the row may be rendered taller than this
*  specified size.
*  
*  If records will be variable height,
*  you should switch on GridRenderer.virtualScrolling.
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number
*/
getRowHeight(record:ListGridRecord, rowNum:number): number;

/**
*  Return how many rows this cell should span. Default is 1. 
*  
*  NOTE: if using horizontal incremental rendering, getRowSpan() may be called for
*  a rowNum in the middle of a spanning cell, and should return the remaining span from
*  that rowNum onward.
*  
*  NOTE: if a cell spans multiple rows, getCellRecord/Style/etc will be called with the topmost
*  row coordinates only.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getRowSpan(record:ListGridRecord, rowNum:number, colNum:number): number;

/**
*  Called when the mouse pointer leaves a cell
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellOut(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when the mouse pointer enters a cell
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellOver(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when the mouse pointer leaves a row
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowOut(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when the mouse pointer enters a row
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowOver(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a cell receives a contextclick event.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellContextClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a row receives a contextclick event.
* 
* @param {ListGridRecord} cell record as returned by getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowContextClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a cell receives a mousedown event.
*  
* 
* @param {ListGridRecord} cell record as returned by getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellMouseDown(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a row receives a mousedown event.
*  
* 
* @param {ListGridRecord} record object returned from 'getCellRecord()'
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowMouseDown(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a cell receives a mouseup event.
* 
* 
* @param {ListGridRecord} Record object (retrieved from getCellRecord(rowNum, colNum))
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellMouseUp(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a row receives a mouseup event.
* 
* 
* @param {ListGridRecord} Record object returned from getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowMouseUp(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a cell receives a click event.
* 
* 
* @param {ListGridRecord} Record object returned from getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a cell receives a double click event.
* 
* 
* @param {ListGridRecord} Record object returned from getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellDoubleClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a row receives a click event.
* 
* 
* @param {ListGridRecord} Record object returned from getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when a row receives a double click event.
* 
* 
* @param {ListGridRecord} Record object returned from getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowDoubleClick(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when the mouse hovers over a cell if this.canHover is true. 
*  Returning false will suppress the hover text from being shown if this.showHover is true.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellHover(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Optional stringMethod to fire when the user hovers over a cell and the value is clipped.
*  If this.showClippedValuesOnHover is true, the default behavior is to show a hover canvas
*  containing the HTML returned by cellValueHoverHTML(). Return false to suppress this default
*  behavior.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellValueHover(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  Called when the mouse hovers over a row if this.canHover is true. 
*  Returning false will suppress the hover text from being shown if this.showHover is true.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
rowHover(record:ListGridRecord, rowNum:number, colNum:number): boolean;

/**
*  StringMethod to dynamically assemble an HTML string to show in a hover window over the
*  appropriate cell/record when this.canHover and this.showHover are both true.
*  Called when the mouse hovers over a cell.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
cellHoverHTML(record:ListGridRecord, rowNum:number, colNum:number): HTMLString;

/**
*  Returns the HTML that is displayed by the default cellValueHover handler. Return null or
*  an empty string to cancel the hover.
*  
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord()
* @param {number} row number for the cell
* @param {number} column number of the cell
* @param {HTMLString} the HTML that would have been displayed by default
*/
cellValueHoverHTML(record:ListGridRecord, rowNum:number, colNum:number, defaultHTML:HTMLString): HTMLString;

/**
*  StringMethod to dynamically create a Canvas-based component to show as a hover window 
*  over the appropriate cell/record when this.canHover and this.showHover are both true and
*  when an override of getCellHoverComponent() is present.
*  Called when the mouse hovers over a cell.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getCellHoverComponent(record:ListGridRecord, rowNum:number, colNum:number): Canvas;

/**
*  Called when (row-based) selection changes within this grid. Note this method fires for
*  each record for which selection is modified - so when a user clicks inside a grid this
*  method will typically fire twice (once for the old record being deselected, and once for
*  the new record being selected).
*  
*  NOTE: For updating other components based on selections or triggering selection-oriented
*  events within an application, see the
*  DataBoundComponent.selectionUpdated event
*  which is likely more suitable. Calls to Selection.getSelection
*  from within this event may not return a valid set of selected records if the event has
*  been triggered by a call to Selection.selectAll or 
*  Selection.deselectAll - in this case use the 
*  DataBoundComponent.selectionUpdated event instead.
* 
* 
* @param {ListGridRecord} record for which selection changed
* @param {boolean} New selection state (true for selected, false for unselected)
*/
selectionChanged(record:ListGridRecord, state:boolean): void;

/**
*  Called when (cell-based) selection changes within this grid.
* 
* 
* @param {Array<any>} Array of cells whos selected state was modified.
*/
cellSelectionChanged(cellList:Array<any>): boolean;

/**
*  Given a pointer to an element in the DOM, this method will check whether this
*  element is contained within a cell of the gridRenderer, and if so return a
*  2 element array denoting the [rowNum,colNum] of the element
*  in question.
* 
* @param {DOMElement} DOM element to test
*/
getCellFromDomElement(element:DOMElement): Array<any>;




}



/**
* A container that allows drag and drop instantiation of visual components from a
*  Palette, and direct manipulation of the position and size of those components.
*  
*  Any drag onto an EditPane from a Palette will add an EditNode created from the dragged
*  PaletteNode.
*  
*  EditPane automatically creates an EditContext and provides several APIs and
*  settings that are passthroughs to the underlying EditContext for convenience.
*/
class EditPane extends Canvas {
/**
* Creates a new EditPane
* 
* @param typeCheckedProperties {Partial<EditPane>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {EditPane} The new instance
*/
static create(typeCheckedProperties? : Partial<EditPane>, uncheckedProperties?: Object): EditPane;


/**
* An EditContext is automatically created to manage EditMode behavior. The public
*  EditContext APIs exposed by the EditPane are passed through to this object.
*  
*  Additional editContext properties can be supplied as
*  EditPane.editContextProperties.
* @type {EditContext}
* @default null
*/
editContext?: EditContext;

/**
* Properties to be applied to the EditPane.editContext when created.
* @type {Partial<EditContext>}
* @default null
*/
editContextProperties?: Partial<EditContext>;

/**
* New nodes added to the editContext are automatically placed
*  into edit mode if the new node's parent is in edit mode. To
*  suppress this action set autoEditNewNodes to false.
* @type {boolean}
* @default null
*/
autoEditNewNodes?: boolean;

/**
* Root of data to edit. Must contain the "type" property, with the name of a
*  valid DataSource or nothing will be able to be dropped on this
*  EditContext. A "liveObject" property representing the rootComponent is also
*  suggested. Otherwise, a live object will be created from the palette node.
*  
*  Can be retrieved at any time. Use EditPane.getRootEditNode to retrieve the
*  EditNode created from the rootComponent.
* @type {PaletteNode}
* @default null
*/
rootComponent?: PaletteNode;

/**
* Palette to use when an EditNode is being created directly by this EditContext,
*  instead of being created due to a user interaction with a palette (eg dragging from
*  a TreePalette, or clicking on MenuPalette).
*  
*  If no defaultPalette is provided, the EditContext uses an automatically created
*  HiddenPalette.
* @type {Palette}
* @default null
*/
defaultPalette?: Palette;

/**
* Additional Palette to consult for metadata when
*  deserializing EditNode. Note that the
*  EditPane.defaultPalette is always consulted and need not be
*  provided again here.
* @type {Array<Partial<Palette>>}
* @default null
*/
extraPalettes?: Array<Partial<Palette>>;

/**
* When enabled, changes to a EditNode.liveObject's position
*  and size will be persisted to their EditNode by default.
*  This applies to both programmatic calls and user interaction (drag reposition
*  or drag resize).
*  
*  This feature can be disabled by either setting this property or
*  EditProxy.persistCoordinates to false. This
*  property affects all nodes within the EditContext whereas the latter
*  property affects children of a single node. 
*  
*  In some use-cases, like VisualBuilder, coordinates should not be
*  persisted except when a component explicitly enables this feature.
*  By setting this property to null no component will
*  persist coordinates of children unless
*  EditProxy.persistCoordinates is explicitly set to
*  true.
* @type {boolean}
* @default true
*/
persistCoordinates?: boolean;

/**
* Controls whether components can be dropped into other components which support child
*  components. 
*  
*  When enabled, during a drop interaction in which a PaletteNode or EditNode
*  is the drop data, the componentSchema of the current
*  candidate drop target is inspected to see whether that parent allows children of the
*  type being dropped. If it does, the drop will result in a call to EditPane.addNode for
*  a paletteNode or for an existing EditNode in the same tree.
*  
*  Specific components can disable nested drops by explicitly setting EditProxy.allowNestedDrops
*  to false.
*  
*  This mode is enabled by default unless explicitly disabled by setting this property to
*  false.
* @type {boolean}
* @default null
*/
allowNestedDrops?: boolean;

/**
* Should the selection outline show a label for selected components? A component may
*  also be highlighted with the selection outline and label to indicate the target of
*  a drop. To suppress showing a label at any time set this property to false.
*  
*  To suppress labels during selection but still show them when targeted for a drop,
*  see EditContext.showSelectedLabelOnSelect.
*  
*  NOTE: A selected component label is only supported when EditProxy.selectedAppearance
*  is "outlineEdges".
* @type {boolean}
* @default null
*/
showSelectedLabel?: boolean;

/**
* Set the CSS border to be applied to the selection outline of the selected components.
*  This property is used when EditProxy.selectedAppearance is outlineMask
*  or outlineEdges.
*  
*  This value is applied as a default to EditProxy.selectedBorder.
* @type {string}
* @default null
*/
selectedBorder?: string;

/**
* The background color for the selection outline label. The
*  default is defined on SelectionOutline.
*  
*  This value is applied as a default to EditProxy.selectedLabelBackgroundColor.
*  
*  NOTE: A selected component label is only supported when EditProxy.selectedAppearance
*  is "outlineEdges".
* @type {string}
* @default null
*/
selectedLabelBackgroundColor?: string;

/**
* Should a group selection outline covering the outermost bounding boxes of all selected
*  components be shown in this container?
*  
*  Treated as true if not set and hoop selection is enabled (see
*  EditProxy.canSelectChildren and
*  EditContext.selectionType.
* @type {boolean}
* @default null
*/
canGroupSelect?: boolean;

/**
* Should the group selection box shown when EditContext.canGroupSelect
*  is true allow dragging the group as a whole?
*  
*  Treated as true if not set and EditContext.canGroupSelect
*  is true.
* @type {boolean}
* @default null
*/
canDragGroup?: boolean;

/**
* Should the group selection box shown when EditContext.canGroupSelect
*  is true be hidden during drag?
*  
*  Treated as true if not explicitly set to false.
* @type {boolean}
* @default null
*/
hideGroupBorderOnDrag?: boolean;

/**
* If set, auto-enables EditProxy.useCopyPasteShortcuts on the EditProxy for the
*  EditPane.getRootEditNode. This works whether there is currently a root editNode
*  or one is added later.
* @type {boolean}
* @default null
*/
useCopyPasteShortcuts?: boolean;







/**
*  Returns the EditContext instance managed by the EditPane.
* 
*/
getEditContext(): EditContext;

/**
*  Returns an Array of PaletteNodes representing all current EditNodes in this
*  pane, suitable for saving and restoring via passing each paletteNode to EditContext.addNode.
* 
*/
getSaveData(): Array<Partial<PaletteNode>>;

/**
*  Returns the root EditNode of the EditContext typically created from EditPane.rootComponent.
* 
* 
*/
getRootEditNode(): EditNode;

/**
*  Creates and returns an EditNode using the EditPane.defaultPalette. Does not add the newly
*  created EditNode to an EditContext.
*  
* 
* @param {PaletteNode} the palette node to use to create the new node
*/
makeEditNode(paletteNode:PaletteNode): EditNode;

/**
*  Gets the tree of editNodes being edited by this editContext. Standard tree
*  traversal methods can then be used to locate desired editNodes for interaction.
*  
*  Note: the returned tree is read-only and must only be modified by calling
*  methods on EditContext like EditContext.addNode or EditContext.setNodeProperties.
* 
* 
*/
getEditNodeTree(): Tree;

/**
*  Add a new EditNode to the EditContext, under the specified parent. If the parentNode
*  is not provided it will be determined from EditContext.defaultParent.
*  
*  The EditContext will interrogate the parent and new nodes to determine what field 
*  within the parent allows a child of this type, and to find a method to add the newNode's 
*  liveObject to the parentNode's liveObject. The new relationship will then be stored
*  in the tree of EditNodes.
*  
*  For example, when a Tab is dropped on a TabSet, the field TabSet.tabs is discovered as
*  the correct target field via naming conventions, and the method TabSet.addTab() is likewise 
*  discovered as the correct method to add a Tab to a TabSet.
* 
* 
* @param {EditNode} new node to be added
* @param {EditNode=} parent to add the new node under.
* @param {Integer=} index within the parent's children array
* @param {string=} the property of the liveParent to which the new node should                 be added, if not auto-discoverable from the schema
* @param {boolean=} whether to skip adding the liveObject to the liveParent                      (default false)
* @param {boolean=} whether to replace existing single field node                       if newNode liveObject is the same (default false)
*/
addNode(newNode:EditNode, parentNode?:EditNode, index?:Integer, parentProperty?:string, skipParentComponentAdd?:boolean, forceSingularFieldReplace?:boolean): EditNode;

/**
*  Creates a new EditNode from a PaletteNode, using the
*  EditPane.defaultPalette. If you have an array of possibly inter-related
*  PaletteNodes, then you should use
*  EditPane.addFromPaletteNodes on the array instead,
*  in order to preserve the relationships.
* 
* 
* @param {PaletteNode} the palette node to use to create the new node
* @param {EditNode=} optional the parent node if the new node should appear                under a specific parent
*/
addFromPaletteNode(paletteNode:PaletteNode, parentNode?:EditNode): EditNode;

/**
*  Add the supplied PaletteNode to the parentNode, preserving internal
*  references from one supplied PaletteNode to another. This method should
*  be used with an array of possibly inter-related PaletteNodes (for
*  instance, those produced as a result of serialization via
*  EditPane.serializeAllEditNodes, rather than
*  calling EditPane.addFromPaletteNode on each
*  individual PaletteNode.
* 
* 
* @param {Array<Partial<PaletteNode>>} array of PaletteNodes
* @param {EditNode=} parent to add to (defaults to the root)
*/
addFromPaletteNodes(paletteNodes:Array<Partial<PaletteNode>>, parentNode?:EditNode): Array<Partial<EditNode>>;

/**
*  Removes EditNode from the EditContext. The editNode
*  liveObject is not destroyed.
* 
* @param {EditNode} node to be removed
*/
removeNode(editNode:EditNode): void;

/**
*  Moves an EditNode from one child index to another in the EditContext under the specified parent.
*  
*  No changes are made to the live objects.
* 
* 
* @param {EditNode} parent to reorder child nodes
* @param {Integer} index within the parent's children array to be moved
* @param {Integer} index within the parent's children array at which to place moved node
*/
reorderNode(parentNode:EditNode, index:Integer, moveToIndex:Integer): void;

/**
*  Removes all EditNode from the EditContext, but does not destroy 
*  the EditNode.liveObject.
* 
*/
removeAll(): void;

/**
*  Removes all EditNode from the EditContext, and calls
*  Canvas.destroy on the
*  EditNode.liveObject.
* 
*/
destroyAll(): void;

/**
*  Returns true if editNode is in edit mode.
* 
* 
* @param {EditNode} the EditNode
*/
isNodeEditingOn(editNode:EditNode): boolean;

/**
*  Enable edit mode for an EditNode. This is a shortcut for calling
*  Canvas.setEditMode.
* 
* 
* @param {EditNode} the EditNode on which to enable editing
*/
enableEditing(editNode:EditNode): void;

/**
*  Update an editNode's serializable "defaults" with the supplied properties. If you
*  wish to remove a property from the defaults (rather than setting it to null), then
*  use EditPane.removeNodeProperties instead.
* 
* @param {EditNode} the editNode to update
* @param {Partial<Canvas>} the properties to apply
* @param {boolean=} whether to skip updating the                     EditNode.liveObject,                     e.g. if you have already updated the liveObject
*/
setNodeProperties(editNode:EditNode, properties:Partial<Canvas>, skipLiveObjectUpdate?:boolean): void;

/**
*  Removes the specified properties from an editNode's serializable "defaults".
*  Note that the EditNode.liveObject is not updated by this method. 
*  To set a property to null (rather than removing it), use
*  EditPane.setNodeProperties instead.
* 
* @param {EditNode} the editNode to update
* @param {Array<Partial<string>>} an array of property names to remove
*/
removeNodeProperties(editNode:EditNode, properties:Array<Partial<string>>): void;

/**
* Palette to use when an EditNode is being created directly by this EditContext,
*  instead of being created due to a user interaction with a palette (eg dragging from
*  a TreePalette, or clicking on MenuPalette).
*  
*  If no defaultPalette is provided, the EditContext uses an automatically created
*  HiddenPalette.
*/
getDefaultPalette(): Palette;

/**
* Palette to use when an EditNode is being created directly by this EditContext,
*  instead of being created due to a user interaction with a palette (eg dragging from
*  a TreePalette, or clicking on MenuPalette).
*  
*  If no defaultPalette is provided, the EditContext uses an automatically created
*  HiddenPalette.
* @param {Palette} the default Palette
*/
setDefaultPalette(palette:Palette): void;

/**
*  Creates a PaletteNode from an EditNode in this context's
*  EditPane.getEditNodeTree.
*  
*  This essentially creates a new PaletteNode with the EditNode.defaults from the
*  passed editNode. The returned paletteNode could then be used with
*  EditContext.addFromPaletteNode to effectively create a copy of the original editNode -
*  specifically a new editNode with a new EditNode.liveObject created from the same
*  defaults.
*  
*  However note that makePaletteNode() does not copy descendant nodes - use
*  EditPane.makePaletteNodeTree for that.
*  
*  May return null if the passed editNode cannot validly by transformed into a paletteNode, for
*  example if EditNode.canDuplicate was set false.
* 
* 
* @param {EditNode} the editNode to use to make a paletteNode
*/
makePaletteNode(editNode:EditNode): PaletteNode;

/**
*  Creates a Tree of PaletteNode from an EditNode in this
*  context's EditPane.getEditNodeTree, by using EditPane.makePaletteNode on the
*  passed EditNode and its descendents within the
*  EditContext.getEditNodeTree.
*  
*  The root node of the returned Tree will be a PaletteNode derived from the passed
*  EditNode.
* 
* 
* @param {EditNode} root editNode to make Tree of PaletteNodes from
*/
makePaletteNodeTree(editNode:EditNode): Tree;

/**
*  Copies the passed editNode or editNodes to an internal "clipboard" space, for later application
*  via EditPane.pasteEditNodes.
* 
* @param {EditNode | Array<Partial<EditNode>>} */
copyEditNodes(editNode:EditNode | Array<Partial<EditNode>>): void;

/**
*  "Pastes" editNodes previously captured via EditPane.copyEditNodes.
*  
*  New editNodes will be added as root-level nodes of the EditPane.getEditNodeTree
*  unless a targetEditNode is passed.
* 
* @param {EditNode=} */
pasteEditNodes(targetEditNode?:EditNode): void;

/**
*  Recreate EditNode from an XML representation of 
*  PaletteNode (possibly created by calling
*  EditPane.serializeAllEditNodes or EditPane.serializeEditNodes.
*  
*  By default, components that have Canvas.ID will not
*  actually be allowed to take those global IDs - instead, only widgets that have one of the
*  global IDs passed as the globals parameter will actually receive their global
*  IDs. To override this behavior, pass the special value RPCManager.ALL_GLOBALS
*  for the globals parameter.
* 
* 
* @param {string} XML string
* @param {EditNode=} parent node (defaults to the root)
* @param {Array<Partial<string>>=} widgets to allow to take their global IDs
* @param {Function=} Callback to fire after nodes have been added
*/
addPaletteNodesFromXML(xmlString:string, parentNode?:EditNode, globals?:Array<Partial<string>>, callback?:Function): void;

/**
*  Recreate EditNode from a JSON representation of 
*  PaletteNode (possibly created by calling
*  EditPane.serializeAllEditNodesAsJSON or EditPane.serializeEditNodesAsJSON.
*  
*  By default, components that have Canvas.ID will not
*  actually be allowed to take those global IDs - instead, only widgets that have one of the
*  global IDs passed as the globals parameter will actually receive their global
*  IDs. To override this behavior, pass the special value RPCManager.ALL_GLOBALS
*  for the globals parameter.
* 
* 
* @param {string} JSON string
* @param {EditNode=} parent node (defaults to the root)
* @param {Array<Partial<string>>=} widgets to allow to take their global IDs
* @param {Function=} Callback to fire after nodes have been added
*/
addPaletteNodesFromJSON(jsonString:string, parentNode?:EditNode, globals?:Array<Partial<string>>, callback?:Function): void;

/**
*  Obtain PaletteNode from an XML representation,
*  but do not add them to the EditContext.
* 
* 
* @param {string} XML string
* @param {PaletteNodeCallback} Callback used to return the PaletteNodes
*/
getPaletteNodesFromXML(xmlString:string, callback:(paletteNodes:Array<Partial<PaletteNode>>) => void): void;

/**
*  Obtain PaletteNode from a JavaScript source representation.
*  
*  By default, components that have Canvas.ID will not
*  actually be allowed to take those global IDs - instead, only widgets that have one of the
*  global IDs passed as the globals parameter will actually receive their global
*  IDs. To override this behavior, pass the special value RPCManager.ALL_GLOBALS
*  for the globals parameter.
* 
* 
* @param {string} JavaScript code to eval.
* @param {PaletteNodeCallback} Callback used to return the PaletteNodes
* @param {Array<Partial<string>>=} widgets to allow to take their global IDs
*/
getPaletteNodesFromJS(jsCode:string, callback:(paletteNodes:Array<Partial<PaletteNode>>) => void, globals?:Array<Partial<string>>): void;

/**
*  Add PaletteNode from a JavaScript source representation.
*  
*  By default, components that have Canvas.ID will not
*  actually be allowed to take those global IDs - instead, only widgets that have one of the
*  global IDs passed as the globals parameter will actually receive their global
*  IDs. To override this behavior, pass the special value RPCManager.ALL_GLOBALS
*  for the globals parameter.
* 
* 
* @param {string} JavaScript code to eval.
* @param {EditNode=} parent node (defaults to the root)
* @param {Array<Partial<string>>=} widgets to allow to take their global IDs
*/
addPaletteNodesFromJS(jsCode:string, parentNode?:EditNode, globals?:Array<Partial<string>>): void;

/**
*  Serialize the tree of EditNode to an XML representation
*  of PaletteNode. The result can be supplied to 
*  EditPane.addPaletteNodesFromXML to recreate
*  the EditNodes.
* 
* 
* @param {SerializationSettings=} Additional serialization settings
*/
serializeAllEditNodes(settings?:SerializationSettings): string;

/**
*  Encode the tree of EditNode to a JSON representation
*  of PaletteNode. The result can be supplied to 
*  EditPane.addPaletteNodesFromJSON to recreate
*  the EditNodes.
* 
* 
* @param {SerializationSettings=} Additional serialization settings
*/
serializeAllEditNodesAsJSON(settings?:SerializationSettings): string;

/**
*  Serialize the provided EditNode to an XML
*  representation of PaletteNode. Note that the
*  EditNodes must have been added to this EditContext. The result can be
*  supplied to EditPane.addPaletteNodesFromXML to
*  recreate the EditNodes.
* 
* 
* @param {Array<Partial<EditNode>>} EditNodes to be serialized
* @param {SerializationSettings=} Additional serialization settings
*/
serializeEditNodes(nodes:Array<Partial<EditNode>>, settings?:SerializationSettings): string;

/**
*  Serialize the provided EditNode to a JSON
*  representation of PaletteNode. Note that the
*  EditNodes must have been added to this EditContext. The result can be
*  supplied to EditPane.addPaletteNodesFromJSON to
*  recreate the EditNodes.
* 
* 
* @param {Array<Partial<EditNode>>} EditNodes to be serialized
* @param {SerializationSettings=} Additional serialization settings
*/
serializeEditNodesAsJSON(nodes:Array<Partial<EditNode>>, settings?:SerializationSettings): string;




}

/**
* 
*  A JavaScript Object where each property name is a facetId and each property value is a
*  facetValueId for that facet.
*  
*  
*  
*  The facetId &rarr; facetValueId mappings in a FacetValueMap describe a specific slice of the
*  dataset. If mappings are included for all facets, a FacetValueMap describes a unique
*  cell. If some facets are omitted, it describes a row, column, or set of rectangular
*  areas, or equivalently, a particular row or column header (if all facetIds in the map are
*  displayed on the same axis)
*  
*  FacetValueMaps are used in various contexts to describe headers, datasets to be loaded,
*  screen regions, etc.
*/
class FacetValueMap {











}



/**
* Policy controlling how the Layout will manage member sizes on this axis.
*  
*  Note that, by default, Layouts do not automatically expand the size of all members
*  to match a member that overflows the layout on the breadth axis. This means that a
*  DynamicForm or other component that can't shrink beyond a minimum width will 
*  "stick out" of the Layout, wider than any other member and wider than automatically
*  generated components like resizeBars or sectionHeaders (in a SectionStack).
*  
*  This is by design: matching the size of overflowing members would cause expensive redraws
*  of all members in the Layout, and with two or more members potentially overflowing, could
*  turn minor browser size reporting bugs or minor glitches in custom components into
*  infinite resizing loops.
*  
*  If you run into this situation, you can either:
*  
* - set the overflowing member to Canvas.overflow: "auto", so that it
*  scrolls if it needs more space
*  
* - set the Layout as a whole to Canvas.overflow:"auto", so that the
*  whole Layout scrolls when the member overflows
*  
* - define a Canvas.resized handler to manually update the breadth
*  of the layout
*  
* - set Layout.minBreadthMember to ensure that the available breadth used to
*  expand all (other) members is artificially increased to match the current breadth of the
*  minBreadthMember member; the layout will still be overflowed in this case
*  and the reported size from Canvas.getWidth or Canvas.getHeight won't
*  change, but all members should fill the visible width or height along the breadth axis
*  
*  For the last approach, given the VLayout myLayout and a member 
*  myWideMember, then we could define the following Canvas.resized handler on myLayout:
*  
*  
*  resized : function () {
*    var memberWidth = myWideMember.getVisibleWidth();
*    this.setWidth(Math.max(this.getWidth(), memberWidth + offset));
*  }
*  where offset reflects the difference in width (due to margins, padding,
*  etc.) between the layout and its widest member. In most cases, a fixed offset can
*  be used, but it can also be computed via the calculation:
*  
*  
*    myLayout.getWidth() - myLayout.getViewportWidth()
*  
*  in an override of Canvas.draw
*  for myLayout. (That calculation is not always valid inside the 
*  Canvas.resized handler itself.)
*  
*  Note: the HLayout case is similar- just substitute height where width appears above.
*  
*  See also Layout.overflow.
*/
type LayoutPolicy = 
/**
* Layout does not try to size members on the axis at all, merely stacking them (length
*  axis) and leaving them at default breadth.
*/
"none" |

/**
* Layout sizes members so that they fill the specified size of the layout. The rules
*  are:
*  
*  
* - Any component given an initial pixel size, programmatically resized to a specific
*  pixel size, or drag resized by user action is left at that exact size
*  
* - Any component that Button.autoFit is given exactly the space it
*  needs, never forced to take up more.
*  
* - All other components split the remaining space equally, or according to their
*  relative percentages.
*  
* - Any component that declares a Canvas.minWidth or Canvas.minHeight
*  will never be sized smaller than that size
*  
* - Any component that declares a Canvas.maxWidth or Canvas.maxHeight
*  will never be sized larger than that size
*  
*  In addition, components may declare that they have 
*  Canvas.canAdaptWidth, and may coordinate with the Layout to render
*  at different sizes according to the amount of available space.
*/
"fill";




/**
* The data URL MIME type to use when DrawPane.getDataURL is called to convert the
*  drawing to a data URL.
*/
type DataURLFormat = 
/**
* Any MIME type supported by the browser is acceptable. Note: The exact MIME type used may depend on the browser, and may change from version to version of SmartClient.
*/
"any" |

/**
* Generate an image/png data URL.
*/
"png";




/**
* Different styles of determining intersection: with mouse or entire rect of target
*/
type DragIntersectStyle = 
/**
* Look for drop targets that are under the current mouse cursor position.
*/
"mouse" |

/**
* Look for drop targets by intersection of the entire rect of the drag        target with the droppable target.
*/
"rect";


/**
* what to do if there is more than one possible childrenProperty when using
*  scanMode "branch" or "level"
*  "node": continue, but pick childrenProperty on a per-node basis (will detect
*        mixed) 
*  "highest": continue, picking the childrenProperty that occurred most as the single
*       choice
*  "stop": if there's a tie, stop at this level (assume no further children)
*/
type TieMode = 
/**
* continue, but pick childrenProperty on a per-node basis (will detect mixed)
*/
"node" |

/**
* continue, picking the childrenProperty that occurred most as the single choice
*/
"highest" |

/**
* if there's a tie, stop at this level (assume no further children)
*/
"stop";






/**
* Subclass of Window used to display events within a Calendar. Customize via
*  Calendar.eventWindow.
* @deprecated * in favor of EventCanvas

*/
class EventWindow extends Window {
/**
* Creates a new EventWindow
* 
* @param typeCheckedProperties {Partial<EventWindow>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {EventWindow} The new instance
*/
static create(typeCheckedProperties? : Partial<EventWindow>, uncheckedProperties?: Object): EventWindow;











}



/**
* For "fetch" operations, how simple search criteria should be interpreted for text fields 
*  (and to some extent numeric fields, see the notes below). For greater flexibility, use 
*  AdvancedCriteria.
*/
type TextMatchStyle = 
/**
* case-insensitive exact match ("foo" matches "foo" and "FoO", but not "FooBar")
*/
"exact" |

/**
* case-sensitive exact match ("foo" matches only "foo")
*/
"exactCase" |

/**
* case-insenstive substring match ("foo" matches "foobar" and "BarFoo")
*/
"substring" |

/**
* case-insensitive prefix match ("foo" matches "FooBar" but not "BarFoo") 
*  textMatchStyle behavior is consistent across the client and all three of the built-in
*  server DataSource implementations - SQL, Hibernate and JPA - to the maximum extent that is
*  practical. Known inconsistencies, some of which can be tweaked with configuration settings,
*  are described below
*  
* - textMatchStyle can be disabled for any individual field with the 
*  DataSourceField.ignoreTextMatchStyle declaration. This behavior is consistent
*  across client and server implementations, with the proviso that case-sensitive behavior 
*  is inherently inconsistent in the server-side implementations (see below for a more
*  detailed discussion of this)
*  
* - textMatchStyle is forced to "exactCase" for DataSourceField.primaryKey 
*  fields on update and remove operations, unless
*  OperationBinding.allowMultiUpdate is in force. This means that a normal update
*  or remove operation - which is constrained by a unique primaryKey value or combination, to 
*  affect only a single data row - will use an equality match on all primaryKey fields. This
*  is desirable because it makes index usage more likely. Note that this means the key values
*  "foo" and "FOO" are distinct, which is a change from previous behavior. For most databases,
*  this change will have no effect because most databases do not by default allow primaryKey 
*  values that only differ in case anyway - if a table has a row with a PK of "foo" and you 
*  try to insert a row with a PK of "FOO", most databases will fail with a constraint violation.
*  However, if you are using a database hat allows key values that differ only in case, and you 
*  need to support this for some reason, you can switch the framework back to the previous 
*  behavior by setting property allow.case.sensitive.pks to true in
*  your server.properties file. Note, we do not recommend this; in fact, we do 
*  not recommend the use of text primaryKey fields at all. Where you have control ,we would 
*  always recommend the use of automatically incrementing numeric sequence keys.
*  
* - textMatchStyle does not apply to boolean or temporal (date, datetime, time)
*  DataSourceField.type. Such fields are always checked for equality. If 
*  you have a need for more complex matching, such as date-range checking, use AdvancedCriteria
*  
* - Critiera values for temporal fields must be real dates (Javascript Date
*  objects on the client, instances of java.util.Date on the server). Attempting
*  to match against a string or any other type of value will result in an "always fails" 
*  clause being generated. This behavior is how the client code has always worked, but is a 
*  change for the server-side DataSources, in the interests of consistency. The previous
*  behavior was:
*   
* - SQLDataSource used to attempt to match the date column to the criteria value provided;
*   if that value was an appropriately-formatted string, that attempt might have 
*   worked, but it would be database-specific. If you have an application that depends on 
*   this old behavior, you can switch it back on by adding the following to your
*   server.properties file: 
*   sql.temporal.fields.require.date.criteria: false
*   
* - HibernateDataSource and JPADataSource used to skip such clauses altogether, which 
*   results in the opposite behavior (there is an implied "always succeeds" clause, because 
*   the clause is not present in the query at all). If you have an application that depends on 
*   this old behavior, you can switch it back on by adding the following to your
*   server.properties file: 
*   jpa.temporal.fields.require.date.criteria: false or 
*   hibernate.temporal.fields.require.date.criteria: false, as appropriate
*  
*  
* - textMatchStyle does not apply to fields used in the joins generated to satisfy 
*  DataSourceField.includeFrom declarations or entity relations in
*  Hibernate or JPA. Again, such comparisons are always for equality
*  
* - textMatchStyle is applied to numeric fields only if the supplied criteria value
*  is a string. So for a numeric field "myNumber", we will match 10, 100, 110 and 1000 if a 
*  criteria value of "10" (the quoted string) is supplied, but we will only match 10 if a 
*  criteria value of 10 (the actual number) is supplied
*   
* - This is a change in behavior for the server-side DataSource implementations, in the 
*   interests of consistency with the client behavior. Previously, all three of the built-in
*   DataSource implementations applied textMatchStyle to numeric fields regardless of the 
*   criteria value's type. If you have an application that depends on this old behavior, 
*   you can switch it back on by adding the following to your server.properties
*   file: sql.substring.numeric.criteria: true or 
*   hibernate.substring.numeric.criteria: true or 
*   jpa.substring.numeric.criteria: true, as appropriate
*  
*  
* - "exactCase" matching cannot be made consistent across all database providers because 
*  each database has its own idea of whether case-sensitive matching should be the default, 
*  and how to switch it on and off (if that is even possible)
*   
* - Microsoft SQL Server must have case-sensitive matching enabled by setting a 
*   case-sensitive collation when the database is created (there are more fine-grained 
*   options, but we do not recommend them because of the scope for further inconsistency, 
*   where "exactCase" works correctly on table A but not on table B). See 
*   MSDN for details
*   
* - HSQLDB has case sensitivity set with the SET IGNORECASE directive -
*   see here
*   
* - MySQL is case-insensitive, but since it provides an easy way to do per-query,
*   case-sensitive matching in SQL, we work around it for that database, but only in
*   SQL DataSources
*  
*  
* - If you are using Hibernate or JPA, whether "exactCase" works as expected depends 
*  entirely on the underlying database. Unlike the SQL DataSource, where we have fine-grained
*  control of the SQL that the database executes, we are constrained by what the ORM will 
*  allow. This prevents us from compensating for database quirks, like we do with MySQL's
*  case-insensitivity in the SQL DataSource
*  
*/
"startsWith";


/**
* The technique SmartClient Server's SQL DataSource should use to select a "page" of data 
*  from a table.
*/
type SQLPagingStrategy = 
/**
* Specify the paging directly in the SQL query we generate. The way this is done varies 
*  considerably from database to database: with some it is a straightforward built-in 
*  facility while others require arcane tricks or simply don't support the idea. This is 
*  the most efficient method, where available. Note that this strategy is not supported 
*  for operations that make use of a OperationBinding.customSQL clause, 
*  because it depends upon being able to determine the size of the whole dataset without
*  actually retrieving the whole dataset. Ordinary operations do this by means of an 
*  automatically-generated "row count query", but we cannot generate such a query for a 
*  customSQL operation.
*/
"sqlLimit" |

/**
* Implement the paging behavior by use of the absolute() method of the
*  JDBC ResultSet.
*/
"jdbcScroll" |

/**
* Implement the paging behavior by fetching the entire resultset from the database and
*  dropping any unnecessary rows on the server before returning the data to the client.
*  This approach is extremely inefficient, but also extremely straightforward; it is 
*  intended as a fallback option, for situations where the more sophisticated approaches
*  cause problems (a JDBC driver that throws vague exceptions when absolute()
*  is called, for example)
*/
"dropAtServer" |

/**
* No paging behavior: we always return the entire resultset to the client.
*/
"none";






/**
*/
type Visibility = 
/**
* The widget visibility will match that of its parent                 (usually visible).
*/
"inherit" |

/**
* The widget will always be visible whether its parent is                 or not.
*/
"visible" |

/**
* The widget will always be hidden even when its parent is                 visible.
*/
"hidden";


/**
* A class used to define a shadow used in a Draw&lt;Shape&gt; Types.
*/
class Shadow {


/**
* @type {CSSColor}
* @default "black"
*/
color?: CSSColor;

/**
* @type {Integer}
* @default "10"
*/
blur?: Integer;

/**
* @type {Point}
* @default "[0,0]"
*/
offset?: Point;










}

/**
* Policy for laying out tiles. 
*  
*  Because a TileLayout can be either horizontally or vertically oriented, the general term
*  "line" is used to mean either a row or column of tiles. 
*  
*  NOTE: for typical form layouts (a variety of input fields and buttons in a
*  tabular layout with col-spanning and row-spanning support), use a DynamicForm and
*  see the formLayout topic.
*/
type TileLayoutPolicy = 
/**
* Each line has the same number of tiles, based on TileLayout.tilesPerLine if set, otherwise,
*  based on fitting as many tiles per line as possible consistent with
*  TileLayout.tileSize and TileLayout.tileMargin. 
*  
*  Tiles that do not specify a size will be sized to fill available space.
*/
"fit" |

/**
* Tiles are laid out with varying numbers of tiles per line according to each tile's size,
*  the TileLayout.tileMargin, and the available space.
*  
*  Tiles are never resized by the TileLayout and TileLayout.tileSize is ignored.
*/
"flow";




/**
* A instance of Process represents a stateful process executing a series of Tasks, 
*  which may be:
*  
*  
* - user interactions
*  
* - calls to DataSources (hence: any database or web service)
*  
* - arbitrary code
*  
* - other Processes
*  
*  A Process is stateful in the sense that it maintains Process.state
*  across the different tasks that are executed. This allows you to maintain context as you
*  walk a user through a multi-step business process in your application, which may involve
*  multiple operations on multiple entities. Each Task that executes can use the Process state
*  as inputs, and can output a result which is stored in the Process state - see
*  taskIO.
*  
*  A Process can have multiple branches, choosing the next Task to execute based on
*  Criteria - see XORGateway and DecisionGateway.
*  
*  Because a Process may return to a previous Task in various situations, the data model of a
*  Process is strictly speaking a graph (a set of nodes connected by arbitary
*  interlinks). However, most processes have sequences of several tasks in a row, and the
*  definition format allows these to be represented as simple Arrays called "sequences",
*  specified via Process.sequences. This reduces the need to manually specify IDs and
*  interlinks for Tasks that simply proceed to the next task in a sequence.
*  
*  Processes follow all the standard rules for encoding as componentXML, however,
*  note that the &lt;Process&gt; tag allows any kind of ProcessElement (tasks, gateways
*  and sequences) to appear as a direct subelement of the &lt;Process&gt; tag without the need
*  for an intervening &lt;elements&gt; or &lt;sequences&gt; tag. The example below
*  demonstrates this shorthand format.
*  
*  &lt;Process ID="processId"&gt;
*    &lt;ServiceTask ID="serviceTaskId" nextElement="sequenceId" ..&gt;
*      &lt;inputFieldList&gt;
*        &lt;value&gt;order.countryName&lt;/value&gt;
*      &lt;/inputFieldList&gt;
*      &lt;outputFieldList&gt;
*        &lt;value&gt;order.countryName&lt;/value&gt;
*        &lt;value&gt;order.continent&lt;/value&gt;
*      &lt;outputFieldList&gt;
*    &lt;/ServiceTask&gt;
*    &lt;sequence ID="sequenceId" &gt;
*      &lt;StateTask ../&gt;
*      &lt;StateTask ../&gt;
*      &lt;StateTask ../&gt;
*      &lt;StateTask nextElement="userTaskId" ../&gt;
*    &lt;/sequence&gt;
*    &lt;UserTask ID="userTaskId" ../&gt;
*    ...
*  &lt;/Process&gt;
*  
*  NOTE: you must load the Workflow module
*  loadingOptionalModules before you can use Process.
*/
class Process extends Task {
/**
* Creates a new Process
* 
* @param typeCheckedProperties {Partial<Process>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Process} The new instance
*/
static create(typeCheckedProperties? : Partial<Process>, uncheckedProperties?: Object): Process;


/**
* Sequences of ProcessElements. By defining a sequences of elements you can make the
*  ProcessElement.nextElement implicit.
*  
*  You do not have to explicitly create a ProcessSequence,
*  you can instead use the shorthand:
*  
*  isc.Process.create({
*    startElement:"firstSequence", 
*    sequences: [
*      { ID:"something", elements: [ ... ] },
*      { ID:"somethingElse", elements: [ ... ] },
*      ...
*    ]
*    ...
*  });
*  
*  .. this is equivalent to ..
*  
*  isc.Process.create({
*    startElement:"firstSequence", 
*    sequences: [
*      isc.ProcessSequence.create({ 
*        ID:"something", 
*        elements: [ ... ] 
*      }),
*      isc.ProcessSequence.create({ 
*        ID:"somethingElement", 
*        elements: [ ... ] 
*      }),
*      ...              
*    ]
*    ...
*  });
*  
*  
*  
* @type {Array<Partial<ProcessSequence>>}
* @default null
*/
sequences?: Array<Partial<ProcessSequence>>;

/**
* Elements involved in this Process. You can also group elements into Process.sequences
*  to reduce the need to explicitly define IDs for elements and interlink them.
* @type {Array<Partial<ProcessElement>>}
* @default null
*/
elements?: Array<Partial<ProcessElement>>;

/**
* The ID of either a Process.sequences or an Process.elements which should
*  be the starting point of the process. If not specified, the first sequence is chosen,
*  or if there are no sequences, the first element.
*  - log a warning and do nothing if there are neither sequences or elements
*  
*  - an example of how a Process would be defined
*  
*  isc.Process.create({
*    startElement:"firstSequence", 
*    sequences: [
*      { 
*       id:"firstSequence",
*       elements : [
*         isc.ServiceTask.create({ .. }),
*         isc.DecisionGateway.create({ .. })
*       ]
*      },
*      {
*       id:"errorFlow",
*       elements : [ ... ]
*       
*      }
*    ],
*    elements: [
*     // standalone process elements not part of sequences
*     isc.ServiceTask.create({ .. })
*    ],
*    state : {
*      someField:"someValue"
*    }
*  })
*  
*  
* @type {string}
* @default null
*/
startElement?: string;

/**
* If wizard is set then current workflow will be handled as wizard. Every userTask will
*  hide associated form after user finished step.
* @type {boolean}
* @default false
*/
wizard?: boolean;

/**
* Identifier of canvas where UI elements created by using
*  UserTask.inlineView property should be added
*  using addMember.
* @type {GlobalId}
* @default null
*/
containerId?: GlobalId;

/**
* Current state of a process. As with Records in general, any field of a Record may
*  contain a nested Record or Array of Records, so the process state is essentially a
*  hierarchical data structure.
*  
*  Transient state
*  In addition to the explicit process state there is a "transient state." The transient
*  state represents the complete output of each of the last tasks of each type within the current
*  process execution. This allows easy reference to the previous task output with
*  taskInputExpression.
* @type {Record}
* @default null
*/
state?: Record;

/**
* Canvas.ID of the component that manages "rule context" for which
*  this process participates. The rule context can be used in
*  taskInputExpression.
* @type {string}
* @default null
*/
ruleScope?: string;

/**
* Cause the process to automatically call Process.start as soon as it is created.
* @type {boolean}
* @default false
*/
autoStart?: boolean;

/**
* Context object to be passed to Process.traceElement during process
*  execution.
* @type {object}
* @default null
*/
traceContext?: object;





/**
*  Loads an XML process definition stored in XML from the server.
*  
*  This method requires server-side support included in SmartClient Pro Edition or better.
*  
*  Process files are stored as .proc.xml files in componentXML
*  format, in the directory indicated by the project.processes setting in
*  server_properties
*  (webroot/processes by default). To load a process
*  saved in a file processId.proc.xml, pass just processId to this method.
*  
* 
* @param {Identifier | Array<Partial<Identifier>>} process IDs to load
* @param {ProcessCallback} called when the process is loaded with argument              "process", the first process. Other processes can be looked               up via Process.getProcess.
*/
static loadProcess(processId:Identifier | Array<Partial<Identifier>>, callback:(dsResponse:DSResponse, process:Process) => void): void;

/**
*  Get a Process instance by it's ID. See Process.loadProcess.
* 
* @param {Identifier} process IDs to retrieve
*/
static getProcess(processId:Identifier): Process;



/**
*  Retrieve a ProcessElement by it's ID
* 
* @param {string} id of the process element
*/
getElement(ID:string): ProcessElement;

/**
*  Set process state for current process
* 
* @param {Record} the new process state
*/
setState(state:Record): void;

/**
*  Starts this task by executing the Process.startElement. Also used by asynchronous
*  tasks to restart the workflow.
* 
*/
start(): void;

/**
*  Reset process to it's initial state, so process can be started again.
* 
* @param {Record} new state of the process
*/
reset(state:Record): void;

/**
*  StringMethod called when a process completes, meaning the process executes a 
*  ProcessElement with no next element.
* 
* @param {Record} the final process state
*/
finished(state:Record): void;

/**
*  StringMethod called during process execution before each task
*  element is processed.
* 
* @param {Task} the Task being executed
* @param {object} the Process.traceContext, if set
*/
traceElement(element:Task, context:object): void;




}



/**
* Hide or show a form field.
*/
class FormHideFieldTask extends ComponentTask {
/**
* Creates a new FormHideFieldTask
* 
* @param typeCheckedProperties {Partial<FormHideFieldTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FormHideFieldTask} The new instance
*/
static create(typeCheckedProperties? : Partial<FormHideFieldTask>, uncheckedProperties?: Object): FormHideFieldTask;


/**
* Field to show/hide.
* @type {FieldName}
* @default null
*/
targetField?: FieldName;

/**
* Should the target form item be hidden?
* @type {boolean}
* @default null
*/
hide?: boolean;










}

/**
*/
type LabelAlignment = 
/**
* Justify label's left edge against its left coordinate              (right in RTL configurations)
*/
"start" |

/**
* Justify label's right edge against its left coordinate              (left in RTL configurations)
*/
"end" |

/**
* Center label about its left coordinate
*/
"center" |

/**
* Justify label's left edge against its left coordinate
*/
"left" |

/**
* Justify label's right edge against its left coordinate
*/
"right";




/**
*/
type VerticalAlignment = 
/**
* At the top of the container
*/
"top" |

/**
* Center within container.
*/
"center" |

/**
* At the bottom of the container
*/
"bottom";




/**
*/
type Overflow = 
/**
* Content that extends beyond the widget's width or height is               displayed.
*                Note: To have the content be sized only by the drawn size of
*                the content set the overflow to be Canvas.VISIBLE and specify
*                a small size, allowing the size to expand to the size required
*                by the content.
*                Leaving the width / height for the widget undefined will use the
*                default value of 100, and setting the size to zero may cause the
*                widget not to draw.
*/
"visible" |

/**
* Content that extends beyond the widget's width or height is               clipped (hidden).
*/
"hidden" |

/**
* Horizontal and/or vertical scrollbars are displayed only if               necessary. Content that extends beyond the remaining visible
*                area is clipped.
*/
"auto" |

/**
* Horizontal and vertical scrollbars are always drawn inside the               widget. Content that extends beyond the remaining visible area
*                is clipped, and can be accessed via scrolling.
*/
"scroll" |

/**
* Clip horizontally but extend the canvas's clip region               vertically if necessary. Note: only supported for
*                specific widget subclasses.
*/
"clip-h" |

/**
* Clip vertically but extend the canvas's clip region               horizontally if necessary. Note: only supported
*                for specific widget subclasses.
*/
"clip-v";


/**
* LoadProjectSettings is the bundle of settings that can be passed to loadProject() as
*  the "settings" argument, including optional http parameters for the request to 
*  servletDetails.
*  
*  There is no need to instatiate an LoadProjectSettings instance. Just pass a normal
*  JavaScript object with the desired properties.
*/
class LoadProjectSettings {


/**
* The name of the screen within the project to draw after loading. 
*  A null value means to use the currentScreenName as specified in the project file of the
*  first project specified in the projectNames argument to RPCManager.loadProject.
* @type {string}
* @default null
*/
currentScreenName?: string;

/**
* A comma-separated string containing the names of screens within the project that should be loaded. 
*  A null value causes all screens to be loaded.
* @type {string}
* @default null
*/
screenNames?: string;

/**
* Determines whether the LoadProjectSettings.currentScreenName screen is drawn after all screens have been loaded.
* @type {boolean}
* @default true
*/
drawFirstScreen?: boolean;

/**
* The name of a locale to use for resolving i18n tags in the component XML of the screen. 
*  The default value of null omits locale loading, which effectively means the framework default "en" locale is used.
* @type {string}
* @default null
*/
locale?: string;

/**
* Use this attribute to specify a project owner. 
*  Only applicable if project source supports owner identification.
* @type {string}
* @default null
*/
ownerId?: string;










}

/**
* Allows a user to select an absolute or relative range of dates via two
*  RelativeDateItems (if DateRangeItem.allowRelativeDates is true) or two
*  DateItem.
*  
*  The item's FormItem.type is expected to be one of "date" or "datetime" and
*  dictates whether the dates in the range include a time portion. If unset and the item's 
*  form is databound, the data type is detected from the associated 
*  DataSourceField. If there is no such field, or the form is not 
*  databound, the default data type value is "date".
*  
*  DateRangeItem is just a convenience relative to using two RelativeDateItem or
*  DateItem controls in a form, then using FormItem.operator and
*  FormItem.criteriaField to cause them to produce a date range. If you need more
*  control over layout, validation, event handling or any other aspect of appearance or
*  behavior, stop using DateRangeItem and use two DateItem/RelativeDateItem controls directly
*  instead.
*/
class DateRangeItem extends CanvasItem {
/**
* Creates a new DateRangeItem
* 
* @param typeCheckedProperties {Partial<DateRangeItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DateRangeItem} The new instance
*/
static create(typeCheckedProperties? : Partial<DateRangeItem>, uncheckedProperties?: Object): DateRangeItem;


/**
* Controls the placement of the DateRangeItem.toField and DateRangeItem.fromField in the 
*  DateRangeItem.dateRangeForm.
*  
*  Note that we don't recommend "horizontal" placement for mobile, and we also don't recommend 
*  it for DateRangeItem.allowRelativeDates mode, since RelativeDateItem changes width 
*  drastically during editing, which causes a lot of unpleasant side-to-side shifting of 
*  controls.
* @type {DateFieldLayout}
* @default "vertical"
*/
fieldLayout?: DateFieldLayout;

/**
* Allow dateRangeItems' values to show up in the form's values array, or if 
*  DynamicForm.getValuesAsCriteria is called, for the criterion to be included
*  in the returned AdvancedCriteria object
* @type {boolean}
* @default true
*/
shouldSaveValue?: boolean;

/**
* The title for the DateRangeItem.fromField part of the range.
* @type {string}
* @default "From"
*/
fromTitle?: string;

/**
* The title for the DateRangeItem.toField part of the range.
* @type {string}
* @default "To"
*/
toTitle?: string;

/**
* The title orientation for the to / from sub-items. If unset this will be derived from
*  FormItem.titleOrientation or 
*  DynamicForm.titleOrientation.
* @type {TitleOrientation}
* @default null
*/
innerTitleOrientation?: TitleOrientation;

/**
* Whether to allow the user to specify relative dates (via RelativeDateItems) or whether
*  dates are absolute (via DateItems).
* @type {boolean}
* @default false
*/
allowRelativeDates?: boolean;

/**
* The field for the "from" date - a RelativeDateItem or DateItem according to
*  DateRangeItem.allowRelativeDates.
* @type {FormItem}
* @default null
*/
fromField?: FormItem;

/**
* The field for the "to" date - a RelativeDateItem or DateItem according to
*  DateRangeItem.allowRelativeDates.
* @type {FormItem}
* @default null
*/
toField?: FormItem;

/**
* Initial value for the "from" date.
* @type {Date | RelativeDateString | TimeUnit}
* @default "today"
*/
fromDate?: Date | RelativeDateString | TimeUnit;

/**
* Initial value for the "to" date.
* @type {Date | RelativeDateString | TimeUnit}
* @default "today"
*/
toDate?: Date | RelativeDateString | TimeUnit;

/**
* Format for direct user input of date values.
*  
*  If unset, the input format will be determined based on the specified
*  DateDisplayFormat if possible, otherwise picked up from the Date class (see
*  DateUtil.setInputFormat).
* @deprecated * This property is supported but the standard FormItem.dateFormatter and DateRangeItem.inputFormat 
*  may now be used to specify date formatting parsing behavior for dateRangeItems

* @type {DateInputFormat}
* @default null
*/
dateInputFormat?: DateInputFormat;

/**
* For fields of type "date", if this is an editable field such as a
*  TextItem, this property 
*  allows you to specify the DateItem.inputFormat applied to the item.
* @type {DateInputFormat}
* @default null
*/
inputFormat?: DateInputFormat;

/**
* Format for displaying dates in to the user. 
*  Defaults to the system-wide default established by DateUtil.setNormalDisplayFormat.
* @deprecated * This property is supported but the standard FormItem.dateFormatter and DateRangeItem.inputFormat 
*  may now be used to specify date formatting parsing behavior for dateRangeItems

* @type {DateDisplayFormat}
* @default null
*/
dateDisplayFormat?: DateDisplayFormat;

/**
* Not applicable to a DateRangeItem.
* @type {UserFormula}
* @default null
*/
formula?: UserFormula;

/**
* Not applicable to a DateRangeItem.
* @type {UserSummary}
* @default null
*/
textFormula?: UserSummary;

/**
* If this attribute is set to true when FormItem.getCriterion is
*  called, the item will validate the "to" and "from" fields and
*  return null if either field fails validation.
*  See DateRangeItem.validateRange
* @type {boolean}
* @default false
*/
validateCriteria?: boolean;

/**
* DynamicForm AutoChild automatically created by the dateRangeItem and applied
*  to the item as CanvasItem.canvas.
*  This DynamicForm contains the "from" and "to" fields the user will interact with to actually
*  select a date-range. Note that as a standard autoChild, developers may customize this form 
*  by modifying dateRangeProperties.
* @type {DynamicForm}
* @default null
*/
dateRangeForm?: DynamicForm;

/**
* The FormItem class to create when DateRangeItem.allowRelativeDates is true.
* @type {string}
* @default "RelativeDateItem"
*/
relativeItemConstructor?: string;

/**
* The FormItem class to create when DateRangeItem.allowRelativeDates is false,
*  but the DateRangeItem does not have type "datetime".
* @type {string}
* @default "DateItem"
*/
absoluteItemConstructor?: string;

/**
* The FormItem class to create when DateRangeItem.allowRelativeDates is false,
*  and the DateRangeItem's type is "datetime".
* @type {string}
* @default "DateTimeItem"
*/
absoluteDateTimeItemConstructor?: string;

/**
* Error message to display if the user enters a date range where the "To" field value
*  is earlier than the "From" field value.
* @type {string}
* @default '"To" field value cannot be earlier than "From" field value.'
*/
invalidRangeErrorMessage?: string;







/**
*  Sets the DateRangeItem.fromDate for this DateRangeItem.
* 
* @param {Date | RelativeDateString | TimeUnit} the date from which this item should start it's range
*/
setFromDate(fromDate:Date | RelativeDateString | TimeUnit): void;

/**
*  Sets the DateRangeItem.toDate for this DateRangeItem.
* 
* @param {Date | RelativeDateString | TimeUnit} the date at which this item should end it's range
*/
setToDate(fromDate:Date | RelativeDateString | TimeUnit): void;

/**
*  Overridden to return true: dateRangeItems always generate AdvancedCriteria.
* 
*/
hasAdvancedCriteria(): boolean;

/**
*  Returns the Criterion entered in the date field.
*  
*  A Criterion with an "and" OperatorId will be
*  returned with both a "greaterOrEqual" and "lessOrEqual" sub-criteria. If either date is 
*  omitted, only the "greaterOrEqual" (from date) or "lessOrEqual" (to date) Criterion is 
*  included.
* 
* 
*/
getCriterion(): Criterion;

/**
*  Validate both "to" and "from" date-fields.
* 
*/
validateRange(): boolean;

/**
*  Returns true if the specified criterion contains:
*  
* - A single "lessOrEqual" or "greaterOrEqual" criterion on this field
*    
* - An "and" type criterion containing a "lessOrEqual" and a "greaterOrEqual" criterion
*      on this field
*    
* - A single "equals" criterion. Internally, this will be converted into a range
*      by constructing an "and" type criterion containing both a "lessOrEqual" and 
*      a "greaterOrEqual" criterion on this field. Note that subsequent calls to 
*      DateRangeItem.getCriterion will return this more complex 
*      criterion.
*  
* 
* @param {Criterion} criterion to test
*/
canEditCriterion(criterion:Criterion): boolean;

/**
*  Applies the specified criterion to this item for editing. Applies any specified 
*  "greaterOrEqual" operator criterion or sub-criterion to our DateRangeItem.fromField and any
*  specified "lessOrEqual" operator criterion or sub-criterion to our DateRangeItem.toField.
*  
*  Note that a single "equals" criterion can also be passed. See 
*  DateRangeItem.canEditCriterion for more detail.
* 
* @param {Criterion} criterion to edit
*/
setCriterion(criterion:Criterion): void;

/**
*  Sets the value for this dateRangeItem. The value parameter is a 
*  DateRange object that optionally includes both start and end values. If
*  passed null, both start- and end-range values are cleared.
* 
* @param {DateRange} the new value for this item
*/
setValue(value:DateRange): void;

/**
*  Retrieves the current value of this dateRangeItem. The return value is a 
*  DateRange object that excludes start and end values if they aren't
*  set.
* 
*/
getValue(): DateRange;




}

/**
* Item for picking a number. Includes arrow buttons to increase / decrease the value
*/
class SpinnerItem extends TextItem {
/**
* Creates a new SpinnerItem
* 
* @param typeCheckedProperties {Partial<SpinnerItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SpinnerItem} The new instance
*/
static create(typeCheckedProperties? : Partial<SpinnerItem>, uncheckedProperties?: Object): SpinnerItem;


/**
* Not applicable to a SpinnerItem.
* @type {string}
* @default null
*/
mask?: string;

/**
* Not applicable to a SpinnerItem.
* @type {boolean}
* @default null
*/
maskSaveLiterals?: boolean;

/**
* Not applicable to a SpinnerItem.
* @type {string}
* @default " "
*/
maskPadChar?: string;

/**
* Not applicable to a SpinnerItem.
* @type {string}
* @default "_"
*/
maskPromptChar?: string;

/**
* Not applicable to a SpinnerItem.
* @type {boolean}
* @default null
*/
maskOverwriteMode?: boolean;

/**
* In SpinnerItem.writeStackedIcons, the base CSS class name for the
*  SpinnerItem's text box element.
*  
*  NOTE: See the CompoundFormItem_skinning discussion for special
*  skinning considerations.
* @type {FormItemBaseStyle}
* @default "textItem"
*/
unstackedTextBoxStyle?: FormItemBaseStyle;

/**
* In SpinnerItem.writeStackedIcons, the base CSS class name for the
*  SpinnerItem's text box element when printed. If unset, then SpinnerItem.unstackedTextBoxStyle
*  is used.
* @type {FormItemBaseStyle}
* @default null
*/
unstackedPrintTextBoxStyle?: FormItemBaseStyle;

/**
* @type {FormItemBaseStyle}
* @default null
*/
unstackedReadOnlyTextBoxStyle?: FormItemBaseStyle;

/**
* How much should the value be incremented / decremented when the user hits the icons
*  to increase / decrease the value?
*  
*  
*  When overriding SpinnerItem.getNextValue and/or SpinnerItem.getPreviousValue,
*  the sign of the step value determines whether these methods
*  
*  are expected to induce monotonically increasing or decreasing functions.
* @type {Double}
* @default "1"
*/
step?: Double;

/**
* Maximum valid value for this item. If this and SpinnerItem.min are both
*  null or unspecified, then
*  
*  SpinnerItem.getNextValue and SpinnerItem.getPreviousValue
*  are used to increase or decrease the value and these methods
*  
*  are also used to determine the maximum value.
* @type {Double}
* @default null
*/
max?: Double;

/**
* Minimum valid value for this item. If this and SpinnerItem.max are both
*  null or unspecified, then
*  
*  SpinnerItem.getNextValue and SpinnerItem.getPreviousValue
*  are used to increase or decrease the value and these methods
*  
*  are also used to determine the minimum value.
* @type {Double}
* @default null
*/
min?: Double;

/**
* In SpinnerItem.writeStackedIcons, the icon to increase the spinner's
*  value (an up arrow by default). This icon is generated automatically using
*  the AutoChild pattern. For skinning purposes, increaseIconDefaults
*  may be modified using Class.changeDefaults. 
*  
*  If sizes for the increase and decrease icons are not explicitly specified in their
*  autoChild configuration, they will be derived from the specified 
*  SpinnerItem.stackedIconsHeight and SpinnerItem.stackedIconsWidth 
*  properties.
*  
*  See the
*  skinning for details on how to provide a sprited
*  image for these icons.
* @type {FormItemIcon}
* @default null
*/
increaseIcon?: FormItemIcon;

/**
* FormItemIcon properties applied to the SpinnerItem.increaseIcon AutoChild of this
*  SpinnerItem.
* @type {Partial<FormItemIcon>}
* @default null
*/
increaseIconProperties?: Partial<FormItemIcon>;

/**
* In SpinnerItem.writeStackedIcons, the icon to decrease the spinner's
*  value (a down arrow by default). This icon is generated automatically using
*  the AutoChild pattern. For skinning purposes, decreaseIconDefaults
*  may be modified using Class.changeDefaults. 
*  
*  If sizes for the increase and decrease icons are not explicitly specified in their
*  autoChild configuration, they will be derived from the specified 
*  SpinnerItem.stackedIconsHeight and SpinnerItem.stackedIconsWidth 
*  properties.
*  
*  See the
*  skinning for details on how to provide a sprited
*  image for these icons.
* @type {FormItemIcon}
* @default null
*/
decreaseIcon?: FormItemIcon;

/**
* FormItemIcon properties applied to the SpinnerItem.decreaseIcon AutoChild of this
*  SpinnerItem.
* @type {Partial<FormItemIcon>}
* @default null
*/
decreaseIconProperties?: Partial<FormItemIcon>;

/**
* When set to true, the increase and decrease icons are stacked on top of
*  each other, also called stacked mode. When false, the increase and decrease
*  icons are placed on the same line as the SpinnerItem's text box, also called
*  unstacked mode. When null, a default setting depending on Browser.isTouch
*  is used (for touch browsers, the default is false/unstacked mode).
*  
*  In stacked mode, SpinnerItem.increaseIcon and SpinnerItem.decreaseIcon
*  control the appearance of the stacked icons.
*  
*  In unstacked mode, SpinnerItem.unstackedIncreaseIcon and SpinnerItem.unstackedDecreaseIcon
*  control the appearance of the unstacked icons.
* @type {boolean}
* @default null
*/
writeStackedIcons?: boolean;

/**
* In SpinnerItem.writeStackedIcons this property can be 
*  used to specify the width of both the increase and decrease icon. 
*  If a width property is explicitly
*  set for the icon via SpinnerItem.increaseIconProperties,
*  SpinnerItem.decreaseIconProperties, or related Defaults property
*  blocks, that will take precedence over any specified stackedIconsWidth.
*  
*  See also SpinnerItem.stackedIconsHeight.
* @type {Integer}
* @default "16"
*/
stackedIconsWidth?: Integer;

/**
* In SpinnerItem.writeStackedIcons this property can be
*  used to specify the height of both the increase and decrease icon. Since the
*  icons are stacked vertically, each icon will be sized to half this specified value.
*  If a height property is explicitly
*  set for the icon via SpinnerItem.increaseIconProperties,
*  SpinnerItem.decreaseIconProperties, or related Defaults property
*  blocks, that will take precedence over any specified stackedIconsHeight.
*  
*  See also SpinnerItem.stackedIconsWidth.
* @type {Integer}
* @default "18"
*/
stackedIconsHeight?: Integer;

/**
* In SpinnerItem.writeStackedIcons, the icon to increase the
*  SpinnerItem's value.
*  
*  By default, "[SKIN]/DynamicForm/Spinner_increase_icon.png" is stretched to
*  an 18x18 icon.
*  
*  When skinning is enabled, this property will not 
*  be used to locate an image, instead, the image is drawn via CSS based on the 
*  FormItemIcon.baseStyle property.
* @type {FormItemIcon}
* @default null
*/
unstackedIncreaseIcon?: FormItemIcon;

/**
* In SpinnerItem.writeStackedIcons, the icon to decrease the
*  SpinnerItem's value.
*  
*  By default, "[SKIN]/DynamicForm/Spinner_decrease_icon.png" is stretched to
*  an 18x18 icon.
*  
*  When skinning is enabled, this property will not 
*  be used to locate an image, instead, the image is drawn via CSS based on the 
*  FormItemIcon.baseStyle property.
* @type {FormItemIcon}
* @default null
*/
unstackedDecreaseIcon?: FormItemIcon;







/**
*  When SpinnerItem.min and SpinnerItem.max are both null or unspecified,
*  this method is called to get the next higher value from the currentValue. The default
*  implementation returns (currentValue&nbsp;+&nbsp;step).
*  
*  To indicate that the given currentValue is the maximum value, return currentValue again.
*  
*  Implementations should expect to be passed any value for currentValue. Also, if
*  SpinnerItem.step is non-negative, getNextValue() must induce a
*  monotonically increasing (non-decreasing) function;
*  otherwise, getNextValue() must induce a monotonically decreasing function.
* 
* @param {number} the current value of this SpinnerItem
* @param {number} a suggested step value based on SpinnerItem.step and how long the user has been continuously increasing the value.
*/
getNextValue(currentValue:number, step:number): number;

/**
*  When SpinnerItem.min and SpinnerItem.max are both null or unspecified,
*  this method is called to get the previous lower value from the currentValue. The default
*  implementation returns (currentValue&nbsp;+&nbsp;step) because the step parameter
*  is based on the opposite of SpinnerItem.step.
*  
*  To indicate that the given currentValue is the minimum value, return currentValue again.
*  
*  Implementations should expect to be passed any value for currentValue. Also, if
*  SpinnerItem.step is non-negative, getPreviousValue() must induce a
*  monotonically decreasing (non-increasing) function;
*  otherwise, getPreviousValue() must induce a monotonically increasing function.
* 
* @param {number} the current value of this SpinnerItem
* @param {number} a suggested step value based on the opposite of SpinnerItem.step and how long the user has been continuously decreasing the value.
*/
getPreviousValue(currentValue:number, step:number): number;




}

/**
* Name of a SmartClient Class that subclasses FormItem. Some values with this type:
*  
* - TextItem
*  
* - SliderItem, 
*  
* - CanvasItem
*  
*/
type FormItemClassName = string;








/**
* An object representing a user-created and user-modifiable formula, which can be created and
*  edited with a FormulaBuilder either directly or via the
*  ListGrid.canAddFormulaFields behavior.
*/
class UserFormula {


/**
* Formula to be evaluated.
*  
*  There are two contexts where a UserFormula is used: ListGridField.userFormula and
*  FormItem.formula or ListGridField.editorFormula. For the grid field formula
*  all variables used by the formula must be single-letter capital characters (eg A). These
*  are derived from field values for the record in question - see UserFormula.formulaVars.
*  
*  In addition to these variables, the keyword record may be used to
*  refer directly to the record for which the formula is being displayed.
*  
*  In the second usage context variables are dot-separated (.) names representing the nested
*  hierarchy path to the desired value within the Canvas.ruleScope. No
*  mapping with UserFormula.formulaVars is needed.
* @type {string}
* @default null
*/
text?: string;

/**
* Object mapping from variable names to fieldNames. All variable names must be single-letter
*  capital characters (eg A). For example, for a formula that should divide the field
*  "population" over the field "area", the formula might be "E/L" and formula vars would be:
*  
*  
*   {
*     E: "population",
*     L: "area"
*   }
*  
*  
*  
*  
*  When used in ListGridField.userFormula context, field names are evaluated against the
*  grid record.
*  
*  When used in FormItem.formula or ListGridField.editorFormula this property is
*  not used for formula mapping. Instead, field names are evaluated directly against the
*  current Canvas.ruleScope.
* @type {Map<any>}
* @default null
*/
formulaVars?: Map<any>;










}

/**
* Request sent to the server to initiate a 
*  dataSourceOperations. All properties which are legal on
*  RPCRequest are legal, in addition to the properties listed here.
*/
class DSRequest extends RPCRequest {


/**
* Should the HTTP response to this request be formatted using the
*  strict JSON subset of the javascript language?
*  If set to true, responses returned by the server should match the format described
*  here.
*  
*  Only applies to requests sent a server with
*  DataSource.dataFormat set to "json" or "iscServer".
* @type {boolean}
* @default null
*/
useStrictJSON?: boolean;

/**
* Whether formatting settings should be applied to data being exported. Default behavior and
*  the effect of setting of exportRawValues is described in the
*  exportFormatting.
* @type {boolean}
* @default null
*/
exportRawValues?: boolean;

/**
* Determines the PropertyIdentifier to be used in the exported data. This
*  essentially means, should we export internal field names like "countryCode" or
*  "EMPLOYEE_NO", or localized descriptive field titles like "code du pays" or
*  "Employee Number". This setting has a lot in common with DSRequest.exportRawValues;
*  both are largely dependent on whether the exported data is intended for direct consumption by
*  an end user (in which case it is appropriate to export formatted values and localized field
*  titles), or for interface to some downstream computer system (in which case you probably
*  want raw, unformatted values and internal field names).
*  
*  If this property is not set, the following defaults apply:
*  
* - If the export format is a native spreadsheet format (XLS or OOXML), localized field 
*  titles are used
*  
* - If the export format is CSV, XML or JSON and this is a client-driven export (ie it was
*  initiated by a call to ListGrid.exportClientData), 
*  localized field titles are used
*  
* - If the export format is CSV, XML or JSON and this is not a client-driven export,
*  internal field names are used 
*  
* @type {PropertyIdentifier}
* @default null
*/
exportPropertyIdentifier?: PropertyIdentifier;

/**
* DataSource this DSRequest will act on.
*  
*  This property is generally automatically populated, for example when calling
*  DataSource.fetchData the dataSource property is set to the target
*  DataSource.
* @type {string}
* @default null
*/
dataSource?: string;

/**
* Type of operation being performed: "fetch", "add", "remove", "update" or "custom".
*  
*  This property is generally automatically populated, for example when calling
*  fetchData() on a DataSource or DataBound component the operationType is
*  automatically set to "fetch". Note that "custom" operations are never generated
*  automatically, they are always fired by your code.
* @type {DSOperationType}
* @default null
*/
operationType?: DSOperationType;

/**
* Data, such as search criteria or an updated record, to be acted upon. Contents differ by
*  operationType, see dataSourceOperations for
*  details.
*  
*  This field is generally filled in by passing the "data" argument to methods such as
*  ListGrid.fetchData.
* @type {object | Array<Partial<object>>}
* @default null
*/
data?: object | Array<Partial<object>>;

/**
* A callback method that will be called with an instance of DSResponse, as sent by the server.
*  Queuing does not affect callbacks in any way - your specified callback will be invoked for 
*  each DSRequest that contained a callback regardless of whether the request was sent as part
*  of a queue or not.
*  
*  Note that if the request encounters an error (such as 500 server error), by default the
*  callback will not be fired. Instead, RPCManager.handleError is called to
*  invoke the default system-wide error handling. Set RPCRequest.willHandleError:true to have your
*  callback invoked regardless of whether there are errors; however, make sure your callback
*  properly handles malformed responses when RPCResponse.status is non-zero. See the
*  errorHandling below for more details.
* @type {DSCallback}
* @default null
*/
callback?: DSCallback;

/**
* Starting row of requested results, used only with fetch operations. If unset, 0 is assumed.
*  
*  Note that startRow and endRow are zero-based, inclusive at the beginning and exclusive at
*  the end (like substring), so startRow: 0, endRow: 1 is a request for the first record.
* @type {number}
* @default null
*/
startRow?: number;

/**
* End row of requested results, used only with fetch operations.
*  
*  Note that startRow and endRow are zero-based, inclusive at the beginning and exclusive at
*  the end (like substring), so startRow: 0, endRow: 1 is a request for the first record.
* @type {number}
* @default null
*/
endRow?: number;

/**
* Field name to sortBy, prefixed with optional "-" indicating descending sort. For example, to
*  sort by the field "userName" in ascending order, set sortBy to just
*  "userName". For descending sort on "userName", set sortBy to "-userName". 
*  
*  To sort by multiple fields, an array of field names is also supported. For example, to sort
*  by the field "department" in ascending order, followed by the field "userName" in descending
*  order, set sortBy to: 
*  
*  [ "department", "-userName" ]
*  
*  Additionally, this property supports an array of SortSpecifier objects. Setting 
*  sortBy to the following SortSpecifier array results in the same multi-level
*  sort mentioned above:
*  
*  
*  [
*  &nbsp;&nbsp;{ property: "department", direction: "ascending" },
*  &nbsp;&nbsp;{ property: "userName", direction: "descending" }
*  ]
*  
* @type {FieldName | Array<Partial<FieldName>> | Array<Partial<SortSpecifier>>}
* @default null
*/
sortBy?: FieldName | Array<Partial<FieldName>> | Array<Partial<SortSpecifier>>;

/**
* A mapping from field names to SummaryFunction to be applied
*  to each field.
*  
*  Valid only for an operation of type "fetch". See the 
*  serverSummaries for examples of usage.
* @type {Map<SummaryFunction>}
* @default null
*/
summaryFunctions?: Map<SummaryFunction>;

/**
* When using RPCManager.exportContent to produce a .pdf from a SmartClient UI, this property allows 
*  dynamic CSS to be passed to the server. Since the exportContent() system already provides 
*  a way to specify a custom skin or additional stylesheet for export, exportCSS should only be
*  used for small bits of CSS that are necessarily dynamic.
*  
*  For example, when printing a very wide page, such as a grid with many columns or a very wide chart,
*  you could send the string "@page {size: A4 landscape; }" as exportCSS to cause the
*  generated PDF to use landscape mode, so that all content fits without clipping.
* @type {string}
* @default null
*/
exportCSS?: string;

/**
* List of fields to group by when using serverSummaries.
*  
*  Valid only for an operation of type "fetch". See the 
*  serverSummaries for details and examples of usage.
* @type {Array<Partial<string>>}
* @default null
*/
groupBy?: Array<Partial<string>>;

/**
* The list of fields to return in the response, specified as a comma-separated string (eg,
*  "foo, bar, baz"). You can use this property to indicate to the server that 
*  you are only interested in a subset of the fields that would normally be returned.
*  
*  Note that you cannot use this property to request a superset of the fields that
*  would normally be returned, because that would be a security hole. It is possible to 
*  configure individual OperationBindings to return extra fields, but this must
*  be done in the server's DataSource descriptor; it cannot be altered on the fly
*  from the client side.
* @type {string}
* @default null
*/
outputs?: string;

/**
* For fetch, add or update operation, an optional comma separated list of fields to fetch from another,
*  related DataSource.
*  
*  Fields should be specified in the format 
*  "localFieldName!relatedDataSourceID.relatedDataSourceFieldName".
*  where relatedDataSourceID is the ID of the related dataSource, and
*  relatedDataSourceFieldName is the field for which you want to
*  fetch related values. The returned field values will be stored on 
*  the data returned to the client under the specified localFieldName.
*  Note that this will be applied in addition to any specified DSRequest.outputs.
*  
*  Note that as with DataSourceField.includeFrom, the related dataSource must be
*  linked to the primary datasource via a foreignKey relationship.
*  
*  Note additionalOutputs sent in request from the browser can be completely disabled in 
*  server_properties by setting
*  datasource.allowClientAdditionalOutputs:
*  
*    datasource.allowClientAdditionalOutputs: false
*  
*  In this case DSRequest.additionalOutputs sent from the browser will be cleared before 
*  executing request. Note that programatically configured additionalOutputs are always allowed, but 
*  you can't modify them from within a DMI method, so the only way to execute a request 
*  with additionalOutputs that differ from what was sent by the client is to create a new DSRequest
* @type {string}
* @default null
*/
additionalOutputs?: string;

/**
* Specifies should related updates have to be generated.
*  If not set (or set to null) then related updates will be generated only for
*  "add" and "update" operations. This property has to be explicitly set to true
*  to generate related updates for "remove" operation.
*  This functionality loads related objects from database thus affecting operation performance.
*  For "add" and "update" operations related objects are loaded anyway and performance impact is minimal.
*  Simple "remove" operation does not need to load related objects. Depending on database structure
*  performance impact can be significant if this property is set to true.
*  
*  Note this feature works only with Hibernate/JPA data sources, see 
*  jpaHibernateRelations for instructions how to 
*  set up relations. Table below uses "country -&gt; cities" sample data model.
*  
*  
*   
*    Relation and Operation type
*    Loading complete related objects
*    Loading related IDs
*   
*   
*    Many-to-one (cities -&gt; country): ADD/UPDATE
*    If operation affected country, for example new city added with existing 
*  countryId, then relatedUpdate is generated. Otherwise if city is added or updated without 
*  countryId set, relatedUpdate is not generated.
*  Note that if provided countryId does not exist, it is created.
*    Same as with complete related objects, except if provided countryId does not exist, 
*  then it is not created, but reset to NULL.
*   
*   
*    Many-to-one (cities -&gt; country): REMOVE
*    Removes record, depending on setting generates or not relatedUpdate for parent record.
*  For example if city record is removed and countryId is sent to the server in remove request, then 
*  country record will be generated in relatedUpdates.
*   
*   
*    One-to-many (country -&gt; cities): ADD/UPDATE
*    If add or update operation provides value sets for cities as well as for country, then 
*  cities are created/updated if necessary and relatedUpdates are generated.
*  Note that all fields in cities value sets can be sent to server.
*    Same as with complete related objects, except you can only sent primary key values for cities.
*   
*    One-to-many (country -&gt; cities): REMOVE
*    Removes country, depending on setting returns or not relatedUpdates for the cities of removed country, 
*  which can be either REMOVE operations of all cities if cascade enabled, or UPDATE operations setting countryId=null to 
*  all cities if cascade is disabled
*   
*  
*  
*  Note that Many-to-Many works the same way as One-to-Many.
* @type {boolean}
* @default null
*/
generateRelatedUpdates?: boolean;

/**
* For "fetch" operations, how search criteria should be interpreted for text fields: one of
*  "exact" for exact match, "exactCase" for case-sensitive exact match, "startsWith" for 
*  matching at the beginning only, or "substring" for substring match. All 
*  textMatchStyle settings except "exactCase" are case-insensitive; use
*  AdvancedCriteria for greater control over matching.
*  
*  This property defaults to the value of DataSource.defaultTextMatchStyle if it is not
*  explicitly provided on the DSRequest. Note, however, that DSRequests issued 
*  by ListGrids and other DataBoundComponent will generally have a 
*  setting for textMatchStyle on the component itself 
*  (see ListGrid.autoFetchTextMatchStyle, for example).
*  
*  This setting is respected by the built-in server-side connectors for SQL, JPA and Hibernate.
*  A custom server-side DataSource implementation should generally respect this flag as well,
*  or server-side filtering will not match client-side filtering, which will require
*  ResultSet.useClientFiltering, a huge performance
*  loss.
* @type {TextMatchStyle}
* @default null
*/
textMatchStyle?: TextMatchStyle;

/**
* For an update or remove operation, the original values from the
*  record that is being updated or removed. oldValues is automatically added to
*  DSRequests submitted by DataBound Components. Available on the server via
*  DSRequest.getOldValues().
*  
*  The server can compare the oldValues to the most recent stored values in order
*  to detect that the user was looking at stale values when the user submitted changes (NOTE:
*  this means of detecting concurrent edit is sometimes called "optimistic concurrency" or
*  "long transactions"). 
*  
*  In applications where a policy of "last update wins" is not appropriate when updating
*  certain fields, special UI can be shown for this case. For example, on detecting concurrent
*  edit, the server may send back a special dsResponse.status code that the client
*  application detects, offering the user a choice of proceeding with the operation,
*  discarding edits, or reconciling new and old values in a special interface.
* @type {Record}
* @default null
*/
oldValues?: Record;

/**
* For requests submitted by a DataBoundComponent, the Canvas.ID of the
*  submitting component.
*  
*  This ID will be present for operations including automatic saves by a ListGrid 
*  editing, or calls to
*  DynamicForm.saveData. It will not be present for a direct
*  call to a DataSource method such as DataSource.fetchData.
*  
*  Note this is the component's String ID - you can retrieve the component itself
*  via Canvas.getById.
*  
*  This property should be used for debugging purposes only - do not use it to trigger
*  differences in server-side behavior, instead, use DSRequest.operationId because only
*  operationId is considered when assessing
*  dsRequestEquivalence.
* @type {string}
* @default null
*/
componentId?: string;

/**
* When a DataBoundComponent sends a DSRequest, the
*  dsRequest.operationId will be automatically picked up from the
*  fetchOperation, addOperation, etc properties of the
*  DataBoundComponent. 
*  
*  The operationId serves as an identifier that you can use to create
*  variations on the 4 basic DataSource operations that are used by different components in
*  different parts of your application. For example, you may be using a standard
*  fetch operation in one part of your application, however on another screen you
*  want to perform a fetch operation on the same DataSource but interpret search
*  criteria differently (eg full text search). 
*  
*  If you declare more than one OperationBinding for the same
*  OperationBinding.operationType, you can specify an operationId
*  OperationBinding.operationId which will cause that
*  operationBinding to be used for dsRequests containing a matching operationId.
*  This allows all the possible settings of an operationBinding, including
*  OperationBinding.wsOperation or dmiOverview settings, to be switched on a
*  per-component or per-request basis. 
*  
*  For example, by setting the fetchOperation on a particular ListGrid, you could
*  cause it to invoke a different server method via DMI, different
*  OperationBinding.dataURL or different 
*  OperationBinding.wsOperation.
*  
*  The operationId can also be directly received by the server in order to affect
*  behavior. When using the SmartClient Server, operationId can be accessed via
*  dsRequest.getOperationId(). The RestDataSource will also send the
*  operationId to the server as part of the
*  RestDataSource.metaDataPrefix. 
*  
*  Note that if you DataSource.fetchData a DataSource operation, you can
*  also specify operationId via the requestProperties parameter.
*  
*  Note that the operationId has special significance in terms of whether two
*  DSRequests are considered equivalent for caching and synchronization purposes - see
*  dsRequestEquivalence.
* @type {string}
* @default null
*/
operationId?: string;

/**
* For a databound Tree.isMultiLinkTree, this is the 
*  operationId to use for the separate fetch on the ResultTree.linkDataSource
*  that will be generated if LinkDataFetchMode is "separate". This property 
*  overrides the ResultTree.linkDataFetchOperation property on 
*  ResultTree, for this fetch only.
*  
*  Ignored if this DSRequest is not a fetch against a multi-link tree.
* @type {string}
* @default null
*/
linkDataFetchOperation?: string;

/**
* DataSource.dataProtocol for this particular request.
*  
*  Note: Typically developers should use DataSource.operationBindings
*  to specify an explicit data protocol for a request.
*  
*  One exception: advanced developers may wish to have a custom 
*  DataSource.transformRequest with entirely client-side
*  handling for some requests. This may be achieved by setting the request's
*  dataProtocol to DSProtocol within transformRequest,
*  and also triggering application code which will fire DataSource.processResponse
*  when complete.
*  
*  The DataSource.getDataProtocol method may be used to determine what data protocol
*  will be used to handle a specific request based on this property (if set), otherwise the
*  settings at the OperationBinding.dataProtocol or 
*  DataSource.dataProtocol levels.
* @type {DSProtocol}
* @default null
*/
dataProtocol?: DSProtocol;

/**
* Automatically generated unique ID for this request. This ID will be required by developers 
*  making use of the DSProtocol.
* @type {string}
* @default "varies"
*/
requestId?: string;

/**
* An object to be held onto for the duration of the DSRequest turnaround to track
*  application-specific context.
*  
*  When a DataSource request completes, the clientContext is available in the
*  DSCallback as dsResponse.clientContext. The
*  clientContext is never sent to the server.
*  
*  The clientContext is useful for holding onto state that will be used when the
*  DSCallback fires, such as the name of a component that will receive the
*  returned data.
* @type {object}
* @default "{}"
*/
clientContext?: object;

/**
* For advanced use in integrating trees that 
*  TreeGrid.loadDataOnDemand with web services,
*  parentNode is automatically set in "fetch" DSRequests issued by a
*  databound TreeGrid that is loading children for that parentNode.
*  
*  This is sometimes needed if a web service requires that additional properties beyond the
*  ID of the parentNode must be passed in order to accomplished level-by-level loading.
*  A custom implementation of DataSource.transformRequest can access
*  dsRequest.parentNode and add any such properties to DSRequest.data.
*  
*  parentNode will also be automatically set by a TreeGrid performing databound
*  reparenting of nodes, as implemented by TreeGrid.folderDrop.
*  
*  This property can only be read. There is no meaning to setting this property
*  yourself.
* @type {TreeNode}
* @default null
*/
parentNode?: TreeNode;

/**
* For advanced use in integrating trees that 
*  ResultTree.loadDataOnDemand with web services,
*  the ResultTree that issued this "fetch" DSRequest is automatically made available as the
*  resultTree property.
*  
*  This property can only be read. There is no meaning to setting this property
*  yourself.
* @type {ResultTree}
* @default null
*/
resultTree?: ResultTree;

/**
* This property is for advanced use in integrating trees that 
*  TreeGrid.loadDataOnDemand using data paging.
*  When this flag is set, a server fetch operation is expected to return all of the tree nodes
*  that either match the provided criteria or have one or more children that match the
*  criteria.
*  
*  A ResultTree with ResultTree.fetchMode and with
*  ResultTree.keepParentsOnFilter enabled will automatically set
*  this property to true on all DSRequests that it sends to the server.
*  
*  Currently, no built-in server-side connectors (SQL, JPA, Hibernate) implement support for
*  the keepParentsOnFilter flag.
* @type {boolean}
* @default null
*/
keepParentsOnFilter?: boolean;

/**
* For advanced use in integrating dataset paging with web services,
*  the ResultSet that issued this "fetch" DSRequest is automatically made available as the
*  resultSet property.
*  
*  This property can only be read. There is no meaning to setting this property
*  yourself.
* @type {ResultSet}
* @default null
*/
resultSet?: ResultSet;

/**
* This is a per-request flag for explicitly controlling whether the cache is used (bypassing
*  it when not wanted, or using it when settings would indicate otherwise). See 
*  DataSource.cacheAllData, DataSource.cacheAllOperationId and
*  DataSource.cacheAcrossOperationIds for caching management for all requests of a 
*  dataSource.
* @type {boolean}
* @default null
*/
shouldUseCache?: boolean;

/**
* When useFlatFields is set for a request to be sent to a WSDL web service, when
*  creating the input XML message to send to the web service, properties in
*  DSRequest.data will be used as the values for XML elements of the same name, at
*  any level of nesting.
*  
*  useFlatFields allows you to ignore gratuitous XML message structure, such as
*  extra levels of nested elements, and provides some insulation against changes in the
*  required structure of the input message.
*  
*  For example, given this input message:
*  
*  &lt;FindServices&gt;
*    &lt;searchFor&gt;search text&lt;/searchFor&gt;
*    &lt;Options&gt;
*      &lt;caseSensitive&gt;false&lt;/caseSensitive&gt;
*    &lt;/Options&gt;
*    &lt;IncludeInSearch&gt;
*      &lt;serviceName&gt;true&lt;/serviceName&gt;
*      &lt;documentation&gt;true&lt;/documentation&gt;
*      &lt;keywords&gt;true&lt;/keywords&gt;
*    &lt;/IncludeInSearch&gt;
*  &lt;/FindServices&gt;
*  
*  If useFlatFields were not set, in order to fill out this message
*  correctly, request.data would need to be:
*  {
*   searchFor: "search text",
*   Options : {
*     caseSensitive: false,
*   },
*   IncludeInSearch : {
*     serviceName: true,
*     documentation : true,
*     keywords : true
*   }
*  }
*  However if useFlatFields were set, request.data could be just:
*  {
*   searchFor: "search text",
*   caseSensitive: false,
*   serviceName: true,
*   documentation : true,
*   keywords : true
*  }
*  useFlatFields is often set when the input data comes from a DynamicForm
*  to avoid the cumbersome and fragile process of mapping input fields to an XML structure.
*  
*  OperationBinding.useFlatFields can also be set to cause all dsRequests of a
*  particular type to useFlatFields automatically.
*  
*  For DataBoundComponent, 
*  DataBoundComponent.useFlatFields can be set use "flattened"
*  binding to fields of a WSDL message or XML Schema.
*  
*  Note that useFlatFields is not generally recommended for use with XML input
*  messages where multiple simple type fields exist with the same name, however if used in this
*  way, the first field to use a given name wins. "first" means the first field encountered in a
*  depth first search. "wins" means only the first field will be populated in the generated
*  XML message.
* @type {boolean}
* @default null
*/
useFlatFields?: boolean;

/**
* Cause the DSRequest.useFlatFields XML serialization behavior to be used
*  for all soap headers in the request. See also DSRequest.headerData.
* @type {boolean}
* @default null
*/
useFlatHeaderFields?: boolean;

/**
* For DataSources using SOAP messaging with a WSDL web service, data to be serialized to form
*  SOAP headers, as a map from the header part name to the data. See
*  WSRequest.headerData for more information.
*  
*  SOAP headers typically contain request metadata such as a session id for authentication, and
*  so dsRequest.headerData is typically populated by
*  DataSource.transformRequest, or, for data that applies to every request sent to the
*  server, by WebService.getHeaderData.
*  
*  Note that this only applies to SOAP headers. General HTTP headers for requests may be
*  modified using RPCRequest.httpHeaders.
* @type {object}
* @default null
*/
headerData?: object;

/**
* When set, causes the results of the DSRequest to be exported to a file, whose name 
*  and format are indicated by DSRequest.exportFilename and 
*  DSRequest.exportAs respectively. When no exportFilename is provided, the default is
*  Results.csv and the default value of exportAs is csv. 
*  
*  The export field-list can also be configured, see DSRequest.exportFields. Formats 
*  for exported date and numeric are controlled by several settings - see 
*  exportFormatting for an overview.
*  
*  Once the operation completes, DSRequest.exportDisplay specifies whether the exported
*  data should be downloaded to the file-system or displayed in a new window. The default value
*  of exportDisplay is "download" which displays the Save As dialog. See ExportDisplay 
*  for more information.
*  
*  You can configure the style of LineBreakStyle to use when 
*  generating the output, the DSRequest.exportDelimiter to use when exporting
*  to CSV and the DSRequest.exportTitleSeparatorChar to use in 
*  field-titles when exporting to XML.
*  
*  Additionally, you can output arbitrary text before and after the exported data by setting 
*  DSRequest.exportHeader and DSRequest.exportFooter.
*  
*  Note that for security reasons, an export initiated using dsRequest properties does not
*  provide support for JSON format (see 
*  this post for more detail). 
*  However, you can use operationBinding.exportAs:"json" in a server-side .ds.xml file to force 
*  JSON export to be allowed.
*  
*  As well as setting dsRequest.exportResults and related properties, exports can be initiated
*  in two other ways, via OperationBindings and via custom server code which sets 
*  export-related properties on the DSResponse. Both of those methods support exporting
*  to JSON format.
*  
*  Format Examples
*  XML format
*  
*    &lt;List&gt;
*      &lt;Object&gt;
*        &lt;id&gt;10101&lt;/id&gt;
*        &lt;displayName&gt;Record 10101&lt;/displayName&gt;
*      &lt;/Object&gt;
*   &lt;/List&gt;
*  
*  JSON Format
*  
*    [
*      { id: 10101, displayName: "Record 10101" }
*    ]
*  
*  CSV Format
*  
*    id,displayName
*    10101,"Record 10101"
*  
* @type {boolean}
* @default false
*/
exportResults?: boolean;

/**
* The format in which the data should be exported. See ExportFormat for more 
*  information.
* @type {ExportFormat}
* @default "csv"
*/
exportAs?: ExportFormat;

/**
* The name of the file to save the exported data into. If 
*  DSRequest.exportToFilesystem is set, this is the name of the 
*  file the server creates on its filesystem. If DSRequest.exportToClient
*  is set, this is the filename that will appear to the browser.
*  
*  If the exportFilename that you specify does not include an extension, one will be added to
*  it based on the ExportFormat specified by DSRequest.exportAs. Filename is
*  forced to have the correct extension to work around bugs in IE, but if you don't want the
*  filename to be manipulated, use "custom" ExportFormat, see example.
* @type {string}
* @default null
*/
exportFilename?: string;

/**
* The image format in which the SVG graphic should be exported.
* @type {ExportImageFormat}
* @default "png"
*/
exportImageFormat?: ExportImageFormat;

/**
* If exporting in ExportImageFormat, the output JPEG quality level.
*  This is a number from 0 to 1, with 1 representing the best quality and 0 representing the
*  least quality but smallest file size.
* @type {Float}
* @default null
*/
exportImageQuality?: Float;

/**
* If DSRequest.exportToFilesystem is set, optionally specifies a 
*  path to use when saving the file. This path is relative to the default export path, which
*  is set using the server_properties setting export.location; this
*  is the project webRoot by default. For example, with the default setting of 
*  export.location, an exportPath of "shared/ds" and 
*  an DSRequest.exportFilename of "exportedData.csv",
*  SmartClient Server would export to file $webRoot/shared/ds/exportedData.csv.
*  
*  If you do not specify this property, SmartClient Server will export to the file indicated 
*  by exportFilename directly in the default export location.
*  
*  This property is only applicable when DSRequest.exportToFilesystem
*  is set.
* @type {string}
* @default null
*/
exportPath?: string;

/**
* If set, SmartClient Server will export data to a file on the server filesystem. The
*  file we export to is determined by the DSRequest.exportFilename and 
*  DSRequest.exportPath. Note that filesystem exports are disabled by 
*  default, for security reasons. To enable them, set export.allow.filesystem to
*  true in your server.properties file. If you enable filesystem exports, you 
*  should also consider setting a default export path, as described in the 
*  DSRequest.exportPath documentation.
*  
*  Note that it is perfectly valid to specify both this property and 
*  DSRequest.exportToClient; in this case the data is both exported to a
*  file on the server filesystem and downloaded to the client. If you specify 
*  neither property, the export no-ops.
*  
*  It is possible to redirect the filesystem export to make use of an OutputStream
*  you provide. You use this when you want to make some use of the export document other than 
*  writing it to a disk file - for example, attaching it to an email or writing it to a 
*  database table. See the server-side Javadocs for DSRequest.setExportTo().
* @type {boolean}
* @default false
*/
exportToFilesystem?: boolean;

/**
* If set, SmartClient Server will export data back to the client, either as a file download 
*  or as content in a new browser window, depending on the setting of
*  DSRequest.exportDisplay.
*  
*  Note that it is perfectly valid to specify both this property and 
*  DSRequest.exportToFilesystem; in this case the data is both 
*  exported to a file on the server filesystem, and downloaded to the client. If you specify 
*  neither property, the export no-ops.
* @type {boolean}
* @default true
*/
exportToClient?: boolean;

/**
* The character with which to replace spaces in field-titles when exporting to XML. If not
*  specified in the request, the server uses "".
* @type {string}
* @default null
*/
exportTitleSeparatorChar?: string;

/**
* Specifies whether the exported data will be downloaded as an attachment or displayed in a 
*  new browser window. See ExportDisplay for more information.
* @type {ExportDisplay}
* @default "download"
*/
exportDisplay?: ExportDisplay;

/**
* The style of line-breaks to use in the exported output. See LineBreakStyle for
*  more information.
* @type {string}
* @default null
*/
lineBreakStyle?: string;

/**
* The character to use as a field-separator in CSV exports. The default delimiter is comma.
* @type {string}
* @default ","
*/
exportDelimiter?: string;

/**
* When you erxport a ListGrid that has ListGrid.headerSpans,
*  should headerSpans also be exported. See DSRequest.exportSpanTitleSeparator for details of
*  of what it means to export headerSpans to different export targets.
*    
*  Note that DSRequest.exportPropertyIdentifier controls whether field names or titles
*  are appended to the headerSpan titles (and used for fields without headerSpans).
* @type {boolean}
* @default true
*/
exportShowHeaderSpanTitles?: boolean;

/**
* When you export a ListGrid that has ListGrid.headerSpans
*  defined and DSRequest.exportShowHeaderSpanTitles is true, the behavior is dependent
*  on the export type. Direct exports to Excel formats (both XLS and OOXML) place the 
*  headerSpans in merged cells in the spreadsheet, giving the same visual effect as the 
*  original ListGrid. This is not possible with exports to CSV format; instead, we alter the 
*  exported headers so that they contain the titles of the ancestor headerSpan(s).
*  
*  For example, if you had a field titled "Population" inside a headerSpan titled "National",
*  nested inside another headerSpan titled "Demographics", that would result in the exported
*  field being titled "Demographics - National - Population".
*  
*  The exportSpanTitleSeparator property allows you to override the separator 
*  string used when constructing these amalgamated headers.
* @type {string}
* @default " - "
*/
exportSpanTitleSeparator?: string;

/**
* The list of field names to export. If provided, the field list in the exported output is 
*  limited and sorted as per the list.
*  
*  If exportFields is not provided: 
*  
* - If we are exporting via DSRequest.exportData, the field list in the exported
*  output is every non-hidden field defined in the DataSource, in DataSource definition order
*  
* - If we are exporting via ListGrid.exportClientData and we are not 
*  exporting to OOXML, or we are exporting to OOXML but we are not 
*  DSRequest.exportStreaming, the field list in the exported output is based on the 
*  client data sent up, taking every row into account (so if there is a value for field "foo"
*  only in row 57, we will output a column "foo", the cells of which are empty except for 
*  row 57)
*  
* - If we are exporting via ListGrid.exportClientData and we are 
*  exporting to OOXML and streaming is in force (the default for OOXML), the field list in 
*  the exported output is based on the client data sent up, taking just the first row into 
*  account (so if there is a value for field "foo" only in row 57, we will not output a column
*  "foo" at all) 
*  
* @type {Array<Partial<string>>}
* @default null
*/
exportFields?: Array<Partial<string>>;

/**
* Optional text to appear at the beginning of the file.
* @type {string}
* @default null
*/
exportHeader?: string;

/**
* This property allows omitting column names from CSV and Excel exports (no effect on JSON or XML exports).
* @type {boolean}
* @default false
*/
exportHeaderless?: boolean;

/**
* Optional text to appear at the end of the file.
* @type {string}
* @default null
*/
exportFooter?: string;

/**
* Only applies to request properties passed to ListGrid.exportClientData.
*  If specified this property contains an arbitrary set of data to be exported.
* @type {Array<Partial<Record>>}
* @default null
*/
exportData?: Array<Partial<Record>>;

/**
* This flag has a different meaning depending on whether you are doing a client-driven or
*  server-driven export.
*  
*  For ListGrid.exportClientData calls (client-driven), ordinarily
*  any fields that have a ListGridField.displayField defined
*  have the value of that displayField exported, rather than the underlying value in the 
*  ListGridField.valueField. If you set the exportValueFields
*  property, we export both the underlying value and the displayField value.
*  
*  Again for exportClientData() calls, any fields that have a
*  ListGridField.valueMap defined ordinarily have the mapped value of the 
*  field exported, rather than the underlying data value. If you set the 
*  exportValueFields property, we instead export the underlying data value. 
*  Note, there is only one field in this scenario, not a valueField and a 
*  separate displayField, so we export either the underlying data value or
*  the mapped value, not both as in the displayField/valueField case 
*  described above.
*  
*  For DataBoundComponent.exportData calls (server-driven), we ordinarily
*  export the underlying data value of all fields. However, if you set the 
*  exportValueFields property explicitly to false, any fields that
*  have a DataSource-defined DataSourceField.valueMap will have the mapped
*  value exported instead. This is similar to the client-side treatment of valueMaps, except
*  that the defaults are reversed.
* @type {boolean}
* @default null
*/
exportValueFields?: boolean;

/**
* When exporting via ListGrid.exportClientData to an XLS or
*  OOXML spreadsheet, forces dates to export as a string rather than a true date
*  value.
*  
*  If a date value is provided to a spreadsheet as a string, Excel or other spreadsheet
*  applications may not recognize them as being date values that are valid for use in
*  date-specific functions in formulas, filters, etc.
*  
*  For this reason, the default behavior of exportClientData is to provide date
*  values to the spreadsheet as true date values. If
*  FormatString are provided via properties like
*  DataSourceField.format these will be translated to Excel /
*  OpenOffice format strings and used when generating spreadsheets. Other formatting logic,
*  such as ListGridField.formatCellValue, will not be used since they
*  cannot be automatically translated to an Excel format string. If no translatable format
*  string is available, date values will be provided to the spreadsheet with no formatter and
*  the spreadsheet program's default formatting for date values will be used.
*  
*  If exportDatesAsFormattedString is set to true, date fields will appear as
*  strings that exactly match the formatting shown in the DataBoundComponent.
*  As noted above, this means the spreadsheet program will not recognize the value as
*  a date.
* @type {boolean}
* @default null
*/
exportDatesAsFormattedString?: boolean;

/**
* When exporting via ListGrid.exportClientData to an XLS or
*  OOXML spreadsheet, forces numbers to export as a string rather than a true
*  numerical value.
*  
*  If a number is provided to a spreadsheet as a string, Excel or other spreadsheet
*  applications may not recognize them as being numbers that are valid for use in
*  numerical formulas, filters, etc.
*  
*  For this reason, the default behavior of exportClientData is to provide
*  numerical values to the spreadsheet as true numbers. If
*  FormatString are provided via properties like
*  DataSourceField.format these will be translated to Excel /
*  OpenOffice format strings and used when generating spreadsheets. Other formatting logic,
*  such as ListGridField.formatCellValue, will not be used since they
*  cannot be automatically translated to an Excel format string. If no translatable format
*  string is available, numbers will be provided to the spreadsheet with no formatter and
*  the spreadsheet program's default formatting for numerical values will be used.
*  
*  If exportNumbersAsFormattedString is set to true, numbers will appear as
*  strings that exactly match the formatting shown in the DataBoundComponent.
*  As noted above, this means the spreadsheet program will not recognize the value as
*  a number.
* @type {boolean}
* @default null
*/
exportNumbersAsFormattedString?: boolean;

/**
* For server-side export with ExportFormat "xls" or "ooxml" only, timezone to use
*  when saving values from FieldType "datetime" to the spreadsheet.
*  
*  This setting exists because MS Excel&trade; has no concept of providing a true datetime value
*  that is timezone-independent and will display in the local timezone where the Excel program is
*  launched. Instead, datetime values must be provided as a rendered string, which implies
*  rendering in a particular timezone when the spreadsheet is generated.
*  
*  exportTZ can either be specified as a timezone offset in the same format
*  expected by Time.setDefaultDisplayTimezone (for example, "+01:00" for one hour after
*  GMT) or as the special constants "client" (meaning the current client display timezone) or
*  "server" (meaning the timezone of the server).
*  
*  Default if unspecified is "server".
*  
*  This setting does not affect fields of type "date" or "time", which are timezone-independent
*  values. See dateFormatAndStorage for more information on how SmartClient
*  handles date, time and datetime values.
*  
*  All non-spreadsheet export formats always use UTC. This setting also does not affect
*  client-driven exports (DataSource.exportClientData), which always use client-side
*  time.
* @type {string}
* @default null
*/
exportTZ?: string;

/**
* When exporting to OOXML format (this is the standard file format used by Excel 2007 and 
*  later), we default to using streaming mode, for memory efficiency. You can override this 
*  for individual exports by setting this flag false. You may wish to do this if you need to
*  grab the spreadsheet object in a DMI and do something with it. The underlying object in 
*  use - POI's SXSSFWorkbook - is intended for write only and cannot usefully be
*  read.
*  
*  You can switch off Excel streaming altogether by setting "excel.useStreaming" false in 
*  server.properties.
*  
*  Note, OOXML is the only native Excel format that supports streaming: when exporting to
*  the older XLS format, we build the spreadsheet in its entirety in server-side memory 
*  before writing it to disk or returning it to the client. This is unlikely to change:
*  streaming the XLS format is impractical bcause it is a self-referential binary format, and
*  in any case the problem of huge exports overflowing JVM memory is less likely to arise with 
*  XLS, because it is innately limited to 65535 rows.
* @type {boolean}
* @default true
*/
exportStreaming?: boolean;

/**
* Mode of validation for entered data.
* @type {ValidationMode}
* @default "full"
*/
validationMode?: ValidationMode;

/**
* Indicates that a validation request is being made for a record that will ultimately be 
*  saved with an "add" request, as opposed to an "update" request. This context is necessary
*  for some validators because the nature of the validation depends on whether we are adding
*  or updating a record. The system sets this flag when processing interim validations, such
*  as those fired when DynamicForm.validateOnChange is in force.
* @type {boolean}
* @default null
*/
pendingAdd?: boolean;

/**
* If true, results will be streamed on the server, rather than all records being read into 
*  server memory at once; this approach is appropriate for retrieving or exporting large
*  datasets without swamping the server. 
*  
*  Although this property can be set without any particular concerns (small datasets can be 
*  streamed just as readily as large ones), bear in mind that although streaming enables the
*  processing of very large datasets, processing and downloading very large datasets in a 
*  normal client/server flow will very rarely give an acceptable user experience. Streaming
*  is of more practical use in a batch setting - for example, a disconnected 
*  DSRequest.exportToFilesystem.
*  
*  Note that streaming requires specific server support; of SmartClient's built-in DataSource
*  types, only SQLDataSource is able to stream results. This property is ignored
*  by other DataSource types. If you wish to implement the necessary server-side behavior to 
*  support streaming with a custom DataSource, see the the server-side Javadocs for 
*  DSResponse.hasNextRecord() and DSResponse.nextRecordAsObject().
*  
*  See also the server-side documentation for DSResponse, 
*  SQLDataSource and StreamingResponseIterator.
*  
*  Note, that streaming results does not support fields with SummaryFunction 
*  on non-Oracle databases. Such fields will be skipped.
* @type {boolean}
* @default false
*/
streamResults?: boolean;

/**
* Sets DataSource.progressiveLoading for this specific
*  request, overriding the OperationBinding- and DataSource-level settings. Note that this 
*  setting applies only to fetch requests - it has no effect if specified on any other kind 
*  of request.
* @type {boolean}
* @default null
*/
progressiveLoading?: boolean;

/**
* A set of key:value pairs, mapping field names to expressions that will be evaluated 
*  server-side to derive a value for that field. This property allows for client-driven 
*  transactionChaining, with some restrictions for security
*  reasons:
*  
* - Normal OperationBinding.values for a 
*  field take precedence over this property, so server-defined rules cannot be overridden 
*  from the client
*  
* - Arbitrary Velocity expressions are not allowed in DSRequests sent from the client 
*  (fieldValueExpressions is also a valid property on a server-side DSRequest,
*  and normal Velocity expressions are allowed in that case - see the server-side
*  Javadoc for DSRequest.setFieldValueExpressions()). For client-originated 
*  requests, only the following bindings are allowed - see the 
*  velocitySupport for details of what these values mean:
*   
* - $currentDate
*   
* - $currentDateUTC
*   
* - $transactionDate
*   
* - $transactionDateUTC
*   
* - $userId
*   
* - $masterId - see DSRequestModifier.value for details
*   
* - References to specific fields in prior responses, via $responseData.first and 
*     $responseData.last, with or without parameters. For example, 
*     $responseData.first("myDataSource", "fetch")[0].myField. See the
*     velocitySupport for details of $responseData
*   
* - References to certain metadata properties of prior responses, via $responses.first 
*     and $responses.last, with or without parameters. For example, 
*     $responses.last("myDataSource", "fetch").totalRows. Note that the only 
*     properties allowed in a client-driven fieldValueExpression are:
*     "startRow", "endRow", "totalRows" and "status"; this restriction does not apply to 
*     server-driven fieldValueExpressions. See the Velocity overview for 
*     details of $responses
*  
*  
* - Normal DataSourceField.editRequiresRole apply: if a
*  field is not valid for writing, its fieldValueExpression will be ignored.
*  Again, this only applies to client-originated requests.
*  
*  Note, it is possible to globally disable fieldValueExpression in 
*  client-originated requests by setting a flag in your server.properties file:
*   dataSource.allowClientFieldValueExpressions: false
* @type {object}
* @default null
*/
fieldValueExpressions?: object;










}

/**
* PrintCanvas is a subclass of canvas which renders printable content HTML and 
*  provides APIs for printing this content as a separate document.
*/
class PrintCanvas extends Canvas {
/**
* Creates a new PrintCanvas
* 
* @param typeCheckedProperties {Partial<PrintCanvas>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {PrintCanvas} The new instance
*/
static create(typeCheckedProperties? : Partial<PrintCanvas>, uncheckedProperties?: Object): PrintCanvas;


/**
* Location of the special printFrame html file provided as part of the SmartClient libraries.
*  This file must be present at the specified location for the printCanvas printing APIs.
* @type {string}
* @default "[HELPERS]printFrame.html"
*/
printFrameURL?: string;

/**
* Setting this property will cause the specified stylesheet to be loaded in this print
*  canvas's frame.
*  The stylesheet should be specified as a URL to load.
* @type {string}
* @default null
*/
externalStylesheet?: string;







/**
*  Update the HTML content displayed in this print canvas. If the printCanvas is not yet
*  drawn the HTML will be displayed when the canvas is drawn.
*  
*  Note that if the printCanvas is Canvas.redraw, or 
*  Canvas.clear and then Canvas.draw again, the HTML will
*  be redisplayed inside the print canvas, and the specified callback will be fired again.
* 
* 
* @param {string} HTML to show in this print canvas
* @param {PrintCanvasCallback} callback function to fire when the HTML is displayed. The callback will be passed a pointer to this print canvas as the first parameter with the  name printPreview. If this canvas is not drawn when this method is called,  the callback will not be fired until the canvas is drawn and the HTML rendered out into  the page.
*/
setHTML(HTML:string, callback:(printPreview:PrintCanvas) => void): void;

/**
*  Show the native print dialog and allow the user to print the current HTML for
*  this printCanvas. Note that the PrintCanvas must be drawn to be printed.
* 
*/
print(): void;




}

/**
* Sets the text of a label.
*/
class SetLabelTextTask extends ComponentTask {
/**
* Creates a new SetLabelTextTask
* 
* @param typeCheckedProperties {Partial<SetLabelTextTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SetLabelTextTask} The new instance
*/
static create(typeCheckedProperties? : Partial<SetLabelTextTask>, uncheckedProperties?: Object): SetLabelTextTask;


/**
* Value to assign to label text contents. To assign a dynamic value see SetLabelTextTask.textFormula.
* @type {HTMLString}
* @default null
*/
value?: HTMLString;

/**
* Formula to be used to calculate the label text contents. Use SetLabelTextTask.value property
*  to assign a static value instead.
*  
*  Available fields for use in the formula are the current Canvas.ruleScope.
* @type {UserSummary}
* @default null
*/
textFormula?: UserSummary;










}



/**
* Possible values for the current pane showing in a SplitPane. See
*  SplitPane.currentPane for details.
*/
type CurrentPane = 
/**
* SplitPane.navigationPane is the most recently shown
*/
"navigation" |

/**
* SplitPane.listPane is the most recently shown
*/
"list" |

/**
* SplitPane.detailPane is the most recently shown
*/
"detail";


/**
* The IButton widget class is a class that implements the same APIs as the 
*  Button class. Depending on the current skin, IButtons may be
*  on the StretchImgButton component, which renders via images, or may be based on the
*  Button component, which renders via CSS styles.
*/
class IButton extends Button {
/**
* Creates a new IButton
* 
* @param typeCheckedProperties {Partial<IButton>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {IButton} The new instance
*/
static create(typeCheckedProperties? : Partial<IButton>, uncheckedProperties?: Object): IButton;











}

/**
* Used to name a validator or reference a standard, built-in Validator - see list below.
*  
*  To make use of a standard validator type for a field in a DataSource, or 
*  DynamicForm instance, specify the validators property to an array 
*  containing a validator definition where the type property is set to 
*  the appropriate type. 
*  
*  A custom error message can be specified for any validator type by setting the
*  errorMessage property on the validator definition object, and some
*  validator types make use of additional properties on the validator definition 
*  object such as max or min.
*  
*  For example, to use the integerRange validator type:
*  &nbsp;&nbsp;field:{
*  &nbsp;&nbsp;&nbsp;&nbsp;validators:[
*  &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{type:"integerRange", min:1, max:100}
*  &nbsp;&nbsp;&nbsp;&nbsp;]
*  &nbsp;&nbsp;}
*  
*  
*  Custom validators can be reused on the client by adding them to the global validator list,
*  via the Validator.addValidatorDefinition method.
*/
type ValidatorType = 
/**
* Validation will fail if this field is non-empty and has a non-boolean value.
*/
"isBoolean" |

/**
* Validation will fail if the value is not a string value.
*/
"isString" |

/**
* Tests whether the value for this field is a whole number. If 
*  validator.convertToInteger is true, float values will be converted 
*  into integers and validation will succeed.
*/
"isInteger" |

/**
* Tests whether the value for this field is a valid floating point number.
*/
"isFloat" |

/**
* Tests whether the value for this field is a valid expression or function; if it is
*  valid, creates a stringMethods object with the value
*  and set the resultingValue to the StringMethod.
*/
"isFunction" |

/**
* RequiredIf type validators should be specified with an expression
*  property set to a stringMethods, which takes four
*  parameters:
*  
* - item - the DynamicForm item on which the error occurred (may be null)
*  
* - validator - a pointer to the validator object
*  
* - value - the value of the field in question
*  
* - record - the "record" object - the set of values being edited by the widget
*  
*  When validation is performed, the expression will be evaluated (or executed). If it
*  returns true, the field will be treated as a required field, so validation
*  will fail if the field has no value, or, in the case of a FileItem or UploadItem
*  and if client-side validation is supported by the browser, if no file is selected for upload
*  or the selected file is empty.
*  To allow server-side enforcement, a required validator can be used instead.
*  With the exception of "binary" fields, conditional criteria can be specified with the
*  Validator.applyWhen property.
*  See ${isc.DocUtils.linkForExampleId('conditionallyRequired')}.
*  
*  
*  NOTE: A requiredIf validator cannot be used to guarantee that a non-empty
*  file is uploaded. The user's browser might not support client-side file validation.
*  Using a requiredIf validator on a "binary" field may be appropriate in scenarios where the
*  application does not technically require a non-empty file to be uploaded by the user.
*  For example, in a bug tracking application, a file upload may be required if the
*  "Have a test case?" checkbox is checked, but the value of the "Have a test case?" checkbox is
*  not actually saved by the application; instead, whether the user is providing a test case is
*  inferred by whether a non-empty test case file was uploaded.
*/
"requiredIf" |

/**
* Tests whether the value for this field matches the value of some other field.
*  The field to compare against is specified via the otherField property
*  on the validator object (should be set to a field name).
*  See ${isc.DocUtils.linkForExampleId('matchValue')}.
*/
"matchesField" |

/**
* Tests whether the value for this field matches some value specified via
*  value.
*/
"equals" |

/**
* Tests whether the value for this field does not match some value specified via
*  value.
*/
"notEqual" |

/**
* [deprecated in favor of inSet]
*  Tests whether the value for this field matches any value from an arbitrary
*  list of acceptable values. The set of acceptable values is specified via
*  the list property on the validator, which should be set to an array of
*  values. If validator.list is not supplied, the valueMap for the field will be used.
*  If there is no valueMap, not providing validator.list is an error.
*/
"isOneOf" |

/**
* Tests whether the value for this field matches any value from an arbitrary
*  list of acceptable values. The set of acceptable values is specified via
*  the list property on the validator, which should be set to an array of
*  values. If validator.list is not supplied, the valueMap for the field will be used.
*  If there is no valueMap, not providing validator.list is an error.
*/
"inSet" |

/**
* Tests whether the value for this field does not match any value from an arbitrary
*  list of unacceptable values. The set of unacceptable values is specified via
*  the list property on the validator, which should be set to an array of
*  values. Not providing validator.list is an error.
*/
"notInSet" |

/**
* Tests whether the value for this field is a whole number within the range 
*  specified. The max and min properties on the validator
*  are used to determine the acceptable range, inclusive. To specify the range as
*  exclusive of the min/mix values, set exclusive to true.
*  See ${isc.DocUtils.linkForExampleId('validationBuiltins')}.
*/
"integerRange" |

/**
* This validator type applies to string values only. If the value is a string value
*  validation will fail if the string's length falls outside the range specified by 
*  validator.max and validator.min.
*  
*  Note that non-string values will always pass validation by this validator type.
*  
*  Note that the errorMessage for this validator will be evaluated as
*  a dynamicString - text within &#36;{...} will be evaluated as JS code
*  when the message is displayed, with max and min available as
*  variables mapped to validator.max and validator.min.
*/
"lengthRange" |

/**
* Determine whether a string value contains some substring specified via 
*  validator.substring.
*/
"contains" |

/**
* Determine whether a string value does not contain some substring specified via 
*  validator.substring.
*/
"doesntContain" |

/**
* Determine whether a string value contains some substring multiple times.
*  The substring to check for is specified via validator.substring.
*  The validator.operator property allows you to specify how to test
*  the number of substring occurrences. Valid values for this property are
*  ==, !=, &lt;, &lt;=,
*  &gt;, &gt;=.
*  
*  The number of matches to check for is specified via validator.count.
*/
"substringCount" |

/**
* regexp type validators will determine whether the value specified 
*  matches a given regular expression. The expression should be specified on the
*  validator object as the expression property.
*  See ${isc.DocUtils.linkForExampleId('formsRegularExpression')}.
*/
"regexp" |

/**
* Validate against a regular expression mask, specified as validator.mask.
*  If validation is successful a transformation can also be specified via the
*  validator.transformTo property. This should be set to a string in the
*  standard format for string replacement via the native JavaScript replace()
*  method.
*  See ${isc.DocUtils.linkForExampleId('formsValueTransform')}.
*/
"mask" |

/**
* Tests whether the value for a date field is within the range specified.
*  Range is inclusive, and is specified via validator.min and
*  validator.max, which should be specified in
*  XML Schema
*  date format or as a live JavaScript Date object (for client-only validators only).
*  To specify the range as exclusive of the min/mix values, set exclusive
*  to true.
*  
*  Note that the errorMessage for this validator will be evaluated as
*  a dynamicString - text within &#36;{...} will be evaluated as JS code
*  when the message is displayed, with max and min available as
*  variables mapped to validator.max and validator.min.
*/
"dateRange" |

/**
* Validate a field as a valid floating point value within a value range.
*  Range is specified via validator.min and validator.max.
*  Also checks precision, specified as number of decimal places in 
*  validator.precision. If validator.roundToPrecision is set 
*  a value that doesn't match the specified number of decimal places will be rounded
*  to the nearest value that does.    
*  
*  For backwards compatibility only. Use "floatRange" and/or "floatPrecision" instead.
*/
"floatLimit" |

/**
* Tests whether the value for this field is a floating point number within the range 
*  specified. The max and min properties on the validator
*  are used to determine the acceptable range, inclusive. To specify the range as
*  exclusive of the min/mix values, set exclusive to true.
*  
*  Note that the errorMessage for this validator will be evaluated as
*  a dynamicString - text within &#36;{...} will be evaluated as JS code
*  when the message is displayed, with max and min available as
*  variables mapped to validator.max and validator.min.
*/
"floatRange" |

/**
* Tests whether the value for this field is a floating point number with the 
*  appropriate number of decimal places - specified in validator.precision
*  If the value is of higher precision and validator.roundToPrecision 
*  is specified, the value will be rounded to the specified number of decimal places
*  and validation will pass, otherwise validation will fail.
*/
"floatPrecision" |

/**
* A non-empty value is required for this field to pass validation.
*  In the case of a "binary" field, a non-empty file must be uploaded.
*/
"required" |

/**
* Change the state/appearance of this field. Desired appearance is specified via
*  the fieldAppearance property on the validator object. See
*  FieldAppearance type for choices.
*  
*  If fieldAppearance is not specified, the default is "readOnly".
*/
"readOnly" |

/**
* Change the visibility of this field.
*/
"visibility" |

/**
* Change the enable/disable state of this field.
*/
"enable" |

/**
* Returns true if the value for this field is unique. The uniqueness check is performed across
*  the whole DataSource unless you specify property validator.criteriaFields as a 
*  comma-separated string of field names; in that case, the uniqueness check is done in the 
*  context of those extra criteria, allowing you to check, for example, whether an employee 
*  number is unique for the department and location found on the record being validated.
*  By default the uniqueness check is not case sensitive but this can be controlled through
*  the Validator.caseSensitive attribute. You can specify the 
*  DataSource.operationBindings to use for the uniqueness check with the 
*  Validator.operationId attribute.
*  
*  Validators of this type have ValidatorDefinition.requiresServer 
*  set to true and do not run on the client.
*  
*  Note when isUnique validator is executed as part of validation process during update 
*  operation, it will perform uniqueness check only for single row updates. If update targets 
*  OperationBinding.allowMultiUpdate, then isUnique validator will 
*  be skipped. If uniqueness check is needed when updating multiple records, consider using 
*  dmiOverview approach to add this check manually.
*  See ${isc.DocUtils.linkForExampleId('uniqueCheckValidation')}.
*/
"isUnique" |

/**
* Returns true if the record implied by a relation exists. The relation can be 
*  derived automatically from the DataSourceField.foreignKey attribute of 
*  the field being validated, or you can specify it manually via 
*  validator.relatedDataSource and validator.relatedField.
*  
*  You can specify at DataSource level that this validator should be automatically 
*  applied to all fields that specify a DataSourceField.foreignKey -
*  see DataSource.validateRelatedRecords.
*  
*  By default the uniqueness check is not case sensitive but this can be controlled through
*  the Validator.caseSensitive attribute.
*  
*  Validators of this type have ValidatorDefinition.requiresServer 
*  set to true and do not run on the client.
*  
*  Note that this validation is generally unnecessary for data coming from a UI. The 
*  typical UI uses a SelectItem or ComboBoxItem with an 
*  FormItem.optionDataSource for user entry, such that the user 
*  can't accidentally enter a related record if that doesn't exist, and a typical SQL 
*  schema will include constraints that prevent a bad insert if the user attempts to 
*  circumvent the UI. The primary purpose of declaring this validation explicitly is 
*  to provide clear, friendly error messages for use cases such as BatchUploader, 
*  where values aren't individually chosen by the user. See also the example
*  ${isc.DocUtils.linkForExampleId('hasRelatedValidation', 'Related Records')}.
*/
"hasRelatedRecord" |

/**
* This validator type is not for direct usage, instead DataSourceField.maxFileSize can 
*  be set and maxFileSize validator will be added automatically. Use 
*  DataSource.maxFileSizeExceededMessage to customize validation error message.
*  
*  In supported browsers (Internet Explorer 10+, Chrome, Firefox, Safari 6+, Opera 11.1+), returns
*  true if the file(s) selected by the user are not larger than the field's
*  DataSourceField.maxFileSize. If not supported by the browser, the validator will always
*  return true.
*  
*  Note that server-side enforcement of the maxFileSize is always required because
*  the user's browser might not support client-side file size checks. Also, any client-side
*  check can be bypassed by a malicious user.
*/
"maxFileSize" |

/**
* Custom client-side validator. 
*  Validator.condition will be called to verify data.
*  
*/
"custom" |

/**
* Custom server-side validator that either evaluates the Velocity expression provided in 
*  Validator.serverCondition (see ${isc.DocUtils.linkForExampleId('velocityValidation')})
*  or makes DMI call to Validator.serverObject to evaluate condition
*  (see ${isc.DocUtils.linkForExampleId('dmiValidation')}).
*  
*  Validators of this type have ValidatorDefinition.requiresServer 
*  set to true and do not run on the client.
*/
"serverCustom";




/**
* Singleton class with static APIs for managing automatically assigned tab order for
*  SmartClient components and other focusable elements.
*  
*  The TabIndexManager separates the logic required to maintain a sensible tab-order for
*  a page's components from the logic to handle allocation of actual tab index values.
*  It is common to have non-focusable components with an implied position in the page's
*  tab order - for example Layouts containing focusable buttons, or DynamicForms containing
*  focusable items, and this class handles maintaining relative tab order within such
*  groups, and supplying explicit TabIndex values for the items which actually need them.
*  
*  Entries are registered with the TabIndexManager via the TabIndexManager.addTarget 
*  API. A 
*  numeric tab index for each entry will be lazily generated when requested via 
*  TabIndexManager.getTabIndex. The class provides APIs to modify the position of entries 
*  in the tab tree. When a target is registered, a couple of custom callback functions
*  can be provided. The first is a notification method for the tab index being updated
*  (due to, for example, a parent being repositioned and all its children having new
*  tab indices assigned), and can be used to take an appropriate action such as updating
*  the tab index of an element in the DOM. The second callback will be fired when a
*  call to the special TabIndexManager.focusInTarget or 
*  TabIndexManager.shiftFocus API requests focus be passed to an entry. This
*  allows a developer to take an appropriate action (such as programmatically focussing
*  in some DOM element).
*  
*  For standard SmartClient components (focusable Canvas 
*  and FormItem), developers will typically use APIs directly on the widget
*  to customize tab sequence behavior rather than interacting with the TabIndexManager
*  class. See the tabOrderOverview topic for more 
*  information on tab order management for components in SmartClient.
*  Developers wishing to embed focusable components into a page which are not
*  SmartClient components (native HTML elements and third party widgets), may use
*  TabIndexManager APIs to do so. This process is described in 
*  customTabElements.
*/
class TabIndexManager extends Class {
/**
* Creates a new TabIndexManager
* 
* @param typeCheckedProperties {Partial<TabIndexManager>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {TabIndexManager} The new instance
*/
static create(typeCheckedProperties? : Partial<TabIndexManager>, uncheckedProperties?: Object): TabIndexManager;






/**
*  Register a target to have its tab order position managed by the TabIndexManager.
* 
* @param {string} Unique ID to associate with a tab position. For a Canvas this  would typically be the Canvas.ID but any unique string is valid.
* @param {boolean} Is this target directly focusable? Governs whether an   explicit tabIndex will be created for this target. This parameter should be   passed as false for targets which do not require an explicit tabIndex   as they are not focusable, or not explicit tab-stops for the user tabbing through the   page. They will still have an implicit tab order position which   governs where descendants appear, and would be used to generate a tabIndex if   canFocus is subsequently updated via TabIndexManager.setCanFocus.
* @param {string=} For cases where the tab position should be treated part of a   group to be moved together, the ID of the parent target containing all members of this   group. An example of this would be a Layout managing the tab order of all its members.   If present, the passed parentID must already be being managed by this TabIndexManager.   May be updated for registered targets via TabIndexManager.moveTarget.
* @param {Integer=} Position in the tab-order within the specified parent [or  within top level widgets]. Omitting this parameter will add the target to the end of   the specified parent's tab group.   May be updated for registered targets via TabIndexManager.moveTarget.
* @param {TabIndexUpdatedCallback=} This notification method will   be fired when the tabIndex is actually updated, typically due to the target, or some   parent of it being re-positioned in the managed Tab order. In some cases tab indices   may also be updated to make space for unrelated entries being added to the   TabIndexManager. This notification is typically used to update the appropriate element   in the DOM to reflect a new tab index.
* @param {ShiftFocusCallback=} This notification method will be fired  when the special TabIndexManager.shiftFocus method is called to   programmatically move focus through the registered targets (simulating the user tabbing   through elements in the tab index chain). The implementation should attempt to update   the UI state by focusing in the appropriate UI for this target -- typically this means   putting browser focus into a DOM element, and return true to indicate success.   Returning false indicates the element is currently not focusable (disabled, masked, etc),   and cause the TabIndexManager to call the shiftFocusCallback on the next registered   entry (skipping over this entry).   If this method was not supplied, calls to TabIndexManager.shiftFocus will simply skip   this entry.
*/
static addTarget(ID:string, canFocus:boolean, parentID?:string, position?:Integer, tabIndexUpdatedCallback?:(ID:string) => void, shiftFocusCallback?:(ID:string) => void): void;

/**
*  Has the specified target been added to this TabIndexManager via
*  TabIndexManager.addTarget?
* 
* @param {string} Unique ID to test for.
*/
static hasTarget(ID:string): boolean;

/**
*  Move a target to the newly specified parent / position. This method may change the
*  calculated tab index for this entry, or other canFocus:true entries which already
*  have a calculated tabIndex. The registered tabIndexUpdated notification method will
*  for for any entry where this occurs.
* 
* 
* @param {string} ID of the target to move
* @param {string=} ID of the new parent (if null, will move to the top level)
* @param {Integer=} Position within the specified parent. If null will be the  last entry.
*/
static moveTarget(ID:string, parentID?:string, position?:Integer): void;

/**
*  Move a list of targets to the newly specified parent / position. This method may change the
*  calculated tab index for these entries, or other canFocus:true entries which already
*  have a calculated tabIndex. The registered tabIndexUpdated notification method will
*  for for any entry where this occurs.
* 
* 
* @param {Array<Partial<string>>} IDs of the targets to move
* @param {string=} ID of the new parent (if null, will move to the top level)
* @param {Integer=} Position within the specified parent. If null will be added at the end
*/
static moveTargets(IDs:Array<Partial<string>>, parentID?:string, position?:Integer): void;

/**
*  Temporarily suppress firing any tabIndexChanged callback passed into 
*  TabIndexManager.addTarget for the 
*  specified targets should their tab index change.
*  
*  This is useful for cases where a developer is managing a list of items and wants to
*  avoid any potential for multiple notifications until the entire list has been managed
* 
* @param {Array<Partial<string>>} targets for which callbacks should be suppressed
*/
static suppressCallbacks(targets:Array<Partial<string>>): void;

/**
*  Resume firing any callbacks suppressed by TabIndexManager.suppressCallbacks
* 
* @param {Array<Partial<string>>} targets for which callbacks should be resumed
*/
static resumeCallbacks(targets:Array<Partial<string>>): void;

/**
*  Removes a target from this TabIndexManager.
*  Any children of this target will also be removed - developers wishing to preserve
*  children should first call TabIndexManager.moveTarget to move the children out of this parent
* 
* @param {string} target to remove
*/
static removeTarget(ID:string): void;

/**
*  Modifies whether or not some specified target should be treated as focusable and
*  provide a meaningful TabIndex on a call to TabIndexManager.getTabIndex.
* 
* @param {string} target ID
* @param {boolean} new value for canFocus
*/
static setCanFocus(ID:string, canFocus:boolean): void;

/**
*  Returns a tabIndex number for some target ID registered via TabIndexManager.addTarget.
*  Generated tab indices are guaranteed to be in order.
*  
*  As targets are added to, or moved within the TabIndexManager, their tab index may become invalid.
*  The tabIndexUpdated notification will be fired when this occurs, giving developers
*  a way to pick up the new tab index, and assign it to the appropriate DOM element if appropriate.
* 
* @param {string} ID of the target for which you want to get a numeric tabIndex.
*/
static getTabIndex(ID:string): Integer;

/**
*  Request the TabIndexManager shift focus to a registered focus target.
*  
*  This method does not directly change the focus within the DOM - instead it invokes the 
*  shiftFocusCallback registered for the specified target if it is marked as
*  canFocus:true.
*  
*  Returns false if the target had no no shiftFocusCallback, 
*  the shiftFocusCallback returned false, or if the target is marked
*  as not canFocus:true
* 
* @param {string} target to shift focus to
*/
static focusInTarget(ID:string): boolean;

/**
*  Method to shift focus to the next registered focusable target.
*  
*  This method does not directly change the focus within the DOM - instead it finds the 
*  next target marked as canFocus:true, and invokes the 
*  shiftFocusCallback registered for that target.
*  This callback is expected to take the appropriate action (typically shifting native
*  focus to an element in the DOM), and return true (or return false, if the target
*  could not receieve focus for some reason, in which case we'll find the next
*  canFocus:true target and repeat the action there.
*  
*  Targets with no shiftFocusCallback will be skipped entirely in this process.
*   
* 
* @param {string} current focus target. If null, focus will be applied to the first  focusable target (or the last if the forward parameter is false).
* @param {boolean} should focus move forward to the next focusable target, or  backward to the previous focusable target.
*/
static shiftFocus(ID:string, forward:boolean): boolean;

/**
*  Method to shift focus to the next registered focusable target within some group.
*  This method will move focus forward or backward, considering only the specified
*  target and any targets within its group (registered as children of the target via
*  TabIndexManager.addTarget or TabIndexManager.moveTarget).
*  
*  The second parameter can be passed to specify an explicit starting position to
*  shift focus from. If this is not present, this method will attempt to focus into the
*  group target itself if moving forward (or its last child, if moving backward) and
*  failing that, shift focus from there.
*  
*  This method does not directly change the focus within the DOM - instead it finds the 
*  next target marked as canFocus:true, and invokes the 
*  shiftFocusCallback registered for that target.
*  This callback is expected to take the appropriate action (typically shifting native
*  focus to an element in the DOM), and return true (or return false, if the target
*  could not receieve focus for some reason, in which case we'll find the next
*  canFocus:true target and repeat the action there.
*  
*  Targets with no shiftFocusCallback will be skipped entirely in this process.
*  
*  A return value of false indicates that this method was unable to shift focus to a new 
*  target.
*   
* 
* @param {string} ID of registered target. Focus will be shifted within  this target and its descendants only.
* @param {string} Optional ID of current focus target within the group  focus will be shifted in the specified direction from this node.
* @param {boolean} should focus move forward to the next focusable target, or  backward to the previous focusable target.
*/
static shiftFocusWithinGroup(targetGroup:string, currentTarget:string, forward:boolean): boolean;

/**
*  Method to shift focus to the next registered focusable target beyond some registered
*  target and any targets registered as children within its group via 
*  TabIndexManager.addTarget or TabIndexManager.moveTarget.
*  
*  This method does not directly change the focus within the DOM - instead it finds the 
*  next target marked as canFocus:true, and invokes the 
*  shiftFocusCallback registered for that target.
*  This callback is expected to take the appropriate action (typically shifting native
*  focus to an element in the DOM), and return true (or return false, if the target
*  could not receieve focus for some reason, in which case we'll find the next
*  canFocus:true target and repeat the action there.
*  
*  Targets with no shiftFocusCallback will be skipped entirely in this process.
*  
*  A return value of false indicates that this method was unable to shift focus to a new 
*  target.
*   
* 
* @param {string} ID of registered target. Focus will be shifted to the  next registered focusable element, skipping this group and its descendants.
* @param {boolean} should focus move forward to the next focusable target, or  backward to the previous focusable target.
*/
static shiftFocusAfterGroup(targetGroup:string, forward:boolean): boolean;

/**
*  Should focus navigation be achieved by explicitly calling the TabIndexManager
*  TabIndexManager.shiftFocus method for the specified node?
*  
*  Developers integrating custom focusable element's into a SmartClient based application 
*  can use this method to ensure the elements in question interact correctly with 
*  Canvas.showClickMask and ListGrid.canEdit.
*  See the tabOrderOverview topic for more information on
*  integrating custom focusable UI into a SmartClient application.
*  
*  This method will return true if the TabIndexManager.setAlwaysUseExplicitFocusNavigation
*  has been set to true (typically because a Canvas.showClickMask is
*  showing), or if the entry or some ancestor has been marked as 
*  TabIndexManager.setUseExplicitFocusNavigation.
*  Note that this is the case for entries registered under a canvas
*  with Canvas.alwaysManageFocusNavigation set to true.
* 
* 
* @param {string} TabIndexManager registered target ID
*/
static useExplicitFocusNavigation(ID:string): boolean;

/**
*  Should TabIndexManager.useExplicitFocusNavigation to always return true?
* 
* @param {boolean} whether we should always use explicit focus navigation
*/
static setAlwaysUseExplicitFocusNavigation(newValue:boolean): void;

/**
*  Mark the specified node (and its descendents) as using explicit focus navigation
*  rather than relying on native browser Tab event handling behavior. See
*  TabIndexManager.useExplicitFocusNavigation for more information.
* 
* @param {string} registered TabIndexManager target
* @param {boolean} should explicit focus navigation be used for the specified  target and its descendents
*/
static setUseExplicitFocusNavigation(ID:string, newValue:boolean): void;

/**
*  Show the current hierarchy of targets passed to TabIndexManager.addTarget together with
*  current canFocus state and tabIndex (if assigned). Results are output to the developer console.
* 
*/
static showAllocatedTabChain(): void;

/**
*  Get a report of the current hierarchy of targets passed to TabIndexManager.addTarget
*  together with current canFocus state and tabIndex (if assigned).
* 
*/
static getAllocatedTabChain(): string;






}

/**
* Maintains a 'selected' subset of a Tree.multiLinkTree. Multi-link
*  trees cannot use the Selection because they allow 
*  multiple occurences of the same node; therefore, selection caching must be done in a way 
*  that allows duplicate nodes to be unambiguously identified. Because the base 
*  Selection class caches pointers to the selected nodes or records directly, it 
*  is fundamentally unable to do this. MultiLinkSelection, by contrast, caches 
*  NodeLocator objects.
*  
*  Includes methods for programmatically selecting node occurences and checking which node
*  occurences are selected, and also for selecting node occurences as a result of mouse events,
*  including drag selection support.
*  The selection object is used automatically to handle selection APIs on TreeGrid
*  instances where the data model is multi-linked (see ResultTree.linkDataSource and 
*  Tree.linkData for further details).
*  
*  Note that selection and deselection are skipped for nodes that aren't enabled, or that are
*  marked as non-selectable; the relevant properties are ListGrid.recordEnabledProperty 
*  and ListGrid.recordCanSelectProperty. The recommended approach to affect disabled 
*  objects via the Selection APIs is to temporarily enable them beforehand.
*/
class MultiLinkSelection extends Class {
/**
* Creates a new MultiLinkSelection
* 
* @param typeCheckedProperties {Partial<MultiLinkSelection>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {MultiLinkSelection} The new instance
*/
static create(typeCheckedProperties? : Partial<MultiLinkSelection>, uncheckedProperties?: Object): MultiLinkSelection;








/**
*  Select a particular node occurence. Note if you do not pass a NodeLocator, 
*  the recordNum parameter is required.
* 
* @param {Node | NodeLocator} node to select, or a NodeLocator that                         identifies it
* @param {Integer=} Optional index into the underlying Tree's                         openList (which will be the same as the record                         number in a TreeGrid)
*/
select(node:Node | NodeLocator, recordNum?:Integer): boolean;

/**
*  Deselect a particular node occurence. Note if you do not pass a NodeLocator, 
*  the recordNum parameter is required.
* 
* @param {Node | NodeLocator} node to deselect, or a NodeLocator that                         identifies it
* @param {Integer=} Optional index into the underlying Tree's                         openList (which will be the same as the record                         number in a TreeGrid)
*/
deselect(node:Node | NodeLocator, recordNum?:Integer): boolean;

/**
*  Select a single node occurence and deselect everything else. Note if you do not pass a 
*  NodeLocator, the recordNum parameter is required.
* 
* @param {Node | NodeLocator} node to select, or a NodeLocator that                         identifies it
* @param {Integer=} Optional index into the underlying Tree's                         openList (which will be the same as the record                         number in a TreeGrid)
*/
selectSingle(node:Node | NodeLocator, recordNum?:Integer): boolean;

/**
*  Returns the selected nodes in this grid as a list of NodeLocators.
* 
* 
*/
getSelection(): Array<Partial<NodeLocator>>;

/**
*  Returns the selected nodes in this grid as a direct array of records. Contrast this with 
*  MultiLinkSelection.getSelection, which returns a list of NodeLocators.
*  Note, because this is MultiLinkSelection, this method may return an array 
*  containing the same node multiple times, with no way of discerning which particular 
*  occurence(s) are selected. If you need an unambiguous list of selected node occurences, use
*  getSelection().
* 
* 
*/
getSelectedRecords(): Array<Partial<TreeNode>>;




}



/**
* Interface to show a drop-down list of pickable options. Used by the SelectItem and
*  ComboBoxItem classes. Depending on the value of PickList.dataSetType, the 
*  generated drop down list of options must be an instance of PickListMenu or
*  PickTreeMenu, or a subclass thereof.
*/
interface PickList {
/**
* Determines the behavior of the click-mask thrown up when this pickList is visible.
*  
*  The default value, "hard", matches the familiar behavior of combos and selects on 
*  Windows, Mac and other platforms - mouse-events such as rollovers are blocked and, when 
*  a click is received, the picker is hidden and the event is cancelled.
*  
*  When clickMaskMode is "soft", mouse-events continue to fire, meaning that 
*  rollover styles, for example, continue to be updated. When a click is received in this
*  mode, the picker is hidden and the click event is allowed to proceed to its target - 
*  this mesns that clicking an item with an open picker will re-open the picker.
* @type {ClickMaskMode}
* @default "hard"
*/
clickMaskMode?: ClickMaskMode;

/**
* Maximum height to show the pick list before it starts to scroll.
*  Note that by default the pickList will be sized to the height required by its content
*  so it will be taller when more rows are available as selectable options
* @type {number}
* @default "300"
*/
pickListHeight?: number;

/**
* Height for an empty pick list (showing the empty message), if the pick list has no
*  records and PickList.hideEmptyPickList is false.
* @type {number}
* @default "100"
*/
emptyPickListHeight?: number;

/**
* Message to display in the pickList if there's no data and 
*  PickList.hideEmptyPickList is false.
* @type {string}
* @default "No items to show"
*/
emptyPickListMessage?: string;

/**
* If this pickList contains no options, should it be hidden?
*  If unset, default behavior is to allow the empty pickList to show if it is databound.
* @type {boolean}
* @default null
*/
hideEmptyPickList?: boolean;

/**
* Default width to show the pickList.
*  If not specified, the width of this form item's element will be used instead.
*  
*  Note that this is a minimum value - by default if the values displayed in this pickList
*  are wider than the specified width the list will expand to accomodate them.
* @type {number}
* @default null
*/
pickListWidth?: number;

/**
* Maximum width for this item's pickList.
*  By default if the values displayed in this pickList are wider than the specified
*  PickList.pickListWidth the pickList will render wide enough to accomodate them.
*  This property allows the developer to limit how wide the pickList will render.
* @type {number}
* @default "400"
*/
pickListMaxWidth?: number;

/**
* Base Style for pickList cells. See cellStyleSuffixes for details on how
*  stateful suffixes are combined with the pickListBaseStyle to generate stateful cell styles.
*  
*  Note: if PickList.pickListTallBaseStyle is specified, this property will be used as the
*  ListGrid.normalBaseStyle and that property will be applied
*  to cells that do not match the default cell height, or if ListGrid.fastCellUpdates
*  is true for the pickList in Internet Explorer.
* @type {CSSStyleName}
* @default "pickListCell"
*/
pickListBaseStyle?: CSSStyleName;

/**
* Optional ListGrid.tallBaseStyle for pickList cells. If unset,
*  PickList.pickListBaseStyle will be applied to all cells.
* @type {CSSStyleName}
* @default null
*/
pickListTallBaseStyle?: CSSStyleName;

/**
* Default value for ListGrid.applyRowNumberStyle for this item's generated
*  pickList.
* @type {boolean}
* @default false
*/
pickListApplyRowNumberStyle?: boolean;

/**
* If true, when the pickList is shown, it will be shown via an animated reveal effect
* @type {boolean}
* @default false
*/
animatePickList?: boolean;

/**
* If PickList.animatePickList is true, this attribute specifies the millisecond 
*  duration of the animation effect.
* @type {number}
* @default "200"
*/
pickListAnimationTime?: number;

/**
* For a ComboBox or other pickList that accepts user-entered criteria, how many
*  milliseconds to wait after the last user keystroke before fetching data from the server.
*  The default setting will initiate a fetch if the user stops typing or pauses briefly.
* @type {number}
* @default "200"
*/
fetchDelay?: number;

/**
* Cell Height for this item's pickList.
* @type {number}
* @default "16"
*/
pickListCellHeight?: number;

/**
* If specified this properties block will be applied to the PickListMenu
*  created for this FormItem.
*  
*  Note: Not every ListGrid property is supported when assigned to a pickList.
*  Where there is a dedicated API on the form item (such as 
*  PickList.pickListCellHeight), we recommend that be used in 
*  favor of setting the equivalent property on the pickListProperties block.
*  
*  PickLists and ListGrid.showFilterEditor:
*  ComboBoxItem do not support setting showFilterEditor 
*  to true on pickListProperties. This combination of settings leads to an ambiguous user
*  exprience as the two sets of filter-criteria (those from the text-box and those from the 
*  pickList filter editor) interact with each other.
*  SelectItem.pickListProperties is a valid way to create a filterable pickList, on 
*  a SelectItem, but this setting is not supported on a SelectItem with 
*  SelectItem.multiple set to true - this combination of settings can cause a 
*  selected value to be filtered out of view at which point further selection changes will 
*  discard that value.
*  In general we recommend the ComboBoxItem class (with ComboBoxItem.addUnknownValues 
*  set as appropriate) as a better interface for filtering pickList data.
* @type {Partial<ListGrid>}
* @default null
*/
pickListProperties?: Partial<ListGrid>;

/**
* If this pick list is showing multiple fields, this property determines the height of
*  the column headers for those fields. Set to zero to suppress the headers entirely.
* @type {number}
* @default "22"
*/
pickListHeaderHeight?: number;

/**
* When this item is showing a PickList.dataSetType, this is 
*  the SelectItem.valueField of the record to use as the 
*  Tree.rootValue node.
* @type {string | number}
* @default null
*/
rootNodeId?: string | number;

/**
* When this item is showing a PickList.dataSetType, which 
*  nodes should be opened automatically. Options are:
*  
*  
* - "none" - no nodes are opened automatically
*  
* - "root" - opens the PickList.rootNodeId - in databound 
*        tree-pickers, this node is always hidden
*  
* - "all" - when ResultTree.loadDataOnDemand, opens the
*        PickList.rootNodeId and all of it's direct
*        descendants - otherwise, opens all loaded nodes 
*  
* @type {string}
* @default "none"
*/
autoOpenTree?: string;

/**
* If this form item maps data values to display values by retrieving the 
*  FormItem.displayField values from an 
*  FormItem.optionDataSource, this property 
*  denotes the the field to use as the underlying data value in records from the 
*  optionDataSource.
*  If not explicitly supplied, the valueField name will be derived as
*  described in FormItem.getValueFieldName.
* @type {string}
* @default null
*/
valueField?: string;

/**
* If set, this item will display a value from another field to the user instead of
*  showing the underlying data value for the FormItem.name.
*  
*  This property is used in two ways:
*  
*  The item will display the displayField value from the 
*  DynamicForm.getValues if 
*  FormItem.useLocalDisplayFieldValue is true, (or if unset and the conditions
*  outlined in the documentation for that property are met).
*  
*  If this field has an FormItem.optionDataSource, this property is used by
*  default to identify which value to use as a display value in records from this
*  related dataSource. In this usage the specified displayField must be 
*  explicitly defined in the optionDataSource to be used - see 
*  PickList.getDisplayFieldName for more on this behavior.
*  If not using PickList.useLocalDisplayFieldValue, the display
*  value for this item will be derived by performing a fetch against the 
*  FormItem.getOptionDataSource 
*  to find a record where the FormItem.getValueFieldName matches 
*  this item's value, and use the displayField value from that record.
*  In addition to this, PickList-based form items that provide a list of possible
*  options such as the SelectItem or ComboBoxItem will show the 
*  displayField values to the user by default, allowing them to choose
*  a new data value (see FormItem.valueField) from a list of user-friendly
*  display values.
*  
*  This essentially allows the specified optionDataSource to be used as
*  a server based valueMap.
*  
*  If FormItem.useLocalDisplayFieldValue
*  are being used and FormItem.storeDisplayValues is true, selecting a new value
*  will update both the value for this field and the associated display-field value
*  on the record being edited.
*  
*  Note: Developers may specify the FormItem.foreignDisplayField property 
*  in addition to displayField. This is useful for cases where the
*  display field name in the local dataSource differs from the display field name in
*  the optionDataSource. See the documentation for
*  DataSourceField.foreignDisplayField for more on this.
*  
*  Note that if optionDataSource is set and no valid display field is
*  specified,
*  FormItem.getDisplayFieldName will return the dataSource title 
*  field by default.
*  
*  If a displayField is specified for a freeform text based item (such as a 
*  ComboBoxItem), any user-entered value will be treated as a display value.
*  In this scenario, items will derive the data value for the item from the
*  first record where the displayField value matches the user-entered value.
*  To avoid ambiguity, developers may wish to avoid this usage if display values
*  are not unique.
* @type {string}
* @default null
*/
displayField?: string;

/**
* If FormItem.displayField is specified for a field, should the
*  display value for the field be picked up from the 
*  DynamicForm.getValues?
*  
*  This behavior is typically valuable for dataBound components where the
*  displayField is specified at the DataSourceField level. See
*  DataSourceField.displayField for more on this.
*  
*  Note that for DataSources backed by the
*  serverDataIntegration, fields with a specified
*  DataSourceField.foreignKey and DataSourceField.displayField will
*  automatically have this property set to true if not explicitly set to false
*  in the dataSource configuration.
*  
*  Otherwise, if not explicitly set, local display value will be used unless:
*  
*  
* - This item has an explicitly specified optionDataSource, rather than
*    deriving its optionDataSource from a specified 
*    DataSourceField.foreignKey specification
*  
* - The FormItem.name differs from the 
*    FormItem.getValueFieldName for the item
*  
* @type {boolean}
* @default null
*/
useLocalDisplayFieldValue?: boolean;

/**
* Whether to show the picker as a flat list, or a collapsible tree.
*  
*  The default value, "list", will use an instance of the
*  PickList.pickListConstructor as the picker - "tree" will 
*  show an instance of PickList.pickTreeConstructor.
* @type {string}
* @default "list"
*/
dataSetType?: string;

/**
* The Class to use when creating a picker of PickList.dataSetType for 
*  a FormItem. Must be a subclass of the builtin default, 
*  PickListMenu.
* @type {SCClassName}
* @default "PickListMenu"
*/
pickListConstructor?: SCClassName;

/**
* The Class to use when creating a picker of PickList.dataSetType for 
*  a FormItem. Must be a subclass of the builtin default, 
*  PickTreeMenu.
* @type {SCClassName}
* @default "PickTreeMenu"
*/
pickTreeConstructor?: SCClassName;

/**
* This property allows the developer to specify which field[s] will be displayed in the 
*  drop down list of options.
*  
*  Only applies to databound pickLists (see PickList.optionDataSource, or pickLists
*  with custom data set up via the advanced PickList.getClientPickListData method.
*  
*  If this property is unset, we display the PickList.displayField, if specified, 
*  otherwise the PickList.valueField.
*  
*  If there are multiple fields, column headers will be shown for each field, the
*  height of which can be customized via the PickList.pickListHeaderHeight attribute.
*  
*  Each field to display should be specified as a ListGridField object. Note that
*  unlike in ListGrid, dataSource fields marked as 
*  DataSourceField.hidden will be hidden by default in pickLists. To
*  override this behavior, ensure that you specify an explicit value for 
*  ListGridField.showIf.
* @type {Array<Partial<ListGridField>>}
* @default null
*/
pickListFields?: Array<Partial<ListGridField>>;

/**
* For Databound formItems, this property determines which column 
*  FormItem.valueIcons should show up in for this formItem's pickList.
*  If unset, valueIcons show up in the PickList.displayField column if specified, 
*  otherwise the PickList.valueField column.
*  In most cases only the displayField or valueField will be visible.
*  This property is typically only required if custom PickList.pickListFields 
*  have been specified for this item.
* @type {string}
* @default null
*/
valueIconField?: string;

/**
* If this item has a databound pickList (for example PickList.optionDataSource is
*  set), this property can be used to provide static filter criteria when retrieving the data
*  for the pickList.
* @type {Criteria}
* @default null
*/
pickListCriteria?: Criteria;

/**
* If set, this FormItem will derive data to show in the PickList by fetching records from
*  the specified optionDataSource. The fetched data will be used as a
*  FormItem.valueMap by extracting the
*  FormItem.valueField and FormItem.displayField in
*  the loaded records, to derive one valueMap entry per record loaded from the
*  optionDataSource. Multiple fields from the fetched data may be shown in the pickList by
*  setting PickList.pickListFields.
*  
*  The data will be retrieved via a "fetch" operation on the DataSource, passing the 
*  PickList.pickListCriteria (if set) as criteria, and passing
*  PickList.optionFilterContext (if set) as DSRequest Properties.
*  
*  The fetch will be triggered when the pickList is first shown, or, you can set
*  SelectItem.autoFetchData to fetch when the FormItem is
*  first drawn. 
*  
*  Note that providing an initial value when
*  FormItem.fetchMissingValues is enabled, or enabling
*  SelectItem.defaultToFirstOption, can also cause a fetch to
*  be initiated immediately upon form creation. You can also call PickList.fetchData
*  at any time to manually trigger a fetch.
*  
*  Data paging is automatically enabled if the optionDataSource supports it. As the
*  pickList is scrolled by the user, requests for additional data will be automatically
*  issued.
*  
*  For a pickList attached to a ComboBoxItem, new fetches are 
*  issued as the user types, with criteria set as described under 
*  ComboBoxItem.getPickListFilterCriteria.
*  If your dataSource is not capable of filtering results by search criteria (eg, the
*  dataSource is backed by an XML flat file), you can set PickList.filterLocally to have the
*  entire dataset loaded up front and filtering performed in the browser. This disables
*  data paging.
*  
*  Note that if a normal, static FormItem.valueMap is also specified
*  for the field (either directly in the form item or as part of the field definition in
*  the dataSource), it will be preferred to the data derived from the optionDataSource for
*  whatever mappings are present.
* @type {DataSource | string}
* @default null
*/
optionDataSource?: DataSource | string;

/**
* If this item is part of a databound form, and has a specified valueMap,
*  by default we show the valueMap options in the pickList for the item.
*  Setting this property to true will ensure that the options displayed in our pickList
*  are derived from the form's dataSource.
* @type {boolean}
* @default null
*/
showOptionsFromDataSource?: boolean;

/**
* If this item has a specified optionDataSource and this property is
*  true, the list of fields used by this pickList will be passed to
*  the datasource as DSRequest.outputs. If the datasource supports this feature
*  the returned fields will be limited to this list. A custom datasource will need to
*  add code to implement field limiting.
*  
*  This list of used fields consists of the values of FormItem.valueField,
*  FormItem.displayField and PickList.pickListFields.
*  
*  NOTE: When enabled, FormItem.getSelectedRecord will only include the
*  fetched fields.
* @type {boolean}
* @default null
*/
fetchDisplayedFieldsOnly?: boolean;

/**
* If this item has a specified optionDataSource, and this property is not
*  null, this will be passed to the datasource as DSRequest properties when
*  performing the filter operation on the dataSource to obtain the set of options for the
*  list. This provides, among other capabilities, a way to trigger the server to return
*  summary records.
* @type {Partial<DSRequest>}
* @default null
*/
optionFilterContext?: Partial<DSRequest>;

/**
* If filterLocally is set for this item, and this item is showing options 
*  from a dataSource, fetch the entire set of options from the server, and use these values
*  to map the item value to the appropriate display value. Also use "local"
*  type filtering on drop down list of options.
*  
*  This means data will only be fetched once from the server, and then filtered on the
*  client.
*  
*  Note - when this property is set to false, filtering will still be 
*  performed on the client if a complete set of data for some criteria has been cached
*  by a fetch, and a subsequent fetch has more restrictive criteria. To explicitly
*  disable client-side filtering set the PickList.useClientFiltering property to false.
* @type {boolean}
* @default false
*/
filterLocally?: boolean;

/**
* Specifies one or more fields by which this item should be initially sorted. It can be 
*  a ListGridField.name, or an array of field names - but note that, if
*  multiple fields are supplied, then each will be sorted in the same 
*  ListGrid.sortDirection.
*  
*  For full sorting control, set PickList.initialSort to a list of
*  custom SortSpecifier.
*  
*  This attribute can also be set to the index of a field in the fields array, but note 
*  that it will be converted to a string (field name) after initialization.
* @type {string | Array<Partial<string>> | Integer}
* @default null
*/
sortField?: string | Array<Partial<string>> | Integer;

/**
* An array of SortSpecifier objects used to set up the initial sort configuration 
*  for this pickList. If specified, this will be used instead of any 
*  PickList.sortField specified.
* @type {Array<Partial<SortSpecifier>>}
* @default null
*/
initialSort?: Array<Partial<SortSpecifier>>;

/**
* For PickList.optionDataSource items, this property will be passed
*  to the generated ResultSet data object for the pickList as ResultSet.useClientFiltering.
*  Setting to false will disable filtering on the client and ensure criteria are
*  always passed to the DataSource directly.
* @type {boolean}
* @default null
*/
useClientFiltering?: boolean;

/**
* For databound pickLists (see PickList.optionDataSource), by default SmartClient
*  will cache and re-use datasets shown by pickLists in an LRU (least recently used) caching
*  pattern.
*  
*  Setting this flag to false avoids this caching for situations where it is too aggressive.
* @type {boolean}
* @default true
*/
cachePickListResults?: boolean;

/**
* For PickList items with PickListItemIconPlacement set such that the pickList does
*  not render near-origin, should specified FormItem.icons be rendered inline within
*  the formItem itself, or within the ComboBoxItem.pickerNavigationBar.
*  
*  May be overridden at the icon level via FormItemIcon.iconPlacement.
*  
*  For mobile browsing with limited available screen space, icons rendered in
*  the navigation bar may be easier for the user to interact with.
* @type {PickListItemIconPlacement}
* @default "both"
*/
iconPlacement?: PickListItemIconPlacement;

/**
* When applying filter criteria to pickList data, what type of matching to use.
*  
*  For a databound pickList (PickList.optionDataSource set), textMatchStyle is
*  sent to the server as DSRequest.textMatchStyle.
*  
*  For a non-databound pickList, textMatchStyle is applied by 
*  PickList.filterClientPickListData.
* @type {TextMatchStyle}
* @default "startsWith"
*/
textMatchStyle?: TextMatchStyle;

/**
* If true, even non-matching options will be shown, with configurable 
*  PickList.separatorRows in between. Not valid for
*  PickList.optionDataSource.
* @type {boolean}
* @default null
*/
showAllOptions?: boolean;

/**
* Array of records to show between matching and non-matching rows in the PickList.
*  
*  Not valid for PickList.optionDataSource.
* @type {Array<Partial<ListGridRecord>>}
* @default "[{isSeparator:true}]"
*/
separatorRows?: Array<Partial<ListGridRecord>>;

/**
* A set of "special" values such as "All", "None" or "Invalid" that do not appear in the normal
*  ValueMap or in the data returned by the PickList.optionDataSource.
*  
*  Like other uses of ValueMap, either a list of values or a mapping from stored to
*  display value can be provided.
*  
*  These values can either be shown at the top of the list of values (in the order specified), or
*  can be shown in a separate, non-scrolling region - the setting
*  SelectItem.separateSpecialValues controls this. Note that data 
*  paging can only be used if separateSpecialValues is enabled.
*  
*  If specialValues are configured, SelectItem.allowEmptyValue is ignored - an empty
*  value, if desired, must be included in the specialValues. To provide a
*  specialValue which clears the value of the field, use the special constant
*  PickList.emptyStoredValue.
*  
*  specialValues can also be used to take a value that does appear in the
*  normal data and redundantly display it at the top of the list to make it more accessible. Note
*  that in this case it is expected that the special value appears both at the top of the
*  list and in it's normal position in the list, so this works best with
*  SelectItem.separateSpecialValues mode enabled.
*  
*  Also, if an PickList.optionDataSource is used, PickList.specialValues that appear in the normal
*  dataset will be updated by automatic cacheSync (if 
*  the PickList.displayField is updated). However when using a distinct PickList.valueField and
*  PickList.displayField, you are required to provide PickList.specialValues as a map (there is no
*  support for FormItem.fetchMissingValues automatically fetching appropriate display values).
*  
*  Note that specialValues are not supported in conjunction with MultiComboBoxItem. Whereas with
*  SelectItem.multiple, specialValues will never be normal values 
*  that may be selected. So, specialValues should have options such as "Select All", "Select None" and others.
* @type {ValueMap}
* @default null
*/
specialValues?: ValueMap;



/**
*  Only applies to databound items (see PickList.optionDataSource).
*  Performs a fetch type operation on this item's DataSource to retrieve the set of valid
*  options for the item, based on the current PickList.pickListCriteria.
* 
* @param {DSCallback=} Callback to fire when the fetch completes. Callback will        fire with 4 parameters:  
- item a pointer to the form item  
- dsResponse the DSResponse returned by the server  
- data the raw data returned by the server  
- dsRequest the DSRequest sent to the server  
* @param {Partial<DSRequest>=} properties to apply to the       dsRequest for this fetch.
*/
fetchData?(callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  PickLists can derive their data directly from a valueMap, or retrieve data from a 
*  dataSource to display as options.
*  
*  This method will return the dataSource used to populate the pickList, or null if 
*  none is specified (implies this list will derive its data from the valueMap for the item).
*  Default implementation will return PickList.optionDataSource if specified,
*  otherwise if this is a field with a specified foreignKey in a databound
*  form, returns the dataSource for the foreignKey.
*  Otherwise picks up this.form.dataSource if set.
* 
* 
*/
getOptionDataSource?(): DataSource;

/**
*  stringMethods to return a set of filter criteria to be applied to
*  the data displayed in the pickList when it is shown.
*  
*  If this is a databound item the criteria will be passed as criteria to
*  DataSource.fetchData. Otherwise an equivalent client-side filter will be
*  performed on the data returned by PickList.getClientPickListData.
*  
*  By default combines FormItem.optionCriteria with
*  PickList.pickListCriteria if specified, otherwise an empty 
*  set of criteria so all records will be displayed.
* 
* 
*/
getPickListFilterCriteria?(): Criteria;

/**
*  Getter method to retrieve the FormItem.valueField for this item. For
*  items with a specified FormItem.optionDataSource, this determines which
*  field in that dataSource corresponds to the value for this item.
*  
*  If unset, if a DataSourceField.foreignKey exists
*  between this field and the optionDataSource, this will be used,
*  otherwise default behavior will return the FormItem.name of this field.
* 
* 
*/
getValueFieldName?(): string;

/**
*  Returns the displayField for this item.
*  
*  Behavior varies based on the configuration of this item, as follows:
*  
* - If this item has an PickList.optionDataSource and an explicit 
*  FormItem.foreignDisplayField is specified, this will be returned.
*  
* - Otherwise if an explicit PickList.displayField is specified it will be returned 
*  by default. If the displayField was specified on the underlying 
*  dataSource field, and no matching field is present in the PickList.optionDataSource for
*  the item, we avoid returning the specified displayField value and instead return the
*  title field of the option DataSource. We do this to
*  avoid confusion for the case where the displayField is intended as a display-field
*  value for showing another field value within the same record in the underlying
*  dataSource only.
*  
* - If no explicit foreignDisplay or displayField 
*  specification was found, and the FormItem.valueField for this item is hidden in the 
*  FormItem.optionDataSource, this method will return the title field for 
*  the optionDataSource.
* 
* 
*/
getDisplayFieldName?(): FieldName;

/**
*  Returns the set of data to be displayed in this item's PickList.
*  
*  This method will be called for non-databound form items implementing the PickList
*  interface. The default implementation will derive data from the item's valueMap - 
*  can be overridden to allow a custom set of options to be displayed.
*  
*  Note that for PickLists that filter data based on user input
*  (ComboBoxItem), this method should return the data before
*  filtering. To customize the data returned after filtering, override
*  PickList.filterClientPickListData instead.
*  
*  As an example, for a formItem with PickList.valueField set to "valueFieldName", the
*  default implementation would take a valueMap like the following:
*  
*    valueMap: { value1: "display 1", value2: "display 2" }
*  
*  .. and returning the following set of records: 
*  
*    [
*      { valueFieldName : "value1" },
*      { valueFieldName : "value2" }
*    ]
*  
*  Due to the valueMap, these records will appear as a two row pickList displayed as
*  "display 1" and "display 2".
* 
* 
*/
getClientPickListData?(): Array<Partial<ListGridRecord>>;

/**
*  If this item is showing a dataBound pickList, this notification method will be fired 
*  when new data arrives from the server.
* 
* @param {Integer} index of first row returned by the server
* @param {Integer} index of last row returned by the server
* @param {ResultSet} pointer to this pickList's data
*/
dataArrived?(startRow:Integer, endRow:Integer, data:ResultSet): void;

/**
*  Returns the data to display in the pick list.
*  
*  The default implementation applies the criteria returned by 
*  PickList.getPickListFilterCriteria to the data returned by
*  PickList.getClientPickListData. A record passes the filter if it has a
*  matching value for all fields in the criteria object. Matching is performed according
*  to TextMatchStyle.
*  
*  If PickList.showAllOptions is set, all values are shown, with matching values
*  shown below a PickList.separatorRows.
* 
* 
*/
filterClientPickListData?(): Array<Partial<ListGridRecord>>;




}

/**
* Provides information about the page you're loaded in. We define "page" here to be roughly
*  equivalent to the browser window or frame the libraries have been loaded in.
*/
class Page extends Class {
/**
* Creates a new Page
* 
* @param typeCheckedProperties {Partial<Page>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Page} The new instance
*/
static create(typeCheckedProperties? : Partial<Page>, uncheckedProperties?: Object): Page;
/**
* On a call to Page.checkBrowserAndRedirect(), if no explicit URL
*    is passed in, and the browser is not supported by ISC, redirect to this URL.
* @type {URL}
* @default "[SKIN]/unsupported_browser.html"
*/
static defaultUnsupportedBrowserURL?: URL;

/**
* If a URL provided to various Page APIs begins with one of these Strings, it is treated
*  as an absolute URL.
*  
*  The default of protocols is:
*  
*    ["http://","https://","file://","mailto:", "app-resource:", "data:"]
*  
*  .. and can be replaced via Class.addClassProperties or
*  via setting the global variable isc_protocolURLs before SmartClient loads.
* @type {Array<Partial<string>>}
* @default "[...]"
*/
static protocolURLs?: Array<Partial<string>>;

/**
* Action to take when Page.checkBrowserAndRedirect is called in a browser for which
*  support is not guaranteed. Valid settings are:
*  
*  
* - "continue" Load the page without notifying the user of potential issues
*  
* - "confirm" Notify the user via a standard confirm dialog that their browser is
*  not supported. Provide options to continue anyway, or redirect to another page.
*  
* - "redirect" Automatically redirect to the another URL
*  
* @deprecated * As discussed in browserSupport, developers are recommended to consider which browsers they wish to support within their specific application rather than relying on framework
*  supported browser detection.

* @type {string}
* @default "continue"
*/
static unsupportedBrowserAction?: string;

/**
* A declared value of the enum type 
*  FireStyle.
* @type {Constant}
* @default "once"
*/
static FIRE_ONCE?: Constant;

/**
* By default most modern browsers will navigate back one page when the user hits the
*  backspace key.
*  
*  Setting this attribute to true will suppress this native behavior. Alternatively, 
*  developers can explicitly return false from the keyPress event
*  (either via event handlers applied to specific widgets, or via Page.registerKey
*  for example) to suppress the native navigation behavior.
* @type {boolean}
* @default true
*/
static suppressBackspaceNavigation?: boolean;







/**
*  Has the page finished loading?
* 
* 
*/
static isLoaded(): boolean;

/**
*  Set the title of the page, which is typically shown as part of the browser window title
* 
*/
static setTitle(): void;

/**
*  Returns the base URL of the application, which is the page URL minus the last non-directory 
*  path component. For example, if the page is loaded from
*  http://foo.com/bar/zoo.jsp, appDir will be http://foo.com/bar/.
*  
*  If other page-wide URLs such as Page.setIsomorphicDir are specified as
*  relative paths, they are considered relative to this URL.
* 
* 
*/
static getAppDir(): string;

/**
*  Specify the directory for app-specific images.
*  
*  This becomes the default location where any SmartClient component will load images from
*  unless the special "[SKIN]" prefix is used to indicate that an image is part of a skin.
*  
*  Default is "[APP]images/"
* 
* 
* @param {string=} New imgDir URL.
*/
static setAppImgDir(URL?:string): void;

/**
*  Return the directory for app-specific images.
* 
* 
*/
static getAppImgDir(): string;

/**
*  Specify the directory for miscellaneous app-specific files other than images, such as
*  HTMLFlow.contentsURL, ViewLoader, 
*  XML or JSON flat data files, videos, etc.
*  
*  This URL also becomes available via the prefix "[APPFILES]" for RPCRequest.actionURL.
*  
*  Defaults to the value of Page.getAppDir, that is, the current directory.
* 
* 
* @param {string=} New app files URL.
*/
static setAppFilesDir(URL?:string): void;

/**
*  Returns the directory for application-specific files (other than images).
* 
* 
* @param {string=} New app files URL.
*/
static getAppFilesDir(URL?:string): void;

/**
*  Specify the root directory for Isomorphic-supplied files - the directory containing 
*  the modules/ and system/ subdirectories shipped as part of
*  the SmartClient package.
*  
*  Note that this property is commonly specified directly in the bootstrap HTML file
*  by setting window.isomorphicDir before loading the SmartClient library files.
* 
* 
* @param {string=} New IsomorphicDir URL.
*/
static setIsomorphicDir(URL?:string): void;

/**
*  Return the root directory for Isomorphic-specific files.
* 
* 
*/
static getIsomorphicDir(): string;

/**
*  Specify the root directory for Isomorphic-supplied tools. Typicall tools/ under webRoot.
*  
*  Note that this property is commonly specified directly in the bootstrap HTML file
*  by setting window.isomorphicToolsDir before loading the SmartClient library
*  files. If unset, it defaults to $isomorphicDir/../tools/
* 
* 
* @param {string=} New IsomorphicToolsDir URL.
*/
static setIsomorphicToolsDir(URL?:string): void;

/**
*  Return the root directory for Isomorphic-supplied tools dir.
* 
* 
*/
static getIsomorphicToolsDir(): string;

/**
*  Return the images directory used by Isomorphic-supplied tools.
* 
* 
*/
static getToolsImgDir(): string;

/**
*  Specify the URL for media that's part of the skin
* 
* 
* @param {string=} New skinDir URL
*/
static setSkinDir(URL?:string): void;

/**
*  Return the directory for media that's part of the skin
* 
* 
*/
static getSkinDir(): string;

/**
*  Return the directory for a skin image.
* 
* 
* @param {URL=} Partial URL (relative to Page._skinDir) where the image lives.     If not supplied, will use "images/"
*/
static getSkinImgDir(imgDir?:URL): string;

/**
*  Return the full URL for app-specific or skin image.
*  
*  To use a skin image, start the URL with "[SKIN]". Any other relative URL is assumed
*  relative to the Page.getAppImgDir.
* 
* 
* @param {SCImgURL} Local file name for the image.
* @param {string=} User-specified image directory,     local to
*/
static getImgURL(src:SCImgURL, imgDir?:string): string;

/**
*  Return a full URL for a relative path that uses a special prefix such as "[APPFILES]" or
*  "[SKIN]".
*  
*  For images, use Page.getImgURL instead.
* 
* 
* @param {string} Local file name for the image.
*/
static getURL(fileName:string): string;

/**
* 
*  Return whether the page text direction is right to left. If you set "DIR=RTL" in the BODY tag of
*  the page, then this method will return true. If you set "DIR=LTR" then this method will return
*  false.
* 
* 
*/
static isRTL(): boolean;

/**
*  Load a styleSheet for this application. 
* 
*  The styleSheetURL parameter can use any special directories, eg:
*    &nbsp;&nbsp;Page.loadStylesheet("[SKIN]/skin_styles.css")
*    or
*    &nbsp;&nbsp;Page.loadStylesheet("[APP]/app_styles.css").
*    
*  If you don't specify a special directory, the app directory
*  	will be assumed.
*  
*  Note:  If the document's ONLOAD handler has already fired, this
*   will have no effect.
* 
* 
* @param {URL} URL to the stylesheet.
*/
static loadStyleSheet(styleSheetURL:URL): void;

/**
*  Resize the outer portion of the window to a specific width and height.
* 
* @param {number} new width for the window
* @param {number} new height for the window
*/
static resizeTo(width:number, height:number): void;

/**
*  Move the window to a specified top and left in screen coordinates.
* 
* 
* @param {number} new left coordinate for window
* @param {number} new top coordinate for window
*/
static moveTo(left:number, top:number): void;

/**
*  Scroll the window to a specified top and left coordinate.
* 
* 
* @param {number} new left coordinate for window
* @param {number} new top coordinate for window
*/
static scrollTo(left:number, top:number): void;

/**
*  Get the width of the visible portion of the window, not including browser chrome or the
*  scrollbar area.
*  
*  See also Page.getOrientation.
* 
* @param {object=} the window object
*/
static getWidth(wd?:object): number;

/**
*  Get the height of the visible portion of the window, not including browser chrome or the
*  scrollbar area.
*  
*  See also Page.getOrientation.
* 
* @param {object=} the window object
*/
static getHeight(wd?:object): number;

/**
*  Is the current page wider than it is tall ("landscape" orientation) or the reverse 
*  ("portrait" orientation). Note that the PageEvent
*  will be fired whenever the page orientation changes.
*  
*  This method is typically useful for apps developed for display on mobile devices,
*  though it will also return a valid value when running against a desktop browser.
*  See also mobileDevelopment on building applications
*  for mobile devices
* 
*/
static getOrientation(): PageOrientation;

/**
*  This method only applies to browsers that support the special viewport meta tag, such as
*  Mobile Safari running on the iPhone.
*  
*  This method will dynamically change the viewport configuration, allowing you to set an
*  initial size or scale level and enable / disable user-scaling. Typically this method will
*  be called with a value for scale, width or height rather than passing in values for all
*  three properties.
*  
*  See Apple's Safari Web Content Guide on configuring the viewport for more information:
*  https://developer.apple.com/library/safari/documentation/AppleApplications/Reference/SafariWebContent/UsingtheViewport/UsingtheViewport.html
*  
*  Note: Modifying the width/height or initial scale of the viewport has two user-visible
*  effects:
*  
*  
* - HTML elements may reflow to fit the specified size (or the implied size calculated
*    from the specified scale level and the native device size).
*  
* - If the user has not scaled the application explicitly, and no other scaling or sizing
*    attributes were specified via a viewport meta tag for this page, the application will
*    zoom to specified scale level (or the scale level required to fit the specified viewport
*    size to the device's screen).
*  
* 
* @param {Float=} Desired scale level where 1 indicates no scaling (each css pixel  will be displayed using 1px on the physical device). Pass in null to avoid setting  this property.
* @param {Integer=} Desired viewport width in pixels. This indicates how many pixels  should fit within the device screen. Pass in null to avoid setting this property.
* @param {Integer=} Desired viewport height in pixels. This indicates how many pixels  should fit within the device screen. Pass in null to avoid setting this property.
* @param {boolean=} Should the user be able to scale the application (using  pinch gestures, double tapping, rotating the device, etc.)? Pass in null to avoid setting  this property.
*/
static updateViewport(scale?:Float, width?:Integer, height?:Integer, scalable?:boolean): void;

/**
*  Get the width of the window contents as they have been drawn. If the page scrolls, this may
*  be larger than the Page.getWidth.
* 
* 
*/
static getScrollWidth(): number;

/**
*  Get the height of the window contents as they have been drawn. If the page scrolls, this may
*  be larger than the Page.getHeight.
* 
* 
*/
static getScrollHeight(): number;

/**
*  Get the amount that the browser window has been scrolled horizontally.
* 
*/
static getScrollLeft(): number;

/**
*  Get the amount that the browser window has been scrolled vertically.
* 
*/
static getScrollTop(): number;

/**
*  Get the width of the user's screen, in pixels.
* 
*/
static getScreenWidth(): void;

/**
*  Get the height of the user's screen, in pixels.
* 
*/
static getScreenHeight(): void;

/**
*  Go back in the browser's history.
* 
*  If the history is empty and the window.opener is set, we assume we're a child window and just
*  close the window.
* 
* 
*/
static goBack(): void;

/**
*  Method to set up a static Class.observe on some target object. 
*  This allows developers to perform some action every time a particular method is invoked 
*  on a target object.
*  
*  This method returns a unique observation identifier string. To cancel the observation, 
*  pass this identifier to Page.ignore.
*  
*  If multiple observations are set up for the same target object and method, the 
*  notification actions will be fired in the order in which they were registered.
*  
*  This method is available as isc.Page.observe() or just isc.observe()
*  
*  Note [potential memory leak]: If the target object is a simple JavaScript object
*  (not an instance of a SmartClient class), developers should always call 
*  Page.ignore to stop observing the object if an observation is no longer necessary.
*  
*  This ensures that if the object is subsequently allowed to go out of scope by application 
*  code, the Page level observation system will not retain a reference to it (so the browser
*  can reclaim the allocated memory).
*  While cleaning up observations that are no longer required is always good practice, this
*  memory leak concern is not an issue if the target object is an instance of a SmartClient
*  class. In that case the observation is automatically released when the 
*  target is Class.destroy.
* 
* 
* @param {object} Object to observe. This may be any JavaScript object with the specified  target method, including native arrays, and instances of SmartClient classes such as   Canvas.
* @param {string} Name of the method to observe. Every time this method is invoked  on the target object the specified action will fire   (after the default implementation completes).
* @param {Function | string} Action to take when the observed method is invoked.  If action is a string to execute, certain keywords are available for context:    
- observed is the target object being observed (on which the method was invoked).  
- returnVal is the return value from the observed method (if there is one)  
- For functions defined with explicit parameters, these will also be available as keywords within    the action string    If action is a function, the arguments for the original method will also be   passed to this action function as arguments. If developers need to access the target   object being observed from the action function they may use native javascript techniques  such as   javascript closure  to do so. The return value from the observed method is not available to the action function.
*/
static observe(object:object, methodName:string, action:Function | string): string;

/**
*  Clear an observation set up by Page.observe.
*  
*  This method is available as isc.Page.ignore() or just isc.ignore()
* 
* 
* @param {string} ID returned from Page.observe call we want to clear
*/
static ignore(observerID:string): void;

/**
*  Wait for a method to fire on an object.
*  
*  waitFor is similar Class.observe, but fires once only.
* 
* 
* @param {object} any SmartClient object, eg, a ListGrid
* @param {string} name of a method on that object
* @param {Function} Callback to fire when the observed method completes
* @param {number=} Optional timeout period (in milliseconds). If you want a timeout,              you must also provide a timeoutCallback
* @param {Function=} Callback to fire if the timeout period elapses before the                   observed method completes
*/
static waitFor(object:object, methodName:string, callback:Function, timeout?:number, timeoutCallback?:Function): boolean;

/**
*  Wait for methods to fire on multiple objects.
*  
*  waitForMultiple is similar to Page.waitFor, except that it does not fire
*  its callback until all of the provided methods have fired.
* 
* 
* @param {Array<any>} an array of objects, each of which consists of two properties:            "object": any SmartClient object, eg a ListGrid             "method": name of a method on that object
* @param {Function} Callback to fire when all observed methods have fired
* @param {number=} Optional timeout period (in milliseconds). If you want a timeout,              you must also provide a timeoutCallback
* @param {Function=} Callback to fire if the timeout period elapses before all                   observed methods have fired
*/
static waitForMultiple(objects:Array<any>, callback:Function, timeout?:number, timeoutCallback?:Function): boolean;

/**
*  Check whether the browser is supported by the Isomorphic SmartClient system. Behavior depends
*  upon the value of Page.unsupportedBrowserAction:
*  
*  
* - "continue" Load the page without notifying the user of potential issues
*  
* - "confirm" Notify the user via a standard confirm dialog that their browser is
*  not supported. Provide options to continue anyway, or redirect to another page. Text of the
*  confirm dialog is retrieved by calling Page.getUnsupportedBrowserPromptString.
*  
* - "redirect" Automatically redirect to the another URL
*  
*  If redirecting to another page is necessary, and no explicit URL is provided we will use
*  Page.defaultUnsupportedBrowserURL.
*  
*  This method is commonly called as part of the skinning logic after page
*  load.
* 
* 
* @param {string=} URL of redirect page. May include Isomorphic special directories   such as [SKIN].
* @deprecated * As discussed in browserSupport, developers are recommended to consider which browsers they wish to support within their specific application rather than relying on framework
*  supported browser detection.

*/
static checkBrowserAndRedirect(URL?:string): void;

/**
*  Returns the text for the prompt shown to user from Page.checkBrowserAndRedirect
*  if they are accessing this page in an unsupported
*  browser and Page.unsupportedBrowserAction is set to "confirm". May be
*  overridden to return a different message.
* 
*/
static getUnsupportedBrowserPromptString(): string;

/**
*  Setting this to true will cause Page.loadStyleSheet to append
*  an "isc_version" parameter to the end of the url when loading a stylesheet.
* 
* 
* @param {boolean} pass in true to turn on automatic adding of version                       parameter to css urls.
*/
static setAddVersionToSkinCSS(addVersionToSkinCss:boolean): void;

/**
*  Returns true if add version to skin CSS is currently turned on.
* 
* 
*/
static isAddVersionToSkinCSS(): boolean;

/**
*  Register to be called whenever a given type of event occurs, at the page level.
*  
*  This includes events that also occur on widgets (like "click") and events that only occur at
*  the page level ("resize" and "load").
*  
*  For events that also occur on widgets, page level event registrations will fire BEFORE the
*  event handlers on Canvases.  If your action returns false, this will prevent
*  the event from getting to the intended Canvas.
*  
*  Capturing events on widgets can be done by setting one of the event methods available on Canvas
*  (and hence available to all widget classes). See widgetEvents.
* 
* 
* @param {PageEvent} event type to register for ("mouseDown", "load", etc)
* @param {string} string to be eval'd when event fires   (function) 	function to be executed when event fires            (object)      an object to call on which a method named "page" +                      eventType will be called
* @param {FireStyle=} Flag to set automatic removal of the event after      it fires one or more times
* @param {string=} optional - if an object was passed in as the second                     parameter, this is a name of a method to call on that                      object.
*/
static setEvent(eventType:PageEvent, action:string, fireStyle?:FireStyle, functionName?:string): number;

/**
* 	Clear event(s) under the given eventType.
* 	To clear all events, omit the ID parameter. To clear a specific event,
* 	pass the ID that was returned by Page.setEvent().
* 
* @param {PageEvent | Event} event type to clear
* @param {number=} ID of the event to clear.     	If not specified, all events in eventType will be cleared.
*/
static clearEvent(eventType:PageEvent | Event, ID?:number): void;

/**
*  Fire some action when the Page receives a keyPress event from a certain key.
*  Note that if a widget has keyboard focus, this action will fire only after any widget-level
*  keyPress handlers have fired and bubbled the event up to the top of their ancestor chain.
*  Multiple actions can be registered to fire on a single keyPress using this method, and can
*  be associated with different target objects (which will then be available as
*  a parameter when the action is fired).
*  
*  
*  This differs from calling Page.setEvent with the "keyPress"
*  events registered via setEvent() will fire before widget level handlers 
*  respond to the event, and will fire for every keyPress event, not just those
*  triggered by some specific key or key-combination.
*  
*  
*  
* 
* @param {KeyIdentifier} key name or identifier object.
* @param {string} Action to fire when key is pressed.       This can be a string of script to evaluate or a javascript function.        This action will be passed 2 parameters: The name of the key pressed will be        available as the first parameter or key keyword. The target        passed into this method will be available as the second parameter or        target keyword.
* @param {any=} If specified this object will be made available to the                 action fired as a parameter.
*/
static registerKey(key:KeyIdentifier, action:string, target?:any): void;

/**
*  Clears an action registered to fire on a specific a keyPress event via the Page.registerKey
*  method. 
* 
* @param {KeyName} Name of key to clear registry entries for.
* @param {object=} target specified when the action was registered for the key.
*/
static unregisterKey(actionID:KeyName, target?:object): void;






}





/**
* EditProxy that handles SectionStack objects when editMode is enabled.
*/
class SectionStackEditProxy extends LayoutEditProxy {
/**
* Creates a new SectionStackEditProxy
* 
* @param typeCheckedProperties {Partial<SectionStackEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SectionStackEditProxy} The new instance
*/
static create(typeCheckedProperties? : Partial<SectionStackEditProxy>, uncheckedProperties?: Object): SectionStackEditProxy;











}

/**
* Button that resets the form to default values, by calling
*  DynamicForm.resetValues()
*  
*  If you define a click handler on this item, you can return false to cancel the reset.
*/
class ResetItem extends ButtonItem {
/**
* Creates a new ResetItem
* 
* @param typeCheckedProperties {Partial<ResetItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ResetItem} The new instance
*/
static create(typeCheckedProperties? : Partial<ResetItem>, uncheckedProperties?: Object): ResetItem;











}

/**
*/
type CharacterCasing = 
/**
* No character translation
*/
"default" |

/**
* Map characters to uppercase
*/
"upper" |

/**
* Map characters to lowercase
*/
"lower";






/**
* DynamicForms automatically choose the FormItem type for a field based on the
*  type property of the field. The table below describes the default FormItem
*  chosen for various values of the type property.
*  
*  You can also set FormItem.editorType to the classname of a
*  FormItem to override this default mapping. You can alternatively override
*  DynamicForm.getEditorType to create a form with different rules for which
*  FormItems are chosen.
*  
*/
type FormItemType = 
/**
* Rendered as a TextItem, unless the length of the field (as specified by DataSourceField.length attribute) is larger than the value
*  specified by DynamicForm.longTextEditorThreshold, a
*  TextAreaItem is shown.
*/
"text" |

/**
* Rendered as a CheckboxItem
*/
"boolean" |

/**
* Same as text by default.           Consider setting editorType:SpinnerItem.
*/
"integer" |

/**
* Same as text by default.           Consider setting editorType:SpinnerItem.
*/
"float" |

/**
* Rendered as a DateItem
*/
"date" |

/**
* Rendered as a TimeItem
*/
"time" |

/**
* Rendered as a SelectItem. Also true for any field that          specifies a FormItem.valueMap. 
*           Consider setting editorType:ComboBoxItem.
*/
"enum" |

/**
* Same as text
*/
"sequence" |

/**
* If DataSourceField.canEdit:false is set on the field,          the value is rendered as a LinkItem. Otherwise the field 
*           is rendered as a TextItem.
*/
"link" |

/**
* Rendered as an image if not editable, or as a TextItem to edit          the URL or partial URL if editable
*/
"image" |

/**
* Rendered as a FileItem, or a ViewFileItem if not editable
*/
"imageFile" |

/**
* Rendered as a FileItem, or a ViewFileItem if not editable
*/
"binary";


/**
* Event on a listGrid header to trigger auto-fit of the listgrid field.
*/
type AutoFitEvent = 
/**
* React to a double click on the listGrid header.
*/
"doubleClick" |

/**
* React to a click on the listGrid header.
*/
"click" |

/**
* No event will trigger auto-fit.
*/
"none";


/**
* An object that modifies a dsRequest based on several available 
*  values, including preceding responses from the same queue. 
*  
*  Some elements of this feature are only available with Power or better licenses. See
*  smartclient.com/product for details.
*  
*  You provide a list of DSRequestModifiers as the 
*  OperationBinding.criteria or OperationBinding.values
*  of an OperationBinding. The ability to modify a request based on responses to 
*  earlier requests in the same queue is called transactionChaining.
*  
*  A DSRequestModifier consists of a DSRequestModifier.fieldName, usually a
*  DSRequestModifier.value and possibly an DSRequestModifier.operator
*  and DSRequestModifier.start and/or DSRequestModifier.end values 
*  (applicable to advanced criteria only). The value, start and end settings can be static, or - 
*  with Power or better licenses - they can be expressions in the Velocity template language, 
*  which will be resolved at runtime, immediately before the DSRequest is executed.
*  
*  In addition to the standard velocitySupport, special Velocity 
*  variables are available for the value field when used in a &lt;values&gt; declaration - 
*  see DSRequestModifier.value for details.
*  
*  See below some examples of OperationBinding.criteria declarations:
*  
*  &lt;operationBindings&gt;
*    &lt;operationBinding operationType="fetch" operationId="..."&gt;
*      &lt;criteria fieldName="lifeSpan" value="10"/&gt;
*      &lt;criteria fieldName="scientificName" value="Gazella thomsoni"/&gt;
*    &lt;/operationBinding&gt;
*    
*    &lt;operationBinding operationType="fetch" operationId="..."&gt;
*      &lt;criteria fieldName="lifeSpan" operator="greaterThan" value="10" /&gt;
*    &lt;/operationBinding&gt;
*  
*    &lt;operationBinding operationType="fetch" operationId="..."&gt;
*      &lt;criteria _constructor="AdvancedCriteria" operator="or"&gt;
*        &lt;criteria&gt;
*          &lt;Criterion fieldName="lifeSpan" operator="greaterThan" value="10" /&gt;
*          &lt;Criterion fieldName="scientificName" operator="contains" value="Octopus" /&gt;
*        &lt;/criteria&gt;
*      &lt;/criteria&gt;
*    &lt;/operationBinding&gt;
*  &lt;/operationBindings&gt;
*  
*  
*  And an example of OperationBinding.values declarations:
*  
*  &lt;operationBinding operationType="add"&gt;
*    &lt;values fieldName="orderID" value="$responseData.last('queuedAdd_order','add').orderID" /&gt;
*  &lt;/operationBinding&gt;
*  
*  
*  See the Master/Detail Add Example 
*  to see this in action.
*/
class DSRequestModifier {


/**
* The name of the field to add or replace on the DSRequest - whether this appears in the DSRequest's
*  values or criteria depends on whether this is part of a OperationBinding.values or
*  OperationBinding.criteria attribute.
* @type {FieldName}
* @default null
*/
fieldName?: FieldName;

/**
* The name of the operator to apply when constructing criteria. This property only applies 
*  to criteria; it is ignored if specified as part of a OperationBinding.values 
*  attribute.
* @type {string}
* @default null
*/
operator?: string;

/**
* The value to assign to the field named by DSRequestModifier.fieldName. This
*  value can be static, and for Pro licenses that is the only option. With Power and better
*  licenses, this value can be an expression in the Velocity template language. In this latter
*  case, all the standard velocitySupport provided by
*  SmartClient Server are available to you.
*  
*  Note, dsRequestModifier values are evaluated during DSRequest 
*  setup, before the request's execute() method is called. This means that 
*  variables added to the Velocity context by calling addToTemplateContext()
*  from a dmiOverview method or 
*  DataSource.serverConstructor will not be available.
*  In this case, you can either
*  
* - Apply the variable values directly to the DSRequest's criteria and values
*  from your Java code. See the server-side Javadoc for DSRequest
*  
* - Add your template variables to the DSRequest's template context before 
*  dsRequestModifier evaluation takes place, in a custom override of 
*  serverDataIntegration
*  
*  masterId and responseData
*  There is also one additional Velocity context variable available in this specific case: 
*  $masterId. If there is a DataSourceField.foreignKey from the 
*  DataSource for the current operation to another DataSource for which an add or update 
*  operation has taken place earlier in the queue, this is the value of the target field of 
*  the foreign key, taken from the response data of that earlier operation (the most recent 
*  one, if there are several). This is useful because it will typically yield the (possibly 
*  just generated) primary key of the "master" record.
*  
*  Consider a queued batch of "add" operations for an order header and its details. The 
*  detail additions need to know the unique primary key that was assigned to the order, but 
*  this will typically be generated at the time of inserting the order row into the database,
*  so it is not known up-front. However, this value will be in the response to the DSRequest
*  that added the order header, so it is accessible via $responseData; if there is a 
*  declared foreign key relationship from the detail DataSource to the header DataSource, the 
*  header's unique key value will also be accesible as $masterId. See this example: 
*  ${isc.DocUtils.linkForExampleId('queuedAdd')}.
*  
*  $responseData - which is an instance of 
*  com.isomorphic.velocity.ResponseDataHandler - exposes various overloads of
*  first() and last() APIs that can be called to obtain the actual 
*  record data of prior responses. These methods return an instance of 
*  com.isomorphic.velocity.ResponseDataWrapper, which allows convenient handling
*  of response data whether it is a single record or a list. Response data can be treated as
*  a single record even if it is a List, so you can access the response data directly, with no
*  need for an index; when you do this, and the data is actually a List or array, you get the 
*  first record. If the response data is a list or array, you can also access individual 
*  records in that list using Velocity index notation, and you can use the special value 
*  "last" to access the last element of a List or array.
*  
*  Examples of the Velocity syntax needed:
*  
*  $responseData.first.myField is the myField property of the first response in
*  the queue. Note, this works whether that response returned a single record or a list. If it
*  returned a list, this Velocity expression gets the first record in the list. This is a
*  particularly useful shorthand for 'add' and 'update' operations, where the response data 
*  is typically a List containing a single record
*  
*  $responseData.first('order').myField is the myField property of the first response to an 
*  operation (any operation) on the "order" DataSource
*  
*  $responseData.first('order', 'add').myField is the myField property of the first 
*  response to an "add" operation on the "order" DataSource
*  
*  $responseData.first('order', 'fetch').last.myField
*  is the myField property of the last record in the response data of the first fetch in the 
*  queue (fetch operations always return a List of records)
*  
*  $responseData.first('order', 'fetch')[0].myField is the myField property of a
*  specific record (in this case, the first) in the response data of the first response in the 
*  queue. Note that this is shown for completeness only: there is no need to use index 
*  notation to explicitly request the first record, unless you are iterating over the entire 
*  list or have some other out-of-the-ordinary use case. The first record is assumed if you 
*  omit the index notation, so this example is equivalent to the simpler: 
*  $responseData.first('order', 'fetch')[0].myField
*  
*  All of these syntactic variations are also available on the $responseData.last object - 
*  "last" here meaning the most recent response matching the DataSource and operation type (if 
*  applicable). Note, "last" potentially has three different meanings, depending on context:
*  If your DataSource contains a field that is actually called "last", the following expression 
*  would be the correct way to obtain the value of the field called "last", on the last record
*  of the last (most recent) response:
*  $responseData.last.last.last
*  
*  Please see the server-side Javadoc for the 
*  com.isomorphic.velocity.ResponseDataHandler class.
* @type {string}
* @default null
*/
value?: string;

/**
* The value to use for the start of a range. This property only applies to criteria, and it 
*  only applies to operators of type "rangeCheck" - for example, the "between" operator.
*  It is ignored if specified as part of a OperationBinding.values attribute, or for 
*  an inapplicable operator type.
*  
*  The same rules apply to this attribute as apply to DSRequestModifier.value, so
*  you can use Velocity expressions if you have a Power or better license.
* @type {string}
* @default null
*/
start?: string;

/**
* The value to use for the end of a range. This property only applies to criteria, and it 
*  only applies to operators of type "rangeCheck" - for example, the "between" operator.
*  It is ignored if specified as part of a OperationBinding.values attribute, or for 
*  an inapplicable operator type.
*  
*  The same rules apply to this attribute as apply to DSRequestModifier.value, so
*  you can use Velocity expressions if you have a Power or better license.
* @type {string}
* @default null
*/
end?: string;










}

/**
*/
type ProportionalResizeMode = 
/**
* proportional resizing is never used
*/
"none" |

/**
* proportional resizing is always used
*/
"always" |

/**
* proportional resize is off by default, but holding down one of the Canvas.proportionalResizeModifiers turns proportional resize on for a given
*  resize interaction
*/
"modifier" |

/**
* proportional resize is on by default, but holding down one of the Canvas.proportionalResizeModifiers turns proportional resize off for a given
*  resize interaction
*/
"modifierOff";




/**
* An axis or "side" of a table.
*/
type Axis = 
/**
* Row axis
*/
"row" |

/**
* Column axis
*/
"column";




/**
* Encapsulates an RPC response from the server. Instances of this class are automatically created
*  and optionally passed to you in the callback you specify as part of your RPCRequest.
*/
class RPCResponse extends Class {
/**
* Creates a new RPCResponse
* 
* @param typeCheckedProperties {Partial<RPCResponse>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {RPCResponse} The new instance
*/
static create(typeCheckedProperties? : Partial<RPCResponse>, uncheckedProperties?: Object): RPCResponse;
/**
* Indicates successful completion of the request. This is the default status and is
*  automatically used by the RPCResponse on the server unless you override it with
*  setStatus().
*  
*  See the error handling section in RPCManager
*  for more information.
* @type {Integer}
* @default "0"
*/
static STATUS_SUCCESS?: Integer;

/**
* Indicates that the browser is currently offline, and that we do not hold a cached 
*  response for the request.
* @type {Integer}
* @default "1"
*/
static STATUS_OFFLINE?: Integer;

/**
* Indicates a generic failure on the server. 
*  See the error handling section in RPCManager
*  for more information.
* @type {Integer}
* @default "-1"
*/
static STATUS_FAILURE?: Integer;

/**
* Indicates a declarativeSecurity failure on the server.
*  See the error handling section in RPCManager
*  for more information.
* @type {Integer}
* @default "-3"
*/
static STATUS_AUTHORIZATION_FAILURE?: Integer;

/**
* Indicates a validation failure on the server.
*  See the error handling section in RPCManager
*  for more information.
* @type {Integer}
* @default "-4"
*/
static STATUS_VALIDATION_ERROR?: Integer;

/**
* Indicates that the RPC has been intercepted by an authenticator that requires the user
*  to log in.
* @type {Integer}
* @default "-5"
*/
static STATUS_LOGIN_INCORRECT?: Integer;

/**
* Indicates that too many authentication attempts have been made and the server refuses to
*  accept any more login attempts.
* @type {Integer}
* @default "-6"
*/
static STATUS_MAX_LOGIN_ATTEMPTS_EXCEEDED?: Integer;

/**
* Indicates that a login is required before this RPCRequest can proceed.
*  
*  Applications do not directly set this status code, instead, to trigger the relogin flow,
*  return the loginRequiredMarker in the response sent by your server when login is
*  required. See the relogin for details.
* @type {Integer}
* @default "-7"
*/
static STATUS_LOGIN_REQUIRED?: Integer;

/**
* Indicates that the login succeeded.
* @type {Integer}
* @default "-8"
*/
static STATUS_LOGIN_SUCCESS?: Integer;

/**
* Indicates that the client attempted an update or remove operation without providing 
*  primary key field(s)
* @type {Integer}
* @default "-9"
*/
static STATUS_UPDATE_WITHOUT_PK_ERROR?: Integer;

/**
* Indicates that the request was either never attempted or was rolled back, because 
*  automatic or user transactions are in force and another request in the same transaction
*  failed. Note that the request(s) that actually failed will have a code specific to the
*  failure; it is only the requests that would otherwise have succeeded that are marked 
*  with this failure code.
* @type {Integer}
* @default "-10"
*/
static STATUS_TRANSACTION_FAILED?: Integer;

/**
* Indicates that an uploaded file's size exceeded the maximum file size allowed.
* @type {Integer}
* @default "-11"
*/
static STATUS_MAX_FILE_SIZE_EXCEEDED?: Integer;

/**
* Indicates that the total size of the data sent to the server was more than the server is
*  configured to allow. Most servers limit the post size to prevent out of memory style
*  attack vectors that push a bunch of data at the server. Apache Tomcat, for example,
*  is pre-configured to limit post size to 2mb.
*  
*  On internal networks, these limits can typically be safely raised or removed. With
*  Tomcat, for example, you can remove the post limit by specifying the following attribute
*  on the &lt;Connector&gt; element in conf/server.xml:
*  
*  maxPostSize="-1"
*  
*  
*  NOTE: this status code is used whenever the server framework receives a request
*  where the POST data has been removed, however, there are other possible causes,
*  including:
*  
*  
* - security software installed on the server or network that erroneously detects some
*    kind of exploit attempt, if its behavior is to just strip the POST data but allow
*    the rest of the request through (SiteMinder is one product known to do this)
*  
* - incorrectly written filter servlets that drop POST'd data
*  
* @type {Integer}
* @default "-12"
*/
static STATUS_MAX_POST_SIZE_EXCEEDED?: Integer;

/**
* Indicates that an operation binding configured to require OperationBinding.requiredCriterion has received none.
* @type {Integer}
* @default "-13"
*/
static STATUS_CRITERIA_REQUIRED_ERROR?: Integer;

/**
* Indicates that an empty file was uploaded for a required 'binary' field.
* @type {Integer}
* @default "-15"
*/
static STATUS_FILE_REQUIRED_ERROR?: Integer;

/**
* Indicates that a response with invalid format has been received from server.
*  If the datasource is using "iscServer" dataFormat, this means that the response is
*  not recognized as a valid ISC frame.
*  
*  One possible cause for this error can be the reception of a RestDataSource JSON
*  response that lacks a valid RestDataSource.jsonPrefix and/or RestDataSource.jsonSuffix
*  
*  If it is using "xml" or "json" dataFormat, the response could not be parsed as XML or JSON.
* @type {Integer}
* @default "-16"
*/
static INVALID_RESPONSE_FORMAT?: Integer;

/**
* This response code is usable only with the XMLHttpRequest transport and indicates that
*  the server returned an HTTP response code outside the range 200-299 (all of these statuses
*  indicate success, but ordinarily only 200 is used). To get the actual
*  response code, you can query rpcResponse.httpResponseCode in your callback.
*  
*  Note that currently this error code will never occur for the hiddenFrame
*  transport - instead, use RPCResponse.STATUS_SERVER_TIMEOUT to detect
*  hiddenFrame transport errors.
* @type {Integer}
* @default "-90"
*/
static STATUS_TRANSPORT_ERROR?: Integer;

/**
* This response code only occurs when using the HTTP proxy. It is issued by the proxy 
*  servlet when the target host is unknown (ie, cannot be resolved through DNS). This
*  response probably indicates that you are attempting to contact a nonexistent server 
*  (though it might mean that you have DNS problems).
* @type {Integer}
* @default "-91"
*/
static STATUS_UNKNOWN_HOST_ERROR?: Integer;

/**
* This response code only occurs when using the HTTP proxy. It is issued by the proxy
*  servlet when the attempt to contact the target server results in a Java SocketException.
*  This response probably indicates that the target server is currently down.
* @type {Integer}
* @default "-92"
*/
static STATUS_CONNECTION_RESET_ERROR?: Integer;

/**
* Indicates a request timed out with no server response.
*  
*  This is a client-only error code - never sent by the server (since it's the server
*  that times out).
*  
*  NOTE that if using hiddenFrame as the transport (not the default), a
*  malformed response such as a "500 Server Error" or 404 errors will be reported as a
*  timeout.
* @type {Integer}
* @default "-100"
*/
static STATUS_SERVER_TIMEOUT?: Integer;



/**
* The data sent by the server.
*  
*  When communicating with the SmartClient server, rpcResponse.data is the data passed to the
*  server-side method RPCResponse.setData() by your Java code.
*  This data is translated into JavaScript
*  objects by the rules described under RPCRequest.data.
*  
*  
*  When not communicating with the SmartClient server rpcResponse.data contains the
*  raw HTTP response body. See RPCRequest.useSimpleHttp,
*  RPCRequest.serverOutputAsString, RPCRequest.evalResult for details.
* @type {string | object}
* @default false
*/
data?: string | object;

/**
* Status code for this response. Status codes less than zero are considered errors by the
*  RPCManager, those greater than or equal to zero are considered successes. Please see the
*  error handling section the RPCManager for more information on
*  what the RPCManager does with the status code and how you can override this behavior.
*  
*  When using the SmartClient server you can set the rpcResponse.status by calling the
*  server-side method RPCResponse.setStatus(). 
*  
*  When not using the SmartClient server, the RPCManager makes no assumptions about the
*  structure of the response, so the status code just reflects the
*  RPCResponse.httpResponseCode: status will be 
*  RPCResponse.STATUS_TRANSPORT_ERROR if an HTTP-level error
*  occurred such as "500 server error". If you have a status code you need to transmit you can
*  simply embed it in the response (as part of RPCResponse.data) and interpret it from
*  the callback.
*  
*  With or without the SmartClient server, the relogin status codes (such as 
*  RPCResponse.STATUS_LOGIN_REQUIRED) are triggered whenever special markers, such as the
*  loginRequiredMarker, appear in the body of the response. See the relogin for details.
* @type {Integer}
* @default "0"
*/
status?: Integer;

/**
* This attribute (available when using the the xmlHttpRequest transport) contains
*  the HTTP response code sent by the server.
*  
*  Note that this is different from RPCResponse.status - that attribute is used to
*  indicate a status code for the RPC itself whereas httpResponseCode is the raw HTTP response
*  code for the HTTP request that contained the RPCRequest.
*  
*  This feature relies on the XMLHttpRequest object which can be disabled by end-users in some
*  supported browsers. See platformDependencies for more information.
*  
*  If you're using this attribute, you'll typically want to avoid the default error
*  handling response of RPCManager. To do so, set
*  RPCRequest.willHandleError to true.
* @type {Integer}
* @default null
*/
httpResponseCode?: Integer;

/**
* HTTP headers returned by the server as a map from header name to header value.
*  
*  Headers are available only when the default RPCTransport "xmlHttpRequest" is in use,
*  and browsers may limit access to headers for cross-domain requests or in other
*  security-sensitive scenarios.
* @type {object}
* @default null
*/
httpHeaders?: object;

/**
* The actual text of the HTTP response. Only available when the default 
*  RPCTransport "xmlHttpRequest" transport is in use,
* @type {string}
* @default null
*/
httpResponseText?: string;

/**
* The RPCRequest.clientContext object as set on the RPCRequest.
* @type {object}
* @default null
*/
clientContext?: object;

/**
* ID of the transaction sent to the server via RPCManager.sendQueue containing the
*  RPCRequest associated with this response.
* @type {Integer}
* @default null
*/
transactionNum?: Integer;










}

/**
* Ask the user to input a value.
*/
class AskForValueTask extends UserConfirmationGateway {
/**
* Creates a new AskForValueTask
* 
* @param typeCheckedProperties {Partial<AskForValueTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AskForValueTask} The new instance
*/
static create(typeCheckedProperties? : Partial<AskForValueTask>, uncheckedProperties?: Object): AskForValueTask;


/**
* Default value.
* @type {string}
* @default null
*/
defaultValue?: string;










}

/**
* Default class used by a TileGrid to render each tile. See TileGrid.tile.
*  
*  SimpleTiles should not be created directly, instead, use a TileGrid and provide data and
*  SimpleTile instances are created for you.
*/
class SimpleTile extends StatefulCanvas {
/**
* Creates a new SimpleTile
* 
* @param typeCheckedProperties {Partial<SimpleTile>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SimpleTile} The new instance
*/
static create(typeCheckedProperties? : Partial<SimpleTile>, uncheckedProperties?: Object): SimpleTile;


/**
* CSS style for the tile as a whole. As with StatefulCanvas.baseStyle, suffixes
*  are appended to this style to represent various states ("Over", "Selected", etc).
* @type {CSSStyleName}
* @default "simpleTile"
*/
baseStyle?: CSSStyleName;

/**
* The TileGrid that created this SimpleTile. This property will be null
*  if the tile was created by a user-provided TileGrid.createTile method.
* @deprecated * As of SmartClient 10.0, use SimpleTile.tileGrid.

* @type {TileGrid}
* @default null
*/
creator?: TileGrid;

/**
* The TileGrid that created this SimpleTile.
* @type {TileGrid}
* @default null
*/
tileGrid?: TileGrid;







/**
*  The default implementation will call TileGrid.getTileHTML.
* 
*/
getInnerHTML(): HTMLString;

/**
*  Return the record that this tile should render.
*  
*  NOTE: a TileGrid that is doing data paging may reuse tiles with different records, so a
*  subclass of SimpleTile should not cache the record returned by getRecord().
*  
* 
*/
getRecord(): TileRecord;




}





/**
* Starts queuing all DataSource operations so they can be sent to the server all together
*  as a transaction.
*/
class StartTransactionTask extends ProcessElement {
/**
* Creates a new StartTransactionTask
* 
* @param typeCheckedProperties {Partial<StartTransactionTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {StartTransactionTask} The new instance
*/
static create(typeCheckedProperties? : Partial<StartTransactionTask>, uncheckedProperties?: Object): StartTransactionTask;











}

/**
* A DataView coordinates the asynchronous loading of WSDL WebService and XML Schema
*  definitions in applications created by Visual Builder.
*  
*  For applications that do not use WSDL Web Services and were not created by Visual Builder,
*  DataView is equivalent to it's superclass VLayout.
*/
class DataView extends VLayout {
/**
* Creates a new DataView
* 
* @param typeCheckedProperties {Partial<DataView>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DataView} The new instance
*/
static create(typeCheckedProperties? : Partial<DataView>, uncheckedProperties?: Object): DataView;








/**
* 
*  Executed when the dataView has loaded all dependencies (such as DataSources or WebServices).
*  No default implementation.
* 
* 
*/
dataViewLoaded(): void;




}

/**
* 
*  The ISC system provides a predictable cross-browser event-handling mechanism for ISC
*  widgets. Events can be handled both at the page level (i.e., globally), and at the level of
*  individual widgets.
*  
*  With the exception of a few page-specific events ('load', 'unload', 'idle' and 'resize'),
*  events are processed in the following sequence:
*  
*  1. The event is sent to any global (page-level) event handlers. These handlers can cancel
*  further propagation of the event by returning false. You can register to listen for any of the
*  events linked in the seeAlso section (below) by calling Page.setEvent
*  method.
*  
*  2. If the event occurred on a form element or a link, it is passed on to the browser so that
*  the element will perform its default action. No widget receives the event.
*  
*  3. If the event occurred on an enabled widget (but not on a form element or link inside
*  the widget), it is sent to that widget's event handler, if any. This handler can cancel
*  further propagation of the event by returning false. An "enabled" widget is any widget that
*  defines an event handler for one of the supported events. Interceptable events are defined in
*  the "widgetEvents" section of Canvas.
*  
*  4. The event is "bubbled" up to the widget's parent in the containment hierarchy, if any.
*  Again, the parent's handler for the event can cancel further propagation by returning
*  false. This step is repeated, with the event "bubbling" up through the containment
*  hierarchy, until a top-level widget is reached or the event is explicitly canceled.
*  In brief, the ISC event model offers the best features of browser event models:
*  
*  
* - Page-first event handling allows you to reliably process or cancel any event before it
*    affects the objects on the page.
*  
* - Event "bubbling" ensures that parent widgets receive events sent to their children,
*    and allows you to create generalized parent-level handlers rather than duplicating
*    code in each child.
*  
*  Note: Canceling propagation of an event may cancel its side effects as well, including the
*  generation of other (synthetic) events. For example, if a global mouseDown handler returns
*  false, drag-and-drop events will not be generated. Specific effects are discussed in the
*  descriptions of the various events in the following sections.
*  
*  SmartClient libraries will not interfere with native event handling when events occur
*  outside of a target widget. You can therefore have HTML that is not ISC-based on the same
*  page as widget objects that will react to native events as you would expect.
*  
*  You can use isc.Event as an alias for isc.EventHandler.
*  
*  
*/
class EventHandler extends Class {
/**
* Creates a new EventHandler
* 
* @param typeCheckedProperties {Partial<EventHandler>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {EventHandler} The new instance
*/
static create(typeCheckedProperties? : Partial<EventHandler>, uncheckedProperties?: Object): EventHandler;
/**
* amount of time between idle messages (msec)
* @type {Integer}
* @default "10"
*/
static IDLE_DELAY?: Integer;

/**
* Return this constant from a child event to stop the event propagating to its parent,
*  without suppressing any native browser handling associated with the event.
*  Developers should not need to modify this value - it should be treated as read-only
*  in most circumstances.
* @type {string}
* @default "***STOP***"
*/
static STOP_BUBBLING?: string;

/**
* Constant containing the full set of edges a component may be resized from.
*  When a component is marked as canDragResize, this will be the default set of edges
*  from which it may be resized.
* @type {Array<Partial<string>>}
* @default "["T","L","B","R","TL","TR","BL","BR"]"
*/
static ALL_EDGES?: Array<Partial<string>>;

/**
* If set to true, when the user drags a Canvas.canDrop canvas
*  over any component with Canvas.canAcceptDrop set to false or where 
*  Canvas.willAcceptDrop returns false, the Canvas.noDropCursor
*  will be shown automatically to indicate this is not a valid drop point.
*  
*  This property can be modified at runtime, meaning a developer could choose to show the
*  no drop indicator for specific drag/drop interactions by changing the value from
*  a dragStart handler or similar.
*  
*  Note that when this property is false, developers may still use the
*  Canvas.dropMove handler for potential drop targets and use Canvas.setCursor
*  to explicitly indicate invalid drop areas within a widget. This is the approach used
*  by default for TreeGrid drag/drop interactions, for example.
* @type {boolean}
* @default false
*/
static showNoDropIndicator?: boolean;

/**
* Advanced property governing whether focus and blur notifications throughout the
*  SmartClient system should be fired synchronously in Internet Explorer and 
*  Microsoft Edge, as they are in other browsers.
*  
*  Internet Explorer differs from other supported browsers in that the native 
*  onfocus and onblur browser events are fired asynchronously.
*  In all other browsers these handlers are fired synchronously.
*  Historically, SmartClient focus change event notifications such as 
*  Canvas.focusChanged and FormItem.focus / FormItem.blur were
*  fired from these native event handlers, meaning that they would also be asynchronous in
*  Internet Explorer and synchronous in all other browsers.
*  Internet Explorer does provide developers with a separate focus-change notification 
*  which fires synchronously in the form of the focusin and focusout
*  events (documented here).
*  When synchronousFocusNotifications is set to true the SmartClient system will leverage
*  these events to provide synchronous notifications in Internet Explorer.
*  
*  For example, consider a Canvas with a 'focusChanged' handler, as follows:
*  
*  
*  // ... Canvas definition
*  isc.Canvas.create({
*    ID:"testCanvas", backgroundColor:"lightblue",
*    contents:"testCanvas",
*    canFocus:true,
*    focusChanged:function (hasFocus) {
*      this.logWarn('focusChanged:' + hasFocus);
*    },
*    autoDraw:true
*  });
*  
*  
*  
*  ...along with the following code to put focus into that canvas:
*  
*  
*  // ... Code to execute in the flow of the application
*  isc.logWarn("Before calling focus");
*  testCanvas.focus();
*  isc.logWarn("After calling focus");
*  
*  
*  
* 
*  If synchronousFocusNotifications is false, the focus changed notification will
*  be fired asynchronously in Internet Explorer, meaning the order of events logged
*  in the developer console would be:
*  
*  WARN:Log:Before calling focus
*  WARN:Log:After calling focus
*  WARN:Canvas:testCanvas:focusChanged:true
*  
*  In all other browsers, the focus changed notification is synchronous:
*  
*  WARN:Log:Before calling focus
*  WARN:Canvas:testCanvas:focusChanged:true
*  WARN:Log:After calling focus
*  
*  Setting synchronousFocusNotifications to true makes event notifications synchronous
*  in Internet Explorer as well.
*  
*  As of SmartClient version 11.1 (SmartGWT version 6.1), this property is true
*  by default. For backwards compatibility purposes, it may be explicitly set
*  to false to reinstate the previous asynchronous focus notification behavior 
*  in Internet Explorer, should application code depend on this behavior.
* @type {boolean}
* @default true
*/
static synchronousFocusNotifications?: boolean;







/**
*  Returns the natively reported target (or source) DOM element for the current mouse event.
*  NOTE: SmartClient cannot guarantee that the same element will
*  be reported in all browser/platform configurations for all event types.
*  If you wish to make use of this value, we recommend testing your use case 
*  in all target browser configurations.
* 
* 
*/
static getNativeMouseTarget(): Element;

/**
*  Method to return the Canvas.canFocus canvas with 
*  current keyboard focus.
* 
* 
*/
static getFocusCanvas(): Canvas;

/**
*  This API may be called to set the native HTML5 drag tracker image. The x and
*  y parameters may be specified to affect the placement of the drag tracker image
*  relative to the mouse cursor. The size of the drag tracker image is the intrinsic size of the
*  image. Browsers may apply certain visual effects (such as a slight transparency) to this image.
*  
*  Can only be called during the Canvas.dragStart event (or methods called during the
*  handling of that event).
*  
*  NOTES:
*  
*  
* - Not supported in Opera 12.x or Safari.
*  
* - For best results, this image should be preloaded. Otherwise, the image might not appear
*  for the first drag using this image.
*  
* - This API does not work in Chrome or Firefox on Windows 7 if the "Use visual styles on windows and buttons"
*  setting is turned off.
*  
* 
* @param {SCImgURL} image source
* @param {Integer=} offset-x from the mouse cursor
* @param {Integer=} offset-y from the mouse cursor
*/
static setDragTrackerImage(src:SCImgURL, x?:Integer, y?:Integer): void;

/**
*  For a cross-frame drag, retrieves the data made available when the drag was initiated in the
*  foreign frame via EventHandler.setNativeDragData.
*  
*  Can only be called during the Canvas.drop event (or methods called during the handling
*  of that event, such as ListGrid.recordDrop); will return null if called at any other
*  time, or if called during a non-HTML5 drag and drop.
* 
*/
static getNativeDragData(): object;

/**
*  Sets the data available in a cross-frame HTML5 drag (see Canvas.useNativeDrag).
*  
*  Data provided to this method must be valid for serialization to JSON via the
*  JSONEncoder, or can simply be a String.
*  
*  Can only be called during the Canvas.dragStart event (or methods called during the
*  handling of that event).
*  
*  Do not pass in sensitive data (e.g. passwords, auth/session tokens, credit card numbers, SSNs, etc.).
* 
* @param {object | string} data to make available to foreign frames
* @param {string=} text data to set. This is the text that users may see if the drag is dropped into an external application such as Notepad or a non-Smart&#67;lient/Smart&nbsp;GWT web application.
*/
static setNativeDragData(data:object | string, strData?:string): void;

/**
*  Applies to Canvas.mouseWheel events only.
*  Returns a numeric value indicating how far the mouse wheel was rotated. This value will be
*  positive if the user scrolled the mousewheel forward or up, or negative if scrolled in the
*  other direction. For a standard wheel-mouse, an increment of 1 relates to the smallest
*  possible rotation of the mouse wheel. For other scrolling devices, such as scroll 
*  gestures on a track pad, wheel delta may be reported in finer grained increments 
*  (causing this method to return a fractional value).
*  
*  Note that behavior for trackpad scroll-gestures may differ by browser, but where 
*  separate vertical and horizontal scroll information is available, this method
*  refers to a vertical scroll gesture.
*  
*  Developers should also be aware
*  that some browsers and operating systems allow the user to configure the sensitivity
*  of the mouse wheel or trackpad, which may change this value.
* 
* 
* @deprecated * in favor of EventHandler.getWheelDeltaY

*/
static getWheelDelta(): Float;

/**
*  Horizontal scroll delta reported by a Canvas.mouseWheel event
*  (such as a horizontal swipe on a track-pad).
*  
*  Returns a numeric value indicating how far the mouse wheel was rotated / the magnitude
*  of the scroll gesture. This value will be
*  positive if the user scrolled the mousewheel to the right, negative if scrolled in the
*  other direction. 
*  
* 
*/
static getWheelDeltaX(): Float;

/**
*  Applies to Canvas.mouseWheel events only.
*  Returns a numeric value indicating how far the mouse wheel was rotated. This value will be
*  positive if the user scrolled the mousewheel forward or up, or negative if scrolled in the
*  other direction. For a standard wheel-mouse, an increment of 1 relates to the smallest
*  possible rotation of the mouse wheel. For other scrolling devices, such as scroll 
*  gestures on a track pad, wheel delta may be reported in finer grained increments 
*  (causing this method to return a fractional value).
*  
*  Note that behavior for trackpad scroll-gestures may differ by browser, but where 
*  separate vertical and horizontal scroll information is available, this method
*  refers to a vertical scroll gesture.
*  
*  Developers should also be aware
*  that some browsers and operating systems allow the user to configure the sensitivity
*  of the mouse wheel or trackpad, which may change this value.
* 
* 
*/
static getWheelDeltaY(): Float;

/**
*  Sets the initial coordinate offset of the last event, typically a mouseDown or touchStart,
*  from the drag target. For example, when grabbing and dragging a Scrollbar thumb with
*  the mouse, you'd expect positive coordinates that reflect your position relative to the top,
*  left corner of the thumb. If a drag tracker will be used, call EventHandler.setDragTracker
*  instead, which takes optional arguments offsetX and offsetY that
*  act similarly to those passed to this method.
*  
*  Your canvas can call this method to set the initial drag offset to whatever you want like so:
*  
*   dragStart : function () {
*     isc.EventHandler.setDragOffset(5, 20);
*   }
* 
* 
* @param {Integer} initial x-offset for the drag
* @param {Integer} initial y-offset for the drag
*/
static setDragOffset(offsetX:Integer, offsetY:Integer): void;

/**
*  Set the HTML for the drag tracker that follows the mouse during a drag and drop interaction.
*  
*  Your canvas can use this routine to set the drag tracker to whatever HTML you want like so:
*  
*   dragStart : function () {
*     isc.EventHandler.setDragTracker('Your contents here');
*   }
*  
* 
* 
* @param {string} HTML for the tracker
* @param {Integer=} new width for the tracker. Default value: 10
* @param {Integer=} new height for the tracker. Default value: 10
* @param {Integer=} x-offset for the tracker
* @param {Integer=} y-offset for the tracker
* @param {Partial<Canvas>=} properties to configure the dragTracker
*/
static setDragTracker(html:string, newWidth?:Integer, newHeight?:Integer, offsetX?:Integer, offsetY?:Integer, properties?:Partial<Canvas>): void;

/**
*  During a drag with Canvas.dragAppearance of either "target" or
*  "outline", returns the page-relative coordinates of whatever element is being dragged.
*  
*  Calling this method allows you to write drag and drop logic that works identically even if
*  dragAppearance is subsequently changed.
* 
* 
*/
static getDragRect(): any;

/**
*  Return the canvas that is the target of the mouse event.
*  Returns null if no canvas found.
* 
* 
*/
static getTarget(): Canvas;

/**
* 
*  Returns the current dragTarget. This is the component on which the drag and drop
*  interaction was initiated. This only returns something meaningful during a drag and drop
*  interaction.
* 
* 
*/
static getDragTarget(): Canvas;

/**
*  	Return the page-relative X (horizontal) coordinate of an event.
* 
* 
*/
static getX(): Integer;

/**
*  	Return the page-relative Y (vertical) coordinate of an event.
* 
* 
*/
static getY(): Integer;

/**
*  	Returns true if the left mouse button is being pressed.
* 
* 
*/
static leftButtonDown(): boolean;

/**
*  	Returns true if the right mouse button is being pressed.
* 
* 
*/
static rightButtonDown(): boolean;

/**
*  Returns true if the middle mouse button is being pressed.
*  
*  Checking whether the middle mouse button is pressed can be used to implement power user
*  shortcuts; however, note that many pointing devices do not have a middle button. Thus, the
*  application should not require the user to press a middle button in order
*  to perform some action.
* 
*/
static middleButtonDown(): boolean;

/**
*      Returns the numeric characterValue reported by the browser.
*      Only available on keyPress events, and only for character (or ascii control) keys
* 
*/
static getKeyEventCharacterValue(): Integer;

/**
*  	Return the character for the current key being pressed.
*  	Note that this is only set reliably for keyPress events on character keys.
*          
* 
*/
static getKeyEventCharacter(): string;

/**
*  	Return the name of the key for the event passed in.
*  	Note that this is only set reliably for keyboard events.
* 
* 
*/
static getKey(): KeyName;

/**
*  	Return true if the shift key is being held down.
*  	Note that this is only set reliably for keyboard events.
* 
* 
*/
static shiftKeyDown(): boolean;

/**
*  	Return true if the control key is being held down.
*  	Note that this is only set reliably for keyboard events.
* 
* 
*/
static ctrlKeyDown(): boolean;

/**
*  	Return true if the alt (option) key is being held down.
*  	Note that this is only set reliably for keyboard events.
* 
* 
*/
static altKeyDown(): boolean;

/**
*  Return whether this Canvas is masked by a clickMask (see Canvas.showClickMask).
* 
* 
* @param {Canvas} widget to check
*/
static targetIsMasked(target:Canvas): boolean;






}







/**
* Represents a field in a DataBoundComponent.
*/
class DBCField {











}

/**
* A logging system similar to the Java log4j package: messages are logged with a "category" and
*  "priority", and developers can dynamically set which log messages are being displayed.
*  
*  5 log priorities are available, with the following general meaning:
*  
*  
* - "debug": diagnostic info which is only likely to be understood by a developer with
*  source access, or would occur too frequently for normal usage
*  
* - "info": reports of significant events in the normal operation of the subsystem
*  
* - "warn": some kind of problem is likely to occur, an API appears is apparently being
*  misused or will yield a partial or very slow result
*  
* - "error": a definite error has occurred which may be recoverable
*  
* - "fatal": total failure with no possibility of recovery
*  
*  
*  Log categories do not need to be declared in advance - you can simply make up a category name and
*  start logging to it, and control whether that category's messages will be displayed via
*  setPriority().
*  
*  NOTE: to open the Developer Console in any page that loads ISC, type
*  javascript:isc.Log.show() in the URL bar - this URL is bookmarkable. 
*  
*  The Developer Console should always be open while developing any ISC-enabled application,
*  because ISC logs many important errors and warnings to the Developer Console.
*  
*  NOTE: if you have the Microsoft JavaScript Debugger installed, ISC will be unable to log stack
*  traces on JS errors until you go to Tools->Internet Options->Advanced Tab and check "Disable
*  script debugging". The ability to see stack traces in the Developer Console is generally much
*  more useful for debugging ISC-based applications than the generic Javascript Debugging
*  facilities.
*/
class Log extends Class {
/**
* Creates a new Log
* 
* @param typeCheckedProperties {Partial<Log>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Log} The new instance
*/
static create(typeCheckedProperties? : Partial<Log>, uncheckedProperties?: Object): Log;
/**
* A declared value of the enum type 
*  LogPriority.
* @type {Integer}
* @default "1"
*/
static FATAL?: Integer;

/**
* A declared value of the enum type 
*  LogPriority.
* @type {Integer}
* @default "2"
*/
static ERROR?: Integer;

/**
* A declared value of the enum type 
*  LogPriority.
* @type {Integer}
* @default "3"
*/
static WARN?: Integer;

/**
* A declared value of the enum type 
*  LogPriority.
* @type {Integer}
* @default "4"
*/
static INFO?: Integer;

/**
* A declared value of the enum type 
*  LogPriority.
* @type {Integer}
* @default "5"
*/
static DEBUG?: Integer;

/**
* Any logs below this priority will be suppressed, unless a more specific setting exists for
*  the category.
* @type {LogPriority}
* @default 3
*/
static defaultPriority?: LogPriority;

/**
* At this priority and above, a stack trace will be included automatically along with the log
*  message itself.
* @type {LogPriority}
* @default 2
*/
static stackTracePriority?: LogPriority;

/**
* Maximum number of logged messages to retain in memory.
*  
*  Note that if the Developer Console is open, it will accumulate an unbounded number of
*  messages in the "Log Messages" area. messageCount only affects the number
*  of messages held in memory in the main application's browser window or tab.
* @type {Integer}
* @default "1000"
*/
static messageCount?: Integer;







/**
*  Returns a one-line summary of the current method call, showing method name and passed
*  arguments. 
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getCallTrace(arguments)" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Note the arguments object is required in most cases for this method to function. In some
*  browsers, it can be derived automatically, but developers looking to debug on multiple platforms
*  should not rely on this.
* 
* 
* @param {any=} arguments object from the call to trace. If ommitted, where supported,  arguments will be derived from the calling function, or if this is not supported, the method  will not function.
*/
static getCallTrace(args?:any): void;

/**
*  Returns a "stack trace" - one line per method in the current call stack, showing the method
*  name and any parameters passed.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getStackTrace" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Platform Notes: In Mozilla Firefox, if Firebug is enabled, a stack trace will be logged
*  to the firebug console in addition to the standard stack trace string returned by
*  this method.
*  
*  In browsers other than Internet Explorer a complete stack trace may not be available - 
*  this occurs when a function is re-entrant (meaning it calls itself). In this case the
*  stack will terminate with text indicating where the recursive function call occurred.
*  
*  See debugging for further information information.
* 
* 
*/
static getStackTrace(): string;

/**
*  Return a very short (generally less than 40 characters) string representation of any object,
*  suitable for viewing by a developer for debugging purposes.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echoLeaf" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
static echoLeaf(obj:any): string;

/**
*  Return a short string representation of any object, suitable for viewing by a developer for
*  debugging purposes.
*  
*  If passed an object containing other objects, echo will not recurse into subobjects,
*  summarizing them instead via echoLeaf().
*  
*  NOTE: echo() is used to generate the output shown in the Log window when evaluating an
*  expression.
*  
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
* 
* @param {any} object to echo
*/
static echo(obj:any): string;

/**
*  Like echo(), except that if passed an Array, echoAll() will echo() every element of the
*  Array.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
static echoAll(obj:any): string;

/**
*  A common usage of Class.logDebug is to call the method directly on
*  the Log class.
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logDebug(message:string, category?:string): void;

/**
*  A common usage of Class.logInfo is to call the method directly on
*  the Log class.
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logInfo(message:string, category?:string): void;

/**
*  A common usage of Class.logWarn is to call the method directly on
*  the Log class.
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logWarn(message:string, category?:string): void;

/**
*  A common usage of Class.logError is to call the method directly on
*  the Log class.
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logError(message:string, category?:string): void;

/**
*  A common usage of Class.logFatal is to call the method directly on
*  the Log class.
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logFatal(message:string, category?:string): void;

/**
*  A common usage of Class.setLogPriority is to call the method
*  directly on the Log class.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
static setLogPriority(category:string, priority:LogPriority): void;

/**
*  A common usage of Class.setDefaultLogPriority is to call the
*  method directly on the Log class.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
static setDefaultLogPriority(category:string, priority:LogPriority): void;

/**
*  A common usage of Class.getDefaultLogPriority is to call the
*  method directly on the Log class.
* 
*/
static getDefaultLogPriority(): LogPriority;

/**
*  Apply a batch a batch of priority settings, as a object mapping category names to priority
*  levels.
* 
* 
* @param {object} priority settings for multiple categories
*/
static applyLogPriorities(settings:object): void;

/**
*  Get all priority settings as an object mapping category names to priority levels.
* 
* 
* @param {Class | object=} Optional param to get priorities specific to                      some ISC class or instance.
* @param {boolean=} If this method is retrieving the priorities specific                  to logging for some class or instance, this parameter                  can be used to view only the overrides to the default                  log priorities on this object.
*/
static getLogPriorities(object?:Class | object, overridesOnly?:boolean): object;

/**
*  Return the priority setting for a particular category.
*  
*  If there is no priority setting specific to this category, null will be
*  returned, NOT Log.defaultPriority.
* 
* 
* @param {string} category name
* @param {Class | object=} Optional class or instance to check for                        specific log priority overrides
*/
static getPriority(category:string, object?:Class | object): LogPriority;

/**
*  Set the priority of messages that will be visible for this log category.
*  
*  After calling setPriority, any messages logged to the given category whose priority is
*  below the specified priority will not appear in the Log.
* 
* 
* @param {string} category name
* @param {LogPriority} priority level to set
* @param {Class | object=} Optional ISC class or instance - if passed the priority will be set for logging    occurring on the class or instance only.
*/
static setPriority(category:string, priority:LogPriority, object?:Class | object): void;

/**
*  Set the default priority of messages that will be visible.
* 
* 
* @param {LogPriority} priority level to set
* @param {Class | object=} Optional ISC class or instance - if passed the default priority will be set for logging    occurring on the class or instance only.
*/
static setDefaultPriority(priority:LogPriority, object?:Class | object): void;

/**
*  Retrieves the default priority of messages that will be visible.
* 
* 
* @param {Class | object=} Optional ISC class or instance - if passed the returns the default priority for    the class or instance only.
*/
static getDefaultPriority(object?:Class | object): LogPriority;

/**
*  Clear the priority setting for a particular category, so that the category's effective
*  priority returns to Log.defaultPriority
*  If the optional second parameter is passed, the specific priority setting for the 
*  category on that object will be cleared, so logs in that category on that object will
*  be logged at the global priority level for the category.
* 
* 
* @param {string} category name
* @param {Class | object=} Optional instance or class object - if passed                    clear logging priority for the appropriate category                     on that object.
*/
static clearPriority(category:string, object?:Class | object): void;

/**
*  Would a message logged to the given category at the given priority appear in the Log?
*  
*  NOTE: if there is no specific priority setting for a given category, the
*  Log.defaultPriority is used.
* 
* 
* @param {string} category name
* @param {LogPriority} priority level to check
*/
static isEnabledFor(category:string, priority:LogPriority): void;

/**
*  Return an Array of the most recently logged messages as an Array of Strings. Up to
*  Log.messageCount messages may be returned.
* 
*/
static getMessages(): Array<Partial<string>>;

/**
*  Open the Developer Console.
*  
*  The Developer Console should always be open while developing any ISC-enabled
*  application, because ISC logs many important errors and warnings to the Developer Console.
*  
*  In Internet Explorer, the Developer Console is able to log a stack trace for every JS error,
*  including errors that occur in non-ISC code.
*  
*  NOTE: if you have the Microsoft JavaScript Debugger installed, ISC will be unable to log
*  stack traces on JS errors until you go to Tools->Internet Options->Advanced Tab and check
*  "Disable script debugging". The ability to see stack traces in the Developer Console is
*  generally much more useful for debugging ISC-based applications than the generic Javascript
*  Debugging facilities.
*  
* 
*/
static show(): void;

/**
*  Clear all currently displayed Log messages
* 
*/
static clear(): void;

/**
*  Causes a stack trace to be logged any time a message containing the provided pattern is
*  logged. This can help figure out the origin of warnings or other mysterious logs in a large
*  complex application.
*  
*  The passed messagePattern is interpreted as a JavaScript regular expression.
*  
*  Note: log messages that do not appear in the Developer Console because of
*  Log.setLogPriority will never trigger a stack trace.
*  
* 
* @param {string} * @param {string=} */
static traceLogMessage(messagePattern:string, prefix?:string): void;

/**
* 
*  Observe a method on an object, logging a stack trace whenever the method is called.
*  
*  Call a second time with identical arguments to disable tracing.
* 
* 
* @param {object} object to observe
* @param {string} name of the method to observe
*/
static traceMethod(object:object, methodName:string): void;

/**
*  
*  Observe a method on an object, logging execution time whenever the method is called.
*  
*  Call a second time with identical arguments to disable tracing.
* 
* 
* @param {object} object to observe
* @param {string} name of the method to observe
*/
static timeMethod(object:object, methodName:string): void;






}

/**
* A simple subclass of Window whose default configuration is appropriate for
*  a standalone, modal window. This includes appropriate default settings for 
*  Window.isModal and Window.autoCenter, and sizing information, as well
*  as having the window be initially ModalWindow.visibility
*  
*  This class is used by some development tools to simplify the creation of a modal window.
*/
class ModalWindow extends Window {
/**
* Creates a new ModalWindow
* 
* @param typeCheckedProperties {Partial<ModalWindow>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ModalWindow} The new instance
*/
static create(typeCheckedProperties? : Partial<ModalWindow>, uncheckedProperties?: Object): ModalWindow;


/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "WindowEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* If true, this Window widget will automatically be centered on the page when shown.
*    If false, it will show up in the last position it was placed (either programmatically,
*    or by user interaction).
*    
*    Note: If an auto-centering Window is either programmatically moved or dragged 
*    by an end user, auto-centering behavior is automatically turned off. To manually center 
*    a Window, you can use Window.centerInPage.
* @type {boolean}
* @default true
*/
autoCenter?: boolean;

/**
* If true, when shown this Window will intercept and block events to all other existing
*  components on the page.
*  
*  Use Window.showModalMask to darken all other elements on the screen when a modal dialog
*  is showing.
*  
*  Chained modal windows - that is, modal windows that launch other modal windows - are
*  allowed. You can accomplish this by simply creating a second modal Window while a modal
*  Window is showing.
*  
*  Note only top-level Windows (Windows without parents) can be modal.
* @type {boolean}
* @default true
*/
isModal?: boolean;

/**
* Controls widget visibility when the widget is initialized. See Visibility type for
*  details.
*  
*  Specifying "visible" sets the CSS visiblity to "visible", forcing a child to be visible
*  even if the parent is hidden. Not supported for use with SmartClient layouts,
*  scrolling or auto-sizing but may be useful when working with third-party or legacy
*  DOM layout systems.
*  
*  Note that if Canvas.hideUsingDisplayNone is set for a hidden ancestor, setting
*  visibility will have no effect at all until that ancestor becomes visible.
* @type {Visibility}
* @default "hidden"
*/
visibility?: Visibility;

/**
* Size for this component's horizontal dimension.
*  
*  Can be a number of pixels, or a percentage like "50%". Percentage sizes are resolved to
*  pixel values as follows:
*  
*  
* - If a canvas has a specified Canvas.percentSource, sizing will be
*    a percentage of the size of that widget (see also Canvas.percentBox).
*  
* - Otherwise, if a canvas has a Canvas.getMasterCanvas, and
*    Canvas.snapTo is set for the widget, sizing will be a percentage of
*    the size of that widget (see also Canvas.percentBox).
*  
* - Otherwise if this is a child of some other canvas, percentages will be based on the
*    inner size of the Canvas.getParentCanvas's viewport.
*  
* - Otherwise, for top level widgets, sizing is calculated as a percentage of page size.
*  
*  Note that if a Canvas.maxWidth or Canvas.minWidth are specified
*  (or Canvas.maxHeight / Canvas.minHeight for heights), these
*  properties act as explicit pixel limits on the canvas' size. For example, a canvas
*  with Canvas.maxWidth set to 500, and width specified as 
*  "100%" will not render larger than 500 pixels in width even if there is more
*  space available in the parent canvas or percentSource.
*  
*  Layout may specially interpret percentage sizes on their children,
*  and also allow "*" as a size.
*  
*  Note that if Canvas.overflow is set to "visible", this size is a
*  minimum, and the component may overflow to show all content and/or children.
*  
*  If trying to establish a default width for a custom component, set
*  Canvas.defaultWidth instead.
* @type {number | string}
* @default "75%"
*/
width?: number | string;

/**
* Maximum width available to this Canvas. 
*  
*  The maxWidth and ModalWindow.maxHeight settings apply to:
*  
*  
* - For a canvas being managed as a member of a Layout, the maximum size the
*    layout should apply to the canvas.
*  
* - For a canvas with a width or height specified as a percent value, a maximum
*    numeric pixel value to limit how large the canvas is sized.
*  
* - determining size for a Canvas in a CanvasItem (maxHeight only)
*  
* - end user Window.canDragResize
*  
*  Maximum sizes do not apply in various other circumstances where sizes are being
*  determined, such as ListGrid.showRecordComponents.
* @type {number}
* @default "900"
*/
maxWidth?: number;

/**
* Size for this component's vertical dimension.
*  
*  Can be a number of pixels, or a percentage like "50%". See documentation for
*  Canvas.width for details on how percentage values are resolved actual size.
*  
*  Note that if Canvas.overflow is set to "visible", this size is a
*  minimum, and the component may overflow to show all content and/or children.
*  
*  If trying to establish a default height for a custom component, set
*  Canvas.defaultHeight instead.
* @type {number | string}
* @default "65%"
*/
height?: number | string;

/**
* Maximum height available to this Canvas. See ModalWindow.maxWidth for details of behavior.
* @type {number}
* @default "750"
*/
maxHeight?: number;










}





/**
* Notify provides a means to display on-screen messages that are automatically dismissed after
*  a configurable amount of time, as an alternative to isc.confirm
*  dialogs that can lower end user productivity. Messages may be shown at a particular
*  location, specified either with viewport-relative coordinates, or as an edge or center
*  location relative to the viewport or a canvas. Messages can be configured to appear and
*  disappear instantly, by sliding into (or out of) view, or by fading in (or out).
*  
*  One or more NotifyAction can be provided when Notify.addMessage is called to display a message. They will be rendered as links on which to
*  click to execute the action.
*  
*  The behavior and appearance of messages are configured per NotifyType, which is simply
*  a string that identifies that message group, similar to Class.logWarn.
*  By calling Notify.configureMessages with the
*  NotifyType, it can be assigned a NotifySettings configuration to control
*  message animation, placement, and the the Label used to render each message, allowing
*  styling and autofit behavior to be configured.
*  
*  Messages of the same NotifyType may be stacked to provide a visible
*  history, with a configurable stacking direction and maximum stacking depth. Details on how
*  to configure messages are provided in the documentation for NotifySettings.
*  
*  Messages for different NotifyTypes are stacked separately and animated by
*  independent Framework pipelines. It's up to you to configure the placement of supported
*  NotifyTypes in your app so that they don't overlap, as the Framework doesn't
*  manage it. For example, separate NotifyTypes could be assigned separate screen
*  edges, or assigned a different NotifySettings.positionCanvas.
*  
*  To dismiss a message manually before its scheduled duration expires, you may call
*  Notify.dismissMessage with a NotifyType (to dismiss
*  all such messages) or an ID previously returned by Notify.addMessage
*  (to dismiss that single message).
*  
*  Warnings and Errors
*  
*  Each notification may be assigned a NotifySettings.messagePriority in
*  the settings passed to Notify.addMessage. By default, all
*  NotifyTypes are configured to have priority Notify.MESSAGE, except for
*  "error" and "warn" NotifyTypes, which are configured with priority
*  Notify.ERROR and Notify.WARN, respectively.
*  
*  The NotifySettings.messagePriority determines the default styling of a
*  message, and which message to remove if a new message is sent while the message stack is
*  already at its limit. We recommended applying a NotifySettings.messagePriority as the best approach for showing pre-styled warnings and errors, since that
*  allows you to interleave them with ordinary messages in a single NotifyType.
*  
*  Alternatively, you can display pre-styled warnings and errors by calling
*  Notify.addMessage with the separate NotifyTypes
*  "warning" and "error", respectively, but then you must take care to
*  Notify.configureMessages used to a separate screen
*  location to avoid one rendering on top of the other.
*  
*  Viewport Considerations
*  
*  Messages are edge or corner-aligned based on the Page.getScrollWidth
*  and Page.getScrollHeight of the current page rather than screen, so
*  you may need to scroll to see the targeted corner or edge. Note that widgets placed
*  offscreen below or to the right of a page may cause the browser to report a larger viewport,
*  and prevent messages from being visible, even if no scrollbars are present. If you need to
*  stage widgets offscreen for measurement or other reasons, place them above or to the left.
*  
*  Modal Windows and Click Masks
*  
*  Messages are always shown above all other widgets, including Window.isModal and Canvas.showClickMask. This is because it's expected
*  that messages are "out of band" and logically indepedent of the widget hierarchy being shown.
*  We apply this layering policy even for windows and widgets created by NotifyActions.
*  If there may a scenario where a message can block a window created by an action, set
*  NotifySettings.canDismiss to true so that an unobstructed view of the underlying
*  widgets can be restored.
*  
*  In the linked sample, note how we take care to reuse the existing modal window, if any, if
*  the "Launch..." link is clicked, so that repeated clicks never stack windows over each other.
*/
class Notify extends Class {
/**
* Creates a new Notify
* 
* @param typeCheckedProperties {Partial<Notify>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Notify} The new instance
*/
static create(typeCheckedProperties? : Partial<Notify>, uncheckedProperties?: Object): Notify;
/**
* Highest priority. Default priority of NotifyType: "error".
* @type {MessagePriority}
* @default "1"
*/
static ERROR?: MessagePriority;

/**
* Second-highest priority. Default priority of NotifyType: "warn".
* @type {MessagePriority}
* @default "2"
*/
static WARN?: MessagePriority;

/**
* Third-highest priority. Default priority for all NotifyTypes other than "error"
*  and "warn".
* @type {MessagePriority}
* @default "3"
*/
static MESSAGE?: MessagePriority;







/**
*  Displays a new message, subject to the Notify.configureMessages
*  for the passed notifyType, overridden by any passed settings.
*  Returns an opaque MessageID that can be passed to Notify.dismissMessage
*  to clear it.
*  
*  Note that an empty string may be passed for contents if actions
*  have been provided, so you may have the message consist only of your specified actions.
*  
*  Most users should do all configuration up front via a call to Notify.configureMessages.
*  The settings argument in this method is provided to allow adjustment of
*  properties that affect only one message, such as NotifySettings.autoFitWidth, NotifySettings.styleName, or 
*  NotifySettings.labelProperties. Making changes to 
*  MultiMessageMode-related properties via this argument isn't supported,
*  unless specifically documented on the property.
*  
* 
* @param {HTMLString} message to be displayed
* @param {Array<Partial<NotifyAction>>=} actions (if any) for this message
* @param {NotifyType=} category of message; default "message"
* @param {NotifySettings=} display and behavior settings for                        this message that override the                        Notify.configureMessages                        settings for the notifyType
*/
static addMessage(contents:HTMLString, actions?:Array<Partial<NotifyAction>>, notifyType?:NotifyType, settings?:NotifySettings): MessageID;

/**
*  Dismisses one or more messages currently being shown, subject to the existing settings
*  for their NotifyType. You may either pass the opaque message identifier returned
*  from the call to Notify.addMessage to dismiss a single message, or a
*  NotifyType to dismiss all such messages.
* 
* @param {MessageID | NotifyType} message identifier or category to dismiss
*/
static dismissMessage(messageID:MessageID | NotifyType): void;

/**
*  Can the message corresponding to the messageID be dismissed? Returns false
*  if the message is no longer being shown. The messageID must have been
*  previously returned by Notify.addMessage.
* 
* @param {MessageID} message identifier to dismiss
*/
static canDismissMessage(messageID:MessageID): boolean;

/**
*  Sets the default NotifySettings for the specified NotifyType. This
*  may be overridden by passing settings to Notify.addMessage when the message
*  is shown, but changing MultiMessageMode-related properties via
*  Notify.addMessage isn't supported,
*  
*  By default, the NotifyTypes "message", "warn", and "error" are predefined, each
*  with their own NotifySettings with different NotifySettings.styleNames. When configuring a new (non-predefined) NotifyType with this method, any
*  NotifySettings left unset will default to those for NotifyType "message".
*  
* 
* @param {NotifyType} category of message; null defaults to "message"
* @param {NotifySettings} settings to store for the notifyType
*/
static configureMessages(notifyType:NotifyType, settings:NotifySettings): void;

/**
*  Changes the default settings that are applied when you create a new NotifyType
*  with Notify.configureMessages. If you want to change the defaults for the built-in
*  NotifyTypes "message", "warn", and "error", with this method, it must be called before
*  the first call to Notify.configureMessages or Notify.addMessage. Once a NotifyType
*  has been created, you must use Notify.configureMessages to change its settings.
*  
*  Note that for defaults that depend on priority (and thus differ between the built-in
*  NotifyTypes), this method only sets the defaults for the "message" NotifyType.
*  
* 
* @param {NotifySettings} changes to NotifyType defaults
*/
static configureDefaultSettings(settings:NotifySettings): void;

/**
*  Updates the contents of the message from what was passed originally to
*  Notify.addMessage, while preserving any existing NotifyAction.
*  
*  The purpose of this method is to support messages that contain timer countdowns or other
*  data that perhaps need refreshing during display. If you find yourself replacing the
*  entire content with something new, you should probably just add it as a new message.
*  
*  Note that this method has minimal animation support. The change in message content and
*  corresponding resizing are instant, but the repositioning of the message or stack (if
*  stacked) to keep your requested NotifySettings.position is controlled
*  by NotifySettings.repositionMethod, allowing slide animation. However, that
*  setting is ignored and the repositioning is instant if you've chosen
*  NotifySettings.positionCanvas to a border or corner along
*  the NotifySettings.position viewport edge, or if an animation is
*  already in progress, in which case the instant repositioning will happen after it
*  completes.
* 
* @param {MessageID} message identifier from Notify.addMessage
* @param {HTMLString} updated message
*/
static setMessageContents(messageID:MessageID, contents:HTMLString): void;

/**
*  Updates the actions of the message from those, if any, passed originally to
*  Notify.addMessage, while preserving any existing Notify.addMessage.
*  
*  See Notify.setMessageContents for further guidance and animation details.
* 
* @param {MessageID} message identifier from Notify.addMessage
* @param {Array<Partial<NotifyAction>>} updated actions for this message
*/
static setMessageActions(messageID:MessageID, actions:Array<Partial<NotifyAction>>): void;

/**
* @param {MessageID} message identifier to check
*/
static messageHasActions(messageID:MessageID): boolean;






}

/**
* Specifies the starting point of a move knob with respect to its draw item. The move knob is
*  positioned relative to the move knob point at the DrawItem.moveKnobOffset.
*/
type MoveKnobPoint = 
/**
* Top Left corner
*/
"TL" |

/**
* Top Right corner
*/
"TR" |

/**
* Bottom Left corner
*/
"BL" |

/**
* Bottom Right corner
*/
"BR";


/**
* DrawItem to render a shape defined by executing the series of drawing commands in the
*  DrawShape.commands array.
*/
class DrawShape extends DrawItem {
/**
* Creates a new DrawShape
* 
* @param typeCheckedProperties {Partial<DrawShape>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DrawShape} The new instance
*/
static create(typeCheckedProperties? : Partial<DrawShape>, uncheckedProperties?: Object): DrawShape;


/**
* The mode in which the DrawItem.titleLabel (if shown) is rotated with
*  this draw item.
* @type {TitleRotationMode}
* @default "withItemAlwaysUp"
*/
titleRotationMode?: TitleRotationMode;

/**
* The drawing commands that will be executed to render the shape.
* @type {Array<Partial<DrawShapeCommand>>}
* @default null
*/
commands?: Array<Partial<DrawShapeCommand>>;







/**
*  Sets the DrawShape.commands that define this shape.
* 
* @param {Array<Partial<DrawShapeCommand>>} the new commands.
*/
setCommands(commands:Array<Partial<DrawShapeCommand>>): void;

/**
*  Move the drawShape by the specified delta
* 
* @param {Distance} number of pixels to move horizontally
* @param {Distance} number of pixels to move vertically
*/
moveBy(dX:Distance, dY:Distance): void;

/**
*  Resize by the specified delta
* 
* @param {Distance} number of pixels to resize by horizontally
* @param {Distance} number of pixels to resize by vertically
*/
resizeBy(dX:Distance, dY:Distance): void;




}

/**
* Specifies the layout of the combo box and buttons in a MultiComboBoxItem.
*/
type MultiComboBoxLayoutStyle = 
/**
* use a FlowLayout, showing values first, then the text entry area
*/
"flow" |

/**
* use a FlowLayout, with the text entry first and values shown afterwards
*/
"flowReverse" |

/**
* Use a horizontal layout with the combo box on the right
*/
"horizontal" |

/**
* Use a horizontal layout with the combo box on the left
*/
"horizontalReverse" |

/**
* Use a vertical layout
*/
"vertical" |

/**
* Use a vertical layout with the combo box at the bottom
*/
"verticalReverse";






/**
* Synonym for the Authentication class
*/
class Auth extends Authentication {
/**
* Creates a new Auth
* 
* @param typeCheckedProperties {Partial<Auth>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Auth} The new instance
*/
static create(typeCheckedProperties? : Partial<Auth>, uncheckedProperties?: Object): Auth;











}





/**
* Defines a set of properties that are applicable to
*  FacetChart.extraAxisMetrics of a FacetChart.
*  
*  These properties behave identically to the same-named properties of FacetChart. Two
*  exceptions are the properties
*  MetricSettings.gradationLineProperties and
*  MetricSettings.gradationZeroLineProperties
*  because they are used to draw tick marks along an extra value axis instead of lines
*  through the chart rectangle. Also the default behavior of the
*  MetricSettings.dataColors property is different, in that the
*  default is to use unused colors from, and in the order of, the 
*  FacetChart.dataColors.
*  
*  The extra value axes that are not positioned directly adjacent to the
*  FacetChart.showChartRect of the FacetChart may additionally
*  have an axis line to connect the gradation tick marks. Whether to draw this
*  line is controlled by the MetricSettings.showAxisLine property,
*  and the properties of the line are specified using
*  MetricSettings.axisLineProperties. The default is to draw
*  the axis lines whenever the chart rect. is drawn and to draw the line using the same
*  properties as the lines of the chart rect. (for example, lineColor and lineWidth).
*  
*  The extra value axis charts may be 
*  MetricSettings.multiFacet charts. If the underlying
*  FacetChart.data is multi-facet then the extra value axes may use the facet
*  whose values are labelled on the data axis or the facet used to create the legend (the
*  first and second facets, respectively). Both types of charts vary the first facet to
*  get a data series to plot. Multi-facet charts also vary the second facet, but
*  single-facet charts keep the second facet fixed at a constant FacetValue.
*  
*  The exact facet value to use for the second facet is specified using the
*  MetricSettings.fixedFacetValue property.
*/
class MetricSettings {


/**
* Whether this extra value axis plots values while varying the facet values of just the
*  first facet (single-facet) or both first and second facets (multi-facet).
* @type {boolean}
* @default true
*/
multiFacet?: boolean;

/**
* For a MetricSettings.multiFacet chart of an extra value axis, this property provides a constant facet value
*  for the second facet. By varying the value of the other facet in multi-facet data, the chart
*  obtains the series of values to plot. The default facet value is the first facet value of the second facet.
* @type {string | number}
* @default null
*/
fixedFacetValue?: string | number;

/**
* For MetricSettings.multiFacet charts embedded in a multi-facet main
*  chart, the legendLabel defines the text of the legend label for this chart.
*  The default text is the FacetValue.title of the metric facet value of this
*  value axis concatenated with the FacetValue.title of the
*  MetricSettings.fixedFacetValue in parentheses. Set the
*  legendLabel to provide custom text for the legend label.
* @type {string}
* @default null
*/
legendLabel?: string;

/**
* Whether to show the extra value axis.
* @type {boolean}
* @default null
*/
showAxis?: boolean;

/**
* When this property is set to the metric of another MetricSettings object, the extra
*  value axis and chart corresponding to these settings will use the same scale for the
*  gradations as the extra value axis and chart of the other MetricSettings object. The
*  value of matchGradations can only be one of the metrics of the metric
*  facet whose values will be displayed by the chart.
* @type {string}
* @default null
*/
matchGradations?: string;

/**
* Whether to show an axis line for this extra value axis if it is not placed
*  directly adjacent to the chart rect. The default setting is the value
*  of the FacetChart.showChartRect property of the FacetChart.
* @type {boolean}
* @default null
*/
showAxisLine?: boolean;

/**
* Properties for the axis line drawn if this extra value axis is not positioned
*  directly adjacent to the chart rect. The default is to match the
*  DrawItem.lineCap, DrawItem.lineColor,
*  DrawItem.lineOpacity, DrawItem.linePattern,
*  and DrawItem.lineWidth of the FacetChart's
*  FacetChart.chartRectProperties for each axis line.
* @type {Partial<DrawLine>}
* @default null
*/
axisLineProperties?: Partial<DrawLine>;

/**
* A label for the data values, such as "Sales in Thousands", typically used as the label for
*  the value axis.
* @type {string}
* @default null
*/
valueTitle?: string;

/**
* Whether to show the MetricSettings.valueTitle (or, in the case of
*  MetricSettings.proportional,
*  MetricSettings.proportionalAxisLabel) as a label on this extra value axis.
* @type {boolean}
* @default null
*/
showValueAxisLabel?: boolean;

/**
* Properties for labels of value axis.
* @type {Partial<DrawLabel>}
* @default null
*/
valueAxisLabelProperties?: Partial<DrawLabel>;

/**
* See ChartType for a list of known types - Column, Bar, Line, Pie, Doughnut,
*  Area, Radar, and Histogram charts are supported.
* @type {ChartType}
* @default null
*/
chartType?: ChartType;

/**
* If the ChartType is "Column" then the metric settings may include
*  a setting for FacetChart.stacked.
*  
* @type {boolean}
* @default null
*/
stacked?: boolean;

/**
* Whether shapes are filled, for example, whether a multi-series line chart appears as a
*  stack of filled regions as opposed to just multiple lines.
*  
*  If unset, fills will be automatically used when there are multiple facets and stacking
*  is active (so Line and Radar charts will show stacked regions). 
*  
*  You can explicitly set filled:false to create multi-facet Line or Radar charts where
*  translucent regions overlap, or filled:true to fill in a single-facet Line or Radar
*  chart.
* @type {boolean}
* @default null
*/
filled?: boolean;

/**
* For multi-facet charts, render data values as a proportion of the sum of all data
*  values that have the same label.
*  
*  Gradation labels will be switched to show percentage instead of absolute values.
*  
*  This setting is valid only for Column, Bar, Area and Radar chart types and only in
*  MetricSettings.stacked mode. Stacked columns will be as tall as the chart rect and
*  stacked bars will be as wide as the chart rect. Area and Radar charts will be
*  completely filled except for facet values where all values are 0.
* @type {boolean}
* @default null
*/
proportional?: boolean;

/**
* Default title for the value axis label when the chart is in
*  MetricSettings.proportional. This title will be used unless the
*  FacetChart.getLegendFacet defines a
*  Facet.proportionalTitle.
* @type {string}
* @default "Percent"
*/
proportionalAxisLabel?: string;

/**
* Same as FacetChart.minDataSpreadPercent but affects only one metric. Default of
*  null means that the chart-wide setting facetChart.minDataSpreadPercent
*  will be used.
* @type {Integer}
* @default null
*/
minDataSpreadPercent?: Integer;

/**
* Same as FacetChart.axisStartValue but affects only one metric.
* @type {Float}
* @default null
*/
axisStartValue?: Float;

/**
* Same as FacetChart.xAxisEndValue but affects only one metric.
* @type {Float}
* @default null
*/
xAxisEndValue?: Float;

/**
* Properties for gradation labels
* @type {Partial<DrawLabel>}
* @default null
*/
gradationLabelProperties?: Partial<DrawLabel>;

/**
* Properties for gradation lines
* @type {Partial<DrawLine>}
* @default null
*/
gradationLineProperties?: Partial<DrawLine>;

/**
* Properties for the gradation line drawn for zero (slightly thicker by default).
* @type {Partial<DrawLine>}
* @default null
*/
gradationZeroLineProperties?: Partial<DrawLine>;

/**
* For Line, Area, Radar, Scatter or Bubble charts, whether to show data points for each
*  individual data value.
*  
*  If shown, the MetricSettings.pointClick and MetricSettings.getPointHoverHTML APIs can be used to create
*  interactivity.
* @type {boolean}
* @default false
*/
showDataPoints?: boolean;

/**
* Size in pixels for data points drawn for line, area, radar and other chart types.
* @type {Integer}
* @default "6"
*/
dataPointSize?: Integer;

/**
* Common properties to apply for all data points (see MetricSettings.showDataPoints).
* @type {Partial<DrawItem>}
* @default null
*/
dataPointProperties?: Partial<DrawItem>;

/**
* Whether to use logarithmic scaling for values.
*  
*  Logarithmic scale charts show an equivalent percentage increase as equivalent distance
*  on the chart. That is, 10 and 100 are the same distance apart as 100 and 1000 (each
*  being a 10 times or 1000% increase).
* @type {boolean}
* @default false
*/
logScale?: boolean;

/**
* Whether to use classic logarithmic gradations, where each order of magnitude is shown as
*  a gradation as well as a few intervening lines.  Gradations also begin and end on an order
*  of magnitude. For example, 1,2,4,6,8,10,20,40,60,80,100.
*  
*  Default gradations can be overridden via MetricSettings.logBase and MetricSettings.logGradations.
* @type {boolean}
* @default false
*/
useLogGradations?: boolean;

/**
* When MetricSettings.useLogGradations, base value for logarithmic gradation lines. Gradation
*  lines will be shown at every power of this value plus intervening values specified by
*  MetricSettings.logGradations.
* @type {Integer}
* @default "10"
*/
logBase?: Integer;

/**
* When MetricSettings.useLogGradations is set, gradation lines to show in between powers,
*  expressed as a series of integer or float values between 1 and MetricSettings.logBase.
*  
*  Some common possibilities (for base 10):
*  
*   [ 1 ] // show only orders of magnitude (0.1, 1, 10, 100, etc)
*   [ 1, 5 ] // show only orders of magnitude plus halfway mark
*   [ 1, 2, 4, 8 ] // show powers of 2 between orders
*   [ 1, 2.5, 5, 7.5 ] // show quarters
*  
*  Or base 2:
*  
*   [ 1 ]
*   [ 1, 1.5 ]
*  
* @type {Array<Partial<Float>>}
* @default "[ 1,2,4,6,8 ]"
*/
logGradations?: Array<Partial<Float>>;

/**
* The FacetChart.decimalPrecision used to render the numeric labels of this metric axis.
* @type {Integer}
* @default "2"
*/
decimalPrecision?: Integer;

/**
* Whether to automatically show shadows for various charts.
* @type {boolean}
* @default true
*/
showShadows?: boolean;

/**
* Properties for shadows.
* @type {Partial<DrawOval>}
* @default null
*/
shadowProperties?: Partial<DrawOval>;

/**
* Properties for lines that show data (as opposed to gradations or borders around the data
*  area).
* @type {Partial<DrawLine>}
* @default null
*/
dataLineProperties?: Partial<DrawLine>;

/**
* Properties for lines that outline a data shape (in filled charts such as area or radar
*  charts).
* @type {Partial<DrawItem>}
* @default null
*/
dataOutlineProperties?: Partial<DrawItem>;

/**
* Properties for data shapes (filled areas in area or radar charts).
* @type {Partial<DrawPath>}
* @default null
*/
dataShapeProperties?: Partial<DrawPath>;

/**
* Properties for a "value line" - a line shows where a particular discrete value is
*  placed, eg, vertical lines connecting points of a line chart to the X axis, or radial
*  lines in a Radar chart.
* @type {Partial<DrawLine>}
* @default null
*/
valueLineProperties?: Partial<DrawLine>;

/**
* An array of colors to use for a series of visual elements representing data (eg columns,
*  bars, pie slices), any of which may be adjacent to any other.
*  
*  Colors must be in the format of a leading hash (#) plus 6 hexadecimal digits, for example, 
*  "#FFFFFF" is white, "#FF0000" is pure red.
* @type {Array<Partial<CSSColor>>}
* @default "see below"
*/
dataColors?: Array<Partial<CSSColor>>;

/**
* A dictionary of gradients to use for a series of visual elements representing data (eg columns,
*  bars, pie slices), any of which may be adjacent to any other.
* @type {any}
* @default "see below"
*/
dataGradients?: any;

/**
* Should data values be shown as text labels near the shape representing the value, for example,
*  above columns of a column chart, or adjacent to points in a line chart?
*  
*  If set to false, then data values will not be shown.
*  
*  If set to true, data values will be shown unless the data density is high enough that
*  labels will potentially overlap, in which case, data values will not be shown and
*  hovers will be shown instead, in the same way as MetricSettings.showValueOnHover shows hovers.
* @type {boolean}
* @default null
*/
showDataValues?: boolean;

/**
* Shows the value of the nearest data value in a floating label whenever the mouse moves
*  within the main chart area. The visual element representing the data
*  value will also be emphasized by brightening or highlighting it (appearance differs by
*  chart type).
*  
*  Calculates nearest value based on FacetChart.getNearestDrawnValue.
*  
*  The data value will be formatted using FacetChart.formatDataValue.
*  The label's appearance is controlled by FacetChart.hoverLabelProperties.
* @type {boolean}
* @default null
*/
showValueOnHover?: boolean;







/**
*  Returns the X coordinate where the passed data value either was or would be drawn.
*  For example, this would be the X coordinate where a bar would end in a bar
*  chart.
*  
*  This is only allowed to be called after FacetChart.chartDrawn fires.
*  
*  If the FacetChart.chartType is
*  "Bar", "Bubble", or "Scatter"
*  
*  then the value argument should be a number. For other rectangular charts,
*  this method expects a FacetValueMap that uniquely identifies the data cell whose
*  X-axis coordinate is to be retrieved.
*  
*  Note that when FacetChart.canZoom is enabled, this API is valid only for
*  data values between FacetChart.zoomStartValue and
*  FacetChart.zoomEndValue.
* 
* 
* @param {Float | FacetValueMap} the value to be drawn.
*/
getXCoord(value:Float | FacetValueMap): Float;

/**
*  Returns the Y coordinate where the passed data value either was or would be drawn.
*  For example, this would be the Y coordinate that a line would pass through on
*  a line chart, or the top of a column on a column chart.
*  
*  This is only allowed to be called after FacetChart.chartDrawn fires.
*  
*  If the FacetChart.chartType is
*  "Area", "Bubble", "Column", "Histogram", "Line", or "Scatter"
*  
*  then the value argument should be a number. For
*  "Bar"
*  charts this method expects a FacetValueMap that uniquely identifies the data cell
*  whose Y-axis coordinate is to be retrieved.
* 
* 
* @param {Float} the value to be drawn.
*/
getYCoord(value:Float): Float;

/**
*  Return an array of the gradation values used in the current chart. Pass these values
*  to MetricSettings.getXCoord / MetricSettings.getYCoord (depending on the orientation of the chart)
*  to discover the coordinates where gradations are drawn.
*  
*  This is only allowed to be called when FacetChart.chartDrawn fires.
* 
* 
*/
getGradations(): Array<Partial<Float>>;

/**
*  Method to change the current ChartType.
*  Will redraw the chart if drawn. 
*  Will use default settings for the new chart type for MetricSettings.stacked
*  and MetricSettings.filled if those values are null.
*  
*  Note that for FacetChart.extraAxisMetrics charts this method changes the
*  chartType for the main value axis only.
* 
* 
* @param {ChartType} new chart type
*/
setChartType(chartType:ChartType): void;

/**
*  Method to change MetricSettings.stacked. Use null to
*  apply a default value for the current ChartType.
* 
* 
* @param {boolean} new value
*/
setStacked(stacked:boolean): void;

/**
*  Method to change MetricSettings.filled.
*  Use null to apply a default value for the current ChartType.
* 
* 
* @param {boolean} new value
*/
setFilled(filled:boolean): void;

/**
*  Setter for MetricSettings.proportional.
* 
* @param {boolean} Whether the chart should now use proportional mode.
*/
setProportional(proportional:boolean): void;

/**
*  Setter for MetricSettings.dataColors.
* 
* @param {Array<Partial<CSSColor>>} New set of data colors
*/
setDataColors(dataColors:Array<Partial<CSSColor>>): void;

/**
*  Get a color from the MetricSettings.dataColors Array.
*  
*  
*  Override to provide a dynamic color generation scheme - must return a color in the format of
*  of a leading hash (#) plus 6 hexadecimal digits as specified for MetricSettings.dataColors.
*  
* 
* 
* @param {number} index of the legend facet value to be colored
* @param {string | number | Date} id of the legend facet value to be colored
* @param {string} purpose for the requested color - such as "legend", "line",             "area", "points", etc.
*/
getDataColor(index:number, facetValueId:string | number | Date, purpose:string): CSSColor;

/**
*  Specifies the color to use for data lines in the chart. No default implementation. If not
*  defined or null is returned, the Framework will default to value of MetricSettings.getDataColor.
*  
*  
*  Note that this method is simply an override point, since it has no default implementation -
*  must return a color in the format of of a leading hash (#) plus 6 hexadecimal digits as
*  specified for MetricSettings.dataColors.
*  
* 
* 
* @param {number} index of the legend facet value to be colored
* @param {string | number | Date} id of the legend facet value to be colored
* @param {string} purpose for the requested color - such as "legend", "line",             "area", "points", etc.
*/
getDataLineColor(index:number, facetValueId:string | number | Date, purpose:string): CSSColor;

/**
*  Specifies the width to use for data lines in the chart. No default implementation. If not
*  defined or null is returned, the line width will be set by the appropriate chart properties,
*  such as FacetChart.dataLineProperties, FacetChart.barProperties, or FacetChart.bubbleProperties.
*  
*  
*  Note that this method is simply an override point, since it has no default implementation.
*  
* 
* 
* @param {number} index of the legend facet value to target
* @param {string | number | Date} id of the legend facet value to target
* @param {string} purpose for the requested width - such as "legend", "line",             "area", "points", etc.
*/
getDataLineWidth(index:number, facetValueId:string | number | Date, purpose:string): Integer;

/**
*  Get a gradient from the MetricSettings.dataGradients Array.
*  
*  Override to provide a dynamic gradient generation scheme.
* 
* 
* @param {number} index of the legend facet value to be colored
* @param {string | number | Date} id of the legend facet value to be colored
* @param {string} purpose for the requested gradient - such as "legend", "line",             "area", "points", etc.
*/
getDataGradient(index:number, facetValueId:string | number | Date, purpose:string): string;

/**
*  When MetricSettings.showDataPoints is true, fires when a point is clicked on.
* 
* 
* @param {Float} the value at the point
* @param {Record} the record at the point
* @param {string} the ID of the metric at the point
*/
pointClick(value:Float, record:Record, metricId:string): void;

/**
*  When MetricSettings.showDataPoints is true and the mouse hovers over a point, this method is called
*  and may return HTML to show in a hover.
* 
* 
* @param {Float} the value at the point
* @param {Record} the record at the point
* @param {string} the ID of the metric at the point
*/
getPointHoverHTML(value:Float, record:Record, metricId:string): string;

/**
*  Return the text string to display in MetricSettings.gradationLabelProperties
*  given the raw value for the metric to show on the value axis. This formatter will only
*  be called if the axis has gradation labels, meaning labels drawn at regular intervals not
*  associated with any particular facet value.
*  
*  Note that the rendering of values in hovers or via MetricSettings.showDataValues is handled by
*  MetricSettings.formatDataValue.
* 
* @param {any} raw value of the metric
*/
formatAxisValue(value:any): string;

/**
*  Return the text string to display in MetricSettings.showValueOnHover or via MetricSettings.showDataValues
*  given the raw value for the metric displayed on the value axis.
*  
*  Note that the rendering of values for gradation labels is handled by
*  MetricSettings.formatAxisValue.
* 
* @param {any} raw value of the metric
*/
formatDataValue(value:any): string;




}

/**
* DrawItem subclass to render triangles
*/
class DrawTriangle extends DrawPolygon {
/**
* Creates a new DrawTriangle
* 
* @param typeCheckedProperties {Partial<DrawTriangle>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DrawTriangle} The new instance
*/
static create(typeCheckedProperties? : Partial<DrawTriangle>, uncheckedProperties?: Object): DrawTriangle;


/**
* Array of points of the triangle. specified in the
*  DrawPane.
* @type {Array<Partial<Point>>}
* @default "[[0,0], [50,50], [100,0]]"
*/
points?: Array<Partial<Point>>;







/**
*  Returns the incenter
*  of the triangle in DrawPane.
* 
*/
getCenter(): Point;

/**
*  Resize by the specified delta
* 
* @param {Distance} number of pixels to resize by horizontally
* @param {Distance} number of pixels to resize by vertically
*/
resizeBy(dX:Distance, dY:Distance): void;




}

/**
* ISC abstraction for Flashlets.
*/
class Flashlet extends BrowserPlugin {
/**
* Creates a new Flashlet
* 
* @param typeCheckedProperties {Partial<Flashlet>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Flashlet} The new instance
*/
static create(typeCheckedProperties? : Partial<Flashlet>, uncheckedProperties?: Object): Flashlet;


/**
* Sets the 'name' attribute on the flashlet object. If a name is not provided it will be
*  auto-generated. Note that in general you don't need to set this. If you have a handle to
*  your ISC Flashlet object you can simply call Flashlet.getPluginHandle to get a
*  handle to the element.
* @type {string}
* @default null
*/
name?: string;

/**
* A map of key/value pairs to pass to the flashlet as parameters. Note that these will be set
*  on the outer &lt;object&gt; element as well as the inner &lt;embed&gt; element.
* @type {object}
* @default null
*/
params?: object;

/**
* This attribute specifies the clsid of the outer &lt;object&gt; tag.
*  
*  The default classID is: "clsid:D27CDB6E-AE6D-11cf-96B8-444553540000"
*  
* @type {string}
* @default "see below"
*/
classID?: string;

/**
* This attribute specifies the minimum version of the flash player required to show this
*  flashlet.
*  
*  The default codeBase is: "http://download.macromedia.com/pub/shockwave/cabs/flash/swflash.cab#version=5,0,0,0"
*  
* @type {URL}
* @default "see below"
*/
codeBase?: URL;

/**
* This attribute specifies the page the user should go to to get the plugin required to view
*  this flashlet.
*  
*  The default pluginsPage is: "http://www.macromedia.com/shockwave/download/index.cgi?P1_Prod_Version=ShockwaveFlash"
*  
* @type {URL}
* @default "see below"
*/
pluginsPage?: URL;

/**
* Location from which to load the Flashlet.
* @type {URL}
* @default null
*/
src?: URL;





/**
*  Is Shockwave Flash installed on this browser?
* 
*/
static flashAvailable(): boolean;

/**
*  Which version of Flash is installed on this browser?
* 
*/
static getFlashVersion(): number;



/**
*  
*  Sets the source file for the flash component 
* 
* 
*/
setSrc(): void;

/**
*  
*  Returns a handle to the flashlet DOM element (valid only after the component has been drawn). 
* 
* 
*/
getPluginHandle(): Element;




}

/**
* An object containing the "view state" information for a treeGrid. In addition to the 
*  state data contained by a ListGridViewState object, this will also contain the 
*  current open state of the treeGrid in question.
*  Note that this object is not intended to be interrogated directly, but may be stored 
*  (for example) as a blob on the server for view state persistence across sessions.
*/
type TreeGridViewState = string;




/**
* Approach to force a text value to be interpreted as text rather than parsed as a date, time
*  or other structured types, as can happen with Microsoft Excel. For background information,
*  see excelPasting.
*/
type ForceTextApproach = 
/**
* a leading space character is added
*/
"leadingSpace" |

/**
* text value is turned into a trivial Excel formula (eg "car" becomes ="car"). In Excel, this renders just the value "car" but editing the cell reveals the formula.
*/
"formula";


/**
* A StretchImg that behaves like a button, going through up/down/over state transitions in response
*  to user events. Supports an optional title, and will auto-size to accommodate the title text if
*  overflow is set to "visible".
*  
*  Examples of use include fancy buttons, poplist headers, and tabs.
*/
class StretchImgButton extends StretchImg {
/**
* Creates a new StretchImgButton
* 
* @param typeCheckedProperties {Partial<StretchImgButton>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {StretchImgButton} The new instance
*/
static create(typeCheckedProperties? : Partial<StretchImgButton>, uncheckedProperties?: Object): StretchImgButton;


/**
* The title HTML to display in this button.
* @type {HTMLString}
* @default null
*/
title?: HTMLString;

/**
* If true and the title is clipped, then a hover containing the full title of this button
*  is enabled.
* @type {boolean}
* @default false
*/
showClippedTitleOnHover?: boolean;

/**
* Should the title for this button wrap? If unset, default behavior is to allow wrapping
*  if this.vertical is true, otherwise disallow wrapping
* @type {boolean}
* @default null
*/
wrap?: boolean;

/**
* Optional icon to be shown with the button title text. 
*  
*  Specify as the partial URL to an image, relative to the imgDir of this component.
*  A sprited image can be specified using the SCSpriteConfig format.
*  
*  Note that the string "blank" is a valid setting for this attribute and will always 
*  result in the system blank image, with no state suffixes applied. Typically, this 
*  might be used when an iconStyle is also specified and the iconStyle renders the icon via 
*  a stateful background-image or other CSS approach.
* @type {SCImgURL}
* @default null
*/
icon?: SCImgURL;

/**
* Size in pixels of the icon image.
*  
*  The StatefulCanvas.iconWidth and StatefulCanvas.iconHeight
*  properties can be used to configure width and height separately.
*  
*  Note: When configuring the properties of a StatefulCanvas (or derivative)
*  AutoChild, it is best to set the iconWidth and iconHeight
*  to the same value rather than setting an iconSize. This is because certain
*  skins or customizations thereto might set the iconWidth and iconHeight,
*  making the customization of the AutoChild's iconSize ineffective.
* @type {Integer}
* @default "16"
*/
iconSize?: Integer;

/**
* Width in pixels of the icon image.
*  
*  If unset, defaults to StatefulCanvas.iconSize.
* @type {Integer}
* @default null
*/
iconWidth?: Integer;

/**
* Height in pixels of the icon image.
*  
*  If unset, defaults to StatefulCanvas.iconSize.
* @type {Integer}
* @default null
*/
iconHeight?: Integer;

/**
* If this button is showing an icon should it appear to the left or right of the title?
*  valid options are "left" and "right".
* @type {string}
* @default "left"
*/
iconOrientation?: string;

/**
* If this button is showing an icon should it be right or left aligned?
* @type {string}
* @default null
*/
iconAlign?: string;

/**
* Pixels between icon and title text.
* @type {Integer}
* @default "6"
*/
iconSpacing?: Integer;

/**
* If using an icon for this button, whether to switch the icon image if the button becomes
*  disabled.
* @type {boolean}
* @default true
*/
showDisabledIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image on mouse rollover.
* @type {boolean}
* @default false
*/
showRollOverIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image when the button
*  receives focus.
*  
*  If StatefulCanvas.showFocusedAsOver is true, the "Over" icon will be
*  displayed when the canvas has focus, otherwise a separate "Focused" icon
*  will be displayed
* @type {boolean}
* @default false
*/
showFocusedIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image when the mouse goes
*  down on the button.
* @type {boolean}
* @default false
*/
showDownIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image when the button
*  becomes selected.
* @type {boolean}
* @default false
*/
showSelectedIcon?: boolean;

/**
* If true, ignore the specified size of this widget and always size just large
*  enough to accommodate the title. If setWidth() is explicitly called on an
*  autoFit:true button, autoFit will be reset to false.
*  
*  Note that for StretchImgButton instances, autoFit will occur horizontally only, as 
*  unpredictable vertical sizing is likely to distort the media. If you do want vertical 
*  auto-fit, this can be achieved by simply setting a small height, and having 
*  overflow:"visible"
* @type {boolean}
* @default null
*/
autoFit?: boolean;

/**
* Base CSS style className applied to the component. 
*  
*  As the component changes StatefulCanvas.state and/or is selected, 
*  suffixes will be added to the base style. In some cases more than one suffix will 
*  be appended to reflect a combined state ("Selected" + "Disabled", for example).
*  
*  See StatefulCanvas.getStateSuffix for a description of the default set
*  of suffixes which may be applied to the baseStyle
*  
*  Rotated Titles
*  
*  The Framework doesn't have built-in support for rotating button titles in a fashion
*  similar to FacetChart.rotateLabels. However, you can manually configure
*  a button to render with a rotated title by applying custom CSS via this property.
*  
*  For example, given a button with a height of 120 and a width of 48, if you
*  copied the existing buttonXXX style declarations from skin_styles.css as new,
*  rotatedTitleButtonXXX declarations, and then added the lines:
*  
*    -ms-transform:   translate(-38px,0px) rotate(270deg);
*    -webkit-transform: translate(-38px,0px) rotate(270deg);
*    transform:     translate(-38px,0px) rotate(270deg);
*    overflow: hidden;
*    text-overflow: ellipsis;
*    width:116px;
*  in the declaration section beginning:
*  
*  .rotatedTitleButton,
*  .rotatedTitleButtonSelected,
*  .rotatedTitleButtonSelectedOver,
*  .rotatedTitleButtonSelectedDown,
*  .rotatedTitleButtonSelectedDisabled,
*  .rotatedTitleButtonOver,
*  .rotatedTitleButtonDown,
*  .rotatedTitleButtonDisabled {
*  then applying that style to the button with Canvas.overflow: "clip_h"
*  would yield a vertically-rendered title with overflow via ellipsis as expected, and also
*  wrap with Button.wrap.
*  
*  Note that:
*  
* - The explicit width applied via CSS is needed because rotated
*  elements don't inherit dimensions in their new orientation from the DOM - 
*  the transform/rotation occurs independently of layout.
*  
* - The translation transform required along the x-axis is roughly
*  (width - height) / 2, but may need slight offsetting for optimal centering.
*  
* - We've explicitly avoided describing an approach based on CSS "writing-mode", since
*  support is incomplete and bugs are present in popular browsers such as Firefox and
*  Safari that would prevent it from being used without Framework assistance.
* @type {CSSStyleName}
* @default "stretchImgButton"
*/
baseStyle?: CSSStyleName;

/**
* CSS style applied to the title text only. Defaults to StretchImgButton.baseStyle when unset.
*  
*  With a separate titleStyle and StretchImgButton.baseStyle set, you can provide a
*  backgroundColor via baseStyle that will allow translucent .png media to be
*  "tinted" by the underlying background color, so that a single set of media can provide
*  range of color options. In this usage, the titleStyle should generally not
*  specify a background color as this would block out the media that appears behind the
*  title.
* @type {CSSStyleName}
* @default null
*/
titleStyle?: CSSStyleName;

/**
* Specifies a skinImgDir to apply to the label containing the title of this 
*  StretchImgButton. May be null in which case this.skinImgDir
*  will be applied to the label as well.
*  
*  Note that icons displayed in the title may make use of the skin img dir set here
* @type {URL}
* @default null
*/
labelSkinImgDir?: URL;

/**
* Whether this component is selected. For some components, selection affects appearance.
* @type {boolean}
* @default false
*/
selected?: boolean;

/**
* String identifier for this canvas's mutually exclusive selection group.
* @type {string}
* @default null
*/
radioGroup?: string;

/**
* Behavior on state changes -- BUTTON, RADIO or CHECKBOX
* @type {SelectionType}
* @default "button"
*/
actionType?: SelectionType;

/**
* Current "state" of this widget. The state setting is automatically updated as the 
*  user interacts with the component (see StatefulCanvas.showRollOver, 
*  StatefulCanvas.showDown, StatefulCanvas.showDisabled).
*  
*  StatefulCanvases will have a different appearance based
*  on their current state. 
*  By default this is handled by changing the css className applied to
*  the StatefulCanvas - see StatefulCanvas.baseStyle and 
*  StatefulCanvas.getStateSuffix for a description of how this is done.
*  
*  For Img or StretchImg based subclasses of StatefulCanvas, the 
*  appearance may also be updated by changing the src of the rendered image. See
*  Img.src and StretchImgButton.src for a description of how the URL 
*  is modified to reflect the state of the widget in this case.
* @type {State}
* @default ""
*/
state?: State;

/**
* Should we visibly change state when disabled?
* @type {boolean}
* @default true
*/
showDisabled?: boolean;

/**
* Should we visibly change state when the mouse goes down in this object?
* @type {boolean}
* @default false
*/
showDown?: boolean;

/**
* Should we visibly change state when the canvas receives focus? Note that by default the
*  over state is used to indicate focus.
* @deprecated * as of SmartClient version 6.1 in favor of StatefulCanvas.showFocused

* @type {boolean}
* @default false
*/
showFocus?: boolean;

/**
* Should we visibly change state when the canvas receives focus? If
*  StatefulCanvas.showFocusedAsOver is true, then "over"
*  will be used to indicate focus. Otherwise a separate "focused" state
*  will be used.
* @type {boolean}
* @default false
*/
showFocused?: boolean;

/**
* Should we visibly change state when the mouse goes over this object?
* @type {boolean}
* @default false
*/
showRollOver?: boolean;

/**
* Horizontal alignment of this component's title.
* @type {Alignment}
* @default "center"
*/
align?: Alignment;

/**
* Vertical alignment of this component's title.
* @type {VerticalAlignment}
* @default "center"
*/
valign?: VerticalAlignment;

/**
* Determines whether any specified StatefulCanvas.getTitle will be 
*  displayed for this component.
*  Applies to Image-based components only, where the title will be rendered out in a label
*  floating over the component
* @type {boolean}
* @default true
*/
showTitle?: boolean;

/**
* The padding for a StretchImgButton's label is determined as follows.
*  
*  If labelHPad is set it will specify the horizontal padding applied to the
*  label. Similarly if labelVPad is set it will specify the vertical padding
*  for the label, regardless of the button's StretchImgButton.vertical setting.
*  
*  Otherwise labelLengthPad can be set to specify the label padding along the
*  length axis (ie: horizontal padding if StretchImgButton.vertical is false,
*  otherwise vertical padding), and 
*  labelBreadthPad can be set to specify the label padding along the other axis.
*  
*  Otherwise the padding on the length axis will match the StretchImgButton.capSize and
*  will be set to zero on the breadth axis.
*  
*  So by default the label will be sized to match the center image of the StretchImgButton, but
*  these settings allow the label to partially or wholly overlap the caps.
* @type {number}
* @default null
*/
labelHPad?: number;

/**
* The padding for a StretchImgButton's label is determined as follows.
*  
*  If labelHPad is set it will specify the horizontal padding applied to the
*  label. Similarly if labelVPad is set it will specify the vertical padding
*  for the label, regardless of the button's StretchImgButton.vertical setting.
*  
*  Otherwise labelLengthPad can be set to specify the label padding along the
*  length axis (ie: horizontal padding if StretchImgButton.vertical is false,
*  otherwise vertical padding), and 
*  labelBreadthPad can be set to specify the label padding along the other axis.
*  
*  Otherwise the padding on the length axis will match the StretchImgButton.capSize and
*  will be set to zero on the breadth axis.
*  
*  So by default the label will be sized to match the center image of the StretchImgButton, but
*  these settings allow the label to partially or wholly overlap the caps.
* @type {number}
* @default null
*/
labelVPad?: number;

/**
* The padding for a StretchImgButton's label is determined as follows.
*  
*  If labelHPad is set it will specify the horizontal padding applied to the
*  label. Similarly if labelVPad is set it will specify the vertical padding
*  for the label, regardless of the button's StretchImgButton.vertical setting.
*  
*  Otherwise labelLengthPad can be set to specify the label padding along the
*  length axis (ie: horizontal padding if StretchImgButton.vertical is false,
*  otherwise vertical padding), and 
*  labelBreadthPad can be set to specify the label padding along the other axis.
*  
*  Otherwise the padding on the length axis will match the StretchImgButton.capSize and
*  will be set to zero on the breadth axis.
*  
*  So by default the label will be sized to match the center image of the StretchImgButton, but
*  these settings allow the label to partially or wholly overlap the caps.
* @type {number}
* @default null
*/
labelLengthPad?: number;

/**
* The padding for a StretchImgButton's label is determined as follows.
*  
*  If labelHPad is set it will specify the horizontal padding applied to the
*  label. Similarly if labelVPad is set it will specify the vertical padding
*  for the label, regardless of the button's StretchImgButton.vertical setting.
*  
*  Otherwise labelLengthPad can be set to specify the label padding along the
*  length axis (ie: horizontal padding if StretchImgButton.vertical is false,
*  otherwise vertical padding), and 
*  labelBreadthPad can be set to specify the label padding along the other axis.
*  
*  Otherwise the padding on the length axis will match the StretchImgButton.capSize and
*  will be set to zero on the breadth axis.
*  
*  So by default the label will be sized to match the center image of the StretchImgButton, but
*  these settings allow the label to partially or wholly overlap the caps.
* @type {number}
* @default null
*/
labelBreadthPad?: number;

/**
* If set to true, if the StatefulCanvas.title of this button contains the
*  specified Canvas.accessKey, when the title is displayed to the user
*  it will be modified to include HTML to underline the accessKey.
*  Note that this property may cause titles that include HTML (rather than simple strings)
*  to be inappropriately modified, so should be disabled if your title string includes
*  HTML characters.
* @type {boolean}
* @default true
*/
hiliteAccessKey?: boolean;

/**
* Base URL for the image. By default, StretchImgButtons consist of three image parts: A
*  start image (displayed at the top or left), a scalable central image and an end image
*  displayed at the bottom or right.
*  
*  The images displayed in the stretchImgButton are derived from this property in the 
*  following way:
*  
*  
*  
* - When the button is in its standard state the suffixes "_start", "_end" and 
*    "_stretch" are applied to the src (before the file extension), so by default 
*    the images displayed will be "button_start.gif" (sized to be 
*    this.capSize by the specified width of the stretchImgButton), 
*    "button_stretch.gif" (stretched to the necessary width) and "button_end.gif" 
*    (sized the same as the start image).
*  
* - As the button's state changes, the images will have suffixes appended before
*    the "_start" / "_end" / "_stretch" to represent the button state.
*    See Img.src for an overview of how states are combined into a compound
*    URL.
*  
*  For example the center piece of a selected stretchImgButton with the mouse hovering
*  over it might have the URL: "button_Selected_Down_stretch.gif".
*  
*  Media should be present for each possible state of the _start, _end and _stretch images.
* @type {SCImgURL}
* @default "button.gif"
*/
src?: SCImgURL;

/**
* Default is a horizontal button. Vertical StretchImgButtons are allowed, but title text,
*  if any, will not be automatically rotated.
* @type {boolean}
* @default false
*/
vertical?: boolean;

/**
* How big are the end pieces by default
* @type {number}
* @default "12"
*/
capSize?: number;







/**
*  Return the title - HTML drawn inside the component.
*  
*  Default is to simply return this.title.
* 
*/
getTitle(): HTMLString;

/**
*  Setter for the StatefulCanvas.title.
* 
* @param {HTMLString} the new title HTML.
*/
setTitle(newTitle:HTMLString): void;

/**
*  Changes the orientation of the icon relative to the text of the button.
* 
* 
* @param {string} The new orientation of the icon relative to the text of the button.
*/
setIconOrientation(orientation:string): void;

/**
*  Change the icon being shown next to the title text.
* 
* @param {SCImgURL} URL of new icon
*/
setIcon(icon:SCImgURL): void;

/**
*  Setter method for the StatefulCanvas.autoFit property. Pass in true or false to turn
*  autoFit on or off. When autoFit is set to false, canvas will be resized to
*  it's previously specified size.
* 
* @param {boolean} New autoFit setting.
*/
setAutoFit(autoFit:boolean): void;

/**
*  Sets the base CSS style. As the component changes state and/or is selected, suffixes will be
*  added to the base style.
* 
* @param {CSSStyleName} new base style
*/
setBaseStyle(style:CSSStyleName): void;

/**
*  Sets the StretchImgButton.titleStyle, which is applied to the title text.
* 
* @param {CSSStyleName} new title style
*/
setTitleStyle(style:CSSStyleName): void;

/**
*  setter for StretchImgButton.labelSkinImgDir.
* 
* @param {URL} new skin img dir to apply to the label holding title text for  this widget.
*/
setLabelSkinImgDir(URL:URL): void;

/**
*  Select this object.
* 
*/
select(): void;

/**
*  Select this object.
* 
*/
deselect(): void;

/**
*  Find out if this object is selected
* 
*/
isSelected(): boolean;

/**
*  Select this object.
* 
*/
setSelected(): void;

/**
*  Add this widget to the specified mutually exclusive selection group with the ID
*  passed in.
*  Selecting this widget will then deselect any other StatefulCanvases with the same
*  radioGroup ID.
*  StatefulCanvases can belong to only one radioGroup, so this method will remove from 
*  any other radiogroup of which this button is already a member.
* 
* @param {string} - ID of the radiogroup to which this widget should be added
*/
addToRadioGroup(groupID:string): void;

/**
*  Remove this widget from the specified mutually exclusive selection group with the ID
*  passed in.
*  No-op's if this widget is not a member of the groupID passed in.
*  If no groupID is passed in, defaults to removing from whatever radioGroup this widget
*  is a member of.
* 
* @param {string=} - optional radio group ID (to ensure the widget is removed                    from the appropriate group.
*/
removeFromRadioGroup(groupID?:string): void;

/**
*  Update the 'actionType' for this canvas (radio / checkbox / button)
*  If the canvas is currently selected, and the passed in actionType is 'button'
*  this method will deselect the canvas.
* 
*/
setActionType(): void;

/**
*  Return the 'actionType' for this canvas (radio / checkbox / button)
* 
*/
getActionType(): void;

/**
*  Sets the StatefulCanvas.state of this object, changing its appearance.
*  Note: newState cannot be
*  "Disabled"
*  
*  if StatefulCanvas.showDisabled is false.
* 
* 
* @param {State} the new state.
*/
setState(newState:State): void;

/**
*  Enable or disable this object
* 
* @param {boolean} true if this widget is to be disabled
*/
setDisabled(disabled:boolean): void;

/**
*  Return the state of this StatefulCanvas
* 
*/
getState(): State;

/**
*  This property contains the default 'action' for the Button to fire when activated.
* 
*/
action(): void;

/**
*  Is the title of this button clipped?
* 
*/
titleClipped(): boolean;

/**
*  Returns the HTML that is displayed by the default StretchImgButton.titleHover
*  handler. Return null or an empty string to cancel the hover.
*  
* 
* @param {HTMLString} the HTML that would have been displayed by default
*/
titleHoverHTML(defaultHTML:HTMLString): HTMLString;

/**
*  If this button is showing an StretchImgButton.icon, a separate click
*  handler for the icon may be defined as this.iconClick.
*  Returning false will suppress the standard button click handling code.
* 
*/
iconClick(): boolean;

/**
*  Optional stringMethod to fire when the user hovers over this button and the title is
*  clipped. If StretchImgButton.showClippedTitleOnHover is true, the default behavior is to
*  show a hover canvas containing the HTML returned by StretchImgButton.titleHoverHTML.
*  Return false to suppress this default behavior.
* 
*/
titleHover(): boolean;




}

/**
* Strings to identify the various keys on the keyboard.
*  
*  
* -  For alpha keys, the single (uppercase) character value is used, such as "Q"
*  
* -  For Numeric keys, the number is used as in a single character string, like "1"
*  
* -  Function keys are identified as "f1" - "f12"
*  
* -  Non alpha-numeric character keys (such as the key for "[" / "{") are identified by
*     their unmodified character value (the value obtained by hitting the key without 
*     holding shift down), by default - exceptions are listed below.
*  
* -  Additional key names:
*    - Space  
*    - Tab  
*    - Enter
*    - Escape
*    - Backspace
*    - Insert
*    - Delete
*    - Arrow_Up
*    - Arrow_Down
*    - Arrow_Left
*    - Arrow_Right      
*    - Home
*    - End
*    - Page_Up  
*    - Page_Down  
*    - Shift
*    - Ctrl
*    - Alt
*  
*  [Note: Some keys may not be available for capture on every platform]
*/
type KeyName = string;


/**
*/
type SortDirection = 
/**
* Sort in ascending order (eg: A-Z, larger items later in the list)
*/
"ascending" |

/**
* Sort in descending order (eg: Z-A, larger items earlier in the list)
*/
"descending";


/**
* An application that allows developers to create and manage SmartClient screens and 
*  datasources. This is an internal only class - do not try to use it directly 
*  in your applications. If you want to create visual tools similar to VisualBuilder, 
*  see devTools
*/
class VisualBuilder extends Class {
/**
* Creates a new VisualBuilder
* 
* @param typeCheckedProperties {Partial<VisualBuilder>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {VisualBuilder} The new instance
*/
static create(typeCheckedProperties? : Partial<VisualBuilder>, uncheckedProperties?: Object): VisualBuilder;


/**
* If set to true the built-in save file mechanism is enabled, allowing you to save files to
*  offline storage or the server filesystem (if enabled)
* @type {boolean}
* @default false
*/
saveFileBuiltinIsEnabled?: boolean;

/**
* If set to true the built-in load file mechanism is enabled, allowing you to load files from
*  offline storage or the server filesystem (if enabled)
* @type {boolean}
* @default false
*/
loadFileBuiltinIsEnabled?: boolean;

/**
* If set to true, allows the built-in save file and load file operations to access the server
*  filesystem. Note, this also requires appropriate server-side permission - your 
*  server.properties file must specify 
*  FilesystemDataSource.enabled: true.
*  
*  If this property is false, saving and loading (if enabled) will be to and from local 
*  Offline.
* @type {boolean}
* @default false
*/
filesystemDataSourceEnabled?: boolean;

/**
* The name of a skin to use. Note that Visual Builder may use two skins - the skin
*  specified here, for the elements of the application you are building, and a high-contrast, 
*  white-on-black "ToolSkin" for the elements of Visual Builder itself. When in "Toolskin" 
*  mode (which is switchable at runtime through the UI), the skin property only
*  affects the skin used by the visual elements of the application you are building.
* @type {string}
* @default "Tahoe"
*/
skin?: string;

/**
* The default application mode. Note, this can be changed through the UI - see 
*  VisualBuilder.showModeSwitcher
* @type {ApplicationMode}
* @default "edit"
*/
defaultApplicationMode?: ApplicationMode;

/**
* If this property is not explicitly set to false, Visual Builder shows a UI to allow the 
*  ApplicationMode to be toggled at runtime.
* @type {boolean}
* @default null
*/
showModeSwitcher?: boolean;

/**
* If true, starts Visual Builder in mockup mode
* @type {boolean}
* @default false
*/
mockupMode?: boolean;

/**
* Whether to use the existing browser window or a new one when opening a Mockup Mode screen
*  converted to standard Component XML via "Go to Visual Builder".
* @type {boolean}
* @default true
*/
openFullBuilderSeparately?: boolean;

/**
* A PaletteNode describing a component to add to an empty screen as an initial container.
* @type {PaletteNode}
* @default null
*/
initialComponent?: PaletteNode;

/**
* Relative URL to Visual Builder's defaultComponents.xml configuration file. The 
*  default value makes the framework look wherever Visual Builder itself was loaded from
* @type {string}
* @default "defaultComponents.xml"
*/
defaultComponentsURL?: string;

/**
* Relative URL to Visual Builder's defaultMockupComponents.xml configuration file. The 
*  default value makes the framework look wherever Visual Builder itself was loaded from
* @type {string}
* @default "defaultMockupComponents.xml"
*/
defaultMockupComponentsURL?: string;

/**
* Relative URL to Visual Builder's customComponents.xml configuration file. The 
*  default value makes the framework look wherever Visual Builder itself was loaded from
* @type {string}
* @default "customComponents.xml"
*/
customComponentsURL?: string;

/**
* Relative URL to Visual Builder's globalDependencies.xml configuration file. The 
*  default value makes the framework look wherever Visual Builder itself was loaded from
* @type {string}
* @default "globalDependencies.xml"
*/
globalDependenciesURL?: string;










}

/**
* Put a value in just one field of a form.
*/
class FormSetFieldValueTask extends ComponentTask {
/**
* Creates a new FormSetFieldValueTask
* 
* @param typeCheckedProperties {Partial<FormSetFieldValueTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FormSetFieldValueTask} The new instance
*/
static create(typeCheckedProperties? : Partial<FormSetFieldValueTask>, uncheckedProperties?: Object): FormSetFieldValueTask;


/**
* Field to assign new value.
* @type {FieldName}
* @default null
*/
targetField?: FieldName;

/**
* Value to assign to FormSetFieldValueTask.targetField.
* @type {any}
* @default null
*/
value?: any;










}

/**
* Set of horizontally arranged buttons.
*/
class ToolbarItem extends CanvasItem {
/**
* Creates a new ToolbarItem
* 
* @param typeCheckedProperties {Partial<ToolbarItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ToolbarItem} The new instance
*/
static create(typeCheckedProperties? : Partial<ToolbarItem>, uncheckedProperties?: Object): ToolbarItem;


/**
* List of buttons for the toolbar.
*  Each button should be specified as a simple JS object with properties to apply to the
*  button to be displayed. Note that any click stringMethod applied to the
*  button will be passed 2 parameters: form and item.
* @type {Array<Partial<StatefulCanvas>>}
* @default null
*/
buttons?: Array<Partial<StatefulCanvas>>;

/**
* Don't show a title for toolbars
* @type {boolean}
* @default false
*/
showTitle?: boolean;

/**
* Space between the buttons of this toolbar. Configures the Layout.membersMargin 
*  property on the created ToolbarItem.canvas.
* @type {number}
* @default "4"
*/
buttonSpace?: number;

/**
* these items are in a row by themselves by default
* @type {boolean}
* @default true
*/
startRow?: boolean;

/**
* these items are in a row by themselves by default
* @type {boolean}
* @default true
*/
endRow?: boolean;

/**
* these items span all columns
* @type {Integer | string}
* @default "*"
*/
colSpan?: Integer | string;

/**
* This item is an autoChild generated Canvas displayed by
*  the ToolbarItem and is an instance of Toolbar by default, 
*  customizable via the ToolbarItem.canvasConstructor attribute.
* @type {Canvas}
* @default null
*/
canvas?: Canvas;

/**
* Constructor class for this toolbarItem's ToolbarItem.canvas.
* @type {Canvas}
* @default "Toolbar"
*/
canvasConstructor?: Canvas;

/**
* Should the toolbar stack its buttons vertically or horizontally?
* @type {boolean}
* @default false
*/
vertical?: boolean;

/**
* Use AutoFitButtons rather than simple Buttons for this toolbar's buttons by default.
* @type {Class}
* @default "AutoFitButton"
*/
buttonConstructor?: Class;

/**
* Default properties for this toolbar's buttons.
* @type {object}
* @default null
*/
buttonProperties?: object;

/**
* If specified this baseStyle will be applied to the buttons in this toolbar.
* @type {CSSStyleName}
* @default null
*/
buttonBaseStyle?: CSSStyleName;

/**
* If set to true, causes the toolbar created by this item to create its child buttons 
*  during initialization, instead of waiting until draw().
*  
*  See the corresponding Toolbar.createButtonsOnInit for more
*  information.
* @type {boolean}
* @default null
*/
createButtonsOnInit?: boolean;










}

/**
*/
type SendMethod = 
/**
* GET method (regular URL encoding)
*/
"GET" |

/**
* POST method (form field encoding)
*/
"POST";




/**
* Add a spacer to a Layout that takes up space just like a normal member, without actually
*  drawing anything. A LayoutSpacer is semantically equivalent to using an empty canvas,
*  but higher performance for this particular use case.
*/
class LayoutSpacer extends Canvas {
/**
* Creates a new LayoutSpacer
* 
* @param typeCheckedProperties {Partial<LayoutSpacer>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {LayoutSpacer} The new instance
*/
static create(typeCheckedProperties? : Partial<LayoutSpacer>, uncheckedProperties?: Object): LayoutSpacer;











}

/**
* Sets the title of a SectionStack section. The section is identified by 
*  specifying either the SetSectionTitleTask.targetSectionName or
*  SetSectionTitleTask.targetSectionTitle.
*/
class SetSectionTitleTask extends ComponentTask {
/**
* Creates a new SetSectionTitleTask
* 
* @param typeCheckedProperties {Partial<SetSectionTitleTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SetSectionTitleTask} The new instance
*/
static create(typeCheckedProperties? : Partial<SetSectionTitleTask>, uncheckedProperties?: Object): SetSectionTitleTask;


/**
* Title to assign to section. To assign a dynamic value see SetSectionTitleTask.textFormula.
* @type {HTMLString}
* @default null
*/
title?: HTMLString;

/**
* Formula to be used to calculate the section title contents. Use SetSectionTitleTask.title property
*  to assign a static title instead.
*  
*  Available fields for use in the formula are the current Canvas.ruleScope.
* @type {UserSummary}
* @default null
*/
textFormula?: UserSummary;

/**
* The name of the target section.
* @type {string}
* @default null
*/
targetSectionName?: string;

/**
* The current title of the target section.
* @type {string}
* @default null
*/
targetSectionTitle?: string;










}





/**
* An operationBinding tells a DataSource how to execute one of the basic DS operations: fetch,
*  add, update, remove. See DataSource.operationBindings.
*/
class OperationBinding extends Class {
/**
* Creates a new OperationBinding
* 
* @param typeCheckedProperties {Partial<OperationBinding>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {OperationBinding} The new instance
*/
static create(typeCheckedProperties? : Partial<OperationBinding>, uncheckedProperties?: Object): OperationBinding;


/**
* If set to "false", transformation of values for
*  DataSourceField.multiple fields, normally controlled by
*  DataSourceField.multipleStorage, is instead disabled for this OperationBinding.
* @type {boolean}
* @default null
*/
transformMultipleFields?: boolean;

/**
* Setting skipAudit to true indicates that DataSource.audit 
*  must be skipped for this operationBinding.
*  
*  Note, that this setting can be overrided by server-side API DSRequest.setSkipAudit().
* @type {boolean}
* @default null
*/
skipAudit?: boolean;

/**
* A per-operationBinding setting for beanClassName, otherwise also settable at the top-level
*  DataSource configuration.
* @type {string}
* @default null
*/
beanClassName?: string;

/**
* Which operationType this operationBinding is for. This property is only settable on an
*  operationBinding, not a DataSource as a whole.
* @type {DSOperationType}
* @default null
*/
operationType?: DSOperationType;

/**
* Optional operationId if this DataSource supports two or more variants of one of the basic
*  DataSource operations, for instance, a "fetch" that uses full text search and a "fetch" that
*  accepts per-field search criteria. See DSRequest.operationId for usage.
* @type {string}
* @default null
*/
operationId?: string;

/**
* A comma-separated list of field names that must be present in criteria / advancedCriteria provided by the caller.
*  Failure to provide any one of these will yield a RPCResponse.STATUS_CRITERIA_REQUIRED_ERROR from the server.
* @type {string}
* @default null
*/
requiredCriterion?: string;

/**
* The name of the method to invoke on the ServerObject for this operationBinding.
*  
*  NOTE: If you have a DataSource.serverObject and
*  wish to override this operation so that it simply calls a different method on the same 
*  server object, it is sufficient to specify just this property on the operationBinding:
*  there is no need to redefine the serverObject at the operationBinding level.
* @type {string}
* @default null
*/
serverMethod?: string;

/**
* You can explicitly declare the arguments to be passed to
*  OperationBinding.serverMethod using this attribute. This isn't required - in the
*  absence of methodArguments, the DMI implementation will still automatically
*  pass a stock set of arguments to your method (see the overview in ServerObject), but
*  specifying arguments gives you the ability to call pre-existing methods without adding
*  SmartClient-specific code.
*  
*  The format for specifying methodArguments is as a comma separated list of VTL
*  (Velocity Template Language) expressions. See the
*  VTL Reference
*  and Velocity User Guide 
*  for an overview of how to use VTL.
*  
*  The Velocity context is pre-populated with the following variables - you can pass these
*  verbatim as arguments, or call methods on these objects and pass the resulting values:
*  
*  
* - dsRequest: instance of the current DSRequest
*  
* - request: the current HttpServletRequest
*  
* - response: the current HttpServletResponse
*  
* - rpcManager: the instance of RPCManager for this request
*  
* - dataSource: a DataSource instance for this request
*  
*  So, for example, if you had a method signature like the following:
*  
*  public DSResponse fetch(SupplyItem criteria, long startRow, long endRow)
*  
*  You can invoke it by specifying methodArguments as follows:
*  
*  methodArguments="$dsRequest.criteria, $dsRequest.startRow, $dsRequest.endRow"
*  
*  Without methodArguments, there would be no way for you to specify
*  startRow/endRow as arguments. You could, of course, simply declare the method
*  to take a DSRequest object and call getStartRow()/getEndRow() in
*  the body of the method.
* @type {string}
* @default null
*/
methodArguments?: string;

/**
* Optional ServerObject declaration that specifies the ServerObject configuration for this
*  operationBinding. In the absence of a serverObject specification here, the one specified on
*  the DataSource itself is used by default, if present (DataSource.serverObject). If
*  neither is present, then Direct Method Invocation will not be enabled for this operationBinding.
*  
*  Note that if a dataSource configuration has both a OperationBinding.script block and
*  a specified serverObject for some operation, the
*  script block will be executed, and the serverObject ignored.
* @type {ServerObject}
* @default null
*/
serverObject?: ServerObject;

/**
* Scriptlet to be executed prior to the DataSource operation which is configured by this
*  operationBinding. This setting overrides any 
*  DataSource.script
*  for this operation.
*  
*  Scriptlets are used similarly to DMIs configured via OperationBinding.serverObject -
*  they can add business logic by modifying the DSRequest before it's executed, modifying the
*  default DSResponse, or taking other, unrelated actions. 
*  
*  Scriptlets are used similarly to DMIs configured via DataSource.serverObject or
*  OperationBinding.serverObject -
*  they can add business logic by modifying the DSRequest before it's executed, modifying the
*  default DSResponse, or taking other, unrelated actions. 
*  
*  For example:
*  
*   &lt;operationBindings&gt;
*     &lt;operationBinding operationType="add"&gt;
*       &lt;script language="groovy"&gt;
*        ... Groovy code ...
*       &lt;/script&gt;
*     &lt;/operationBinding&gt;
*   &lt;/operationBindings&gt;
*  
*  
*  Scriptlets can be written in any language supported by the "JSR 223" standard, including Java
*  itself. See the dmiOverview for rules on how to return data,
*  add additional imports, and other settings.
*  
*  The following variables are available for DMI scriptlets:
*  
*  
* - dataSource: the current DataSource
*  
* - dsRequest: the current DSRequest
*  
* - criteria: shortcut to DSRequest.getCriteria() (a Map)
*  
* - values: shortcut to DSRequest.getValues() (a Map)
*  
* - oldValues: shortcut to DSRequest.getOldValues() (a Map)
*  
* - sqlConnection: SQLDataSource only: the current SQLConnection object. If
*  using DataSource.autoJoinTransactions are enabled, this
*  SQLConnection is in the context of the current transaction.
*  
* - beanFactory: the spring BeanFactory (when applicable)
*  
* - cdiBeanManager: the CDI BeanManager (when applicable)
*  
*  
*  Scriptlets also have access to a set of contextual variables related to the Servlets
*  API, as follows:
*  
*  
* - servletRequest: the current ServletRequest
*  
* - session: the current HttpSession
*  
* - rpcManager: the current RPCManager
*  
* - servletResponse: the current ServletResponse (advanced use only)
*  
* - servletContext: the current ServletContext(advanced use only)
*  
*  As with DMI in general, be aware that if you write scriptlets that depend upon these variables,
*  you preclude your DataSource from being used in the widest possible variety of
*  circumstances. For example, adding a scriptlet that relies on the HttpSession
*  prevents your DataSource from being used in a command-line process.
*  
*  Note that if a dataSource configuration has both a &lt;script&gt; block and
*  a specified OperationBinding.serverObject for some operation, the
*  script block will be executed, and the serverObject ignored.
* @type {string}
* @default null
*/
script?: string;

/**
* Name of the web service operation that will be invoked in order to execute this DataSource
*  operation.
*  
*  Valid only for a DataSource that has a WebService (DataSource.serviceNamespace is
*  set). Otherwise, use OperationBinding.dataURL.
*  
*  Setting wsOperation means that DSRequest.data will be serialized as the
*  request message for the specified web service operation, with namespacing and soap encoding
*  handled automatically. See DataSource.transformRequest for how to customize what
*  data is sent to the server.
* @type {string}
* @default null
*/
wsOperation?: string;

/**
* URL to contact to fulfill DSRequests for this operationBinding.
*  
*  dataURL is typically set as DataSource.dataURL rather than on each individual
*  operationBinding.
*  
*  dataURL can be omitted for a DataSource using a Web Service
*  (DataSource.serviceNamespace is set).
* @type {URL}
* @default null
*/
dataURL?: URL;

/**
* Controls the format in which inputs are sent to the dataURL.
*  
*  When a DataSource operation such as fetchData() is invoked on this DataSource or a component
*  bound to this DataSource, the data passed to the operation, if any, will be sent to the
*  dataURL. The dataProtocol property controls the format in
*  which the data is sent: SOAP message, HTTP GET or POST of parameters, etc.
*  
*  The dataProtocol property need not be set for a DataSource with a WebService
*  ( DataSource.serviceNamespace is set), in this case, SOAP messaging is used by
*  default.
*  
*  Developers may completely bypass the SmartClient comm system by setting dataProtocol to
*  "clientCustom". In this case SmartClient will not attempt to send any data
*  to the server after calling DataSource.transformRequest. Instead the developer is
*  expected to implement transformRequest() such that it performs the necessary
*  data action outside of SmartClient, and then calls DataSource.processResponse, 
*  passing in the DSRequest.requestId and an appropriate set of DSResponse properties to
*  indicate the result of the action.
*  
*  NOTE: when OperationBinding.dataFormat is "iscServer", dataProtocol is not consulted.
*  Instead, SmartClient uses a proprietary wire format to communicate with the SmartClient
*  server, and the server-side DSRequest and DSResponse objects should be used to access
*  request data and form responses.
* @type {DSProtocol}
* @default "getParams"
*/
dataProtocol?: DSProtocol;

/**
* Format for response data for this operation.
*  
*  Typically set once for the DataSource as a whole via DataSource.dataFormat.
* @type {DSDataFormat}
* @default "iscServer"
*/
dataFormat?: DSDataFormat;

/**
* Transport to use for this operation. Defaults to DataSource.dataTransport, which in
*  turn defaults to RPCManager.defaultTransport. This would typically only be set to
*  enable "scriptInclude" transport for contacting DataSource.dataFormat web
*  services hosted on servers other than the origin server.
*  
*  When using the "scriptInclude" transport, be sure to set DataSource.callbackParam or
*  OperationBinding.callbackParam to match the name of the query parameter name expected
*  by your JSON service provider.
* @type {RPCTransport}
* @default "xmlHttpRequest"
*/
dataTransport?: RPCTransport;

/**
* Whether to use the RPCManager.sendProxied servlet to send requests
*  described by this operationBinding. If unset, automatically detects whether using the
*  HttpProxy is necessary based on the same-origin policy.
*  
*  Valid only with OperationBinding.dataProtocol settings other than ISCServer.
* @type {boolean}
* @default null
*/
useHttpProxy?: boolean;

/**
* Applies only to dataFormat: "json". Specifies the name of the query parameter that
*  tells your JSON service what function to call as part of the response for this operation.
*  
*  Typically set once for the DataSource as a whole via DataSource.callbackParam.
* @type {string}
* @default "callback"
*/
callbackParam?: string;

/**
* Additional properties to pass through to the DSRequest created for this operation.
*  Note that these will be cumulative with and will override on a per-property basis any
*  properties set via DataSource.requestProperties.
*  
*  These properties are applied before DataSource.transformRequest is called.
* @type {Partial<DSRequest>}
* @default null
*/
requestProperties?: Partial<DSRequest>;

/**
* Configures DataSource.preventHTTPCaching on a per-operationType basis.
* @type {boolean}
* @default null
*/
preventHTTPCaching?: boolean;

/**
* HTTP parameters that should be submitted with every DSRequest.
*  
*  Useful for authenticated services that require a sessionId with every request.
*  
*  Can be set for all operations of a given DataSource as DataSource.defaultParams.
* @type {object}
* @default null
*/
defaultParams?: object;

/**
* Setting useFlatFields on an operationBinding is equivalent to setting
*  DSRequest.useFlatFields on all DataSource requests with the same OperationBinding.operationType
*  as this operationBinding.
*  
*  Typical usage is to combine operationBinding.useFlatFields with 
*  DataBoundComponent.useFlatFields, with the SearchForm
*  bound to the WebService.getInputDS of the web service operation set
*  as OperationBinding.wsOperation. This allows gratuitous nesting to be consistently
*  bypassed in both the user presentation and in the actual XML messaging.
*  
*  Note that useFlatFields is not generally recommended for use with input messages
*  where multiple simple type fields exist with the same name, however if used in this way,
*  the first field to use a given name wins. "first" means the first field encountered in a
*  depth first search. "wins" means only the first field will be available in data binding,
*  and only the first field will be populated in the generated XML message.
* @type {boolean}
* @default false
*/
useFlatFields?: boolean;

/**
* A mapping from field names to SummaryFunction to be applied
*  to each field.
*  
*  Valid only for an operation of type "fetch". See the 
*  serverSummaries for examples of usage.
* @type {object}
* @default null
*/
summaryFunctions?: object;

/**
* List of fields to group by when using serverSummaries.
*  
*  Valid only for an operation of type "fetch". See the 
*  serverSummaries for details and examples of usage.
* @type {Array<Partial<string>>}
* @default null
*/
groupBy?: Array<Partial<string>>;

/**
* For an XML or JSON DataSource, XPath expression used to retrieve the objects that will
*  become DataSource records.
*  
*  For example, an "ItemSearch" web service might return a "Results" structure containing
*  metadata along with the set of Items that one might want to display in a grid. An XPath
*  expression like "/Results/Items" could be used to retrieve just the Items, which would then
*  become DataSource records.
*  
*  For a JSON web service, the recordXPath is applied to the returned JSON data
*  via XMLTools.selectObjects. Only limited XPath syntax is allowed; see 
*  XMLTools.selectObjects for details.
*  
*  For processing XML results, see OperationBinding.xmlNamespaces for information
*  on the namespaces that are available in this XPath expression. If you are contacting a WSDL
*  web service, note that OperationBinding.recordName is an alternative way to specify
*  which records should be selected by their tagName or type, and this is usually simpler.
*  
*  To learn about XPath, try the following search:
*  http://www.google.com/search?q=xpath+tutorial
* @type {XPathExpression}
* @default null
*/
recordXPath?: XPathExpression;

/**
* For an XML DataSource, tagName of the elements to be used as records.
*  
*  This is a simple alternative to OperationBinding.recordXPath when the elements to be used as
*  records all share a tagName.
*  
*  When a DataSource has a WebService, recordName can also be set to the name
*  of any complexType declared within the WebService's WSDL file.
* @type {string}
* @default null
*/
recordName?: string;

/**
* For a DataSource contacting a DataSource.serviceNamespace, setting
*  this flag means the DataSource doesn't actually attempt to contact the server but generates
*  a sample response instead, based on the XML Schema of the response message embedded in the
*  WSDL.
*  
*  The spoofed response will include all complexType elements and will fill in appropriate
*  values by type for all simpleType elements, although the spoofed data will not conform to
*  all xs:restriction declarations (eg xs:pattern).
*  
*  Note that if your WSDL does not fully describe the response format (some WSDL services just
*  have a placeholder &lt;xs:any&gt; element), SmartClient can only produce a partial
*  response. To use a hand-generated sample response, just save an XML file to disk and use
*  the OperationBinding.dataURL setting to point to it.
* @type {boolean}
* @default false
*/
spoofResponses?: boolean;

/**
* Optional object declaring namespace prefixes for use in OperationBinding.recordXPath and
*  DataSourceField.valueXPath XPath expressions.
*  
*  xmlNamespaces should be specified as a mapping from namespace prefix to
*  namespace URI, for example:
*  
*   xmlNamespaces : {
*     az : "http://webservices.amazon.com/AWSECommerceService/2005-03-23"
*   }
*  
*  By default, all namespaces declared on the document element (outermost element of the
*  response) are made available with the prefix used in the document itself.
*  
*  Then, for non-WSDL-described XML results, if there is a default namespace on the document
*  element, it is made available with the special prefix "default".
*  
*  For results of WSDL-described operations, the prefix "service" means the service namespace,
*  that is, the "targetNamespace" on the &lt;definitions&gt; element from the WSDL file. The prefix
*  "schema" means the namespace of the outermost element in the output message for the current
*  operation. "default" will be the schema namespace if there is one, otherwise the service
*  namespace.
*  
*  For basic information on XML Namespaces and their use in XPath, try the following search:
*  http://www.google.com/search?q=XPath+xml+namespaces
* @type {object}
* @default null
*/
xmlNamespaces?: object;

/**
* Optional schema describing how to extract DataSource records from the XML elements selected.
*  
*  Once a set of XML elements have been selected via recordXPath or
*  recordName, those elements are normally transformed to JavaScript objects
*  using the fields of the DataSource that owns the operationBinding. A
*  responseDataSchema can be specified instead if the XML differs in some way between
*  different DataSource operations, such that different values for
*  DataSourceField.valueXPath may be necessary to extract the same
*  DataSource record from slightly different XML structures.
* @type {DataSource}
* @default null
*/
responseDataSchema?: DataSource;

/**
* When set, causes the results of the DataSource Operation to be exported to a file, whose 
*  name and format are indicated by OperationBinding.exportFilename and 
*  OperationBinding.exportAs respectively. When no exportFilename is provided, the 
*  default is Results and the default value of exportAs is csv. Once the Operation 
*  completes, DSRequest.exportDisplay specifies whether the 
*  exported data will be downloaded to the file-system or displayed in a new window.  The 
*  default value of exportDisplay is "download" which displays the Save As dialog. See 
*  ExportDisplay for more information.
*  
*  The export field-list can also be configured, see DSRequest.exportFields.
*  
*  You can also configure the style of line-breaks to use when generating the output. See
*  LineBreakStyle for more information.
*  
*  As well as setting this and other properties on the OperationBinding, Exports can be 
*  initiated in two other ways. You can set properties on the dsRequest by passing 
*  requestProperties into DataSource.exportData. Note that this method does
*  not support exporting to JSON format (see 
*  this post for more detail).
*  Additionally, custom server code may set export-related properties on the 
*  DSResponse.
*  
*  Format Examples
*  
*  XML format
*  
*    &lt;List&gt;
*      &lt;Object&gt;
*        &lt;id&gt;10101&lt;/id&gt;
*        &lt;displayName&gt;Record 10101&lt;/displayName&gt;
*      &lt;/Object&gt;
*   &lt;/List&gt;
*  
*  JSON Format
*  
*    [
*      { id: 10101, displayName: "Record 10101" }
*    ]
*  
*  CSV Format
*  
*    id,displayName
*    10101,"Record 10101"
*  
* @type {boolean}
* @default false
*/
exportResults?: boolean;

/**
* The format in which the data should be exported. Default is "csv".
*  See ExportFormat for more information.
* @type {ExportFormat}
* @default "csv"
*/
exportAs?: ExportFormat;

/**
* The name of the file to save the exported data into.
* @type {string}
* @default null
*/
exportFilename?: string;

/**
* The style of line-breaks to use in the exported output. See LineBreakStyle for
*  more information.
* @type {string}
* @default null
*/
lineBreakStyle?: string;

/**
* The list of field-names to export. If provided, the field-list in the exported output is 
*  limited and sorted as per the list.
*  
*  If exportFields is not provided, the exported output includes all visible fields 
*  from the DataSource (field.hidden=false), sorted in the order they're defined.
* @type {Array<any>}
* @default null
*/
exportFields?: Array<any>;

/**
* If true, causes requests against this operation to automatically start or join a transaction.
*  if false, causes requests against this operation to be committed individually. If null, 
*  falls back to DataSource.autoJoinTransactions.
*  
*  See DataSource.autoJoinTransactions for further details of SmartClient's automatic
*  transaction control.
* @type {boolean}
* @default null
*/
autoJoinTransactions?: boolean;

/**
* Sets or clears the useSpringTransaction flag for this specific operation.
*  
*  See DataSource.useSpringTransaction for details of the Spring transaction integration
*  feature
* @type {boolean}
* @default null
*/
useSpringTransaction?: boolean;

/**
* This setting makes difference if DataSource.useAnsiJoins are in use.
*  If true, causes ansi joins to be included in DefaultQueryClause,
*  otherwise ansi joins will be put into separate DefaultQueryClause.
*  
*  If omitted, system-wide sql.includeAnsiJoinsInTableClause setting from
*  server.properties will be used. If it is missing as well, false
*  is the default.
* @type {boolean}
* @default null
*/
includeAnsiJoinsInTableClause?: boolean;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  SELECT clause to use when constructing the SQL query to perform this operation. The
*  property should be a comma-separated list of column names and/or expressions, and you 
*  can refer to any scalar function supported by the underlying database. The server will 
*  insert the text of this property immediately after the "SELECT" token.
*  
*  Note that if you also specify a OperationBinding.groupClause, you can use aggregate 
*  functions such as SUM and COUNT in the selectClause.
*  
*  This property is only applicable to operationBindings of 
*  OperationBinding.operationType "fetch".
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
selectClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  table clause to use when constructing the SQL query to perform this operation. The
*  property should be a comma-separated list of tables and views, and you can use any
*  special language constructs supported by the underlying database. The server will 
*  insert the text of this property immediately after the "FROM" token.
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
tableClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this
*  property can be specified on an operationBinding to provide the server with a bespoke
*  ANSI-style joins clause to use when constructing the SQL query to perform this operation.
*  The property should be a set of joins implemented with JOIN directives (as opposed to
*  additional join expressions in the where clause), joining related tables to the
*  main table or view defined in OperationBinding.tableClause.
*  The server will insert the text of this property immediately after the
*  OperationBinding.tableClause.
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
ansiJoinClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  WHERE clause to use when constructing the SQL query to perform this operation. The
*  property should be a valid expression in the syntax of the underlying database. The 
*  server will insert the text of this property immediately after the "WHERE" token.
*  
*  You may find the SmartClient-provided $criteria variable of particular use with
*  this property.
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
whereClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  GROUP BY clause to use when constructing the SQL query to perform this operation. The
*  property should be a comma-separated list of column names and/or expressions, forming
*  a valid GROUP BY clause in the syntax of the underlying database. The server will 
*  insert the text of this property immediately after the "GROUP BY" token.
*  
*  Note that specifying this property enables you to use aggregate functions (such as 
*  COUNT and SUM) in your OperationBinding.selectClause. Also note that care is required 
*  when using groupClause to ensure that the selectClause contains the fields you are grouping
*  by. Failure to do this correctly will result in a runtime SQL error.
*  
*  This property is only applicable to operationBindings of 
*  OperationBinding.operationType "fetch".
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
groupClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  ORDER BY clause to use when constructing the SQL query to perform this operation. The
*  property should be a comma-separated list of column names and/or expressions, forming
*  a valid ORDER BY clause in the syntax of the underlying database. The server will 
*  insert the text of this property immediately after the "ORDER BY" token.
*  
*  This property is only applicable to operationBindings of 
*  OperationBinding.operationType "fetch".
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
orderClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql", this 
*  property can be specified on an operationBinding to provide the server with a bespoke
*  set of values to add or update, for use when constructing the SQL query to perform this 
*  operation. The property should be one of the following, depending on the 
*  OperationBinding.operationType:
*  
*  For "add" operations, the syntax that would be valid for an INSERT INTO query: a 
*  comma-separated list of column names enclosed in parentheses, followed by a comma-separated
*  list of new values, enclosed in parentheses and preceded by the token "VALUES". For example:
*  
*  &lt;valuesClause&gt;(name, age) VALUES("Jane Doe", 48)&lt;/valuesClause&gt;
*  
*  For "update" operations, the syntax that would be valid for an UPDATE query: a 
*  comma-separated list of expressions equating a column name to its new value. For example:
*  
*  &lt;valuesClause&gt;name="Jane Doe", age=48&lt;/valuesClause&gt;
*  
*  You may find the SmartClient-provided $values variable of particular use with
*  this property.
*  
*  See the documentation for OperationBinding.customSQL for usage examples
* @type {VelocityExpression}
* @default null
*/
valuesClause?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "sql" or "hibernate", this 
*  property can be specified on an operationBinding to indicate that the server should run 
*  user-specified SQL, rather than the query it would normally generate to satisfy a 
*  dataSource operation. This property allows you to provide a fully-customized query; an
*  alternative is to provide custom "pieces" to the query generator, via properties such as
*  OperationBinding.whereClause and OperationBinding.valuesClause. See the 
*  customQuerying for more details.
*  
*  For a dataSource of type "sql", the SmartClient server generates a number of useful
*  DefaultQueryClause, 
*  and makes them available to your custom SQL code via the Velocity templating 
*  language (note that this is not available for "hibernate" dataSources).
*  
*  We also make the template variables $criteria and $values available, to give
*  you direct access to the supplied criteria, and to the new field values for update and add
*  operations. These variables are available to both "sql" and "hibernate" dataSources.
*  
*  Note that you should use this feature with care. In particular, writing customSQL code
*  that makes use of a particular database engine's features or syntax will make your 
*  application less portable. 
*  
*  See customQuerying for an overview of writing custom queries and clauses.
*  
*  Examples
*  An example using the SmartClient-supplied query pieces. This custom query will give 
*  exactly the same result as the SmartClient-generated query:
*  
*  
*  &lt;operationBinding operationId="customFetch" operationType="fetch"&gt;
*  &nbsp;&nbsp;&lt;customSQL&gt;
*  &nbsp;&nbsp;&nbsp;&nbsp;SELECT $defaultSelectClause FROM $defaultTableClause
*  WHERE $defaultWhereClause ORDER BY $defaultOrderClause
*  &nbsp;&nbsp;&lt;/customSQL&gt;
*  &lt;/operationBinding&gt;
*  
*  
*  An example using the SmartClient-supplied $criteria template variable:
*  
*  
*  &lt;operationBinding operationId="customFetch" operationType="fetch"&gt;
*  &nbsp;&nbsp;&lt;customSQL&gt;
*  &nbsp;&nbsp;&nbsp;&nbsp;SELECT foo, bar, baz FROM MyTable WHERE bar &gt; $criteria.someValue
*  &nbsp;&nbsp;&lt;/customSQL&gt;
*  &lt;/operationBinding&gt;
*  
*  
*  An update example:
*  
*  
*  &lt;operationBinding operationId="myUpdateOp" operationType="update"&gt;
*  &nbsp;&nbsp;&lt;customSQL&gt;
*  &nbsp;&nbsp;&nbsp;&nbsp;UPDATE $defaultTableClause SET $defaultValuesClause WHERE bar 
*  &lt;= $criteria.someValue
*  &nbsp;&nbsp;&lt;/customSQL&gt;
*  &lt;/operationBinding&gt;
*  
* @type {VelocityExpression}
* @default null
*/
customSQL?: VelocityExpression;

/**
* This feature is available with Power or better licenses only. See 
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "hibernate", this 
*  property can be specified on an operationBinding to indicate that the server should run 
*  user-specified HQL (Hibernate Query Language), rather than the Hibernate criteria query or 
*  saveOrUpdate call it would normally generate to satisfy a dataSource operation
*  via Hibernate. 
*  
*  Note that inserting new records via HQL is often impractical, due to intentional restrictions
*  in the language (it is only possible to perform an insert expressed in terms of a SELECT; the
*  "VALUES" construct commonly used when inserting new rows singly is not supported). If you are 
*  intending to use customHQL, we recommend that you avoid doing so for OperationBindings 
*  with OperationBinding.operationType "add", unless you have a special 
*  requirement such as a bulk insert; if you need custom queries to perform inserts 
*  on "hibernate" dataSources, we recommend you use OperationBinding.customSQL,
*  which is valid for "hibernate" DataSources as well as "sql" dataSources.
*  
*  For other operations on "hibernate" dataSources, however, HQL has the advantage of 
*  being more portable across different database engines than is plain SQL.
*  
*  Note that using customHQL affects paging implementation. If you use it, full data set is fetched 
*  from Hibernate and records that aren't in the requested range are dropped at the server side.
* @type {string}
* @default null
*/
customHQL?: string;

/**
* This feature is available with Power or better licenses only. See
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "jpa", this
*  property can be specified on an operationBinding to indicate that the server should run
*  user-specified JQL (Java Persistence Query Language).
*  
*  Note that inserting new records via JQL is often impractical, due to intentional restrictions
*  in the language (it is only possible to perform an insert expressed in terms of a SELECT; the
*  "VALUES" construct commonly used when inserting new rows singly is not supported). If you are
*  intending to use customJQL, we recommend that you avoid doing so for OperationBindings
*  with OperationBinding.operationType "add", unless you have a special
*  requirement such as a bulk insert; if you need custom queries to perform inserts
*  on "jpa" dataSources, we recommend you use OperationBinding.customSQL,
*  which is valid for "jpa" DataSources as well as "sql" dataSources.
*  
*  For other operations on "jpa" dataSources, however, JQL has the advantage of
*  being more portable across different database engines than is plain SQL.
*  
*  Note that using customJQL affects paging implementation. If you use it, full data set is fetched
*  from JPA and records that aren't in the requested range are dropped at the server side.
* @type {string}
* @default null
*/
customJQL?: string;

/**
* This feature is available with Power or better licenses only. See
*  smartclient.com/product for details.
*  
*  For a dataSource of DataSource.serverType "jpa" or "hibernate", this
*  property can be specified on an operationBinding to indicate that the server should execute
*  a named query which has already been defined on an entity.
*  
*   &#64;Entity
*   &#64;Table (name="Countries")
*   &#64;NamedQuery(name = "Country.withPopulationLessThan", query = "SELECT country FROM Country country WHERE country.population &lt; :population")
*   public class Country { ... }
*  
* 
*  
*   &lt;operationBindings&gt;
*     &lt;operationBinding operationType="custom" operationId="withPopulationLessThan" namedQuery="Country.withPopulationLessThan"/&gt;
*   &lt;/operationBindings&gt;
*  
*  Substitution values can be used in order to build more dynamic named queries. When calling
*  DataSource.performCustomOperation the values are passed in using the data argument.
*  
*  Note that value substitution for named queries is slightly different to other custom queries.
*  Because of the way the persistence API works the JQL query written in the &#64;NamedQuery annotation
*  can only contain basic parameter names such as "population". Therefore the value substitution
*  becomes a simple name based mapping.
*  
*  Examples
*  Using Simple Criteria
*  
*  An example using a simple criteria for the above defined Country entity. In this case the named
*  query parameter ":population" will be swapped out for the value of the criteria objects "population"
*  field.
*  
*   var criteria = {
*     population: 596000
*   };
* 
*   countryDataSource.performCustomOperation("withPopulationLessThan", criteria);
*  
*  
*  Using Advanced Criteria
*  
*  If an advanced criteria is detected, access to all "fieldName" variables and their values will
*  be provided but still using simple name based mapping. In the below case only the deep-first
*  occurrence of the "population" fieldName will available. The operator is effectively ignored.
*  
*   var criteria = {
*     _constructor: "AdvancedCriteria",
*     operator:"or",
*     criteria:[
*       {
*         fieldName:"population",
*         operator:"lessThan",
*         value: 12000
*       },
*       {
*         fieldName:"name",
*         operator:"equals",
*         value: "Sweden"
*       },
*       {
*         _constructor: "AdvancedCriteria",
*         operator:"and",
*         criteria:[
*           {
*             fieldName:"population",
*             operator:"lessThan",
*             value: 0
*           }
*         ]
*       }
*     ]
*   };
* 
*   countryDataSource.performCustomOperation("withPopulationLessThan", criteria);
*  
*  
*  Note
*  Using namedQuery affects paging implementation. If you use it, full data set is fetched
*  from JPA and records that aren't in the requested range are dropped at the server side.
* @type {string}
* @default null
*/
namedQuery?: string;

/**
* A SQLDataSource will normally issue two queries for a "fetch" operation when paging is
*  enabled: one to determine the total rows available (the "row count query"), and one to fetch
*  the specific range of rows requested.
*  
*  Setting skipRowCount="true" will avoid the "row count query", but as a consequence
*  DSResponse.totalRows will be set to match the requested DSRequest.endRow since
*  the totalRows is unknown. You can avoid this by using a 
*  OperationBinding.sqlPaging of "jdbcScroll" or "dropAtServer", but be 
*  aware that these paging strategies can introduce significant delays when used with 
*  potentially large datasets (in fact, "dropAtServer" is almost guaranteed to do so if used
*  with datasets of more than 1000 or so rows)
*  
*  As an alternative, consider enabling 
*  DataSource.progressiveLoading, which avoids doing a query for row
*  counts, but will still allow the user to load more results using the scrollbar if viewing
*  results in a ListGrid.
* @type {boolean}
* @default null
*/
skipRowCount?: boolean;

/**
* Indicates that the listed fields should be included in the default
*  OperationBinding.selectClause and OperationBinding.selectClause generated for this 
*  operationBinding, even if they are marked DataSourceField.customSQL="true".
*  
*  If you need to apply different sets of overrides for the selectClause and the
*  whereClause, use OperationBinding.customValueFields and/or 
*  OperationBinding.customCriteriaFields instead. If you specify both
*  customFields and customCriteriaFields or 
*  customValueFields, the more specific variant wins. If you specify both 
*  customFields and OperationBinding.excludeCriteriaFields, 
*  customFields wins (this is another use case when you may wish to use 
*  customValueFields instead)
*  
*  You can specify this property as a comma-separated list (eg, "foo, bar, baz") or by just
*  repeating the &lt;customFields&gt; tag multiple times with one field each.
*  
*  This property is only applicable to DataSources of DataSource.serverType.
* @type {string | Array<any>}
* @default null
*/
customFields?: string | Array<any>;

/**
* Indicates that the listed fields should be included in the default
*  OperationBinding.selectClause and OperationBinding.valuesClause generated for this
*  operationBinding, even if they are marked DataSourceField.customSQL="true".
*  
*  You can specify this property as a comma-separated list (eg, "foo, bar, baz") or by just
*  repeating the &lt;customValueFields&gt; tag multiple times with one field each.
*  
*  This property is only applicable to DataSources of DataSource.serverType.
* @type {string | Array<any>}
* @default null
*/
customValueFields?: string | Array<any>;

/**
* Indicates that the listed fields should be included in the default
*  OperationBinding.whereClause generated for this operationBinding, even if they are marked 
*  DataSourceField.customSQL.
*  
*  You can specify this property as a comma-separated list (eg, "foo, bar, baz") or by just
*  repeating the &lt;customCriteriaFields&gt; tag multiple times with one field each.
*  
*  This property is only applicable to DataSources of DataSource.serverType.
* @type {string | Array<any>}
* @default null
*/
customCriteriaFields?: string | Array<any>;

/**
* Indicates that the listed fields should be excluded from the default
*  OperationBinding.whereClause generated for this operationBinding. 
*  
*  This enables you to use these fields in a customQuerying while
*  still allowing the $defaultWhereClause to be generated for all other fields. For example,
*  you might take a particular field and apply it in the WHERE clause of a subquery.
*  
*  You can specify this property as a comma-separated list (eg, "foo, bar, baz") or by just
*  repeating the &lt;customCriteriaFields&gt; tag multiple times with one field each.
*  Note that if a field is included in both excludeCriteriaFields and
*  OperationBinding.customCriteriaFields, customCriteriaFields
*  wins.
*  
*  This property is only applicable to DataSources of DataSource.serverType.
* @type {string | Array<any>}
* @default null
*/
excludeCriteriaFields?: string | Array<any>;

/**
* For an operationBinding of OperationBinding.operationType "fetch" which specifies no
*  OperationBinding.operationId, this property determines whether the operationBinding 
*  should be used for cache synchronization purposes (ie, to retrieve the record most recently
*  added or changed). This property has no effect on an operationBinding that specifies an 
*  operationId - see OperationBinding.cacheSyncOperation.
*  
*  In order to work correctly with SmartClient's cache synchronization system, an 
*  operationBinding marked useForCacheSync should have the following properties:
*  
*  
* - Able to complete its retrieval using no context other than the values of the primary key
*  fields declared in the dataSource (these will be provided in the $criteria object passed to 
*  the operation)
*  
* - Returns the entire record, including any values that may require joins to other tables
*  or other complexities
*  
*  This property is only applicable to DataSources of type "sql".
* @type {boolean}
* @default true
*/
useForCacheSync?: boolean;

/**
* For an operationBinding of OperationBinding.operationType "add" or "update", this 
*  property is the OperationBinding.operationId of a "fetch" operationBinding to use 
*  for cache synchronization purposes (ie, to fetch the row most recently added or changed).
*  This property, along with OperationBinding.useForCacheSync and 
*  OperationBinding.canSyncCache is provided so that you can use custom database operations
*  without sacrificing the benefits of SmartClient's automatic cache synchronization.
*  
*  This property is only applicable to DataSources of type "sql".
* @type {string}
* @default null
*/
cacheSyncOperation?: string;

/**
* For an operation of type "add" or "update", a SQLDataSource will normally obtain
*  data to return to the client by performing the "cacheSyncOperation": a SELECT statement that
*  retrieves the modified record by primary key. This accommodates sequence columns, columns
*  with default values, database triggers and other database features that may modify data
*  after insertion or update.
*  
*  Certain major SQL customizations can prevent the SQLDataSource from authoritatively
*  determining the primary key used in the SQL statement, such that re-selecting the saved
*  record may fail. By default, when canSyncCache has not been explicitly
*  set, in the following cases it is assumed that the normal cacheSyncOperation cannot be used:
*  
*  
* - &lt;customSQL&gt; has been used to define an entirely custom query
*  
* - a custom &lt;whereClause&gt; has been defined for an "update" or "remove" operation
*  
* - a custom &lt;valuesClause&gt; has been defined for an "add" operation
*  
*  If any of these cases apply or if canSyncCache has been set false, the server
*  will skip the cacheSyncOperation and return a DSResponse where
*  DSResponse.invalidateCache has been set to true to notify client-side components that
*  they may need to refresh their entire cache.
*  
*  Alternatively, if the default re-selection behavior will not work but a customized SQL
*  query would work, you can define that SQL operation as another operationBinding and use
*  OperationBinding.cacheSyncOperation to declare that it should be used. Setting
*  cacheSyncOperation implicitly sets canCacheSync to true.
*  
*  This property is only applicable to DataSources of type "sql".
* @type {boolean}
* @default null
*/
canSyncCache?: boolean;

/**
* For an operation of type "add" or "update", a SQLDataSource will normally obtain data to
*  return to the client by performing the OperationBinding.cacheSyncOperation. In certain edge cases
*  this is either not possible or causes problems (such as record locks); in this case, you 
*  mark the operation with OperationBinding.canSyncCache: false to inhibit this automatic behavior. 
*  However, this gives you an operation that does not properly cooperate with the client in 
*  keeping client-side caches fresh, which in turn leads to an application that must perform
*  more server turnarounds and explicit database fetches. 
*  
*  It is possible that the data needed for cache synchronization could be obtained by 
*  application code running in a DMI or serverScript, 
*  even if the normal automatic cache synchronization will not work. However, such application
*  code is almost certainly going to need to know the primary key(s) of any newly added 
*  record(s).
*  
*  When makeKeysAvailable is true, SmartClient Server will go through the process
*  of obtaining generated keys in accordance with the SequenceMode, even if 
*  canSyncCache is false (note, "in accordance with the sequenceMode"
*  implies that we do not attempt to get keys if the sequenceMode is "none").
*  The keys used in the operation (both generated keys, if any, and any keys provided in the 
*  operation's criteria or values) will be stored on the DSResponse and your 
*  server-side DMI or script will have access to them via the server API 
*  dsResponse.getKeys(). Please see the server-side documentation for that 
*  method for further details.
*  
*  Note, if you are using SequenceMode "jdbcDriver", the keys are available to SmartClient
*  Server at no cost, so makeKeysAvailable is always true. If you are using
*  sequenceMode "native", a separate native query must be run to obtain the keys. 
*  The overhead of this native query is likely to be insignificant, but because it is an extra
*  step that you may not want or need, makeKeysAvailable defaults to false in
*  this case. You can override this by adding setting 
*  sql.always.makeKeysAvailable:true to your server.properties file. 
*  
*  This property is only applicable to DataSources of type "sql".
* @type {boolean}
* @default "See below"
*/
makeKeysAvailable?: boolean;

/**
* For dataSources of DataSource.serverType "sql" and "hibernate" only, this 
*  property determines whether "custom" operations have their custom SQL or HQL sent to the 
*  underlying database via a JDBC executeQuery() or a JDBC 
*  executeUpdate(). The default value of null means the same as "query", so you
*  only need to use this property when your custom SQL or HQL updates data.
* @type {SQLType}
* @default null
*/
sqlType?: SQLType;

/**
* Ordinarily, "update" and "remove" operations are only allowed for DataSources
*  that have a DataSourceField.primaryKey, and all primary key values
*  are present in the request. This is because an update of a DataSource with no primary key, 
*  or an update request that has missing primary key values, cannot be guaranteed to affect 
*  only one record.
*  
*  Setting this property on an operationBinding circumvents this restriction for that operation
*  only.
*  
*  Warning: Be aware that this is a potentially dangerous setting and should be used
*  with care. With this flag set, you have no guarantee that an update will not change or 
*  remove every row in a table.
*  
*  Note, in the case of doing an update or delete operation with a primary key 
*  and additional criteria, allowMultiUpdate must be set or additional criteria 
*  will be dropped and just the primary key fields will be used in criteria.
*  
*  Also, running allowMultiUpdate operations directly from the client is not 
*  straightforward because it requires the ability to specify criteria and values separately
*  in the request, which is not currently supported. This can be worked around in various
*  ways, but really allowMultiUpdate is primarily intended for server-side 
*  operations. Therefore, the recommended pattern is to use a 
*  DataSource.performCustomOperation from the client to invoke a DMI on
*  the server which performs the multi-update operation via a second, server-side DSRequest. 
*  
*  In any case, it's normally a good idea to set OperationBinding.requiredCriterion 
*  on the multi-update operation to ensure that the alternative criteria is present as expected.
* @type {boolean}
* @default null
*/
allowMultiUpdate?: boolean;

/**
* Ordinarily, "update" and "remove" operations are only allowed if all primary key values
*  are present in the request. This is because an update request that has missing primary 
*  key values cannot be guaranteed to affect only one record.
*  
*  Setting this property on an operationBinding circumvents this restriction for that operation
*  only. Note, this property differs from OperationBinding.allowMultiUpdate in its 
*  intent: allowMultiUpdate tells the framework that this operation deliberately 
*  affects multiple records; providesMissingKeys tells the framework that this 
*  operation will only affect one record, and will ensure this by providing values for missing
*  keys during its operation. Unlike allowMultiUpdate, setting this flag does not
*  cause component caches to be ListGrid.invalidateCache
*  
*  Providing values for missing keys can be done in various ways:
*  
* - Operations that specify &lt;OperationBinding.customSQL&gt; or 
*    &lt;OperationBinding.whereClause&gt; can provide missing
*    key values from session storage or elsewhere in the provided record
*  
* - Operations that specify &lt;serverScript&gt; can provide
*    arbitrary code to manipulate the record in whatever way they like before executing the 
*    underlying built-in functionality
*  
* - Operations can specify &lt;OperationBinding.criteria&gt; 
*    to provide missing keys
*  
* - A request can contain DSRequest.fieldValueExpressions, 
*    which can be used to provide values for missing keys
*  
*  Note, you can also use a regular dmiOverview to arbitrarily manipulate the
*  record sent from the client, including providing values for any missing keys. If you do 
*  this, you do not need to specify providesMissingKeys because the request is 
*  not validated for the presence of key values until after the DMI has run.
*  
*  Warning: Be aware that this is a potentially dangerous setting and should be used
*  with care. With this flag set, the framework cannot guarantee that an update will not 
*  change or remove every row in a table: it becomes your code's responsibility to ensure 
*  that all PK values are provided to the operation by the time it actually needs them.
* @type {boolean}
* @default null
*/
providesMissingKeys?: boolean;

/**
* Specifies, for this specific operationBinding, whether to qualify column names with table
*  names in any SQL we generate. Overrides the DataSource.qualifyColumnNames property.
*  Only applicable to dataSources of DataSource.serverType "sql".
* @type {boolean}
* @default true
*/
qualifyColumnNames?: boolean;

/**
* If set, every invocation of this operationBinding will invalidate the local cache, forcing
*  a server visit to refresh the data.
* @type {boolean}
* @default null
*/
invalidateCache?: boolean;

/**
* Specifies, for this operationBinding only, the list of field names that should be returned
*  to the client. Typically this will be a subset of the DataSource.fields, but note 
*  that this is not a requirement; outputs can include fields that are not defined
*  in the DataSource's field list. In this case, the server will return extra fields even if 
*  DataSource.dropExtraFields is true.
*  
*  You specify this property as a string containing a comma-separated list of field names 
*  (eg, "foo, bar, baz")
* @type {string}
* @default null
*/
outputs?: string;

/**
* The paging strategy to use for this specific OperationBinding. If this property is not
*  set, we fall back to the DataSource.sqlPaging value, and the defaults described in
*  the documentation for that property.
* @type {SQLPagingStrategy}
* @default null
*/
sqlPaging?: SQLPagingStrategy;

/**
* If explicitly set true or false, forces the use of a "hint" in the SQL we generate for paged 
*  queries on or off as appropriate. If not set, defaults to the 
*  DataSource.sqlUsePagingHint value. Note this property is only applicable to 
*  DataSource.serverType DataSources, only when a 
*  DataSource.sqlPaging of "sqlLimit" is in force, and it only has an 
*  effect for those specific database products where we employ a native hint in the generated 
*  SQL in an attempt to improve performance.
* @type {boolean}
* @default null
*/
sqlUsePagingHint?: boolean;

/**
* Whether to use the subselect technique (see DataSource.useSubselectForRowCount for
*  details) to derive a rowcount query for this operation. If this property is not
*  set, we fall back to the useSubselectForRowCount setting on the DataSource, 
*  and the defaults described in the documentation for that property.
* @type {boolean}
* @default null
*/
useSubselectForRowCount?: boolean;

/**
* Sets DataSource.progressiveLoading for this particular 
*  operation, overriding the DataSource-level setting. Note that this setting applies only
*  to fetch operations - it has no effect if specified on any other kind of operation.
* @type {boolean}
* @default null
*/
progressiveLoading?: boolean;

/**
* This property indicates whether this operation supports AdvancedCriteria. This setting 
*  overrides DataSource.allowAdvancedCriteria for this operation only. See
*  DataSource.supportsAdvancedCriteria for further information.
*  
*  NOTE: If you specify this property in a DataSource descriptor 
*  (.ds.xml file), it is enforced on the server. This means that if you run 
*  a request containing AdvancedCriteria against an OperationBinding that advertises itself
*  as allowAdvancedCriteria:false, it will be rejected.
* @type {boolean}
* @default null
*/
allowAdvancedCriteria?: boolean;

/**
* Elements of this feature are only available with Power or better licenses. See 
*  smartclient.com/product for details.
*  
*  A list of DSRequestModifiers that will be used to modify the criteria of each 
*  DSRequest that uses this operationBinding. Note that the criteria elements are 
*  applied to DSRequest criteria as follows:
*  
*  
* - Simple criteria: The field and value are just applied as an extra key/value 
*  pair in the criteria map, as long as the DSRequestModifier.operator 
*  attribute is left unset, or is set to "equals". For any other setting of 
*  operator, the criteria is first converted to the equivalent AdvancedCriteria 
*  and then processed as described below
*  
* - AdvancedCriteria: If the topmost operator is "and", we add the new criterion 
*  as an additional criterion directly in the existing list. Otherwise, we create a new 
*  top-level AdvancedCriteria with an operator of "and". This is then set to have two 
*  elements in its criteria: the previous top-level criteria and the new criterion.
*  
*  The effect of this is to apply any criteria specifed here as additional constraints on top
*  of what the user has specified, and of course, the user is unable to affect this. Thus, 
*  this is a suitable and convenient place to enforce rules such as "Users can only ever see
*  their own records". 
*  Below is an example of the XML as it should be defined in your ds.xml, datasource 
*  definitions. 
*  &lt;operationBindings&gt;
*   &lt;operationBinding operationType="fetch" operationId="..."&gt;
*    &lt;criteria fieldName="USER_ROLE" value="ADMIN" operator="equals" /&gt;
*   &lt;/operationBinding&gt;
*  &lt;/operationBindings&gt;
*  
* @type {Array<Partial<DSRequestModifier>>}
* @default null
*/
criteria?: Array<Partial<DSRequestModifier>>;

/**
* Elements of this feature are only available with Power or better licenses. See 
*  smartclient.com/product for details.
*  
*  A list of DSRequestModifiers that will be used to modify the values object of each 
*  DSRequest that uses this operationBinding. See this example:
*  ${isc.DocUtils.linkForExampleId('queuedAdd')}. 
*  Below example of the xml as it should be defined in ds.xml: 
*  &lt;operationBinding operationType="add"&gt;
*   &lt;values fieldName="orderID" value="$responseData.last('queuedAdd_order','add').orderID" /&gt; 
*  &lt;/operationBinding&gt;
* @type {Array<Partial<DSRequestModifier>>}
* @default null
*/
values?: Array<Partial<DSRequestModifier>>;

/**
* Whether a user must be authenticated in order to access this operation. For details of 
*  what is meant by "authenticated", see DataSource.requiresAuthentication.
*  
*  To protect access to an entire operationType (eg, all "fetch" operations), declare an
*  operationBinding with requiresAuthentication="true", OperationBinding.operationType set
*  to the operationType to be protected, but no OperationBinding.operationId. This will then
*  prevent access to the "fetch" operationType unless another OperationBinding
*  declares requiresAuthentication="false" with a specific
*  OperationBinding.operationId.
* @type {boolean}
* @default null
*/
requiresAuthentication?: boolean;

/**
* Comma-separated list of user roles that are allowed to invoke the operation described by
*  this operationBinding. If the current user has any of the roles listed, they can
*  invoke the operation. Also note that authentication.superuserRole can be 
*  specified in the server_properties file. If set this denotes a "super user" 
*  role - any user with that role will have access to all operations, regardless of the
*  "requiresRole" settings for the operation.
*  
*  Whether the current user has a given role is determined by calling the standard Java
*  servlets method httpServletRequest.isUserInRole(), hence works with both simple
*  J2EE security (realms and form-based authentication) and JAAS (Java Authentication &
*  Authorization Service).
*  
*  If you wish to use a role-based security scheme that does not make use of the servlet API's 
*  standards, SmartClient Server also implements the setAuthenticated and
*  setUserRoles methods on RPCManager. 
*  You can use this API to tell SmartClient that all the requests in the queue currently 
*  being processed are associated with a user who has the roles you supply; in this case,
*  SmartClient will not attempt to resolve the user's roles via 
*  httpServletRequest.isUserInRole(). When taking this approach, the 
*  rpcManager.setUserRoles() method should be called on the server for 
*  each transaction received from the client. We recommend doing this by overriding the 
*  special IDACall servlet and checking server side state to determine the current user's 
*  roles, calling the API, and then calling handleDSRequest() or
*  handleRPCRequest() directly to handle the request(s) passed in.
*  Here's an example of this approach which assumes the current user's roles has been
*  set directly on the HttpSession object as a comma-separated-string attribute "roles":
*  
*  public class SecureIDACall extends IDACall {
*  
*    public void processRequest(HttpServletRequest request,
*        HttpServletResponse response)
*     throws ServletException, IOException
*    {
*      HttpSession session = request.getSession();
*      Object roles = session == null ? null : session.getAttribute("roles");
*  
*      if (roles != null) {
*        try {
*          RequestContext context = RequestContext.instance(this, request, response);  
*          RPCManager rpc = new RPCManager(request, response);
*          rpc.setAuthenticated(true);
*          rpc.setUserRoles((String) roles); 
*          
*          // call processRPCTransaction() to iterate through all RPCRequests and
*          // DSRequests and execute them
*          processRPCTransaction(rpc, context);
*  
*        } catch (Throwable e) {
*          handleError(response, e);
*        }
*      } else {
*        super.processRequest(request, response);
*      } 
*    }
*  }
*  
*  
*  If there is an operationBinding declared for a given operationType which does not have an
*  OperationBinding.operationId, that is, it is the default operationBinding for the type, then any other
*  operationBinding of the same type is assumed to have the same setting for
*  requiresRole as the default operationBinding for the operationType. For
*  example, given these declarations:
*  
*    &lt;operationBinding operationType="fetch" requiresRole="manager"&gt;
*       ... settings ...
*    &lt;/operationBinding&gt;
*    &lt;operationBinding operationType="fetch" operationId="fetchWithExtraFields"&gt;
*       ... settings ...
*    &lt;/operationBinding&gt;
*  
*  The second operationBinding requires the "manager" role even though there is no explicit
*  requiresRole declaration. To prevent the "manager" role being required by the
*  second operationBinding, add requireRole="".
*  
*  Note that if DataSource.requiresRole is set, all operations on the DataSource require
*  the roles set for the DataSource as a whole, even if they declare individual
*  requiresRole attributes.
*  
*  This property is valid only for a server-side DataSource when using the SmartClient Server.
*  
*  Special rules for cache sync
*  
*  After successfull "add" or "update" operation cache sync request is performed, which is using 
*  "fetch" operation of the same datasource. It may happen that user is allowed to add records, but 
*  is not allowed to fetch them, for example:
*  
*    &lt;operationBinding operationType="fetch" requiresRole="admin"&gt;
*       ... settings ...
*    &lt;/operationBinding&gt;
*    &lt;operationBinding operationType="add"&gt;
*       ... settings ...
*    &lt;/operationBinding&gt;
*  
*  User without "admin" role will be able to successfully add record, but the cache sync operation 
*  will fail due to security violation. In this case the record will be saved to database, but the 
*  added record will not be fetched from database, instead just DSRequest.oldValues 
*  overlaid with submitted values will be returned. So, any changes made to the new record during 
*  request execution, including generated values for primary key fields of "sequence" type, will not 
*  be returned to the client.
*  
*  However, if "add" or "update" operation explicitly declares OperationBinding.cacheSyncOperation,
*  cache sync request will be executed even if the user does not meet the security checks for the 
*  operationBinding. Note that field-level security still will be respected and disallowed fields will 
*  be excluded from returned data.
* @type {string}
* @default null
*/
requiresRole?: string;

/**
* Indicates that the specified VelocityExpression must be true for a user to access
*  this operationBinding.
*  
*  As with OperationBinding.requiresRole, if there an operationBinding that is the
*  default operationBinding for the operationType, its requires expression is
*  assumed to apply to all other operationBindings of the same type unless they explicitly set
*  requires=""
*  
*  DataSource.requires, if specified, applies before
*  operationBinding.requires is evaluated. In this case, both requires
*  expressions must be true for the request to be accepted.
* @type {VelocityExpression}
* @default null
*/
requires?: VelocityExpression;

/**
* Indicates that field-level declarative security rules are waived for rows that were 
*  created by the current user, as described in the discussion of 
*  DataSource.creatorOverrides. This setting overrides 
*  dataSource.creatorOverrides, for this operation only.
* @type {boolean}
* @default null
*/
creatorOverrides?: boolean;

/**
* Requires that the currently authenticated user match the contents of this
*  field, for client-initiated requests (i.e., where
*  DSRequest.isClientRequest() returns true on the server).
* 
*  When a new row is added by a client-initiated DSRequest, the
*  ownerIdField will be automatically populated with the currently
*  authenticated user (clobbering any value supplied by the client).
*  Client-initiated attempts to update the ownerIdField will also be prevented.
* 
*  If you wish to set the ownerIdField to a different value via an "add" or
*  "update" operation, you can do so in server-side DMI code (possibly
*  consulting DSRequest.getClientSuppliedValues() to get the value
*  that was clobbered).
* 
*  For client-initiated "fetch", "update" or "remove" operations, the server
*  will modify client-supplied criteria so that only rows whose ownerIdField
*  matches the currently authenticated user can be read, updated or deleted. 
*  
*  Overrides the same setting at the DataSource.ownerIdField level.
* 
*  If ownerIdField is specified,
*  DataSource.requiresAuthentication will default
*  to true. If requiresAuthentication is explicitly
*  set to false, then unauthenticated users will be able to see
*  all records. To avoid this, you can use
*  DataSource.guestUserId to specify a default user to apply
*  when no one has authenticated.
* @type {string}
* @default null
*/
ownerIdField?: string;

/**
* Value to use for the OperationBinding.ownerIdField if no one 
*  has authenticated.
* 
*  Overrides the same setting at the DataSource.guestUserId level.
* @type {string}
* @default null
*/
guestUserId?: string;

/**
* Definition of an email message that will be sent as an after-effect of selecting or updating
*  data.
*  
*  Note that if a fixed number of different messages need to be sent, multiple 
*  &lt;mail&gt; tags may be specified. For example, one mail could be sent to an 
*  admin address, and a different message to every member of a user group.
* @type {Mail}
* @default null
*/
mail?: Mail;










}

/**
* Show a currently hidden component.
*/
class ShowComponentTask extends ComponentTask {
/**
* Creates a new ShowComponentTask
* 
* @param typeCheckedProperties {Partial<ShowComponentTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ShowComponentTask} The new instance
*/
static create(typeCheckedProperties? : Partial<ShowComponentTask>, uncheckedProperties?: Object): ShowComponentTask;











}



/**
* HTML elements that make up a complete FormItem (note, not all FormItems use all of 
*  these elements)
*  
*/
type FormItemElementType = 
/**
* The form item as a whole, including the text element, any icons,           and any hint text for the item. This is the cell containing the 
*            form item
*/
"cell" |

/**
* The "control" cell containing the text box and picker
*/
"control" |

/**
* The cell containing the item's picker icon, if it has one
*/
"pickerIcon" |

/**
* The item's native text box, if it has one
*/
"textBox" |

/**
* The cell containing the title
*/
"title";


/**
* The CubeGrid is an interactive grid component that presents very large, multi-dimensional
*  data sets (also known as data cubes) for reporting or analytic applications.
*  
*  CubeGrids are often called crosstabs, for their cross-tabular display of data dimensions in
*  stacked/nested rows and columns, or pivot tables, for their ability to "pivot" dimensions
*  between rows and columns to view a data cube from different perspectives. They are typically
*  used in the querying and reporting front-ends of data warehousing, decision support, OLAP,
*  and business intelligence systems.
*  
*  For example, CubeGrids can be connected to Pentaho Mondrian, Jasper Reports, Microsoft
*  Analysis Services and any other OLAP technology that supports the XMLA standard - the
*  Isomorphic public wiki has 
*  examples.
*  of such integration.
*  
*  NOTE: you must load the Analytics 
*  loadingOptionalModules before you can use CubeGrid.
*  
*  Multi-Dimensional Data Terminology
*  
*  The CubeGrid refers to the dimensions of a data cube as facets, to the possible values in
*  each facet as facet values, and to the values within the data cube as data values or cell
*  values. Equivalent terms that are commonly used in data warehousing or business intelligence
*  systems include:
*  facet: dimension, attribute, feature
*  facet value: dimension member, attribute value, feature value
*  cell value: data value, metric value, measure
*  
*  Visual Structure
*  
*  Like the ListGrid and TreeGrid components, the CubeGrid displays data values in a tabular
*  "body" with adjacent "headers". While the ListGrid and TreeGrid display rows of records
*  with field values, the CubeGrid displays a body of individual cell values, each associated
*  with a combination of facet values. The facet values for a cell are displayed in the column
*  headers above the cell and row headers to the left of the cell. CubeGrids can display an
*  arbitrary number of facets, by stacking multiple levels of row and/or column headers.
*  
*  Except for the innermost column facet, each facet in a CubeGrid has a facet label adjacent
*  to its row or column headers. The facet labels serve two main purposes: they display the
*  titles of the facets, and they provide drag-and-drop reordering or pivoting of facets within
*  the CubeGrid. The row facet labels also provide interactive selection, resizing, and other
*  operations on the columns of row facet values.
*  
*  The innermost column headers provide special behaviors and controls for manipulating the
*  columns of data in a CubeGrid. End users may select, resize, reorder, minimize, maximize, or
*  auto-fit the columns of data via mouse interactions with these headers. Customizable
*  indicators and controls may be included at the top of each innermost column header.
*  
*  If a CubeGrid is not large enough to display all of its cell values, horizontal and/or
*  vertical scrollbars will appear below and to the right of the body. The body of the CubeGrid
*  may be scrolled on either axis. The headers are "frozen" from scrolling on one axis - row
*  headers only scroll vertically, while column headers only scroll horizontally - so the facet
*  values for the visible cells are always displayed.
*  
*  Data Loading
*  
*  Data can be provided to the Cube via CubeGrid.data as an Array of
*  CellRecord, each representing the data for one cell.
*  
*  For large datasets, CubeGrid.dataSource with one field per
*  facetId, and the CubeGrid will load data on demand to fill the visible area, including
*  lazily loading data for expanding/collapsing tree facets and when facetValues are made
*  visible programmatically or via menus.
*  
*  Picking Facets
*  
*  By "facet" we mean an aspect of the data which is orthogonal to other aspects of the data,
*  that is, combining values from any two "facets" should make sense.
*  
*  For example, in sales data, two facets might be "quarter" and "region" - it makes sense to
*  combine any quarter and region, although for some combinations, there may not be data
*  available.
*  
*  An example of two aspects that would not be independent facets are "state" and "city"
*  - it's senseless to combine arbitrary states with arbitrary cities - most combinations are
*  invalid. Consider instead a Facet.isTree that combines "city" and "state"
*  values. 
*  
*  Note that if "city" and "state" are represented as facets, they may look correct if they are
*  both on the same axis of the grid and
*  CubeGrid.hideEmptyFacetValues is used to trim nonsense
*  combinations, but if the data is CubeGrid.canMoveFacets such that "state" and
*  "city" are on opposing axes, there will be a roughly diagonal "stripe" of data for
*  combinations of "state" and "city" that make sense, and all other space will be blank. This
*  is a strong indication that two facets should be represented as a single tree facet instead.
*/
class CubeGrid extends ListGrid {
/**
* Creates a new CubeGrid
* 
* @param typeCheckedProperties {Partial<CubeGrid>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CubeGrid} The new instance
*/
static create(typeCheckedProperties? : Partial<CubeGrid>, uncheckedProperties?: Object): CubeGrid;
/**
* A declared value of the enum type 
*  FacetIndentDirection.
* @type {Constant}
* @default "descending"
*/
static DESCENDING?: Constant;

/**
* A declared value of the enum type 
*  FacetIndentDirection.
* @type {Constant}
* @default "ascending"
*/
static ASCENDING?: Constant;



/**
* CSS class for the CubeGrid as a whole
* @type {CSSStyleName}
* @default "normal"
*/
styleName?: CSSStyleName;

/**
* CSS class for the CubeGrid body
* @type {CSSStyleName}
* @default "cubeGridBody"
*/
bodyStyleName?: CSSStyleName;

/**
* GridRenderer.baseStyle for this listGrid.
*  If this property is unset, base style may be derived from ListGrid.normalBaseStyle
*  or ListGrid.tallBaseStyle as described in
*  ListGrid.getBaseStyle.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined
*  with the base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "cubeCell"
*/
baseStyle?: CSSStyleName;

/**
* Whether alternating rows should be drawn in alternating styles, in order to create a "ledger"
*  effect for easier reading. If enabled, the cell style for alternate rows will have "Dark"
*  appended to it.
* @type {boolean}
* @default true
*/
alternateRecordStyles?: boolean;

/**
* Default directory for skin images (those defined by the class), relative to the
*  Page-wide Page.getSkinDir.
* @type {SCImgURL}
* @default "images/CubeGrid/"
*/
skinImgDir?: SCImgURL;

/**
* Button.baseStyle for the buttons in the innermost column header for 
*  this cubeGrid.
* @type {CSSStyleName}
* @default "innerHeader"
*/
innerHeaderBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the buttons in this grid's column headers.
*  
*  Exception: The innermost column
*  header will always be styled using CubeGrid.innerHeaderBaseStyle.
* @type {CSSStyleName}
* @default "colHeader"
*/
colHeaderBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the buttons in this grid's row headers.
* @type {CSSStyleName}
* @default "rowHeader"
*/
rowHeaderBaseStyle?: CSSStyleName;

/**
* This property is not supported for CubeGrid.
* @type {boolean}
* @default null
*/
rotateHeaderTitles?: boolean;

/**
* If specified and this.showHover is true, this is the default width to apply
*  to hover tips shown for facetValues. If unset, the hover canvas will be sized to 
*  this.hoverWidth if specified instead.
* @type {number | string}
* @default null
*/
facetValueHoverWidth?: number | string;

/**
* If specified and this.showHover is true, this is the default width to apply
*  to hover tips shown for facetLabels. If unset, the hover canvas will be sized to 
*  this.hoverWidth if specified instead.
* @type {number | string}
* @default null
*/
facetLabelHoverWidth?: number | string;

/**
* If specified and this.showHover is true, this is the default height to apply
*  to hover tips shown for facetValues. If unset, the hover canvas will be sized to 
*  this.hoverHeight if specified instead.
* @type {number | string}
* @default null
*/
facetValueHoverHeight?: number | string;

/**
* If specified and this.showHover is true, this is the default height to apply
*  to hover tips shown for facetLabels. If unset, the hover canvas will be sized to 
*  this.hoverHeight if specified instead.
* @type {number | string}
* @default null
*/
facetLabelHoverHeight?: number | string;

/**
* Allows the developer to override the horizontal text alignment of hover tips shown for
*  facet values. If unspecified the hover canvas content alignment will be set by
*  this.hoverAlign if specified.
* @type {Alignment}
* @default null
*/
facetValueHoverAlign?: Alignment;

/**
* Allows the developer to override the horizontal text alignment of hover tips shown for
*  facetLabels. If unspecified the hover canvas content alignment will be set by
*  this.hoverAlign if specified.
* @type {Alignment}
* @default null
*/
facetLabelHoverAlign?: Alignment;

/**
* Allows the developer to override the vertical text alignment of hover tips shown for
*  facet values. If unspecified the hover canvas content alignment will be set by
*  this.hoverVAlign if specified.
* @type {VerticalAlignment}
* @default null
*/
facetValueHoverVAlign?: VerticalAlignment;

/**
* Allows the developer to override the vertical text alignment of hover tips shown for
*  facetLabels. If unspecified the hover canvas content alignment will be set by
*  this.hoverVAlign if specified.
* @type {VerticalAlignment}
* @default null
*/
facetLabelHoverVAlign?: VerticalAlignment;

/**
* Allows the developer to override the css class applied to hover tips shown for
*  facet values. If unspecified, and this.hoverStyle is not null, that 
*  css class will be applied to facet value hovers instead.
* @type {CSSStyleName}
* @default null
*/
facetValueHoverStyle?: CSSStyleName;

/**
* Allows the developer to override the css class applied to hover tips shown for
*  facet labels. If unspecified, and this.hoverStyle is not null, that 
*  css class will be applied to facet label hovers instead.
* @type {CSSStyleName}
* @default null
*/
facetLabelHoverStyle?: CSSStyleName;

/**
* A FacetValueMap describing the set of facet values that should be regarded as
*  "fixed" in this cubeGrid. These are used as fixed criteria for load on demand, and also
*  allow using a dataset with more facets in it than are currently shown in the grid.
* @type {FacetValueMap}
* @default null
*/
fixedFacetValues?: FacetValueMap;

/**
* Hilites to be applied to the data for this component. See hiliting.
* @type {Array<Partial<Hilite>>}
* @default null
*/
hilites?: Array<Partial<Hilite>>;

/**
* DataSource to use to fetch CubeGrid data.
*  
*  The DataSource should have a field named after each facetId. The CubeGrid will submit
*  requests for data as DataSource "fetch" operations that request 
*  CellRecord only for currently visible area
*  (including ListGrid.drawAheadRatio). The Criteria passed
*  in each fetch operation will be a FacetValueMap that corresponds to a rectangular
*  swath of cells the CubeGrid needs data for, for example:
*    { region:"west", product:["chair", "table"], timePeriod:"Q1 2004" }
*  
*  Note that in the criteria above, the Array value for the "product" facet indicates the
*  CubeGrid needs cellRecords for both the "chair" and "table" facetValues.
*  
*  CubeGrid.fixedFacetValues, if any, are always sent in each "fetch" operation.
*  
*  Scrolling, expanding/collapsing tree facets, pivoting, and changing the currently
*  visible facets may all trigger data requests.
*  
*  The CubeGrid will generally attempt to fetch a minimal set of data to fill the viewport,
*  sending multiple fetch operations in a batch (via RPCManager.startQueue)
*  which minimally describe newly revealed areas without specifying each individual cell.
*  The CubeGrid will automatically handle being provided more data than was asked
*  for, so server-side fetch-ahead policies can be implemented without any client-side
*  customization.
*  
*  Note that the DataSource.serverType shipped with the SmartClient
*  SDK is capable of responding to the CubeGrid's data requests without writing any custom
*  server code.
* @type {DataSource}
* @default null
*/
dataSource?: DataSource;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from ListGrid.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only ListGrid.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default true
*/
autoFetchData?: boolean;

/**
* If CubeGrid.autoFetchData is true, this attribute allows the developer to
*  specify a textMatchStyle for the initial ListGrid.fetchData call.
* @type {TextMatchStyle}
* @default "exact"
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Facet definitions for this CubeGrid. Facets, also called "dimensions", are orthogonal
*  aspects of the data model.
*  
*  For example, you can look at profit by the facets "plant and product" or by "product and
*  plant" and it's the same number, because the facets - plant and product - are the same.
*  What would change the profit numbers would be to remove a facet, called "summarizing",
*  or add a new facet, called "drilling down". For example, showing profit by plant and
*  product, you could "drill down" by adding the region facet, which would divide profit
*  among each region. Or you could remove the "plant" facet, showing total profit for each
*  "product", summed across all plants.
*  
*  This property need not be set and will automatically be constructed during widget
*  initialization if data is provided up front and CubeGrid.rowFacets and 
*  CubeGrid.columnFacets have been set. If CubeGrid.facets is not set and
*  there is no initial data but a DataSource is present, drawing the grid will
*  automatically issue a fetch to allow CubeGrid.facets to be resolved.
* @type {Array<Partial<Facet>>}
* @default null
*/
facets?: Array<Partial<Facet>>;

/**
* If true, show facet label context menus with some built-in operations.
*  Otherwise, use generic context menu handling.
* @type {boolean}
* @default false
*/
showFacetContextMenus?: boolean;

/**
* If true, show facet value context menus with some built-in operations.
*  Otherwise, use generic context menu handling.
*  Use this in place of ListGrid.showHeaderContextMenu and
*  ListGrid.showHeaderMenuButton for CubeGrids.
* @type {boolean}
* @default true
*/
showFacetValueContextMenus?: boolean;

/**
* Determines whether cell values in the body can be selected.
* @type {boolean}
* @default true
*/
canSelectValues?: boolean;

/**
* Determines whether row or column facetValue headers can be selected.
* @type {boolean}
* @default true
*/
canSelectHeaders?: boolean;

/**
* If true, when multiple facets appear on one side in a nested headers presentation, the
*  selection state of parent/child headers are automatically kept in sync.
* @type {boolean}
* @default true
*/
autoSelectHeaders?: boolean;

/**
* Whether to select cells in the body when row or column headers are selected.
* @type {AutoSelectionModel}
* @default "both"
*/
autoSelectValues?: AutoSelectionModel;

/**
* If true, clicking on the existing selection causes it to be entirely deselected.
* @type {boolean}
* @default false
*/
simpleDeselect?: boolean;

/**
* The list of Facet.id for facets that will appear on top of the body.
* @type {Array<Partial<string>>}
* @default null
*/
columnFacets?: Array<Partial<string>>;

/**
* The list of Facet.id for facets that will appear to the left of the body.
* @type {Array<Partial<string>>}
* @default null
*/
rowFacets?: Array<Partial<string>>;

/**
* This causes the headers for any combination of facetValues for which there are no 
*  cellRecords to be suppressed.
*  
*  To use this feature, either:
*  
*  
* - all must be provided via ListGrid.setData before the CubeGrid is first drawn, OR
*  
* - all data must be returned by the first DataSource fetch, OR
*  
* - CubeGrid.hideEmptyAxis must be set to either "row" or "column" so that empty values
*  are only automatically hidden for one axis
*  
*  This last point is required because there is no way to determine whether a row is empty
*  unless data for all columns of the row has been loaded (and vice-versa). For this
*  reason if you set hideEmptyFacetValues but do not set hideEmptyAxis, the default
*  behavior of DataSource is automatically disabled and
*  only CubeGrid.fixedFacetValues will be sent as criteria.
* @type {boolean}
* @default null
*/
hideEmptyFacetValues?: boolean;

/**
* With CubeGrid.hideEmptyFacetValues, controls on which axis hiding of empty values is applied,
*  "row" (only empty rows are hidden), "column" (only empty columns are hidden) or both
*  (the default).
* @type {Axis}
* @default null
*/
hideEmptyAxis?: Axis;

/**
* An array of "cellRecords", each of which represents data for one cell of the body area.
* @type {Array<Partial<CellRecord>>}
* @default null
*/
data?: Array<Partial<CellRecord>>;

/**
* Name of the property in a cell record that holds the cell value.
* @type {Identifier}
* @default "_value"
*/
valueProperty?: Identifier;

/**
* Name of the property in a cell record that holds it's unique ID. Note cell record IDs are
*  optional.
* @type {Identifier}
* @default "ID"
*/
cellIdProperty?: Identifier;

/**
* Whether cells can be edited in this grid. Can be overridden on a per-facetValue basis.
* @type {boolean}
* @default false
*/
canEdit?: boolean;

/**
* CubeGrids only support editing by cell.
* @type {boolean}
* @default true
*/
editByCell?: boolean;

/**
* CubeGrids only support editing by cell.
* @type {boolean}
* @default true
*/
saveByCell?: boolean;

/**
* For Browser.isTouch, canDragSelect defaults to false
*  so that touch scrolling can be used to navigate scrollable CubeGrids. In all other
*  browsers it defaults to true.
*  
*  NOTE: If canDragSelect is enabled, it may be desirable to disable
*  Canvas.useTouchScrolling so that touch-dragging cells of the CubeGrid
*  selects them rather than starting a scroll. If Canvas.disableTouchScrollingForDrag
*  is set to true, then touch scrolling will be disabled automatically.
*  However, for accessibility reasons, it is recommended to
*  leave touch scrolling enabled and provide an alternative set of controls that can be
*  used to perform drag-selection.
* @type {boolean}
* @default null
*/
canDragSelect?: boolean;

/**
* If true, body columns can be reordered via the innermost column headers.
* @type {boolean}
* @default null
*/
canReorderColumns?: boolean;

/**
* If true, body columns can be resized via the innermost column headers.
* @type {boolean}
* @default null
*/
canResizeColumns?: boolean;

/**
* If true, hierarchical facets will show expand/collapse controls to allow the user to
*  expand and collapse the tree of facetValues for that facet.
* @type {boolean}
* @default false
*/
canCollapseFacets?: boolean;

/**
* If true, when multiple facets are shown on a side, all facetValues in the second level
*  of headers or higher will show controls to "minimize" the values of the next facet.
*  Minimizing means showing only one, or very few, of the next facet's values.
*  
*  Set FacetValue.isMinimizeValue to indicate which facetValues should be
*  shown when a facet is minimized.
* @type {boolean}
* @default false
*/
canMinimizeFacets?: boolean;

/**
* Turns on indenting of any Facet.isTree column facets. This can be
*  overridden at the facet level via Facet.indentVTree. Setting this property also
*  ensures that the header is sized tall enough to accommodate the fully expanded facet. 
*  
*  The amount of indenting per level can be set with CubeGrid.vTreeFacetIndent, and the
*  direction of the indenting specified with CubeGrid.vTreeFacetIndentDirection.
*  
*  Note that if you specify an explicit height for such a fscet, such as by setting
*  Facet.height or Facet.labelHeight, then the greater of that or the space
*  required to accommodate the fully expanded facet will determine the actual height used.
* @type {boolean}
* @default false
*/
indentVTreeFacets?: boolean;

/**
* Determines how many pixels to move for each level when Facet.isTree
*  column facets are being CubeGrid.indentVTreeFacets.
* @type {Integer}
* @default "8"
*/
vTreeFacetIndent?: Integer;

/**
* Determines layout of facet value titles in each column facet being 
*  CubeGrid.indentVTreeFacets.
* @type {FacetIndentDirection}
* @default "descending"
*/
vTreeFacetIndentDirection?: FacetIndentDirection;

/**
* FacetValueMap of facet values representing a set of facetValues by which the
*  cubeGrid data is sorted.
* @type {FacetValueMap}
* @default null
*/
sortedFacetValues?: FacetValueMap;

/**
* Direction of sorting if sortedFacet or sortedFacetValues is specified.
* @type {SortDirection}
* @default "ascending"
*/
sortDirection?: SortDirection;

/**
* If true, sort controls will be shown on FacetHeaders.
*  
*  When clicked, sort controls call CubeGrid.sortByFacetId.
* @type {boolean}
* @default null
*/
canSortFacets?: boolean;

/**
* If true, sort controls will be shown on facet values.
*  
*  When clicked, sort controls call CubeGrid.sortByFacetValues.
* @type {boolean}
* @default null
*/
canSortData?: boolean;

/**
* If true, allow columns in the grid body to be minimized (reduced to the width of the
*  minimize control) by clicking on a minimize control in the innermost column headers.
* @type {boolean}
* @default null
*/
canMinimizeColumns?: boolean;

/**
* Whether row and column facets can be rearranged by the user, by dragging and dropping the
*  facet labels.
* @type {boolean}
* @default false
*/
canMoveFacets?: boolean;

/**
* If enabled row headers for this cubeGrid will be rendered using a GridRenderer
*  component. This improves performance for very large cubeGrids.
*  
*  Note that this attribute must be set for hierarchical row facets to be indented properly.
* @type {boolean}
* @default false
*/
rowHeaderGridMode?: boolean;

/**
* Automatically size row headers to fit wrapped text.
* @type {boolean}
* @default false
*/
autoSizeHeaders?: boolean;

/**
* Default alignment for facet labels.
* @type {Alignment}
* @default "center"
*/
facetTitleAlign?: Alignment;

/**
* Default alignment for facet values (in headers).
* @type {Alignment}
* @default "center"
*/
facetValueAlign?: Alignment;

/**
* Default align for cell values (in body).
* @type {Alignment}
* @default "center"
*/
cellAlign?: Alignment;

/**
* Whether to allow text wrapping on facet titles.
* @type {boolean}
* @default false
*/
wrapFacetTitles?: boolean;

/**
* Whether to allow text wrapping on facet value titles.
*  
*  Note that this property is incompatible with CubeGrid.indentVTreeFacets column
*  facets.
* @type {boolean}
* @default false
*/
wrapFacetValueTitles?: boolean;

/**
* Whether to pad titles so they aren't flush with header borders.
* @type {boolean}
* @default true
*/
padTitles?: boolean;

/**
* Default width of inner column headers.
* @type {number}
* @default "100"
*/
defaultFacetWidth?: number;

/**
* facetValueId of the default rollupValue for each facet. Can be overridden per facet via
*  facet.rollupValue.
* @type {string}
* @default "sum"
*/
rollupValue?: string;

/**
* In a CubeGrid that displays values of different types (eg "Revenue" and "Income"), the
*  different types of values on display are enumerated as the facet values of the "metric
*  facet". 
*  
*  The metric facet is treated identically to any other facet by the CubeGrid: it can be
*  represented as row or column headers, can be innermost or have other facets under it,
*  can be moved around, etc. However when a metric facet is used, CubeGrid.metricFacetId
*  must be set to allow the CubeGrid to generate meaningful descriptions of values shown in
*  cells for use in hovers and other situations; see CubeGrid.valueTitle for a full
*  explanation.
* @type {string}
* @default "metric"
*/
metricFacetId?: string;

/**
* A label for the data values shown in cells, such as "Sales in Thousands", typically used
*  when the CubeGrid must generate a description for a cell value or set of cell values.
*  
*  For example, in a CubeGrid showing "Revenue" by region and product, a cell with a
*  CellRecord like:
*  
*  {product:"chairs", region:"northwest", _value:"$5k"}
*  
*  Should be described as "Revenue for Chairs for Northwest Region", not "Chairs for
*  Revenue for Northwest Region".
*  
*  For CubeGrids that show multiple types of values at once (eg both "Revenue" and
*  "Income") see CubeGrid.metricFacetId.
* @type {string}
* @default null
*/
valueTitle?: string;

/**
* Minimum width for the body of this cubeGrid.
* @type {number}
* @default null
*/
bodyMinWidth?: number;

/**
* Minimum height for the body of this cubeGrid.
* @type {number}
* @default null
*/
bodyMinHeight?: number;

/**
* This property is not supported for CubeGrid.
*  
*  Consider setting explicit widths via FacetValue.width or CubeGrid.defaultFacetWidth.
* @type {boolean}
* @default null
*/
autoFitFieldWidths?: boolean;

/**
* If set to true, context menu items will be included on the cells and headers providing the
*  user with an option to create a chart of the cubeGrid's data set.
*  See CubeGrid.makeChart for more information.
* @type {boolean}
* @default false
*/
enableCharting?: boolean;

/**
* Name of the SmartClient Class to be used when creating charts. Must support the
*  Chart interface.
* @type {string}
* @default "FacetChart"
*/
chartConstructor?: string;

/**
* Default type of chart to plot.
* @type {ChartType}
* @default "Column"
*/
chartType?: ChartType;

/**
* If CubeGrid.makeChart is called with a chart specification that will show more than
*  chartConfirmThreshold data elements, the user will be presented with a
*  isc.confirm.
*  
*  Set to 0 to disable this confirmation.
* @type {number}
* @default "2000"
*/
chartConfirmThreshold?: number;

/**
* Default separator string used by CubeGrid.exportClientData to separate column
*  and row facet value titles.
* @type {string}
* @default " - "
*/
exportFacetSeparatorString?: string;

/**
* Array of MenuItem to replace the default menu. 
*  Call CubeGrid.getDefaultFacetValueContextItems to get a default set of
*  items to start with.
* @type {Array<Partial<MenuItem>>}
* @default null
*/
facetValueContextItems?: Array<Partial<MenuItem>>;

/**
* Sets the background color for the row and column headers of the cube,
*  if not otherwise set by a more specific property.
*  (see CubeGrid.exportRowFacetBGColor and
*  CubeGrid.exportColumnFacetBGColor).
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportFacetBGColor?: CSSColor;

/**
* Sets the background color for the row headers of the cube.
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportRowFacetBGColor?: CSSColor;

/**
* Sets the background color for the column headers of the cube.
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportColumnFacetBGColor?: CSSColor;

/**
* Sets the text color for the row and column headers of the cube, 
*  if not otherwise set by a more specific property.
*  (see CubeGrid.exportRowFacetTextColor and
*  CubeGrid.exportColumnFacetTextColor).
* @type {CSSColor}
* @default null
*/
exportFacetTextColor?: CSSColor;

/**
* Sets the text color for the row headers of the cube.
* @type {CSSColor}
* @default null
*/
exportRowFacetTextColor?: CSSColor;

/**
* Sets the text color for the column headers of the cube.
* @type {CSSColor}
* @default null
*/
exportColumnFacetTextColor?: CSSColor;

/**
* FormatString for numeric or date formatting. See DataSourceField.format.
* @type {FormatString}
* @default null
*/
valueFormat?: FormatString;

/**
* FormatString used during exports for numeric or date formatting. See
*  DataSourceField.exportFormat.
* @type {FormatString}
* @default null
*/
valueExportFormat?: FormatString;







/**
*  Returns the row number of the most recent mouse event.
* 
* @param {Integer=} optional y-coordinate to obtain row number, in lieu of the y            coordinate of the last mouse event
*/
getEventRow(y?:Integer): Integer;

/**
*  Returns the column number of the most recent mouse event.
* 
* @param {Integer=} optional x-coordinate to obtain column number for, in lieu of the x            coordinate of the last mouse event
*/
getEventColumn(x?:Integer): Integer;

/**
*  Return the CSS class for a cell. By default this method has the following implementation:
*  - return any custom style for the record (GridRenderer.recordCustomStyleProperty) 
*   if defined.
*  - create a style name based on the result of GridRenderer.getBaseStyle and the 
*   state of the record using the rules described in cellStyleSuffixes.
*  
*  Cell Styles are customizable by:
*  
*  
* - attaching a custom style to a record by setting 
*   record[this.recordCustomStyleProperty] to some valid CSS style name. 
*  
* - modifying the base style returned by getBaseStyle() [see that method for further 
*    documentation on this]
*  
* - overriding this function
*  
*  
*  In addition to this, GridRenderer.getCellCSSText may be overriden to
*  provide custom cssText to apply on top of the styling attributes derived from the named
*  style.
*  
* 
* 
* @param {ListGridRecord} record object for this row and column
* @param {number} number of the row
* @param {number} number of the column
*/
getCellStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Return the base stylename for this cell. Default implementation just returns this.baseStyle.
*  See ListGrid.getCellStyle for a general discussion of how to style cells.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getBaseStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Note: This is a ListGrid feature which is inapplicable on this class.
* 
* @param {ListGridViewState} Object describing the desired view state for the grid
*/
setViewState(viewState:ListGridViewState): void;

/**
*  Note: This is a ListGrid feature which is inapplicable on this class.
* 
*/
getViewState(): ListGridViewState;

/**
*  Set the temporary edit values for some cell in the cubeGrid.
*  Note that only the CubeGrid.valueProperty of the object passed in
*  will be displayed in the cell.
* 
* @param {Array<Partial<number>>} 2 element array of the form          [rowNum,colNum] indicating the record being edited
* @param {object} New values for the record
*/
setEditValues(cell:Array<Partial<number>>, values:object): void;

/**
* 
*  Set the edit value for some cell in the cube grid.
*  Note that cubeGrids display one record per cell - the value passed in should be the 
*  desired edit value for the CubeGrid.valueProperty of the record.
* 
* 
* @param {number} Row number
* @param {number} Column number
* @param {any} New value for the record
*/
setEditValue(rowNum:number, colNum:number, value:any): void;

/**
* 
*  Returns the current temporary locally stored edit value for a cell being edited.
*  Note this is the CubeGrid.valueProperty that will be saved for the
*  cell in question.
* 
* 
* @param {number} index of the row for which the editValue should be returned
* @param {number} index of the column for which value should be returned
*/
getEditValue(rowNum:number, colNum:number): any;

/**
*  Returns the combination of unsaved edits (if any) and original values (if any) for a given
*  cell being edited.
*  
*  The returned value is never null, and can be freely modified.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number=} colNum of the record being edited. Only required if the records             rowNum is passed in as the first parameter
*/
getEditedRecord(valuesID:number | object, colNum?:number): object;

/**
*  Returns the current value of a cell. If the cell has an outstanding edit value, this will
*  be returned, otherwise the underlying value of the record will be returned.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number} colNum of the cell. Only required if the first parameter is               a rowNum
*/
getEditedCell(record:number | object, colNum:number): any;

/**
*  Returns the current set of unsaved edits for a given row being edited.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number} colNum of the record being edited. Only required if valuesID             is passed in as a rowNum.
*/
getEditValues(valuesID:number | object, colNum:number): object;

/**
* 
*  Clear a field value being tracked as an unsaved user edit for some cell.
*  The saved record value will be displayed in the the appropriate cell instead.
* 
* 
* @param {number | object} Row number, primary keys object for the record,                       or editValues object
* @param {number=} Column number for the cell in question. Only required if the               first parameter is a row number.
*/
clearEditValue(editValuesID:number | object, colNum?:number): void;

/**
* 
*  Validates and saves edits for some cell. If rowNum and colNum are not passed in, the current
*  edit cell will be saved.
*  
*  The 'callback' parameter provides a notification when the save attempt completes.
*  Cases under which the callback will fire are:
*  
* - Save completed successfully
*  
* - No changes to the edited cell, so save not required
*  
* - Validation failure occurred on the client or on the server
*  Note that if no rowNum/colNum were passed in and the editor is not showing for the cell,
*  the callback will NOT fire - in this case, the method is a no-op.
*  
*  Other, standard callbacks such as ListGrid.editComplete,
*  ListGrid.editFailed and ListGrid.cellChanged will
*  fire normally.
*  
*  Note this method does not hide the inline editors if they are showing - to explicitly save 
*  and end editing, use the method 'endEditing()'
* 
* 
* @param {EditCompletionEvent=} Event used to complete cell editing.     Optional, and defaults to "programmatic". Can be used by the      callback method to perform custom actions such as navigation when the      save completes.
* @param {Callback=} Callback to fire on completion of the saving process.                  If no edits were made or client-side validation fails                   the callback will be fired synchronously at the end of this                   method.                   Takes the following parameters:                   - rowNum (Number) edited row number                   - colNum (Number) edited column number                   - editCompletionEvent (EditCompletionEvent) event                    passed in (defaults to "programmatic")                   - success (boolean) false if the save was unable to                    complete due to a validation failure or server-side                    error.
* @param {number=} Which row should be saved. If unspecified the current edit row              is saved by default. Note that if there is no current edit cell               this method will no op.
* @param {number=} Which row should be saved. If unspecified the current edit column              is saved by default. Note that if there is no current edit cell               this method will no op.
*/
saveEdits(editCompletionEvent?:EditCompletionEvent, callback?:Callback, rowNum?:number, colNum?:number): void;

/**
*  Method to determine which records currently have pending (unsubmitted) edits.
*  Returns an array of 2 element arrays indicating the [rowNum,colNum] of the
*  cells in question.
* 
*/
getAllEditCells(): Array<any>;

/**
* 
* 
* @param {Array<Partial<any>>=} allows you to specify which cell(s) to drop          edits for. Each record should be identified as an array containing          [rowNum,colNum]
* @param {boolean=} By default this method will hide the editor if               it is currently showing for any row in the grid. Passing in                this parameter will leave the editor visible (and just reset                the edit values underneath the editor).
*/
discardAllEdits(records?:Array<Partial<any>>, dontHideEditor?:boolean): void;

/**
*  If this cubeGrid can be edited, this method will return true if the record passed in has been
*  edited, but the edits have not yet been saved to the CubeGrid's data object.
* 
* @param {number} row index of record to check for changes
* @param {number} column index of the record to check for changes
*/
recordHasChanges(rowNum:number, colNum:number): boolean;

/**
*  Determines whether any cells in this cubeGrid have been edited but not yet saved to the
*  underlying data set.
* 
*/
hasChanges(): boolean;

/**
*  Save a number of outstanding edits for this CubeGrid. If no cells are specified, all 
*  outstanding edits will be saved
* 
* @param {Array<Partial<any>>=} optionally specify which cells to save. Each cell should be specified as a 2 element array in the format [rowNum,colNum].
* @param {Callback=} If specified this callback will be fired on a successful save of the specified rows. Note that if there are no pending edits to be saved this callback will  not fire - you can check for this condition using CubeGrid.hasChanges or  CubeGrid.recordHasChanges.
*/
saveAllEdits(cells?:Array<Partial<any>>, saveCallback?:Callback): boolean;

/**
*  Get a facet definition by facetId. Constant time.
* 
* 
* @param {string} the id of the facet to retrieve
*/
getFacet(facetId:string): Facet;

/**
*  Get a facet value definition by facetId and facetValueId. Constant time.
* 
* 
* @param {string} the id of the facet to retrieve
* @param {string} the id of the facet value to retrieve
*/
getFacetValue(facetId:string, facetValueId:string): FacetValue;

/**
*  Get the facetValue definition for the facetValue to show when this facet is "rolled up" under
*  another facet, during a breakout.
* 
*  A facet is not required to have a rollup value, and if it does not have one, then rollups will
*  simply be blank rows. The facetValueId of the rollup value can be declared as
*  cubeGrid.rollupValue or facet.rollupValue. 
* 
* 
*/
getRollupValue(): string;

/**
*  Given a cell coordinate within this CubeGrid return a FacetValueMap
*  indicating the facet values for the cell.
* 
* @param {number} row index of the cell
* @param {number} column index of the cell
*/
getCellFacetValues(rowNum:number, colNum:number): FacetValueMap;

/**
*  Return a FacetValueMap of the facet values for the row field at the specified level
*  containing the requested row number. Note that outer row fields may span several grid rows.
* 
* 
* @param {Integer} 0-based index into the grid rows (and inner row header fields)
* @param {Integer} target header level; 0 represents the outer row header
*/
getRowHeaderFacetValues(rowNum:Integer, level:Integer): FacetValueMap;

/**
*  Return a FacetValueMap of the facet values for the column field at the specified
*  level containing the requested column number. Note that outer column fields may span several
*  grid columns.
* 
* 
* @param {Integer} 0-based index into the grid columns (and inner column header fields)
* @param {Integer} target header level; 0 represents the outer column header
*/
getColumnHeaderFacetValues(colNum:Integer, level:Integer): FacetValueMap;

/**
*  Get the index of the first column in the grid that matches the specified FacetValueMap.
*  
*  The facetValues passed in should contain values for at least one column facet. It may
*  contain properties other than column facets, which will be ignored. If values are 
*  sparse (values not specified for every column facet), the first column matching the specified
*  facet values will be returned.
* 
* 
* @param {FacetValueMap} facet values to find
*/
getFacetValuesColumn(facetValues:FacetValueMap): Integer;

/**
*  Get the index of the first row in the grid that matches the specified FacetValueMap.
*  
*  The facetValues passed in should contain values for at least one row facet. It may contain
*  properties other than row facets, which will be ignored. If values are 
*  sparse (values not specified for every row facet), the first row matching the specified
*  facet values will be returned.
* 
* 
* @param {FacetValueMap} facet values to find
*/
getFacetValuesRow(facetValues:FacetValueMap): Integer;

/**
*  Return a FacetValueMap indicating the facet values for a specific 
*  row in the grid.
* 
* @param {number} index of the row
*/
getRowFacetValues(rowNum:number): FacetValueMap;

/**
*  Return a FacetValueMap indicating the facet values for a specific 
*  column in the grid.
* 
* @param {number} index of the column
*/
getColumnFacetValues(colNum:number): FacetValueMap;

/**
*  Return the pointer to a particular record by record and column number.
* 
* @param {number} row index of record to return.
* @param {number} column index of record to return.
*/
getCellRecord(rowNum:number, colNum:number): ListGridRecord;

/**
*  Given a record in this grid, this method returns the rowNum on which the record is displayed.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellRow(cellRecord:CellRecord): Integer;

/**
*  Given a record in this grid, this method returns the colNum in which the record is displayed.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellColumn(cellRecord:CellRecord): Integer;

/**
*  Given a record in this grid, this method returns the coordinates of the cell in which the 
*  record is displayed as a 2 element array.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellCoordinates(record:CellRecord): Array<any>;

/**
*  Notification method fired when new data arrives from the server to be displayed in this
*  CubeGrid. For example in response to the user openng a collapsed facet, or as a result of
*  an initial fetch request for all data from a CubeGrid where CubeGrid.facets is not
*  set and there is no initial data. Only applies to databound CubeGrids.
* 
*/
dataArrived(): void;

/**
*  Method handler fired when the user clicks on a facet label.
* 
* 
*/
facetLabelClick(): void;

/**
*  Get the HTML for the facet label hover. Default implementation returns null.
*  methodType callback
* 
* @param {string} ID for the facet
*/
facetLabelHoverHTML(facetId:string): void;

/**
*  Get the HTML for the facetValue button hover. Default implementation returns null.
*  
* 
* @param {FacetValueMap} facetValues object for the button
*/
facetValueHoverHTML(facetValues:FacetValueMap): void;

/**
*  Toggles the open state of the specified field. No-ops if it's not showing.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
toggleFieldOpenState(facetValueMap:FacetValueMap): boolean;

/**
*  Return whether the specified CubeGrid field is open, taking into account both
*  Facet.canCollapse and Facet.canMinimize.
*  
*  Note that if you don't already have a FacetValueMap to the field in question, you can
*  get one by calling CubeGrid.getRowHeaderFacetValues or CubeGrid.getColumnHeaderFacetValues,
*  
*  You can also construct a FacetValueMap on your own by using the Facet.ids from
*  CubeGrid.rowFacets or CubeGrid.columnFacets together with the FacetValue.ids of the
*  Facet.values for the row or column that you want to query. Given a Facet.id,
*  you can use CubeGrid.getFacet to obtain the correponding Facet.
* 
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
isFieldOpen(facetValueMap:FacetValueMap): boolean;

/**
*  Expands the specified field. No-ops if it's not showing, or if it's already expanded.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
expandField(facetValueMap:FacetValueMap): boolean;

/**
*  Collapses the specified field. No-ops if it's not showing, or it it's already collapsed.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
collapseField(facetValueMap:FacetValueMap): boolean;

/**
*  auto-size the header facet horizontally
* 
* @param {string} ID of facet to resize.
*/
autoSizeFacet(facetId:string): void;

/**
* 
*  Handler fired when facet is closed
* 
*    methodType handler
* 
* @param {string} ID of facet that was closed
*/
closeFacet(facetId:string): void;

/**
* 
*  Handler fired when column is closed
* 
*    methodType handler
* 
* @param {FacetValue} FacetValues for the appropriate col.
*/
closeColumn(headerFacetValues:FacetValue): void;

/**
*  Hides the specified field if it is currently visible. No-ops if it's already hidden.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
hideFacetValues(facetValueMap:FacetValueMap): boolean;

/**
*  Shows the specified field if it was previsouly hidden. No-ops if it's already showing.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
showFacetValues(facetValueMap:FacetValueMap): boolean;

/**
*  Handler/Notification function for facetValue selection change (no default implementation).
* 
* 
* @param {FacetValueMap} facetValues with new selection state
* @param {boolean} new selection state
*/
facetValueSelectionChanged(facetValues:FacetValueMap, newState:boolean): void;

/**
*  Returns an array of the selected cell records.
* 
*  methodType getter
* 
*/
getSelectedCells(): Array<Partial<CellRecord>>;

/**
*  Returns an array of the IDs of all selected cell records.
* 
*  methodType getter
* 
*/
getSelectedCellIds(): Array<Partial<string>>;

/**
*  Determine whether any cells are selected in this cubeGrid.
* 
*  methodType tester
* 
*/
anyCellSelected(): boolean;

/**
*  Determine whether the cell passed in is selected in this cubeGrid.
* 
*  methodType tester
* 
* @param {CellRecord} cell to test
*/
cellIsSelected(cell:CellRecord): boolean;

/**
*  Select / deselect a single cell - accepts cell ID or CellRecord.
* 
* 
* @param {CellRecord | GlobalId} cell to select / deselect
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectCell(cell:CellRecord | GlobalId, newState?:boolean): void;

/**
*  Deselect a single cell - accepts cell ID or cell record.
* 
* 
* @param {CellRecord | GlobalId} cell to deselect
*/
deselectCell(cell:CellRecord | GlobalId): void;

/**
*  Select/deselect cells that match a FacetValueMap.
*  Also supports an explicit list of CellRecords or cell IDs.
* 
* 
* @param {Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>} cells to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectCells(cellList:Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>, newState?:boolean): void;

/**
*  Deselect cells that match a FacetValueMap.
*  Also supports an explicit list of CellRecords or cell IDs.
* 
* 
* @param {Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>} cells to deselect
*/
deselectCells(cellList:Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>): void;

/**
*  Select all cells.
* 
* 
*/
selectAllCells(): void;

/**
*  Deselect all cells.
* 
* 
*/
deselectAllCells(): void;

/**
*  Returns an array of facetValues objects indicating the headers that are selected in the
*  headerBar for this facet. If facetId is not passed, returns selection for all
*  facets.
* 
* 
* @param {string=} id for facet for which we are getting selected facetValues
*/
getSelectedFacetValues(facetId?:string): Array<Partial<FacetValueMap>>;

/**
*  Return whether the header indicated by the set of facetValues is selected.
* 
* 
* @param {FacetValueMap} facetValues to test
*/
facetValuesSelected(facetValues:FacetValueMap): boolean;

/**
*  Return whether any facet value for this facet is selected in headers. If no facetId passed,
*  return whether any facet has a selection.
* 
*  methodType tester
* 
* @param {string} Id for facet to test
*/
facetHasSelection(facetId:string): boolean;

/**
*  Return the list of facets that have any selection in their headers.
* 
*  methodType getter
* 
*/
getFacetsHavingSelection(): Array<Partial<string>>;

/**
*  Select/deselect the header showing a given set of facet values.
* 
* 
* @param {FacetValueMap} facetValues to select the header of
* @param {boolean=} new selection state - if null defaults to true
*/
selectFacetValues(facetValues:FacetValueMap, newState?:boolean): void;

/**
*  Deselect the header showing a given set of facet values.
* 
*  methodType action
* 
* @param {FacetValueMap} facetValues to deselect
*/
deselectFacetValues(facetValues:FacetValueMap): void;

/**
*  Select/deselect the header for a given facet value.
* 
*  methodType action
* 
* @param {string} ID of facet
* @param {string} ID of facetValue to select
* @param {boolean=} new selection state - if null defaults to true
*/
selectFacetValue(facetId:string, facetValueId:string, newState?:boolean): void;

/**
*  Deselect the header for a given facet value.
* 
*  methodType action
* 
* @param {string} ID of facet
* @param {string} ID of facetValue to select
*/
deselectFacetValue(facetId:string, facetValueId:string): void;

/**
*  Select/deselect all headers in a headerBar (specified by facetId) or
*  all headerBars (if no facetId).
* 
*  methodType action
* 
* @param {string=} ID of facet - if null, selects all headerbars' headers
* @param {boolean=} new selection state - if null defaults to true
*/
selectAllFacetValues(facetId?:string, newState?:boolean): void;

/**
*  Deselect all headers in a headerBar (specified by facetId) or
*  all headerBars (if no facetId).
* 
*  methodType action
* 
* @param {string=} ID of facet - if null, selects all headerbars' headers
*/
deselectAllFacetValues(facetId?:string): void;

/**
*  Deselect all cells and facetValues.
* 
*    methodType action
* 
*/
deselectAll(): void;

/**
*  Resizes all columns for the provided facetValueId, which must be a facetValueId from the
*  innermost column facet.
* 
* 
* @param {Identifier} facetValueId of columns to be resized
* @param {number} column's new width
*/
resizeFacetValue(facetValueId:Identifier, newWidth:number): void;

/**
*    Apply a hilite to a specific cell.
*    Note: can be called either as 
*      hiliteCell(cellObject, hiliteID) or 
*      hiliteCell(row, column, hiliteID)
* 
*    methodType action
* 
* @param {any | any} cell to hilite / row of cell to hilite
* @param {string | any} hilite ID / column of cell to hilite
* @param {string=} optional third parameter - hilite ID.
*/
hiliteCell(cellObj:any | any, hiliteID:string | any, alternateHiliteID?:string): boolean;

/**
*    Apply a hilite to an array of cells.
* 
*    methodType action
* 
* @param {Array<Partial<any>>} cells to hilite
* @param {string} ID of hilite to apply to cells
*/
hiliteCellList(cellObjList:Array<Partial<any>>, hiliteID:string): boolean;

/**
*    Apply a hilite to all cells corresponding to a facetValue.
* 
*    methodType action
* 
* @param {string} facet ID
* @param {string} facet value ID
* @param {string} hilite ID
*/
hiliteFacetValue(facetID:string, facetValueID:string, hiliteID:string): boolean;

/**
*  Set the title of a facet (appears in facet label).
* 
* 
* @param {Identifier} facet to update
* @param {string} title for the facet
*/
setFacetTitle(facetId:Identifier, newTitle:string): void;

/**
*  Set the align of a facet title (appears in facet label).
* 
* 
* @param {Identifier} facet to update
* @param {Alignment} new alignment for facet title
*/
setFacetTitleAlign(facetId:Identifier, align:Alignment): void;

/**
*  Set the title for a facet value.
* 
* 
* @param {Identifier} facet to update
* @param {Identifier} facetValue to update
* @param {string} title for the facet
*/
setFacetValueTitle(facetId:Identifier, facetValueId:Identifier, newTitle:string): void;

/**
*  Set the align for the title for a facet value.
* 
* 
* @param {Identifier} facet to update
* @param {Identifier} facetValue to update
* @param {Alignment} new alignment for facet value title
*/
setFacetValueTitleAlign(facetId:Identifier, facetValueId:Identifier, align:Alignment): void;

/**
*  Get the current visual order and width for the facet values of a facet or facetValueGroup as
*  an object of the form:
*  
*  [ {id:facetValueId, width:currentWidth }, ... ]
*  
* 
* 
* @param {string | string} Which facet do we want details for?
*/
getFacetValueLayout(id:string | string): Array<any>;

/**
*  Get the current widths of the row facets, as an object of the form:
*  
*  [ {facetId:facetId, width:currentWidth }, ... ]
*  
* 
* 
*/
getRowFacetLayout(): Array<any>;

/**
*  Get the current heights of the column facets, as an object of the form:
*  
*  [ {facetId:facetId, height:currentHeight}, ... ]
*  
* 
* 
*/
getColumnFacetLayout(): Array<any>;

/**
*  Setter for the CubeGrid.enableCharting attribute.
* 
* @param {boolean} */
setEnableCharting(enableCharting:boolean): void;

/**
*  Chart the portion of the dataset indicated by fixedFacetValues, for all 
*  values of the variableFacets.
*  
*  One, two or more variableFacets may be passed. Two variable facets for a column chart will
*  result in Chart.stacked or clustering. Three facets or more may be
*  supported by some CubeGrid.chartType or
*  CubeGrid.chartConstructor.
*  
* 
* @param {FacetValueMap} set of facet values to hold constant. Pass null to                     chart the entire dataset.
* @param {Array<Partial<string>>} set of facets to be charted
* @param {Partial<Chart>} properties to pass through to the created            Chart
*/
makeChart(fixedFacetValues:FacetValueMap, variableFacets:Array<Partial<string>>, chartProperties:Partial<Chart>): Chart;

/**
*  Returns a default set of items, which can be updated/modified, and then assigned to
*  CubeGrid.facetValueContextItems to be used in the context menu of the appropriate 
*  header button.
* 
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
getDefaultFacetValueContextItems(facetValues:FacetValueMap): Array<Partial<MenuItem>>;

/**
*  Add a row facet to the view at index "index". Handles the facet already being in the view
*  (does a pivot).
* 
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
*  methodType action
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addRowFacet(facetId:Identifier, index?:Integer): void;

/**
*  Add a column facet to the view at index "index". Handles the facet already being in the view
*  (does a pivot).
* 
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
*  methodType action
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addColumnFacet(facetId:Identifier, index?:Integer): void;

/**
* 
*  Add a facet to the view, into the row or column facets (intoRows true or false), at index
*  "index". Handles the facet already being in the view (does a pivot).
*  
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {boolean=} [true] whether to add facet as a row facet
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addFacet(facetId:Identifier, intoRows?:boolean, index?:Integer): void;

/**
* 
*  Remove a facet from the current view, using a fixed value from that facet. For example, remove
*  the "months" facet from the view, collapsing to just January, or total for all months.
* 
* 
* @param {Identifier} facetId to remove
* @param {Identifier=} New fixed value for the facet, to be added to             CubeGrid.fixedFacetValues. Default is the rollup value for the facet.
*/
removeFacet(facetId:Identifier, fixedFacetValueId?:Identifier): void;

/**
* 
*  Modify fixedFacetValues for this cubeGrid.
* 
* 
* @param {Identifier} facetId
* @param {Identifier} New fixed value for the facet, to be added to             CubeGrid.fixedFacetValues. Default is the rollup value for the facet.
*/
setFixedFacetValue(facetId:Identifier, fixedFacetValueId:Identifier): void;

/**
*  Note that CubeGrid does not support a WYSIWYG print view by default(also used when
*  RPCManager.exportContent). Instead we recommend 
*  DataSource.exportClientData.
* 
* 
* @param {PrintProperties=} properties to configure printing behavior - may be null.
* @param {Callback=} optional callback. This is required to handle cases where HTML         generation is asynchronous - if a method generates HTML asynchronously, it should return          null, and fire the specified callback on completion of HTML generation. The          first parameter HTML should contain the generated print HTML.          The callback is only called if null is returned. Furthermore, the default          getPrintHTML() implementation always returns null and fires the callback          when a callback is provided.
*/
getPrintHTML(printProperties?:PrintProperties, callback?:Callback): HTMLString;

/**
*  Exports this component's data with client-side formatters applied, so is suitable for direct
*  display to users. This feature requires the SmartClient server.
*  
*  The export format will combine the column facet value titles, generating a single row of
*  column headers at the top with titles such as "All Years - Budget" if Time and Scenario were
*  column facets. The row facet value titles for separate facets won't be combined, so that
*  each row facet will have a separate column, with the facet titles at the top in the 
*  "column header" row, and the row facet value titles below their corresponding facet title.
*  Data values each get their own row and column position.
* 
* 
* @param {object} contains configuration settings for the export, including:  
- facetSeparatorString (String) - if specified, the separator to use in favor   of CubeGrid.exportFacetSeparatorString when combining titles from multiple   facet values. 
* @param {Partial<DSRequest>=} Request properties for the export.
* @param {RPCCallback=} Optional callback. If you specify DSRequest.exportToClient: false in the request  properties, this callback will fire after export completes. Otherwise the callback will  fire right before the download request is made to the server.
*/
exportClientData(settings:object, requestProperties?:Partial<DSRequest>, callback?:(response:RPCResponse, rawData:any, request:RPCRequest) => void): void;

/**
*  StringMethod handler fired when mouseover occurs over a facetValues header button
*  
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueOver(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when mouseout occurs for a facetValues header button
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueOut(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when mouse hovers over a facetValue button in a header.
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueHover(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when context click occurs over a facetValue button in a header.
* 
* 
* @param {FacetValueMap} FacetValueMap describing the appropriate header button
*/
facetValueContextClick(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when the user right clicks on a facet label.
* 
* 
* @param {string} ID of facet
*/
facetContextClick(facetId:string): void;

/**
*  
*  Notification fired when a facet or facetValueGroup is reordered
* 
* 
* @param {string} facetValueGroupId or facetId
*/
facetValueReordered(groupId:string): void;

/**
*    Notification fired when a new facet is added.
*  
* 
* @param {string} facetId that was added
*/
facetAdded(facetId:string): void;

/**
*  Notification fired when a facet is removed.
*  
* 
* @param {string} facetId that was removed
*/
facetRemoved(facetId:string): void;

/**
*  Notification fired when a facet is moved.
*  
* 
* @param {string} facetId which moved
*/
facetMoved(facetId:string): void;

/**
*  Notification fired when a fixed facet value is set for some facet.
* 
* @param {string} facetId
* @param {string} new fixed facet value
*/
fixedFacetValueChanged(facetId:string, facetValueId:string): void;

/**
*  Called when a sort control is clicked on a FacetValueHeader. Does nothing by default.
*  
* 
* @param {FacetValueMap} facetValues to sort
* @param {boolean} true for ascending
*/
sortByFacetValues(facetValues:FacetValueMap, sortDirection:boolean): void;

/**
*  Called when a sort control is clicked on a FacetHeader. Does nothing by default.
*  
* 
* @param {string} ID of facet to sort
* @param {boolean} true for ascending
*/
sortByFacetId(facetId:string, sortDirection:boolean): void;

/**
*  StringMethod handler fired when mouseover occurs over a facet label
* 
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelOver(facetId:string): void;

/**
*  StringMethod handler fired when mouseout occurs over a facet label
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelOut(facetId:string): void;

/**
*  StringMethod handler fired from hover over a facet label
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelHover(facetId:string): void;

/**
* 
*    Handler/Notification function for cell selection change
*    May update header button styles.
* 
*    methodType handler
* 
* @param {Array<Partial<CellRecord>>} array of cells with new selection state
*/
cellSelectionChanged(cellList:Array<Partial<CellRecord>>): boolean;




}



/**
* When discovering a tree, the scanMode determines how to scan for the childrenProperty
*  "node": take each node individually
*  "branch": scan direct siblings as a group, looking for best fit
*  "level": scan entire tree levels as a group, looking for best fit
*/
type ScanMode = 
/**
* take each node individually
*/
"node" |

/**
* scan direct siblings as a group, looking for best fit
*/
"branch" |

/**
* scan entire tree levels as a group, looking for best fit
*/
"level";


/**
* Configuration of an HTML list in a RichTextEditor.
*/
class ListProperties {


/**
* The style of list item marker. If "custom-image", the ListProperties.image
*  should be specified.
* @type {ListStyleType}
* @default null
*/
style?: ListStyleType;

/**
* When the list item marker ListProperties.style is "custom-image", the image
*  to use for the markers.
* @type {SCImgURL}
* @default "[SKIN]/RichTextEditor/bullet_blue.png"
*/
image?: SCImgURL;

/**
* For ordered lists, the number to start the first item with. Must be non-negative.
* @type {Integer}
* @default "1"
*/
startNumber?: Integer;










}

/**
* Settings for generating printable HTML for components.
*/
class PrintProperties {


/**
* An array of Strings indicating the classNames of controls that should be omitted from
*  printing. By default, omitControls includes all button-based controls, menus
*  and similar interactive controls that are typically useless in printed output.
*  
*  All subclasses of the specified classes are also omitted.
*  
*  See also PrintProperties.includeControls.
* @type {Array<Partial<string>>}
* @default null
*/
omitControls?: Array<Partial<string>>;

/**
* An array of Strings indicating the classNames of controls that should be specifically
*  included when printing, even if a superclass is listed in PrintProperties.omitControls.
* @type {Array<Partial<string>>}
* @default null
*/
includeControls?: Array<Partial<string>>;

/**
* If true, generates HTML for export.
*  
*  Some components, specifically DrawPane and FacetChart on IE8 and earlier, need to generate
*  different HTML for export versus in-browser print preview. When using RPCManager.exportContent
*  the printForExport property is set to true automatically. If not using RPCManager.exportContent(),
*  but the generated HTML will be sent for export, the PrintProperties passed to
*  Canvas.getPrintHTML must have printForExport:true.
* @type {boolean}
* @default null
*/
printForExport?: boolean;










}



/**
* Flag passed to functions as displayNodeType, telling the function whether it should work on
*  folders, leaves or both at once.
*/
type DisplayNodeType = 
/**
* operate on both folders and leaves
*/
"null/unset" |

/**
* operate on folders only, ignoring leaves
*/
"folders" |

/**
* operate on leaves only, ignoring folders
*/
"leaves";


/**
* The style of line-breaks to use when exporting data
*/
type LineBreakStyle = 
/**
* Use the default line-break style of the server OS
*/
"default" |

/**
* Use UNIX-style line-breaks (LF only)
*/
"unix" |

/**
* Use MAC-style line-breaks (CR only)
*/
"mac" |

/**
* Use DOS-style line-breaks (both CR & LF)
*/
"dos";


/**
* Defines a simple gradient vertical gradient between Gradient.startColor
*  Gradient.endColor, or using Gradient.colorStops. See
*  SimpleGradient, LinearGradient and RadialGradient for further
*  properties to define more advanced gradients.
*/
class Gradient {


/**
* Identifier which can be used by one or more DrawItems when gradient is assigned
*  to DrawPane.gradients. The ID property is optional when gradient is assigned directly
*  to a DrawItem.
*  
*  The ID must be unique within DrawPane.gradients if defined.
* @type {Identifier}
* @default null
*/
id?: Identifier;

/**
* An array of color stops for this gradient.
* @type {Array<Partial<ColorStop>>}
* @default null
*/
colorStops?: Array<Partial<ColorStop>>;

/**
* A start color for the gradient. If both startColor and Gradient.endColor
*  are set then Gradient.colorStops is ignored.
* @type {CSSColor}
* @default null
*/
startColor?: CSSColor;

/**
* An end color for the gradient. If both Gradient.startColor and endColor
*  are set then Gradient.colorStops is ignored.
* @type {CSSColor}
* @default null
*/
endColor?: CSSColor;










}

/**
* The SmartClient system supports hierarchical data (also referred to as tree data
*  due to its "branching" organization) with:
*  
*   
* - the Tree class, which manipulates hierarchical data sets
*   
* - the TreeGrid widget class, which extends the ListGrid class to visually
*     present tree data in an expandable/collapsible format.
*  
*  For information on DataBinding Trees, see treeDataBinding.
*  
*  A TreeGrid works just like a ListGrid, except one column (specified by
*  TreeGridField.treeField) shows a hierarchical Tree. A TreeGrid is not limited
*  to displaying just the Tree column - you can define additional columns (via
*  TreeGrid.fields) which will render just like the columns of a ListGrid, and
*  support all of the functionality of ListGrid columns, such as
*  ListGridField.formatCellValue.
*  
*  Except where explicitly overridden, ListGrid methods, callbacks, and properties
*  apply to TreeGrids as well. The ListGrid defines some methods as taking/returning
*  ListGridField and ListGridRecord. When using those methods in a TreeGrid,
*  those types will be TreeGridField and TreeNode, respectively.
*/
class TreeGrid extends ListGrid implements DataBoundComponent {
/**
* Creates a new TreeGrid
* 
* @param typeCheckedProperties {Partial<TreeGrid>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {TreeGrid} The new instance
*/
static create(typeCheckedProperties? : Partial<TreeGrid>, uncheckedProperties?: Object): TreeGrid;


/**
* The DataSource that this component should bind to for default fields and for performing
*  DSRequest.
*  
*  Can be specified as either a DataSource instance or the String ID of a DataSource.
* @type {DataSource | GlobalId}
* @default null
*/
dataSource?: DataSource | GlobalId;

/**
* A Tree object containing of nested TreeNodes to 
*  display as rows in this TreeGrid. 
*  The data property will typically not be explicitly specified for 
*  databound TreeGrids, where the data is returned from the server via databound component
*  methods such as fetchData()
* @type {Tree}
* @default null
*/
data?: Tree;

/**
* You can specify the initial set of data for a databound TreeGrid using this property.
*  The value of this attribute should be a list of parentId-linked
*  TreeNodes in a format equivalent to that documented on Tree.data or, for
*  TreeGrids with TreeGrid.dataFetchMode set to
*  FetchMode, on ResultTree.data.
*  
*  If you create a standalone Tree or ResultTree as the
*  TreeGrid's TreeGrid.data then you may equivalently specify this initial set
*  of tree nodes in that tree's Tree.data property.
* @type {Array<Partial<TreeNode>>}
* @default null
*/
initialData?: Array<Partial<TreeNode>>;

/**
* For databound treeGrid instances, should the entire tree of data be loaded on initial 
*  fetch, or should folders load their children as they are opened.
*  
*  If unset, calling TreeGrid.fetchData will default it to true, otherwise, if a ResultTree
*  is passed to TreeGrid.setData, the ResultTree.loadDataOnDemand setting is
*  respected. Must be enabled on the underlying ResultTree when using
*  TreeGrid.dataFetchMode: "paged".
*  
*  Note that when using loadDataOnDemand, every node returned by the server is
*  assumed be a folder which may load further children. See
*  ResultTree.defaultIsFolder for how to control this behavior.
* @type {boolean}
* @default null
*/
loadDataOnDemand?: boolean;

/**
* If set, tree-based filtering is performed such that parent nodes are kept as long as
*  they have children that match the filter criteria, even if the parents themselves do not
*  match the filter criteria. If not set, filtering will exclude parent nodes not matching
*  the criteria and all nodes below it in the tree.
*  
*  When keepParentsOnFilter is enabled for paged ResultTrees, server-side
*  filtering is required.
*  
*  When enabled for non-paged trees, FetchMode is automatically
*  enabled so that all filtering behavior shifts to the client-side and full criteria are
*  no longer sent to the server. Instead, server fetches will always load all nodes, or
*  with TreeGrid.loadDataOnDemand:true, will always load all nodes under a given parent.
*  This means that the server does not need to implement special tree filtering logic.
*  
*  Optionally, ResultTree.serverFilterFields can be set to a list of field names that
*  will be sent to the server whenever they are present in the criteria.
* @type {boolean}
* @default null
*/
keepParentsOnFilter?: boolean;

/**
* Mode of fetching records from server.
*  
*  fetchMode:"local" implies that local filtering will always be performed. See
*  TreeGrid.keepParentsOnFilter for additional filtering details.
*  
*  fetchMode:"basic" or "paged" implies that if search criteria change, the entire
*  tree will be discarded and re-fetched from the server. When retrieving the replacement
*  tree data, the default behavior will be to preserve the TreeGrid.getOpenState
*  for any nodes that the server returns which were previously opened by the user. Note
*  that this implies that if TreeGrid.loadDataOnDemand is enabled and the server returns only
*  root-level nodes, open state will be preserved only for root-level nodes, and children
*  of open root-level nodes will be immediately fetched from the server if
*  they are not included in the server's initial response.
*  
*  fetchMode:"paged" enables paging for nodes that have very large numbers of children.
*  Whenever the children of a folder are loaded, the resultTree will set
*  DSRequest.startRow and DSRequest.endRow when requesting children
*  from the DataSource, and will manage loading of further children on demand, similar to
*  how a ResultSet manages paging for lists. For a deeper discussion see the
*  Paging large sets of children section of the treeDataBinding
*  overview.
* @type {FetchMode}
* @default "basic"
*/
dataFetchMode?: FetchMode;

/**
* When TreeGrid.keepParentsOnFilter is enabled for FetchMode
*  ResultTrees, this property lists field names that will be sent to the server if they
*  are present in the criteria.
* @type {Array<Partial<string>>}
* @default null
*/
serverFilterFields?: Array<Partial<string>>;

/**
* With TreeGrid.loadDataOnDemand:true, TreeGrids fetch data by selecting the child nodes of
*  each parent, which should be exact match, so we default to
*  autoFetchTextMatchStyle:"exact" when autoFetchData is true.
*  
*  See ListGrid.autoFetchTextMatchStyle for details.
* @type {TextMatchStyle}
* @default "exact"
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Should children be selected when parent is selected? And should parent be
*  selected when any child is selected?
* @type {boolean}
* @default false
*/
cascadeSelection?: boolean;

/**
* Should partially selected parents be shown with special icon?
* @type {boolean}
* @default false
*/
showPartialSelection?: boolean;

/**
* If specified, the selection object for this list will use this property to mark records
*  as selected. In other words, if this attribute were set to "isSelected"
*  any records in the listGrid data where "isSelected" is true
*  will show up as selected in the grid. Similarly if records are selected within the grid
*  after the grid has been created, this property will be set to true on the selected
*  records.
* @type {string}
* @default null
*/
selectionProperty?: string;

/**
* This property is not supported on TreeGrid, and only applies to the ListGrid
*  superclass.
* @type {boolean}
* @default null
*/
canSelectAll?: boolean;

/**
* For databound trees, use this attribute to supply a DataSourceField.rootValue for this
*  component's generated data object.
*  
*  This property allows you to have a particular component navigate a tree starting from any
*  given node as the root.
* @type {any}
* @default null
*/
treeRootValue?: any;

/**
* An array of field objects, specifying the order, layout, dynamic calculation, and
*  sorting behavior of each field in the treeGrid object. In TreeGrids, the fields
*  array specifies columns. Each field in the fields array is TreeGridField object.
*  
*  If TreeGrid.dataSource is also set, this value acts as a set of overrides as
*  explained in DataBoundComponent.fields.
* @type {Array<Partial<TreeGridField>>}
* @default null
*/
fields?: Array<Partial<TreeGridField>>;

/**
* Visible title for the tree column (field).
* @type {string}
* @default "Name"
*/
treeFieldTitle?: string;

/**
* If this grid was passed an explicit set of fields, but no field was specified as the
*  "tree-field" (showing indentations for tree hierarchy
*  and tree icons), should we assign one of the other fields to be the tree-field?
*  
*  When true, if we're showing a field for the Tree.titleProperty of the tree, 
*  this will be displayed as a Tree Field by default. If not, the first entry in the 
*  specified fields array will be used.
*  
*  This may be set to false to display a tree or partial tree as a flattened list within
*  a TreeGrid.
* @type {boolean}
* @default true
*/
autoAssignTreeField?: boolean;

/**
* Specifies whether the root node should be displayed in the treeGrid.
*  
*  This property is only available for "children" modelType trees, hence is not allowed for
*  trees that load data from the server dynamically via TreeGrid.fetchData. 
*  
*  To get the equivalent of a visible "root" node in a tree that loads data dynamically,
*  add a singular, top-level parent to the data. However, note that this top-level parent
*  will technically be the only child of root, and the implicit root object will be
*  returned by Tree.getRoot.
* @type {boolean}
* @default false
*/
showRoot?: boolean;

/**
* If specified, this attribute will override Tree.separateFolders on the
*  data for this treeGrid.
*  
*  Specifies whether folders and leaves should be segregated in the treeGrid display.
*  Use Tree.sortFoldersBeforeLeaves to customize whether folders appear before 
*  or after their sibling leaves.
*  
*  If unset, at the treeGrid level, the property can be set directly on
*  TreeGrid.data or for dataBound TreeGrids on the
*  TreeGrid.dataProperties.
* @type {boolean}
* @default null
*/
separateFolders?: boolean;

/**
* For a TreeGrid that uses a DataSource, these properties will be passed to
*  the automatically-created ResultTree. This can be used for various customizations such as
*  modifying the automatically-chosen Tree.parentIdField.
* @type {Tree}
* @default null
*/
dataProperties?: Tree;

/**
* If specified, this attribute will override Tree.sortFoldersBeforeLeaves on
*  the data for this treeGrid.
*  
*  Specifies whether when Tree.separateFolders is true, folders should be displayed
*  before or after their sibling leaves in a sorted tree. If set to true, with
*  sortDirection set to Array.ASCENDING, folders are displayed before their sibling leaves
*  and with sort direction set to Array.DESCENDING they are displayed after. To invert
*  this behavior, set this property to false.
* @type {boolean}
* @default null
*/
sortFoldersBeforeLeaves?: boolean;

/**
* Specifies the type of nodes displayed in the treeGrid.
* @type {DisplayNodeType}
* @default null
*/
displayNodeType?: DisplayNodeType;

/**
* For dataBound treeGrids this specifies the ResultTree.autoPreserveOpenState,
*  governing whether the open state of the tree should be preserved when new data
*  arrives due to cache invalidation.
* @type {PreserveOpenState}
* @default null
*/
autoPreserveOpenState?: PreserveOpenState;

/**
* Indicates whether records can be dragged from this listGrid and dropped elsewhere.
*  
*  NOTE: If canDragRecordsOut is initially enabled or might be
*  dynamically enabled after the grid is created, it may be desirable to disable
*  Canvas.useTouchScrolling so that touch-dragging a record starts
*  a drag operation rather than a scroll, but see the discussion of 
*  ListGrid.showDragHandles. If 
*  Canvas.disableTouchScrollingForDrag is set to true, then touch
*  scrolling will be disabled automatically. However, for
*  accessibility reasons, it is recommended to leave touch
*  scrolling enabled and provide an alternative set of controls that can be used to perform
*  drag and drop of records out of the grid.
* @type {boolean}
* @default false
*/
canDragRecordsOut?: boolean;

/**
* Indicates whether records can be dropped into this listGrid.
* @type {boolean}
* @default false
*/
canAcceptDroppedRecords?: boolean;

/**
* Indicates whether records can be reordered by dragging within this ListGrid.
*  
*  NOTE: If canReorderRecords is initially enabled or might be
*  ListGrid.setCanReorderRecords after the grid is created,
*  it may be desirable to disable Canvas.useTouchScrolling
*  so that touch-dragging a record starts a reorder operation rather than a scroll,
*  but see the discussion of ListGrid.showDragHandles. If 
*  Canvas.disableTouchScrollingForDrag is set to true, then touch
*  scrolling will be disabled automatically. However, for accessibility
*  reasons, it is recommended to leave touch scrolling enabled and provide an alternative
*  set of controls that can be used to perform drag-reordering of records.
* @type {boolean}
* @default false
*/
canReorderRecords?: boolean;

/**
* Whether drops are allowed on leaf nodes.
*  
*  Dropping is ordinarily not allowed on leaf nodes unless TreeGrid.canReorderRecords is
*  set. 
*  
*  The default action for a drop on a leaf node is to place the node in that leaf's parent
*  folder. This can be customized by overriding TreeGrid.folderDrop.
*  
*  Note that enabling canDropOnLeaves is usually only appropriate where you
*  intend to add a custom TreeGrid.folderDrop implementation that does not add a
*  child node under the leaf. If you want to add a child nodes to a leaf, instead of
*  enabling canDropOnLeaves, use empty folders instead - see Tree.isFolder for how
*  to control whether a node is considered a folder.
* @type {boolean}
* @default false
*/
canDropOnLeaves?: boolean;

/**
* If set this property allows the user to reparent nodes by dragging them from their
*  current folder to a new folder.
*  Backcompat: For backwards compatibility with versions prior to SmartClient 1.5,
*  if this property is unset, but this.canAcceptDroppedRecords is true, we
*  allow nodes to be dragged to different folders.
* @type {boolean}
* @default null
*/
canReparentNodes?: boolean;

/**
* Specifies what to do with data dragged from this TreeGrid (into another component, or
*  another node in this TreeGrid. The default action is to move the data. A setting of
*  "none" is not recommended for trees because Trees maintain the node open state on the nodes
*  themselves, and hence having multiple Tree objects share a reference to a node can have
*  unintended consequences (such as opening a folder in one tree also triggering an open in
*  another tree that shares the same node).
*  
*  See TreeGrid.folderDrop for a full explanation of default behaviors on drop, and how to
*  customize them.
* @type {DragDataAction}
* @default "move"
*/
dragDataAction?: DragDataAction;

/**
* Message displayed when user attempts to drag a node into a parent that already contains
*  a child of the same name/ID.
* @type {string}
* @default "This item already contains a child item with that name."
*/
parentAlreadyContainsChildMessage?: string;

/**
* Message displayed when user attempts to drop a dragged node onto itself.
* @type {string}
* @default "You can't drag an item into itself."
*/
cantDragIntoSelfMessage?: string;

/**
* Message displayed when user attempts to drop a node into a child of itself.
* @type {string}
* @default "You can't drag an item into one of it's children."
*/
cantDragIntoChildMessage?: string;

/**
* For Tree.isMultiLinkTree only, the message displayed when the 
*  user attempts to drag two or more occurences of the same node into a parent.
* @type {string}
* @default "You can't drag two occurences of the same node into a parent."
*/
cantDragMultipleNodeOccurencesMessage?: string;

/**
* The amount of indentation (in pixels) to add to a node's icon/title for each level
*  down in this tree's hierarchy.
*  
*  This value is ignored when TreeGrid.showConnectors is
*  true because fixed-size images are used to render the connectors.
* @type {number}
* @default "20"
*/
indentSize?: number;

/**
* The amount of gap (in pixels) between the extraIcon (see TreeGrid.getExtraIcon)
*  or checkbox icon and the TreeGrid.nodeIcon/
*  TreeGrid.folderIcon or node text.
* @type {Integer}
* @default "2"
*/
extraIconGap?: Integer;

/**
* The standard size (same height and width, in pixels) of node icons in this
*      treeGrid.
* @type {number}
* @default "16"
*/
iconSize?: number;

/**
* Default width and height in pixels of the opener icons, that is, the icons which show
*  the open or closed state of the node, typically a [+] or [-] symbol, if not overridden
*  by TreeGrid.openerIconWidth/TreeGrid.openerIconHeight.
*  
*  If TreeGrid.showConnectors is true, the opener icon includes the connector line, and
*  defaults to ListGrid.cellHeight.
*  
*  Otherwise, openerIconSize defaults to TreeGrid.iconSize.
* @type {number}
* @default null
*/
openerIconSize?: number;

/**
* Width in pixels of the opener icons, that is, the icons which show the open or closed
*  state of the node, typically a [+] or [-] symbol.
*  
*  If not specified, TreeGrid.openerIconSize is used instead.
* @type {number}
* @default null
*/
openerIconWidth?: number;

/**
* Height in pixels of the opener icons, that is, the icons which show the open or closed
*  state of the node, typically a [+] or [-] symbol.
*  
*  If not specified, TreeGrid.openerIconSize is used instead.
* @type {number}
* @default null
*/
openerIconHeight?: number;

/**
* If set, when a folder is loading its children from the server (Tree.getLoadState
*  returns "loading"), it uses a distinct icon image given by TreeGrid.loadingIcon. This is
*  typically used to show a small animating "spinner" icon to let the user know data is being
*  fetched.
* @type {boolean}
* @default true
*/
showLoadingIcons?: boolean;

/**
* If TreeGrid.showLoadingIcons is set, this icon will be used when the folder is 
*  Tree.getLoadState.
* @type {SCImgURL}
* @default "[SKIN]folder_loading.gif"
*/
loadingIcon?: SCImgURL;

/**
* The URL of the base icon for all folder nodes in this treeGrid. Note that this URL will
*  have TreeGrid.openIconSuffix, TreeGrid.closedIconSuffix or 
*  TreeGrid.dropIconSuffix appended to indicate state changes if appropriate - 
*  see documentation on TreeGrid.showOpenIcons, TreeGrid.showSelectedIcons
*  and TreeGrid.showDropIcons.
* @type {SCImgURL}
* @default "[SKIN]folder.gif"
*/
folderIcon?: SCImgURL;

/**
* If TreeGrid.showDropIcons is true, this suffix will be appended to the
*  TreeGrid.folderIcon when the user drop-hovers over some folder.
* @type {string}
* @default "drop"
*/
dropIconSuffix?: string;

/**
* If TreeGrid.showOpenIcons is true, this suffix will be appended to the
*  TreeGrid.folderIcon for open folders in this grid.
* @type {string}
* @default "open"
*/
openIconSuffix?: string;

/**
* This suffix will be appended to the TreeGrid.folderIcon for closed folders.
*  If TreeGrid.showOpenIcons is set to false this suffix will also be
*  appended to open folders' icons.
* @type {string}
* @default "closed"
*/
closedIconSuffix?: string;

/**
* If TreeGrid.showSelectedIcons is true, this suffix will be appended to the
*  TreeGrid.folderIcon for selected nodes in this grid.
* @type {string}
* @default "selected"
*/
selectedIconSuffix?: string;

/**
* The filename of the default icon for all leaf nodes in this grid. To specify a 
*  custom image for an individual node, set the TreeGrid.customIconProperty directly on
*  the node.
* @type {SCImgURL}
* @default "[SKIN]file.gif"
*/
nodeIcon?: SCImgURL;

/**
* If true, show a different icon for open folders than closed folders.
*  This is achieved by appending the TreeGrid.openIconSuffix onto the 
*  TreeGrid.folderIcon URL [for example "[SKIN]/folder.gif" might be 
*  replaced by "[SKIN]/folder_open.gif".
*  Note If this property is set to false the same icon is shown for
*  open folders as for closed folders, unless a custom folder icon was specified. This will be
*  determined by TreeGrid.folderIcon plus the TreeGrid.closedIconSuffix.
* @type {boolean}
* @default true
*/
showOpenIcons?: boolean;

/**
* If true, show a different icon for selected nodes than unselected nodes.
*  This is achieved by appending the TreeGrid.selectedIconSuffix onto the 
*  TreeGrid.folderIcon URL or TreeGrid.nodeIcon for selected records.
*  
*  If appropriate, this suffix will be combined with the 
*  TreeGrid.openIconSuffix or TreeGrid.closedIconSuffix (see 
*  TreeGrid.showOpenIcons. So a treeGrid with its folderIcon
*  property set to "[SKIN]/folder.gif", with both
*  showSelectedIcons and showOpenIcons set to true would show
*  an icon with the URL "[SKIN]/folder_open_selected.gif" for a
*  folder that was both selected and opened.
* @type {boolean}
* @default false
*/
showSelectedIcons?: boolean;

/**
* If true, when the user drags a droppable target over a folder in this TreeGrid, show 
*  a different folder icon.
*  This is achieved by appending the TreeGrid.dropIconSuffix onto the
*  TreeGrid.folderIcon URL (for example "[SKIN]/folder.gif" may be
*  replaced by "[SKIN]/folder_drop.gif").
* @type {boolean}
* @default true
*/
showDropIcons?: boolean;

/**
* This property allows the developer to rename the 
*  TreeNode.icon property.
* @type {string}
* @default "icon"
*/
customIconProperty?: string;

/**
* This property allows the developer to rename the 
*  TreeNode.showOpenIcon property.
* @type {string}
* @default "showOpenIcon"
*/
customIconOpenProperty?: string;

/**
* This property allows the developer to rename the 
*  TreeNode.showDropIcon property.
* @type {string}
* @default "showDropIcon"
*/
customIconDropProperty?: string;

/**
* This property allows the developer to rename the 
*  TreeNode.showSelectedIcon property.
* @type {string}
* @default "showSelectedIcon"
*/
customIconSelectedProperty?: string;

/**
* Should folder nodes showing custom icons (set via the TreeGrid.customIconProperty),
*  show open state images when the folder is opened.
*  If true, the TreeGrid.openIconSuffix will be appended to the image URL
*  (so "customFolder.gif" might be replaced with 
*  "customFolder_open.gif").
*  Note that the TreeGrid.closedIconSuffix is never appended to custom folder icons.
*  Can be overridden at the node level via the default property TreeNode.showOpenIcon
*  and that property can be renamed via TreeGrid.customIconOpenProperty.
* @type {boolean}
* @default false
*/
showCustomIconOpen?: boolean;

/**
* Should folder nodes showing custom icons (set via the TreeGrid.customIconProperty,
*  default TreeNode.icon),
*  show drop state images when the user is drop-hovering over the folder.
*  If true, the TreeGrid.dropIconSuffix will be appended to the image URL
*  (so "customFolder.gif" might be replaced with 
*  "customFolder_drop.gif").
*  Can be overridden at the node level via the default property TreeNode.showDropIcon
*  and that property can be renamed via TreeGrid.customIconDropProperty.
* @type {boolean}
* @default false
*/
showCustomIconDrop?: boolean;

/**
* Should folder nodes showing custom icons (set via the TreeGrid.customIconProperty),
*  show selected state images when the folder is selected, 
*  if TreeGrid.showSelectedIcons is true?
*  
*  If true, the TreeGrid.selectedIconSuffix will be appended to the image URL
*  (so "customFolder.gif" might be replaced with 
*  "customFolder_selected.gif").
*  Can be overridden at the node level via the default property TreeNode.showSelectedIcon
*  and that property can be renamed via TreeGrid.customIconSelectedProperty.
* @type {boolean}
* @default false
*/
showCustomIconSelected?: boolean;

/**
* Should tree nodes show a disabled checkbox 
*  ListGrid.selectionAppearance:"checkbox" 
*  is set on the treegrid, and a node can't be selected? 
*  
*  If set to false the treeGrid will use 
*  TreeGrid.leaveSelectionCheckboxGap to determine whether to leave
*  a blank space where the checkbox would normally appear.
* @type {boolean}
* @default false
*/
showDisabledSelectionCheckbox?: boolean;

/**
* If ListGrid.selectionAppearance:"checkbox" 
*  is set on the treegrid, and a node can't be selected, should a gap be left where
*  the checkbox icon would normally appear, in order to make the node's icon and title
*  line up with the content for other nodes in the same parent?
*  
*  Has no effect if TreeGrid.showDisabledSelectionCheckbox is true
* @type {boolean}
* @default true
*/
leaveSelectionCheckboxGap?: boolean;

/**
* The filename of the default icon for all open folder nodes in this treeGrid.
* @deprecated * as part of SmartClient release 5.5 in favor of TreeGrid.folderIcon

* @type {string}
* @default null
*/
folderOpenImage?: string;

/**
* The filename of the default icon for all closed folder nodes in this treeGrid. Use
*  the node.icon property (null by default) to specify a custom image for an individual
*  folder node. The same custom image will be used for both the open and closed folder
*  images.
* @deprecated * as part of SmartClient release 5.5 in favor of TreeGrid.folderIcon

* @type {string}
* @default null
*/
folderClosedImage?: string;

/**
* The filename of the icon displayed for a folder node that will accept drag-and-drop
*  data when the mouse is released.
* @deprecated * as part of SmartClient release 5.5 in favor of TreeGrid.folderIcon

* @type {string}
* @default null
*/
folderDropImage?: string;

/**
* The filename of the default icon for all leaf nodes in this treeGrid. Use the
*  node.icon property (null by default) to specify a custom image for an individual
*  node.
* @deprecated * as part of SmartClient release 5.5 in favor of TreeGrid.nodeIcon

* @type {SCImgURL}
* @default "[SKIN]file.gif"
*/
fileImage?: SCImgURL;

/**
* The filename of the icon displayed use as the default drag tracker when for multiple
*  files and/or folders are being dragged.
* @type {SCImgURL}
* @default "[SKIN]folder_file.gif"
*/
manyItemsImage?: SCImgURL;

/**
* Should this treeGrid show connector lines illustrating the tree's hierarchy?
*  
*  For the set of images used to show connectors, see TreeGrid.connectorImage.
*  
*  Note: in order for connector images to be perfectly connected, all styles for
*  cells must have no top or bottom border or padding. If you see small gaps in connector
*  lines, check your CSS files. See the example below for an example of correct
*  configuration, including example CSS.
* @type {boolean}
* @default false
*/
showConnectors?: boolean;

/**
* If TreeGrid.showConnectors is true, this property determines whether we should show
*  vertical continuation lines for each level of indenting within the tree. Setting to
*  false will show only the hierarchy lines are only shown for the most indented path ("sparse"
*  connectors).
* @type {boolean}
* @default true
*/
showFullConnectors?: boolean;

/**
* Should the opener icon be displayed next to folder nodes? This is an icon
*  which visually indicates whether the folder is opened or closed (typically via
*  a [+] or [-] image, or a turn-down arrow) and may be clicked to expand or collapse
*  the folder.
*  
*  For folders with no children, this icon is not shown unless 
*  TreeGrid.alwaysShowOpener is true. Note that for trees which
*  TreeGrid.loadDataOnDemand, we may not know if a folder
*  has any descendants if it has never been opened. As such we will show the
*  opener icon next to the folder. Once the user opens the icon and a fetch occurs,
*  if the folder is empty, and TreeGrid.alwaysShowOpener is false, the opener icon
*  will be hidden.
*  
*  For more information on load on demand trees, and how we determine whether
*  a node is a a folder or a leaf, please refer to the treeDataBinding
*  documentation.
*  
*  The opener icon URL is derived from the specified
*  TreeGrid.openerImage or TreeGrid.connectorImage depending on
*  TreeGrid.showConnectors. If TreeGrid.showSelectedOpener is specified
*  a separate opener icon will be displayed for selected nodes.
* @type {boolean}
* @default true
*/
showOpener?: boolean;

/**
* If TreeGrid.showOpener is true, should we display the opener icon
*  for folders even if they have no children?
*  
*  Note that for trees which
*  TreeGrid.loadDataOnDemand, we may not know if a folder
*  has any descendants if it has never been opened. As such we will show the
*  opener icon next to the folder. Once the user opens the icon and a fetch occurs,
*  if the folder is empty, and this property is false, the opener icon
*  will be hidden.
*  
*  For more information on load on demand trees, and how we determine whether
*  a node is a a folder or a leaf, please refer to the treeDataBinding
*  documentation.
* @type {boolean}
* @default false
*/
alwaysShowOpener?: boolean;

/**
* a If TreeGrid.showOpener is true, should a different opener icon be displayed
*  for selected nodes? This provides a way for developers to show a "selected"
*  version of the opener icon set which looks optimal with the 
*  cellStyleSuffixes applied the selected record.
*  
*  The selected icon URL is created by appending the suffix 
*  "_selected" to the TreeGrid.openerImage or
*  TreeGrid.connectorImage.
* @type {boolean}
* @default false
*/
showSelectedOpener?: boolean;

/**
* The base filename of the opener icon for the folder node when 'showConnectors' is false
*  for this TreeGrid.
*  The opener icon is displayed beside the folder icon in the Tree column for folder nodes.
*  Clicking on this icon will toggle the open state of the folder.
*  The filenames for these icons are assembled from this base filename and the state of the
*  node, as follows:
*  If the openerImage is set to {baseName}.{extension}, 
*  {baseName}_opened.{extension} will be displayed next to opened folders, and
*  {baseName}_closed.{extension} will be displayed next to closed folders, or
*  if this page is in RTL mode, {baseName}_opened_rtl.{extension} and
*  {baseName}_closed_rtl.{extension} will be used.
*  
*  If TreeGrid.showSelectedOpener is true the URL for selected nodes will append
*  the string "_selected" to the image URLs described above.
*  So for an openerImage set to {baseName}.{extension}, the URLs
*  for selected records would be
*  {baseName}_opened_selected.{extension}, 
*  {baseName}_closed_selected.{extension}, etc.
* @type {SCImgURL}
* @default "[SKIN]opener.gif"
*/
openerImage?: SCImgURL;

/**
* The base filename for connector icons shown when TreeGrid.showConnectors is true.
*  Connector icons are rendered into the title field of each row and show the dotted
*  hierarchy lines between siblings of the same parent node. For each node, a connector icon
*  may be shown:
*  
* - As an opener icon for folder nodes, next to the folder icon
*  
* - In place of an opener icon for leaf nodes, next to the leaf icon
*  
* - As a standalone vertical continuation line in the indent to the left of the node, to show
*    a connection between some ancestor node's siblings (only relevant if
*    TreeGrid.showFullConnectors is true).
*  
*  Note that TreeGrid.showFullConnectors governs whether connector lines will be
*  displayed for all indent levels, or just for the innermost level of the tree.
*  
*  The filenames for these icons are assembled from this base filename and the state of the
*  node. Assuming the connectorImage is set to {baseName}.{extension}, the
*  full set of images to be displayed will be:
*  
*  {baseName}_ancestor[_rtl].{extension} if TreeGrid.showFullConnectors
*  is true, this is the URL for the vertical continuation image to be displayed at the
*  appropriate indent levels for ancestor nodes with subsequent children.
*  
*  For nodes with no children:
*  
*  
* - {baseName}_single[_rtl].{extension}: Shown when there is no connector line
*  attached to the parent or previous sibling, and no connector line to the next sibling. For
*  TreeGrid.showFullConnectors trees, there will always be a
*  connector leading to the parent or previous sibling if its present in the tree so this
*  icon can only be displayed for the first row.
*  
* - {baseName}_start[_rtl].{extension}: Shown when the there is no connector
*  line attached to the parent or previous sibling, but there is a connector to the next
*  sibling. As with _single this will only ever be used for the first row if
*  TreeGrid.showFullConnectors is true
*  
* - {baseName}_end[_rtl].{extension}: Shown if we are not showing a connector 
*  line attached to the next sibling of this node (but are showing a connection to the previous
*  sibling or parent).
*  
* - {baseName}_middle[_rtl].{extension}: Shown where the we have a connector
*  line leading to both the previous sibling (or parent) and the next sibling.
*  
*  For folders with children. Note that if TreeGrid.showFullConnectors is false, open
*  folders will never show a connector to subsequent siblings:
*  
*  
* - {baseName}_opened_single[_rtl].{extension} opened folder node with 
*  children when no connector line is shown attaching to either the folder's previous sibling
*  or parent, or to any subsequent siblings.
*  
* - {baseName}_opened_start[_rtl].{extension}: opened folder with children
*  when the there is no connector line attached to the parent or previous sibling, but there 
*  is a connector to the next sibling.
*  
* - {baseName}_opened_end[_rtl].{extension}: opened folder with children 
*  if we are not showing a connector line attached to the next sibling of this node (but are
*  showing a connection to the previous sibling or parent).
*  
* - {baseName}_opened_middle[_rtl].{extension}: opened folder with children 
*  where the we have a connector line leading to both the previous sibling (or parent) and the
*  next sibling.
*  
*  
*  
* - {baseName}_closed_single[_rtl].{extension} closed folder node with 
*  children when no connector line is shown attaching to either the folder's previous sibling
*  or parent, or to any subsequent siblings.
*  
* - {baseName}_closed_start[_rtl].{extension}: closed folder with children
*  when the there is no connector line attached to the parent or previous sibling, but there 
*  is a connector to the next sibling.
*  
* - {baseName}_closed_end[_rtl].{extension}: closed folder with children 
*  if we are not showing a connector line attached to the next sibling of this node (but are
*  showing a connection to the previous sibling or parent).
*  
* - {baseName}_closed_middle[_rtl].{extension}: closed folder with children 
*  where the we have a connector line leading to both the previous sibling (or parent) and the
*  next sibling.
*  
*  (Note '[_rtl]' means that "_rtl" will be attached if isRTL() is true for this widget).
*  
*  If TreeGrid.showSelectedOpener is true the URL for selected nodes will append
*  the string "_selected" to the image URLs described above.
*  So for a connectorImage set to {baseName}.{extension}, the URLs
*  for selected records would be
*  {baseName}_ancestor[_rtl]_selected.{extension}, 
*  {baseName}_single[_rtl]_selected.{extension}, etc.
* @type {SCImgURL}
* @default "[SKIN]connector.gif"
*/
connectorImage?: SCImgURL;

/**
* For TreeGrids with loadDataOnDemand: true, a message to show the user if an attempt is 
*  made to open a folder, and thus load that node's children, while we are offline and 
*  there is no offline cache of that data. The message will be presented to the user in 
*  in a pop-up dialog box.
* @type {string}
* @default "This data not available while offline"
*/
offlineNodeMessage?: string;

/**
* For record components placed "within" the TreeGridField.treeField
*  column, should the component be indented to the position where a title would normally
*  show?
*  
*  For more general placement of embedded components, see
*  ListGrid.addEmbeddedComponent.
* @type {boolean}
* @default true
*/
indentRecordComponents?: boolean;

/**
* If no fields are specified, create a single field with 
*  TreeGridField.treeField set to true to show the tree.
*  
*  This automatically generated field will display values derived by calling
*  TreeGrid.getNodeTitle, and have the column title set to the specified
*  TreeGrid.treeFieldTitle.
*  
*  Has no effect if fields are explicitly specified.
*  
*  This is a convenience setting to allow a TreeGrid to be created without specifying a
*  field list. If fields are specified, refer to the documentation on property
*  TreeGrid.autoAssignTreeField for a way to automatically have one of the fields be
*  use as the tree field if no fields have TreeGridField.treeField set.
*  
*  For databound treeGrids, if there is no explicit fields array specified, developers
*  who wish to pick up all fields from the DataSource definition rather than displaying
*  this single automatically generated tree field may 
*  either set this property to false, or set TreeGrid.useAllDataSourceFields
*  to true.
* @type {boolean}
* @default true
*/
createDefaultTreeField?: boolean;

/**
* If true, the set of fields given by the "default binding" (see 
*  DataBoundComponent.fields) is used, with any fields specified in
*  component.fields acting as overrides that can suppress or modify the
*  display of individual fields, without having to list the entire set of fields that
*  should be shown.
*  
*  If component.fields contains fields that are not found in the DataSource,
*  they will be shown after the most recently referred to DataSource field. If the new
*  fields appear first, they will be shown first.
*  
*  ${isc.DocUtils.linkForExampleId('validationFieldBinding', 'This example')} shows a mixture of component
*  fields and DataSource fields, and how they interact for validation.
*  
*  This setting may be cleared if a FieldPicker is used to edit the component's field
*  order.
* @type {boolean}
* @default null
*/
useAllDataSourceFields?: boolean;

/**
* If true, when folders are opened / closed children will be animated into view.
*  
*  Folder animations are automatically disabled if ListGrid.autoFitData is set to "vertical" or "both", or 
*  if ListGrid.showRecordComponents are used.
* @type {boolean}
* @default true
*/
animateFolders?: boolean;

/**
* If TreeGrid.animateFolders is true for this grid, this number can be set to designate
*  the maximum number of rows to animate at a time when opening / closing a folder.
* @type {Integer}
* @default null
*/
animateFolderMaxRows?: Integer;

/**
* When animating folder opening / closing, if TreeGrid.animateFolderSpeed is not
*  set, this property designates the duration of the animation in ms.
* @type {number}
* @default "100"
*/
animateFolderTime?: number;

/**
* When animating folder opening / closing, this property designates the speed of the
*  animation in pixels shown (or hidden) per second. Takes precedence over the 
*  TreeGrid.animateFolderTime property, which allows the developer to specify a
*  duration for the animation rather than a speed.
* @type {number}
* @default "3000"
*/
animateFolderSpeed?: number;

/**
* When animating folder opening / closing, this property can be set to apply an
*  animated acceleration effect. This allows the animation speed to be "weighted", for
*  example expanding or collapsing at a faster rate toward the beginning of the animation
*  than at the end.
* @type {AnimationAcceleration}
* @default null
*/
animateFolderEffect?: AnimationAcceleration;

/**
* If TreeGrid.canAcceptDroppedRecords is true for this treeGrid, this property governs
*  whether the user can drop between, or over records within the grid.
*  This controls what RecordDropPosition is passed to the TreeGrid.recordDrop
*  event handler.
* @type {RecordDropAppearance}
* @default "both"
*/
recordDropAppearance?: RecordDropAppearance;







/**
*  Opens the folder specified by node if it's closed, and closes it if it's open.
*  TreeGrid will redraw if there's a change in the folder's open/closed state.
* 
* 
* @param {TreeNode | string | Integer | NodeLocator} the node in question, or the                                the node's ID, or a NodeLocator                                object
*/
toggleFolder(node:TreeNode | string | Integer | NodeLocator): void;

/**
*  Opens a folder.
*  
*  Executed when a folder node receives a 'doubleClick' event.
*  If you override this method, the single parameter passed will be
*  a reference to the relevant folder node in the tree's data.
*  
*  See the ListGrid Widget Class for inherited recordClick and recordDoubleClick events.
* 
* 
* @param {TreeNode} node to open
* @param {string=} optional parameter containing the full path to the node.                   This is essential context for a                    Tree.multiLinkTree, but is not                    required in ordinary trees
*/
openFolder(node:TreeNode, path?:string): void;

/**
*  Closes a folder.
* 
* 
* @param {TreeNode} node to close
*/
closeFolder(node:TreeNode): void;

/**
*  Should this folder be animated when opened / closed? Default implementation will
*  return true if TreeGrid.animateFolders is true, the folder being actioned
*  has children and the child-count is less than the result of
*  TreeGrid.getAnimateFolderMaxRows.
* 
* @param {TreeNode} folder being opened or closed.
*/
shouldAnimateFolder(folder:TreeNode): boolean;

/**
*  If TreeGrid.animateFolders is true for this treeGrid, this method returns the 
*  the maximum number of rows to animate at a time when opening / closing a folder.
*  This method will return TreeGrid.animateFolderMaxRows if set. Otherwise
*  the value will be calculated as 3x the number of rows required to fill a viewport,
*  capped at a maximum value of 75.
* 
*/
getAnimateFolderMaxRows(): Integer;

/**
*  Returns a snapshot of the current open state of this grid's data as
*  a TreeGridOpenState object.
*  This object can be passed to TreeGrid.setOpenState to open the same set of folders
*  within the treeGrid's data (assuming the nodes are still present in the data).
* 
*/
getOpenState(): TreeGridOpenState;

/**
*  Reset this set of open folders within this grid's data to match the 
*  TreeGridOpenState object passed in.
*  Used to restore previous state retrieved from the grid by a call to 
*  TreeGrid.getOpenState.
* 
* 
* @param {TreeGridOpenState} Object describing the desired set of open folders.
*/
setOpenState(openState:TreeGridOpenState): void;

/**
*  Returns a snapshot of the current selection within this treeGrid as 
*  a ListGridSelectedState object.
*  This object can be passed to TreeGrid.setSelectedPaths to reset this grid's selection
*  the current state (assuming the same data is present in the grid).
* 
*/
getSelectedPaths(): ListGridSelectedState;

/**
*  Reset this grid's selection to match the ListGridSelectedState object passed in.
*  Used to restore previous state retrieved from the grid by a call to 
*  TreeGrid.getSelectedPaths.
* 
* 
* @param {ListGridSelectedState} Object describing the desired selection state of                       the grid
*/
setSelectedPaths(selectedPaths:ListGridSelectedState): void;

/**
*  Overridden to return a TreeGridViewState object for the grid.
* 
*/
getViewState(): TreeGridViewState;

/**
*  Overridden to take a TreeGridViewState object.
* 
* 
* @param {TreeGridViewState} Object describing the desired view state for the grid
*/
setViewState(viewState:TreeGridViewState): void;

/**
*  Set the Tree object this TreeGrid will view and manipulate.
* 
* 
* @param {Tree} Tree to show
*/
setData(newData:Tree): void;

/**
*  Overridden to disallow editing of the TreeNode.name field of this grid's data
*  tree. Also disallows editing of the auto-generated tree field, which displays the result
*  of Tree.getTitle on the node.
* 
*/
canEditCell(): boolean;

/**
*  This inherited ListGrid.startEditingNew is not supported by the TreeGrid
*  since adding a new tree node arbitrarily at the end of the tree is usually not useful.
*  Instead, to add a new tree node and begin editing it, use either of these two strategies:
*  
*  
* - add a new node to the client-side Tree model via Tree.add, then use
*  ListGrid.startEditing to begin editing this node. Note that if using a DataSource, when the
*  node is saved, an "update" operation will be used since adding a node directly to the
*  client-side ResultTree effectively means a new node has been added server side.
*  
* - use DataSource.addData to immediately save a new node. Automatic cache sync
*  by the ResultTree will cause the node to be integrated into the tree. When the
*  callback to addData() fires, locate the new node by matching primary key and call
*  ListGrid.startEditing to begin editing it.
*  
* 
* 
* @param {object=} Optional initial set of properties for the new record
* @param {boolean=} Whether to suppress the default behavior of moving focus                  to the newly shown editor.
*/
startEditingNew(newValues?:object, suppressFocus?:boolean): void;

/**
* 
*  This override to ListGrid.rowClick. This implementation calls through to the
*  TreeGrid.nodeClick, TreeGrid.folderClick, TreeGrid.leafClick methods, as
*  appropriate unless the click was on the expand/collapse control of a folder - in which case
*  those callbacks are not fired.
*  
*  Do not override this method unless you need a rowClick callback that fires even when the
*  user clicks on the expand/collapse control. If you do override this method, be sure to call
*  return this.Super("rowClick", arguments); at the end of your override to
*  preserver other handler that are called from the superclass (for example,
*  ListGrid.recordClick.
*  
* 
* 
* @param {TreeNode} record that was clicked on
* @param {number} index of the row where the click occurred
* @param {number} index of the col where the click occurred
*/
rowClick(record:TreeNode, recordNum:number, fieldNum:number): void;

/**
* 
*  Handle a doubleClick on a tree node - override of ListGrid stringMethod of same name. If
*  the node is a folder, this implementation calls TreeGrid.toggleFolder on it. If
*  the node is a leaf, calls TreeGrid.openLeaf on it.
*  
* 
*/
recordDoubleClick(): void;

/**
*  Executed when a leaf node receives a 'doubleClick' event. This handler must be
*  specified as a function, whose single parameter is a reference to the relevant leaf node in
*  the tree's data.
*  See the ListGrid Widget Class for inherited recordClick and recordDoubleClick events.
* 
* 
* @param {TreeNode} node to open
*/
openLeaf(node:TreeNode): void;

/**
* 
*  During a drag-and-drop interaction, this method is called to transfer a set of records that
*  were dropped onto some other component. This method is called after the set of records has
*  been copied to the other component. Whether or not this component's data is modified is 
*  determined by the value of DataBoundComponent.dragDataAction.
*  
*  With a dragDataAction of "move", a databound component will issue "remove"
*  dsRequests against its DataSource to actually remove the data, via
*  DataSource.removeData.
* 
* 
*/
transferDragData(): Array<any>;

/**
*  Return an icon to display as a drag tracker when the user drags some node(s).
*  Default implementation:
*  If multiple nodes are selected and TreeGrid.manyItemsImage is defined, this
*  image will be returned.
*  Otherwise returns the result of TreeGrid.getIcon for the first node being 
*  dragged.
*  
*  Note: Only called if ListGrid.dragTrackerMode is set to "icon". 
* 
* @param {Array<Partial<ListGridRecord>>} Records being dragged
*/
getDragTrackerIcon(records:Array<Partial<ListGridRecord>>): string;

/**
* 
*  NOTE: Applicable only to Tree.multiLinkTree; if called on a 
*  regular TreeGrid, returns an empty array.
*  
*  During a drag-and-drop interaction, this method returns the set of node occurences being 
*  dragged out of the component, wrapped inside NodeLocators. In the default 
*  implementation, this is the list of currently selected node occurences
* 
* 
* @param {TreeGrid} source grid from which the records will be transferred
*/
getDraggedNodeLocators(source:TreeGrid): Array<Partial<NodeLocator>>;

/**
*  
*  This method overrides ListGrid.willAcceptDrop and works as follows:
*  
*  First, ListGrid.willAcceptDrop (the superclass definition) is consulted. If it
*  returns false, then this method returns false immediately.
*  This handles the following cases:
*  - reordering of records withing this TreeGrid when ListGrid.canReorderRecords is true
*  - accepting dropped records from another dragTarget when ListGrid.canAcceptDroppedRecords 
*   is true and the dragTarget gives us a valid set of records to drop into place.
*  - disallowing drop over disabled nodes, or nodes with canAcceptDrop:false
*  
*  This method will also return false if the drop occurred over a leaf node whos immediate 
*  parent has canAcceptDrop set to false
*  If TreeGrid.canReparentNodes is true, and the user is dragging a node from one folder
*  to another, this method will return true to allow the change of parent folder.
*  
*  Otherwise this method returns true.
* 
* 
*/
willAcceptDrop(): boolean;

/**
*  Returns the row number of the most recent mouse event.
* 
* @param {Integer=} optional y-coordinate to obtain row number, in lieu of the y            coordinate of the last mouse event
*/
getEventRow(y?:Integer): Integer;

/**
*  Simulates a drag / drop type transfer of the selected records in some other grid to this
*  treeGrid, without requiring any user interaction.
*  See the dragging documentation for an overview of grid drag/drop data
*  transfer.
* 
* @param {ListGrid} source grid from which the records will be transferred
* @param {TreeNode=} parent node into which records should be dropped - if null  records will be transferred as children of the root node.
* @param {Integer=} target index (drop position) within the parent folder
* @param {Callback=} optional callback to be fired when the transfer process has            completed. The callback will be passed a single parameter "records",            the list of nodes actually transferred to this component (it is called             "records" because this logic is shared with ListGrid).
*/
transferSelectedData(sourceGrid:ListGrid, folder?:TreeNode, index?:Integer, callback?:Callback): void;

/**
*  Return the horizontal alignment for cell contents. Default implementation will always
*  left-align the special TreeGridField.treeField [or right-align if the page is in
*  RTL mode] - otherwise will return ListGridField.cellAlign if specified, otherwise
*  ListGridField.align.
* 
* 
* 
* @param {ListGridRecord} this cell's record
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getCellAlign(record:ListGridRecord, rowNum:number, colNum:number): Alignment;

/**
* 
*  Returns the title to show for a node in the tree column. If the field specifies the
*  name attribute, then the current node[field.name] is returned.
*  Otherwise, the result of calling Tree.getTitle on the node is called.
*  
*  You can override this method to return a custom title for node titles in the tree column.
*  
*  Note: if a default tree field is generated for you by TreeGrid.createDefaultTreeField
*  being true, and you've overridden this method, it will be called with recordNum: -1
*  during sorting of the tree field, if the tree is ResultTree.fetchMode.
* 
* 
* @param {TreeNode} The node for which the title is being requested.
* @param {number} The index of the node.
* @param {string} The field for which the title is being requested.
*/
getNodeTitle(node:TreeNode, recordNum:number, field:string): HTMLString;

/**
*  Returns true if the last event occurred over the indented area or over the
*  open / close icon of a folder node in this TreeGrid. Returns false if the event
*  did not occur over a folder node.
* 
* 
*/
isOverOpenArea(): boolean;

/**
*  Returns true if the last event occurred over TreeGrid.getExtraIcon
*  for the current node.
*  
*  Returns false if the event did not occur over an extraIcon, or if no extraIcon is
*  showing for the node in question.
* 
* 
*/
isOverExtraIcon(): boolean;

/**
*  Get the appropriate open/close opener icon for a node. Returns null if TreeGrid.showOpener is
*  set to false.
* 
* 
* @param {TreeNode} tree node in question
*/
getOpenIcon(node:TreeNode): URL;

/**
*  Set the icon for a particular treenode to a specified URL
* 
* 
* @param {TreeNode} tree node
* @param {SCImgURL} path to the resource
*/
setNodeIcon(node:TreeNode, icon:SCImgURL): void;

/**
* 
*  This method is called when a folder is opened either via the user manipulating the
*  expand/collapse control in the UI or via TreeGrid.openFolder. You can return
*  false to cancel the open.
* 
* 
* @param {TreeNode} the folder (record) that is being opened
* @param {string=} optional parameter containing the full path to the node.             This is essential context for a              Tree.multiLinkTree, but is not              required in ordinary trees
*/
folderOpened(node:TreeNode, path?:string): boolean;

/**
* 
*  This method is called when a folder is closed either via the user manipulating the
*  expand/collapse control in the UI or via TreeGrid.closeFolder. You can return
*  false to cancel the close.
* 
* 
* @param {TreeNode} the folder (record) that is being closed
*/
folderClosed(node:TreeNode): boolean;

/**
* 
*  This method is called when a folder record is clicked on.
* 
* 
* @param {TreeGrid} The TreeGrid on which folderClick() occurred.
* @param {TreeNode} The folder (record) that was clicked
* @param {number} Index of the row where the click occurred.
*/
folderClick(viewer:TreeGrid, folder:TreeNode, recordNum:number): void;

/**
* 
*  This method is called when a leaf record is clicked on.
* 
* 
* @param {TreeGrid} The TreeGrid on which leafClick() occurred.
* @param {TreeNode} The leaf (record) that was clicked
* @param {number} Index of the row where the click occurred.
*/
leafClick(viewer:TreeGrid, leaf:TreeNode, recordNum:number): void;

/**
* 
*  This method is called when a leaf or folder record is clicked on. Note that if you set
*  up a callback for nodeClick() and e.g. TreeGrid.leafClick, then
*  both will fire (in that order) if a leaf is clicked on.
* 
* 
* @param {TreeGrid} The TreeGrid on which leafClick() occurred.
* @param {TreeNode} The node (record) that was clicked
* @param {number} Index of the row where the click occurred.
*/
nodeClick(viewer:TreeGrid, node:TreeNode, recordNum:number): void;

/**
* 
*  This method is called when a context click occurs on a folder record.
* 
* 
* @param {TreeGrid} The TreeGrid on which the contextclick occurred.
* @param {TreeNode} The folder (record) on which the contextclick occurred.
* @param {number} Index of the row where the contextclick occurred.
*/
folderContextClick(viewer:TreeGrid, folder:TreeNode, recordNum:number): boolean;

/**
* 
*  This method is called when a context click occurs on a leaf record.
* 
* 
* @param {TreeGrid} The TreeGrid on which the contextclick occurred.
* @param {TreeNode} The leaf (record) on which the contextclick occurred.
* @param {number} Index of the row where the contextclick occurred.
*/
leafContextClick(viewer:TreeGrid, leaf:TreeNode, recordNum:number): boolean;

/**
* 
*  This method is called when a context click occurs on a leaf or folder record. Note that
*  if you set up a callback for nodeContextClick() and
*  e.g. TreeGrid.leafContextClick, then both will fire (in that order) if a leaf
*  is contextclicked - unless nodeContextClick() returns false, in which case
*  no further contextClick callbacks will be called.
* 
* 
* @param {TreeGrid} The TreeGrid on which the contextclick occurred.
* @param {TreeNode} The node (record) on which the contextclick occurred.
* @param {number} Index of the row where the contextclick occurred.
*/
nodeContextClick(viewer:TreeGrid, node:TreeNode, recordNum:number): boolean;

/**
*  Notification method fired whenever this TreeGrid receives new data nodes from the 
*  dataSource. Only applies to databound TreeGrids where TreeGrid.data is a 
*  ResultTree - either explicitly created and applied via TreeGrid.setData or
*  automatically generated via a TreeGrid.fetchData call.
*  
*  Note that dataArrived(), unlike TreeGrid.dataChanged, only fires in limited
*  circumstances - when data for a ResultTree arrives from the server due to a fetch
*  or cache invalidation, or as a result of filtering. If you want to catch all data
*  changes, you should instead react to TreeGrid.dataChanged.
* 
* 
* @param {TreeNode} The parentNode for which children were just loaded
*/
dataArrived(parentNode:TreeNode): void;

/**
*  Notification method fired when the TreeGrid's data changes, for any reason.
*  If overridden (rather than Class.observe), you must Class.Super to ensure proper Framework behavior.
*  
*  Examples of why data changed might be:
*  
* - a call to ListGrid.addData, ListGrid.updateData, or ListGrid.removeData
*  
* - DataSource updates from the server for ResultTree data
*  (triggered by record editing, etc.)
*  
* - fetches arriving back from the server for ResultTree data
*  
* - programmatic changes to Tree data if made through APIs such as
*  Tree.add, Tree.remove, etc.
*  
* - cache invalidation
*  
* - filtering
*  
*  Calling TreeGrid.setData doesn't call this notification directly, but it may
*  fire if one of the above listed events is triggered (e.g. a server fetch for 
*  ResultTree data).
*  
*  Note that the operationType parameter is optional and will be passed and
*  contain the operation (e.g. "update") if this notification was triggered by a fetch,
*  an ListGrid.addData, ListGrid.updateData, or ListGrid.removeData, or a DataSource
*  update for ResultTree data (the first three reasons listed above) but otherwise
*  will be undefined.
* 
* 
* @param {string=} optionally passed operation causing the change
*/
dataChanged(operationType?:string): void;

/**
*  Get an additional icon to show between the open icon and folder/node icon for a particular 
*  node.
*  
*  NOTE: If ListGrid.selectionAppearance is "checkbox", this method will
*  NOT be called. Extra icons cannot be shown for that appearance.
* 
* 
* @param {TreeNode} tree node in question
*/
getExtraIcon(node:TreeNode): URL;

/**
*  Get the appropriate icon for a node.
*  
*  By default icons are derived from TreeGrid.folderIcon and TreeGrid.nodeIcon.
*  Custom icons for individual nodes can be overridden by setting the TreeGrid.customIconProperty
*  on a node.
*  
*  If you want to suppress icons altogether, provide an override of this method that simply
*  returns null.
*  
*  Note that the full icon URL will be derived by applying Canvas.getImgURL to the
*  value returned from this method.
* 
* 
* @param {TreeNode} tree node in question
* @param {Integer=} the row number of the node in the TreeGrid. This additional               context is required for Tree.multiLinkTree               because the same node can appear in multiple places
*/
getIcon(node:TreeNode, rowNum?:Integer): URL;

/**
*  The superclass event ListGrid.recordDrop does not fire on a TreeGrid, use
*  TreeGrid.folderDrop instead.
* 
* 
*/
recordDrop(): void;

/**
* 
*  Process a drop of one or more nodes on a TreeGrid folder.
*  
*  
*  This method can be overridden to provide custom drop behaviors and is a more appropriate
*  override point than the lower level Canvas.drop handler.
*  
*  
*  
*  The default behavior is to simply delegate to the TreeGrid.transferNodes method; thus, the 
*  correct way to perform a programmatic folder drop, with all the built-in behaviors described
*  below, is to call transferNodes()
*  
*  If this is a self-drop, nodes are simply reordered. An "update" operation will
*  be submitted to update the Tree.parentIdField field of the moved node(s). 
*  
*  For a drop from another widget, TreeGrid.transferDragData is called which,
*  depending on the TreeGrid.dragDataAction specified on the source
*  widget, may either remove the source nodes from the original list (dragDataAction:"move")
*  or just provide a copy to this tree (dragDataAction:"copy").
*  
*  In either case the new row(s) appear in the folder at the index
*  specified by the arguments of the same name.
*  
*  If this grid is databound, the new nodes will be added to the dataset by calling
*  DataSource.addData. Further, if the new nodes were dragged from another
*  databound component, and DataBoundComponent.addDropValues
*  is true, DataBoundComponent.getDropValues will be called for every item
*  being dropped.
*  
*  As a special case, if the sourceWidget is also databound and a
*  DataSourceField.foreignKey relationship is declared from the
*  sourceWidget's DataSource to this TreeGrid's DataSource, the interaction will
*  be treated as a "drag recategorization" use case such as files being placed in folders,
*  employees being assigned to teams, etc. "update" DSRequests will be submitted that
*  change the foreignKey field in the dropped records to point to the tree folder that was the
*  target of the drop. In this case no change will be made to the Tree data as such, only to
*  the dropped records. 
*  
*  For multi-record drops, Queuing is automatically used to combine all DSRequests into a
*  single HTTP Request (see QuickStart Guide, Server Framework chapter). This allows the
*  server to persist all changes caused by the drop in a single transaction (and this is
*  automatically done when using the built-in server DataSources with Power Edition and
*  above).
*  
*  If these default persistence behaviors are undesirable, 
*  return false to cancel them
*  , then implement your own behavior, 
*  typically by using grid.updateData() or addData() to add new records.
*  NOTE: the records you receive in this event are the actual Records from the source
*  component. Use DataSource.copyRecords to create a copy before modifying the records
*  or using them with updateData() or addData().
* 
* 
* @param {Array<Partial<TreeNode>>} List of nodes being dropped
* @param {TreeNode} The folder being dropped on
* @param {Integer} Within the folder being dropped on, the index at which the drop is          occurring. Only passed if TreeGrid.canReorderRecords is true.
* @param {Canvas} The component that is the source of the nodes (where the nodes               were dragged from)
*/
folderDrop(nodes:Array<Partial<TreeNode>>, folder:TreeNode, index:Integer, sourceWidget:Canvas): void;

/**
* 
*  Transfer a list of TreeNodes from another component (does not have to be a databound
*  component) into this component. This method is only applicable to list-type components,
*  such as ListGrid, TreeGrid or TileGrid.
*  
*  This method implements the automatic drag-copy and drag-move behavior and calling it is
*  equivalent to completing a drag and drop of the nodes (the default 
*  TreeGrid.folderDrop implementation simply calls transferNodes())
*  
*  Note that this method is asynchronous - it may need to perform server turnarounds to prevent
*  duplicates in the target component's data. If you wish to be notified when the transfer
*  process has completed, you can either pass the optional callback to this method or implement
*  the DataBoundComponent.dropComplete method on this component.
*  
*  For a TreeGrid, see also TreeGrid.transferSelectedData.
* 
* 
* @param {Array<Partial<TreeNode>>} Nodes to transfer to this component
* @param {TreeNode} The target folder (eg, of a drop interaction), for context
* @param {Integer} Insert point within the target folder data for the transferred nodes
* @param {Canvas} The databound or non-databound component from which the nodes               are to be transferred.
* @param {Callback=} optional callback to be fired when the transfer process has            completed. The callback will be passed a single parameter "records",            the list of nodes actually transferred to this component (it is called            "records" because this is logic shared with ListGrid)
*/
transferNodes(nodes:Array<Partial<TreeNode>>, folder:TreeNode, index:Integer, sourceWidget:Canvas, callback?:Callback): void;

/**
*  When the user is dragging a droppable element over this grid, this method returns the folder
*  which would contain the item if dropped. This is the current drop node if the user is hovering
*  over a folder, or the node's parent if the user is hovering over a leaf.
* 
*/
getDropFolder(): Node | NodeLocator;

/**
*  Uses a "fetch" operation on the current DataSource to retrieve data
*  that matches the provided criteria, and displays the matching data in this component as a
*  tree.
*  
*  This method will create a ResultTree to manage tree data, which will
*  subsequently be available as treeGrid.data. DataSource records
*  returned by the "fetch" operation are linked into a tree structure according to
*  DataSourceField.primaryKey and
*  DataSourceField.foreignKey declarations on DataSource fields. See the
*  treeDataBinding topic for complete details.
*  
*  By default, the created ResultTree will use folder-by-folder load on demand, asking the
*  server for the children of each folder as the user opens it.
*  
*  The ResultTree created by fetchData() can be customized by setting
*  ListGrid.dataProperties to an Object containing properties and methods to apply to
*  the created ResultTree. For example, the property that determines whether a node is a
*  folder (Tree.isFolderProperty) can be customized, or
*  level-by-level loading can be disabled via
*  ResultTree.loadDataOnDemand.
*  
*  If TreeGrid.loadDataOnDemand is true, this grid will issue fetch requests each time the
*  user opens a folder to load its child data.
*  The criteria on this fetch request will consist of the appropriate value for the
*  foreignKey field, combined with the criteria passed to fetchData()
*  when the data was first loaded.
*  This allows you to retrieve multiple different tree structures from the same DataSource.
*  However note that the server is expected
*  to always respond with an intact tree - returned nodes which do not have parents are dropped
*  from the dataset and not displayed.
*  
*  The callback passed to fetchData will fire once, the first time that data is
*  loaded from the server. If using folder-by-folder load on demand, use the
*  ResultTree.dataArrived notification to be notified each time new nodes are loaded.
*  
*  Note that when calling 'fetchData()', changes to criteria may or may not result in a
*  DSRequest to the server due to client-side filtering (see ResultTree.fetchMode. 
*  You can call willFetchData(criteria) to determine if new criteria will result in a 
*  server fetch.
*  
*  If you need to force data to be re-fetched, you can call invalidateCache() and 
*  new data will automatically be fetched from the server using the current criteria 
*  and sort direction.
*  When using invalidateCache() there is no need to also call fetchData() and 
*  in fact this could produce unexpected results.
*  
* 
* 
* @param {Criteria=} Search criteria. If a DynamicForm is passed                     in as this argument instead of a raw criteria                      object, will be derived by calling                      DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required
* @param {DSRequest=} additional properties to set on the DSRequest                     that will be issued
*/
fetchData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Retrieves data that matches the provided criteria and displays the matching data in this
*  component.
*  
*  This method behaves exactly like TreeGrid.fetchData except that
*  DSRequest.textMatchStyle is automatically set to "substring" so that String-valued
*  fields are matched by case-insensitive substring comparison.
* 
* 
* @param {Criteria=} Search criteria.            If a DynamicForm is passed in as this argument            instead of a raw criteria object, will be derived by calling            DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required; see                      ListGrid.fetchData for details
* @param {DSRequest=} for databound components only - optional              additional properties to set on the DSRequest that will be issued
*/
filterData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Exports this component's data with client-side formatters applied, so is suitable for direct
*  display to users. See ListGrid.exportClientData for details of the general 
*  requirements and restrictions when exporting client data.
*  
*  The following notes apply when exporting client data from TreeGrids:
*  
* - Export only works correctly if you specify TreeGrid.fields; if you allow it to
*    generate a TreeGrid.createDefaultTreeField, nothing will be exported
*  
* - Only visible nodes are exported; if you close a node, its children are not exported 
*    even if they are loaded and known to the client
*  
* - Tree nodes are exported as a flat list, in the same order they are displayed in the 
*    TreeGrid
*  
*  
*  If your TreeGrid has custom formatters, formatted values will be exported by default, with
*  HTML normalized to text where possible. Since some levels of HTML normalizing aren't 
*  possible, this may result in missing or incorrect export values. In this case, you have 
*  two possible approaches:
*  
* - Set ListGridField.exportRawValues on the field. This will export
*    the raw underlying value of the field; your formatter will not be called
*  
* - Have your formatter call TreeGrid.isExportingClientData
*    and perform whatever alternative formatting you require if that method returns true
*  
* 
* @param {Partial<DSRequest>=} Request properties for the export. Note that specifying DSRequest.exportData on the request properties  allows the developer to pass in an explicit data set to export.
* @param {RPCCallback=} Optional callback. If you specify DSRequest.exportToClient: false in the request  properties, this callback will fire after export completes. Otherwise the callback will  fire right before the download request is made to the server.
*/
exportClientData(requestProperties?:Partial<DSRequest>, callback?:(response:RPCResponse, rawData:any, request:RPCRequest) => void): void;

/**
*  Returns true if this component is currently 
*  TreeGrid.exportClientData. This method can be called from
*  custom cell formatters if you need to return a different formatted value for an export
*  than for a live TreeGrid
* 
*/
isExportingClientData(): boolean;



// stubs from interface: DataBoundComponent
// -----------------------------------------------------------------------------------------------------------


/**
* Before we start editing values in this DataBoundComponent, should we perform a deep clone 
*  of the underlying values. See DataSource.deepCloneOnEdit for details of what this means.
*  
*  If this value is not explicitly set, it defaults to the value of DataSource.deepCloneOnEdit.
*  This value can be overridden per-field with DataSourceField.deepCloneOnEdit.
*  
*  Like the other deepCloneOnEdit settings, this flag only has an effect if you are 
*  editing a values object that contains nested objects or arrays, using 
*  Canvas.dataPaths.
* @type {boolean}
* @default null
*/
deepCloneOnEdit?: boolean;

/**
* When editing values in this DataBoundComponent, should we perform a deep clone of values 
*  that are not associated with a field (ie, attributes on the record that do not map to a 
*  component field either directly by name, or by FormItem.dataPath. If this value is
*  not explicitly set, it defaults to the value of 
*  DataSource.deepCloneNonFieldValuesOnEdit if there is a dataSource, or to the value 
*  of the static DataSource.deepCloneNonFieldValuesOnEdit if there is no
*  dataSource.
*  
*  Like the other deepCloneOnEdit settings, this flag only has an effect if you are 
*  editing a values object that contains nested objects or arrays.
* @type {boolean}
* @default null
*/
deepCloneNonFieldValuesOnEdit?: boolean;

/**
* When using DataBoundComponent.dataFetchMode, how many records to fetch at a time. If set to
*  a positive integer, dataPageSize will override the default 
*  ResultSet.resultSize for ResultSets automatically created when you call
*  ListGrid.fetchData (and similarly for the ResultTree.resultSize of
*  ResultTrees). Leaving dataPageSize at its default means to just use the default page
*  size of the data container.
*  
*  Note that regardless of the dataPageSize setting, a component will always fetch
*  all of data that it needs to draw. Settings such as
*  ListGrid.showAllRecords,
*  ListGrid.drawAllMaxCells and
*  ListGrid.drawAheadRatio can cause more rows than the configured
*  dataPageSize to be fetched.
* @type {Integer}
* @default null
*/
dataPageSize?: Integer;

/**
* Delay in milliseconds before fetching data.
*  
*  This setting only applies to the ResultSet automatically created by calling
*  ListGrid.fetchData. If a pre-existing ResultSet is passed to setData() instead, it's
*  existing setting for ResultSet.fetchDelay applies.
* @type {Integer}
* @default null
*/
dataFetchDelay?: Integer;

/**
* Whether to show fields marked hidden:true when a DataBoundComponent is given a
*  DataSource but no component.fields.
*  
*  The hidden property is used on DataSource fields to mark fields that are
*  never of meaning to an end user.
* @type {boolean}
* @default false
*/
showHiddenFields?: boolean;

/**
* Whether to show fields marked detail:true when a DataBoundComponent is 
*  given a DataSource but no component.fields.
*  
*  The detail property is used on DataSource fields to mark fields that 
*  shouldn't appear by default in a view that tries to show many records in a small space.
* @type {boolean}
* @default false
*/
showDetailFields?: boolean;

/**
* Whether to show fields of non-atomic types when a DataBoundComponent is given a
*  DataSource but no component.fields.
*  
*  If true, the component will show fields that declare a complex type, for example, a
*  field 'shippingAddress' that declares type 'Address', where 'Address' is the ID of a
*  DataSource that declares the fields of a shipping address (city, street name, etc).
*  
*  Such fields may need custom formatters or editors in order to create a usable interface,
*  for example, an Address field in a ListGrid might use a custom formatter to combine the
*  relevant fields of an address into one column, and might use a pop-up dialog for
*  editing.
* @type {boolean}
* @default true
*/
showComplexFields?: boolean;

/**
* DSRequest.operationId this component should use when performing fetch operations.
* @type {string}
* @default null
*/
fetchOperation?: string;

/**
* DSRequest.operationId this component should use when performing update operations.
* @type {string}
* @default null
*/
updateOperation?: string;

/**
* DSRequest.operationId this component should use when performing add operations.
* @type {string}
* @default null
*/
addOperation?: string;

/**
* DSRequest.operationId this component should use when performing remove operations.
* @type {string}
* @default null
*/
removeOperation?: string;

/**
* The list of field-names to export. If provided, the field-list in the exported output is 
*  limited and sorted as per the list.
*  
*  If exportFields is not provided, the exported output includes all visible fields 
*  from this component, sorted as they appear.
* @type {Array<Partial<string>>}
* @default null
*/
exportFields?: Array<Partial<string>>;

/**
* Setting exportAll to true prevents the component from passing it's list of fields to the 
*  export call. The result is the export of all visible fields from DataSource.fields.
*  
*  If exportAll is false, an export operation will first consider 
*  DataBoundComponent.exportFields, if it's set, and fall back on all visible fields from
*  DataSource.fields otherwise.
* @type {boolean}
* @default false
*/
exportAll?: boolean;

/**
* If Summary rows exist for this component, whether to include them when exporting client data.
* @type {boolean}
* @default true
*/
exportIncludeSummaries?: boolean;

/**
* Flag controlling when to recategorize records being dropped on a databound
*  component from another databound component.
* @type {RecategorizeMode}
* @default "checked"
*/
dragRecategorize?: RecategorizeMode;

/**
* If set, detect and prevent duplicate records from being transferred to this component, either via
*  drag and drop or via DataBoundComponent.transferSelectedData. When a duplicate transfer is detected,
*  a dialog will appear showing the DataBoundComponent.duplicateDragMessage.
*  
*  If the component either does not have a DataSource or has a DataSource with no
*  DataSourceField.primaryKey declared, duplicate checking is off by
*  default. If duplicate checking is enabled, it looks for an existing record in the dataset
*  that has all of the properties of the dragged record, and considers that a duplicate.
*  
*  For DragDataAction:"copy" where the target DataSource is related to the source
*  DataSource by foreignKey, a duplicate means that the target list, as filtered by the current
*  criteria, already has a record whose value for the foreignKey field matches the
*  primaryKey of the record being transferred.
*  
*  For example, consider dragging "employees" to "teams", where "teams" has a field
*  "teams.employeeId" which is a foreignKey pointing to "employees.id", and the target
*  grid has search criteria causing it to show all the members of one team. A duplicate -
*  adding an employee to the same team twice - is when the target grid's dataset contains an
*  record with "employeeId" matching the "id" field of the dropped employee.
* @type {boolean}
* @default null
*/
preventDuplicates?: boolean;

/**
* Message to show when a user attempts to transfer duplicate records into this component, and
*  DataBoundComponent.preventDuplicates is enabled.
*  
*  If set to null, duplicates will not be reported and the dragged duplicates will not be
*  saved.
* @type {string}
* @default "Duplicates not allowed"
*/
duplicateDragMessage?: string;

/**
* Indicates whether the text of the offlineMessage property should be displayed if no data is
*  available because we do not have a suitable offline cache
* @type {boolean}
* @default true
*/
showOfflineMessage?: boolean;

/**
* Message to display when this DataBoundComponent attempts to load data that is not available
*  because the browser is currently offline. Depending on the component, the message is either
*  displayed in the component's body, or in a pop-up warning dialog.
* @type {string}
* @default "This data not available while offline"
*/
offlineMessage?: string;

/**
* Message to display when this DataBoundComponent attempts to save data while the application
*  is offline.
* @type {string}
* @default "Data cannot be saved because you are not online"
*/
offlineSaveMessage?: string;

/**
* Indicates whether to add "drop values" to items dropped on this component, if both 
*      the source and target widgets are databound, either to the same DataSource or 
*      to different DataSources that are related via a foreign key. "Drop values" are 
*      properties of the dropped item that you wish to change (and persist) as a 
*      result of the item being dropped on this grid.
*      
*      If this value is true and this component is databound, DataBoundComponent.getDropValues will 
*      be called for every databound item dropped on this grid, and an update performed
*      on the item
* @type {boolean}
* @default true
*/
addDropValues?: boolean;

/**
* When an item is dropped on this component, and DataBoundComponent.addDropValues is true and both 
*      the source and target widgets are databound, either to the same DataSource or 
*      to different DataSources that are related via a foreign key, this object 
*      provides the "drop values" that SmartClient will apply to the dropped object 
*      before updating it.
*      
*      If this property is not defined, SmartClient defaults to returning the selection
*      criteria currently in place for this component. Thus, any databound items (for example, 
*      rows from other grids bound to the same DataSource) dropped on the grid will,
*      by default, be subjected to an update that makes them conform to the grid's 
*      current filter criteria.
* @type {object}
* @default null
*/
dropValues?: object;

/**
* Does this component represent singular or multiple "records" objects?
*  Options are "multiple" or "single", or "either"
* @type {string}
* @default "multiple"
*/
dataArity?: string;

/**
* Indicates whether or not this component will load its data 
*  DataSource.progressiveLoading.
* @type {boolean}
* @default null
*/
progressiveLoading?: boolean;

/**
* If this component is bound to a dataSource, this attribute may be specified to customize
*  what fields from the dataSource may be edited by default. For example the SearchForm
*  class has this attribute set to "canFilter" which allows search forms to edit
*  dataSource fields marked as canEdit:false (but not those marked as
*  canFilter:false).
*  
*  Note that if canEdit is explicitly specified on a field in 
*  the DataBoundComponent.fields array, that property will be respected in preference to 
*  the canEditAttribute value. (See FormItem.canEdit, ListGridField.canEdit).
*  Also note that individual dataBoundComponents may have additional logic around whether a field
*  can be edited - for example ListGrid.canEditCell may be overridden.
* @type {Identifier}
* @default "canEdit"
*/
canEditFieldAttribute?: Identifier;

/**
* The useFlatFields flag causes all simple type fields anywhere in a nested
*  set of DataSources to be exposed as a flat list for form binding. 
*  
*  useFlatFields is typically used with imported metadata, such as 
*  XMLTools.loadXMLSchema from a 
*  XMLTools.loadWSDL, as a means of eliminating levels of XML
*  nesting that aren't meaningful in a user interface, without the cumbersome and fragile
*  process of mapping form fields to XML structures.
*  
*  For example, having called WebService.getInputDS to retrieve the input message
*  schema for a web service operation whose input message looks like this:
*  
*  &lt;FindServices&gt;
*    &lt;searchFor&gt;search text&lt;/searchFor&gt;
*    &lt;Options&gt;
*      &lt;caseSensitive&gt;false&lt;/caseSensitive&gt;
*    &lt;/Options&gt;
*    &lt;IncludeInSearch&gt;
*      &lt;serviceName&gt;true&lt;/serviceName&gt;
*      &lt;documentation&gt;true&lt;/documentation&gt;
*      &lt;keywords&gt;true&lt;/keywords&gt;
*    &lt;/IncludeInSearch&gt;
*  &lt;/FindServices&gt;
*  
*  Setting useFlatFields on a DynamicForm that is bound to this input
*  message schema would result in 5 FormItem reflecting the 5 simple type
*  fields in the message.
*  
*  For this form, the result of DynamicForm.getValues might look
*  like:
*  
*  {
*   searchFor: "search text",
*   caseSensitive: false,
*   serviceName: true,
*   documentation : true,
*   keywords : true
*  }
*  When contacting a WebService, these values can be automatically
*  mapped to the structure of the input message for a web service operation by setting
*  WSRequest.useFlatFields (for use with WebService.callOperation) or by setting
*  DSRequest.useFlatFields (for use with a DataSource that is
*  wsdlBinding via
*  OperationBinding.wsOperation). 
*  
*  Using these two facilities in conjunction (component.useFlatFields and
*  request.useFlatFields) allows gratuitous nesting to be consistently bypassed in both the user
*  presentation and when providing the data for XML messages.
*  
*  You can also set OperationBinding.useFlatFields to automatically enable 
*  "flattened" XML serialization (request.useFlatFields) for all DataSource requests of a
*  particular operationType.
*  
*  Note that useFlatFields is not generally recommended for use with structures
*  where multiple simple type fields exist with the same name, however if used with such a
*  structure, the first field to use a given name wins. "first" means the first field
*  encountered in a depth first search. "wins" means only the first field will be present as a
*  field when data binding.
* @type {boolean}
* @default null
*/
useFlatFields?: boolean;

/**
* If this attribute is set to false, any attributes in the component's values object that do
*  not map to a DataSourceField or FormItem will not 
*  be tracked when checking for changes. You should only set this flag to false if you know 
*  that your code does not store additional, non-field values in the component's data, or if 
*  you do store such values, but you don't care that they are not checked for changes. 
*  This flag is primarily provided to avoid performance issues in cases where developers are
*  storing large numbers of extra attributes in component data; generally speaking, you should
*  only consider setting it to false if you have a use case like this.
*  
*  Note, even with this flag set to false, these extra values will still be managed and stored 
*  by SmartClient; they just will not be checked when the component's values are inspected to 
*  see if they have changed. This may lead to methods like ListGrid.rowHasChanges
*  returning false when you are expecting it to return true. In this case, either switch this 
*  flag back to true (or just do not set it false), or provide a field definition for the 
*  affected attribute(s).
* @type {boolean}
* @default true
*/
canChangeNonFieldValues?: boolean;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from DataBoundComponent.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only DataBoundComponent.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default false
*/
autoFetchData?: boolean;

/**
* Criteria to be used when DataBoundComponent.autoFetchData is set.
*  
*  This property supports dynamicCriteria - use Criterion.valuePath
*  to refer to values in the Canvas.ruleScope.
* @type {Criteria}
* @default null
*/
initialCriteria?: Criteria;

/**
* Criteria that are never shown to or edited by the user and are cumulative with any criteria
*  provided via DataBoundComponent.initialCriteria and related methods
* @type {Criteria}
* @default null
*/
implicitCriteria?: Criteria;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define
*  grid hilites using the HiliteEditor.
*  
*  User-added hilites can be persisted via DataBoundComponent.getHiliteState and 
*  DataBoundComponent.setHiliteState.
* @type {boolean}
* @default false
*/
canEditHilites?: boolean;

/**
* Hilites to be applied to the data for this component. See hiliting.
* @type {Array<Partial<Hilite>>}
* @default null
*/
hilites?: Array<Partial<Hilite>>;

/**
* Specifies a list of icons that can be used in DataBoundComponent.editHilites.
*  
*  hiliteIcons should be specified as an Array of SCImgURL.
*  When present, the hilite editing interface shown when DataBoundComponent.editHilites is called 
*  will offer the user a drop down for picking one of these icons when defining either a 
*  simple or advanced hilite rule.
*  
*  If the user picks an icon, the created hiliting rule will have Hilite.icon set to 
*  the chosen icon. DataBoundComponent.hiliteIconPosition controls where the icon will 
*  appear for that field -- the default is that it appears in front of the normal cell content.
*  This can also be overridden at the field level.
* @type {Array<Partial<string>>}
* @default "["[SKINIMG]/Dialog/notify.png", "[SKINIMG]/Dialog/warn.png", "[SKINIMG]/actions/approve.png"]"
*/
hiliteIcons?: Array<Partial<string>>;

/**
* When DataBoundComponent.hiliteIcons are present, where the hilite icon will be placed 
*  relative to the field value. See HiliteIconPosition. Can be overridden at the
*  field level.
* @type {HiliteIconPosition}
* @default "before"
*/
hiliteIconPosition?: HiliteIconPosition;

/**
* Default width and height of DataBoundComponent.hiliteIcons for this component.
*  Can be overridden at the component level via explicit 
*  DataBoundComponent.hiliteIconWidth and
*  DataBoundComponent.hiliteIconHeight, or at the field level via 
*  ListGridField.hiliteIconSize,
*  ListGridField.hiliteIconWidth and 
*  ListGridField.hiliteIconHeight
* @type {number}
* @default "12"
*/
hiliteIconSize?: number;

/**
* Width for hilite icons for this component.
*  Overrides DataBoundComponent.hiliteIconSize.
*  Can be overridden at the field level.
* @type {number}
* @default null
*/
hiliteIconWidth?: number;

/**
* Height for hilite icons for this listGrid.
*  Overrides DataBoundComponent.hiliteIconSize.
*  Can be overridden at the field level
* @type {number}
* @default null
*/
hiliteIconHeight?: number;

/**
* How much padding should there be on the left of DataBoundComponent.hiliteIcons by default?
*  Can be overridden at the field level
* @type {number}
* @default "2"
*/
hiliteIconLeftPadding?: number;

/**
* How much padding should there be on the right of DataBoundComponent.hiliteIcons by default?
*  Can be overridden at the field level
* @type {number}
* @default "2"
*/
hiliteIconRightPadding?: number;

/**
* Marker that can be set on a record to flag that record as hilited. Should be set to a value
*  that matches Hilite.id for a hilite defined on this component.
* @type {string}
* @default "_hilite"
*/
hiliteProperty?: string;

/**
* Initial hilite state for the grid. 
*  
*  ListGrid.viewState can be used to initialize all view properties of the grid.
*  When doing so, hiliteState is not needed because viewState
*  includes it as well. If both are provided, hiliteState has priority for
*  hilite state.
*  
*  
*  To retrieve current state call DataBoundComponent.getHiliteState.
*  
* @type {string}
* @default null
*/
hiliteState?: string;

/**
* The strategy to use when generating names for new fields in this component. The default 
*  strategy, "simple", combines the field-type with an index maintained by field-type and 
*  component instance. For example, "formulaField1".
* @type {FieldNamingStrategy}
* @default "simple"
*/
fieldNamingStrategy?: FieldNamingStrategy;

/**
* Text for a menu item allowing users to edit grid highlights.
* @type {string}
* @default "Edit Highlights..."
*/
editHilitesText?: string;

/**
* The title for the DataBoundComponent.editHilites dialog.
* @type {string}
* @default "Edit Highlights"
*/
editHilitesDialogTitle?: string;

/**
* The title for the DataBoundComponent.fieldEditorWindow used to edit calculated
*  fields.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will
*  be evaluated as JS code whenever the message is displayed.
*  
*  Two dynamic variables are available - "builderType", either Formula or Summary, and 
*  "fieldTitle", which is the title of the calculated field being edited.
* @type {HTMLString}
* @default "${builderType} Editor [${fieldTitle}]"
*/
fieldEditorWindowTitle?: HTMLString;

/**
* The Window used to edit calculated fields for this component.
* @type {Window}
* @default null
*/
fieldEditorWindow?: Window;

/**
* This component's HiliteEditor instance used to allow the user to create, modify, or delete
*  hilites.
*  
*  The following autoChildUsage apply:
*  
*  
* - DataBoundComponent.hiliteIcons for HiliteEditor.hiliteIcons
*  
* @type {HiliteEditor}
* @default null
*/
hiliteEditor?: HiliteEditor;

/**
* CSS Style to apply to the drag tracker when dragging occurs on this component.
* @type {CSSStyleName}
* @default "gridDragTracker"
*/
dragTrackerStyle?: CSSStyleName;

/**
* If the result of a formula evaluation is invalid (specifically, if isNaN(result)==true),
*  badFormulaResultValue is displayed instead. The default value is ".".
* @type {string}
* @default "."
*/
badFormulaResultValue?: string;

/**
* If a summary format string contains an invalid field reference, replace the reference
*  with the missingSummaryFieldValue. The default value is "-".
* @type {string}
* @default "-"
*/
missingSummaryFieldValue?: string;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define a new
*  field based on values present in other fields, using the FormulaBuilder.
*  
*  User-added formula fields can be persisted via ListGrid.getFieldState and 
*  ListGrid.setFieldState.
* @type {boolean}
* @default false
*/
canAddFormulaFields?: boolean;

/**
* Text for a menu item allowing users to add a formula field
* @type {string}
* @default "Add formula column..."
*/
addFormulaFieldText?: string;

/**
* Text for a menu item allowing users to edit a formula field
* @type {string}
* @default "Edit formula..."
*/
editFormulaFieldText?: string;

/**
* Text for a menu item allowing users to remove a formula field
* @type {string}
* @default "Remove formula"
*/
removeFormulaFieldText?: string;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define a new
*  text field that can contain both user-defined text and the formatted values present in other 
*  fields, using the SummaryBuilder.
*  
*  User-added summary fields can be persisted via ListGrid.getFieldState and 
*  ListGrid.setFieldState.
* @type {boolean}
* @default false
*/
canAddSummaryFields?: boolean;

/**
* Text for a menu item allowing users to add a formula field
* @type {string}
* @default "Add summary column..."
*/
addSummaryFieldText?: string;

/**
* Text for a menu item allowing users to edit the formatter for a field
* @type {string}
* @default "Edit summary format..."
*/
editSummaryFieldText?: string;

/**
* Text for a menu item allowing users to remove a summary field
* @type {string}
* @default "Remove summary format..."
*/
removeSummaryFieldText?: string;

/**
* When assembling a value for a DataBoundComponent.canAddSummaryFields, if a referenced
*  field is hilited, should the hilite HTML be included in the summary field value?
*  
*  To control hilites showing in group summaries, see ListGrid.showHilitesInGroupSummary.
* @type {boolean}
* @default true
*/
includeHilitesInSummaryFields?: boolean;

/**
* Best field to use for a user-visible title for an individual record from this
*  component.
*  
*  This attribute has the same function as DataSource.iconField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
titleField?: string;

/**
* Designates a field of FieldType:"image" as the field to use when rendering a
*  record as an image, for example, in a TileGrid. 
*  
*  This attribute has the same function as DataSource.iconField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
iconField?: string;

/**
* Name of the field that has the second most pertinent piece of textual information in the
*  record, for use when a DataBoundComponent needs to show a short summary of a record.
*  
*  This attribute has the same function as DataSource.infoField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
infoField?: string;

/**
* Name of the field that has the most pertinent numeric, date, or enum value, for use when a
*  DataBoundComponent needs to show a short summary of a record.
*  
*  This attribute has the same function as DataSource.dataField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
dataField?: string;

/**
* Name of the field that has a long description of the record, or has the primary text data
*  value for a record that represents an email message, SMS, log or similar.
*  
*  This attribute has the same function as DataSource.descriptionField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
descriptionField?: string;

/**
* The message to display to the user if an export of a DataBoundComponent's data is attempted
*  while the DataBoundComponent's data is null or an empty list.
* @type {string}
* @default "You are attempting to export an empty dataset"
*/
emptyExportMessage?: string;

/**
* Properties to apply to the MultiSortDialog which gets automatically generated
*  when DataBoundComponent.askForSort is called.
*  
*  See also ListGrid.showHeaderSpanTitlesInSortEditor and ListGrid.sortEditorSpanTitleSeparator
* @type {Partial<MultiSortDialog>}
* @default null
*/
multiSortDialogProperties?: Partial<MultiSortDialog>;

/**
* Class level defaults to apply to the MultiSortDialog which gets automatically
*  generated when DataBoundComponent.askForSort is called.
*  
*  See also ListGrid.showHeaderSpanTitlesInSortEditor and ListGrid.sortEditorSpanTitleSeparator
* @type {Partial<MultiSortDialog>}
* @default null
*/
multiSortDialogDefaults?: Partial<MultiSortDialog>;

/**
* For databound components that support editing, the error message for a failed validator
*  that does not specify its own errorMessage.
* @type {HTMLString}
* @default "Invalid value"
*/
unknownErrorMessage?: HTMLString;

/**
* A message to display to the user if server-side validation fails with an error but the
*  server did not provide an error message
* @type {string}
* @default "Error during validation; no error details were provided"
*/
noErrorDetailsMessage?: string;





/**
*  Returns whether there are any pending DSRequests initiated by this
*  DataBoundComponent. May not include any requests sent by directly calling the
*  DataSource APIs (rather than the DataBoundComponent APIs).
*  
* 
*/
static requestsArePending(): boolean;



/**
*  This method is invoked on the source component whenever a drag operation or 
*  DataBoundComponent.transferSelectedData completes. This method is called when the entire chain of 
*  operations - including, for databound components, server-side updates and subsequent 
*  integration of the changes into the client-side cache - has completed.
*  There is no default implementation of this method; you are intended to override it if you 
*  are interested in being notified when drag operations complete.
* 
* 
*/
dragComplete(): void;

/**
*  This method is invoked whenever a drop operation or DataBoundComponent.transferSelectedData 
*  targeting this component completes. A drop is considered to be complete when all the client-
*  side transfer operations have finished. This includes any server turnarounds SmartClient 
*  needs to make to check for duplicate records in the target component; it specifically does 
*  not include any add or update operations sent to the server for databound components. If 
*  you want to be notified when the entire drag operation - including server updates and cache
*  synchronization - has completed, override DataBoundComponent.dragComplete
*  on the source component.
*  There is no default implementation of this method; you are intended to override it if you 
*  are interested in being notified when drop operations complete.
* 
* 
* @param {Array<Partial<Record>>} The list of records actually transferred to          this component (note that this is not necessarily the same thing as the           list of records dragged out of the source component because it doesn't           include records that were excluded because of collisions with existing           records)
*/
dropComplete(transferredRecords:Array<Partial<Record>>): void;

/**
*  For a component with a specified DataSource, find the associated dataSource field object
*  from a specified DataPath.
* 
* @param {DataPath} dataPath for which the field definition should be returned.
*/
getDataPathField(dataPath:DataPath): void;

/**
*  Can the field be edited? This base method always returns false, but it's
*  overridden by subclasses such as DynamicForm and ListGrid.
* 
* 
* @param {object | number | string} field object or identifier
*/
fieldIsEditable(field:object | number | string): boolean;

/**
*  Return a field by a field index or field name.
* 
* 
* @param {FieldName | number} field index or field.name
*/
getField(fieldID:FieldName | number): object;

/**
*  Find the index of a currently visible field.
* 
* 
* @param {FieldName | string} field name or field
*/
getFieldNum(fieldName:FieldName | string): Integer;

/**
*  This API is equivalent to List.find but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
find(advancedCriteria:AdvancedCriteria): object;

/**
*  This API is equivalent to List.findAll but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
findAll(advancedCriteria:AdvancedCriteria): Array<any>;

/**
*  This API is equivalent to List.findIndex but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
findIndex(advancedCriteria:AdvancedCriteria): Integer;

/**
*  This API is equivalent to List.findNextIndex but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {Integer} first index to consider
* @param {AdvancedCriteria} AdvancedCriteria to use with
* @param {Integer=} last index to consider
*/
findNextIndex(startIndex:Integer, advancedCriteria:AdvancedCriteria, endIndex?:Integer): Integer;

/**
*  Bind to a new DataSource.
*  
*  Like passing the "dataSource" property on creation, binding to a DataSource means that the
*  component will use the DataSource to provide default data for its fields.
*  
*  When binding to a new DataSource, if the component has any existing "fields" or has a dataset,
*  these will be discarded by default, since it is assumed the new DataSource may represent a
*  completely unrelated set of objects. If the old "fields" are still relevant, pass them to
*  setDataSource().
*  
* 
* @param {GlobalId | DataSource} DataSource to bind to
* @param {Array<Partial<DataSourceField>>=} optional array of fields to use
*/
setDataSource(dataSource:GlobalId | DataSource, fields?:Array<Partial<DataSourceField>>): void;

/**
*  Sends the current filter criteria and sort direction to the server, then exports data in the
*  requested DSRequest.exportAs.
*  
*  A variety of DSRequest settings, such as 
*  DSRequest.exportAs and DSRequest.exportFilename, affect the 
*  exporting process: see DSRequest.exportResults for further detail.
*  
*  Note that data exported via this method does not include any client-side formatting and
*  relies on both the SmartClient server and server-side DataSources. To export client-data 
*  with formatters applied, 
*  see ListGrid.exportClientData, which still requires the
*  SmartClient server but does not rely on server-side DataSource definitions (.ds.xml files).
*  
*  For more information on exporting data, see DataSource.exportData.
* 
* 
* @param {DSRequest=} additional properties to set on the DSRequest                      that will be issued
* @param {DSCallback=} callback to invoke on completion. Note that this parameter                 only applies where DSRequest.exportToClient is                  explicitly set to false, because file downloads do not                  provide ordinary SmartClient callbacks
*/
exportData(requestProperties?:DSRequest, callback?:DSCallback): void;

/**
*  Returns an array of Alignment for this grid
* 
*/
getFieldAlignments(): Array<Partial<Alignment>>;

/**
*  Compares the specified criteria with the current criteria applied to this component's
*  data object and determines whether the new criteria could be satisfied from the currently
*  cached set of data, or if a new filter/fetch operation will be required.
*  
*  This is equivalent to calling this.data.willFetchData(...).
*  Always returns true if this component is not showing a set of data from the dataSource.
*  
*  Note that to predict correctly the decision that will be made by filter/fetch, you'll need to
*  pass the same TextMatchStyle that will be used by the future filter/fetch. Fetching
*  manually (e.g. ListGrid.fetchData) will by default use "exact" while filtering
*  (e.g. ListGrid.filterData) will by default use "substring". If the component
*  is configured for autofetch (i.e. ListGrid.autoFetchData: true), that will
*  use ListGrid.autoFetchTextMatchStyle, which defaults to "substring". If nothing/null
*  is passed for the style, this method assumes you want the style from the last filter/fetch.
*  
*  To determine what TextMatchStyle is being used, check the RPC Tab of the
*  debugging and check the relevant DSRequest. 
*  
* 
* @param {Criteria} new criteria to test.
* @param {TextMatchStyle=} New text match style. If not passed assumes    textMatchStyle will not be modified.
*/
willFetchData(newCriteria:Criteria, textMatchStyle?:TextMatchStyle): boolean;

/**
*  Attempt to find the record in the resultSet that has a primary key value that matches the 
*  passed in parameter value. Only the locally cached data will be searched. 
*  Checks only loaded rows and will not trigger a fetch. Returns null if there is no match, 
*  data is not loaded, or there is no ResultSet.dataSource.
*  
*  Note, if you pass a simple value to this method, it will be matched against the first 
*  primaryKey field. For DataSources with a composite primary key (multiple primaryKey fields),
*  pass
*  a criteria object containing just your primaryKeys, like this: 
*  { firstPkField: "value", secondPkField: 25 }.
*  
* 
* 
* @param {object} primary key value to search for
*/
findByKey(keyValue:object): Record;

/**
* 
*  Select/deselect a Record passed in explicitly, or by index.
* 
* 
* @param {Record | number} record (or row number) to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRecord(record:Record | number, newState?:boolean): void;

/**
*  Select a single Record passed in explicitly, or by index, and deselect everything else.
*  When programmatic selection of records is a requirement and 
*  ListGrid.selectionType is "single", use this method rather than 
*  DataBoundComponent.selectRecord to 
*  enforce mutually-exclusive record-selection.
* 
* 
* @param {Record | number} record (or row number) to select
*/
selectSingleRecord(record:Record | number): void;

/**
* 
*  Deselect a Record passed in explicitly, or by index.
*  
*  Synonym for selectRecord(record, false)
* 
* 
* @param {Record | number} record (or row number) to deselect
*/
deselectRecord(record:Record | number): void;

/**
* 
*  Select/deselect a list of Records passed in explicitly, or by index.
*  
*  Note that developers may wish to use DataBoundComponent.selectRange to select a single
*  contiguous range.
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRecords(records:Array<Partial<Record>> | number, newState?:boolean): void;

/**
* 
*  Deselect a list of Records passed in explicitly, or by index.
*  
*  Synonym for selectRecords(records, false)
*  
*  Note that developers may wish to use DataBoundComponent.deselectRange to select a single
*  contiguous range.
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to deselect
*/
deselectRecords(records:Array<Partial<Record>> | number): void;

/**
*  Select all records
* 
* 
*/
selectAllRecords(): void;

/**
* 
*  Deselect all records
* 
* 
*/
deselectAllRecords(): void;

/**
*  Select a contiguous range of records by index
* 
* @param {Integer} start of selection range
* @param {Integer} end of selection range (non-inclusive)
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRange(startRow:Integer, endRow:Integer, newState?:boolean): void;

/**
*  Deselect a contiguous range of records by index.
*  
*  This is a synonym for selectRange(startRow, endRow, false);
* 
* @param {Integer} start of selection range
* @param {Integer} end of selection range (non-inclusive)
*/
deselectRange(startRow:Integer, endRow:Integer): void;

/**
*  Whether at least one item is selected
* 
*/
anySelected(): boolean;

/**
*  Returns the number of selected records.
* 
* 
*/
getSelectionLength(): Integer;

/**
*  Return the set of hilite-objects currently applied to this DataBoundComponent. These
*  can be serialized for storage and then restored to a component later via 
*  DataBoundComponent.setHilites.
* 
* 
*/
getHilites(): Array<Partial<Hilite>>;

/**
*  Accepts an array of hilite objects and applies them to this DataBoundComponent. See also
*  DataBoundComponent.getHilites for a method of retrieving the hilite
*  array for storage, including hilites manually added by the user.
* 
* 
* @param {Array<Partial<Hilite>>} Array of hilite objects
*/
setHilites(hilites:Array<Partial<Hilite>>): void;

/**
*  Get the current hilites encoded as a String, for saving.
* 
* 
*/
getHiliteState(): string;

/**
*  Set the current hilites based on a hiliteState String previously returned from
*  DataBoundComponent.getHiliteState.
* 
* @param {string} hilites state encoded as a String
*/
setHiliteState(hiliteState:string): void;

/**
*  Enable / disable a DataBoundComponent.hilites
* 
* 
* @param {string} ID of hilite to enable
* @param {boolean=} new enabled state to apply - if null, defaults to true
*/
enableHilite(hiliteID:string, enable?:boolean): void;

/**
*  Disable a hilite
* 
* 
* @param {string} ID of hilite to disable
*/
disableHilite(hiliteID:string): void;

/**
*  Enable all hilites.
* 
* 
* @param {boolean=} new enabled state to apply - if null, defaults to true
*/
enableHiliting(enable?:boolean): void;

/**
*  Disable all hilites.
* 
* 
*/
disableHiliting(): void;

/**
*  Shows a HiliteEditor interface allowing end-users to edit
*  the data-hilites currently in use by this DataBoundComponent.
* 
* 
*/
editHilites(): void;

/**
* 
*  Transfer a list of Records from another component (does not have to be a databound
*  component) into this component. This method is only applicable to list-type components,
*  such as ListGrid, TreeGrid or TileGrid
*  
*  This method implements the automatic drag-copy and drag-move behaviors of components like
*  ListGrid, and calling it is equivalent to completing a drag and drop of the
*  dropRecords.
*  
*  Note that this method is asynchronous - it may need to perform server turnarounds to prevent
*  duplicates in the target component's data. If you wish to be notified when the transfer 
*  process has completed, you can either pass the optional callback to this method or implement
*  the DataBoundComponent.dropComplete method on this component.
*  
*  See also DataBoundComponent.transferSelectedData.
* 
* 
* @param {Array<Partial<Record>>} Records to transfer to this component
* @param {Record} The target record (eg, of a drop interaction), for context
* @param {Integer} Insert point in this component's data for the transferred records
* @param {Canvas} The databound or non-databound component from which the records              are to be transferred.
* @param {Callback=} optional callback to be fired when the transfer process has completed
*/
transferRecords(dropRecords:Array<Partial<Record>>, targetRecord:Record, index:Integer, sourceWidget:Canvas, callback?:Callback): void;

/**
*  Returns the "drop values" to apply to a record dropped on this component prior to update. Only
*  applicable to databound components - see DataBoundComponent.dropValues for more details. If multiple records 
*  are being dropped, this method is called for each of them in turn.
*  
*  The default implementation of this method returns the following:
*  
*  
* - Nothing, if DataBoundComponent.addDropValues is false
*  
* - dropValues, if that property is set. If the component's criteria object is applicable (as explained
*  in the next item), it is merged into dropValues, with properties in dropValues taking precedence.
*  
* - The component's criteria object, if the most recent textMatchStyle for the component was "exact" 
*    and it is simple criteria (ie, not an AdvancedCriteria object)
*  
* - Otherwise nothing
*  
*  
*  You can override this method if you need more complex setting of drop values than can be 
*  provided by simply supplying a dropValues object.
*  
* 
* @param {Record} record being dropped
* @param {DataSource} dataSource the record being dropped is bound to
* @param {Record} record being dropped on
* @param {Integer} index of record being dropped on
* @param {Canvas} widget where dragging began
*/
getDropValues(record:Record, sourceDS:DataSource, targetRecord:Record, index:Integer, sourceWidget:Canvas): object;

/**
* 
*  During a drag-and-drop interaction, this method returns the set of records being dragged out
*  of the component. In the default implementation, this is the list of currently selected
*  records.
*  
*  This method is consulted by ListGrid.willAcceptDrop.
*  
*  NOTE: If this component is a Tree.isMultiLinkTree TreeGrid, 
*  this method returns a list of NodeLocators rather than a list of records. Each 
*  nodeLocator contains a pointer to the associated record in its node
*  property.
* 
* 
* @param {DataBoundComponent} source component from which the records will be transferred
*/
getDragData(source:DataBoundComponent): Array<Partial<Record>>;

/**
*  Sets the custom tracker HTML to display next to the mouse when the user initiates a drag
*  operation on this component. Default implementation will examine ListGrid.dragTrackerMode
*  and set the custom drag tracker to display the appropriate HTML based on the selected record.
*  
*  To display custom drag tracker HTML, this method may be overridden - call 
*  EventHandler.setDragTracker to actually update the drag tracker HTML.
* 
*/
setDragTracker(): boolean;

/**
*  Convenience method to display a FormulaBuilder to create a new Formula Field. This 
*  is equivalent to calling DataBoundComponent.editFormulaField with 
*  no parameter.
* 
* 
*/
addFormulaField(): void;

/**
*  Method to display a FormulaBuilder to edit a formula Field. If the function is called
*  without a parameter, a new field will be created when the formula is saved.
* 
* 
* @param {string} Field to edit or null to add a new formula field
*/
editFormulaField(field:string): void;

/**
*  Get the computed value of a DataBoundComponent.canAddFormulaFields.
* 
* @param {string} field that has a formula
* @param {Record} record to use to compute formula value
*/
getFormulaFieldValue(field:string, record:Record): Double | string;

/**
*  Convenience method to display a SummaryBuilder to create a new Summary Field. This 
*  is equivalent to calling DataBoundComponent.editSummaryField with 
*  no parameter.
* 
* 
*/
addSummaryField(): void;

/**
*  Method to display a SummaryBuilder to edit a Summary Field. If the function is called
*  without a parameter, a new field will be created when the summary is saved.
* 
* 
* @param {string} Field to edit or null to add a new summary column
*/
editSummaryField(field:string): void;

/**
*  Get the computed value of a DataBoundComponent.canAddSummaryFields.
* 
* @param {string} field that has a summary format
* @param {Record} record to use to compute formula value
*/
getSummaryFieldValue(field:string, record:Record): string;

/**
*  When assembling a value for a DataBoundComponent.canAddSummaryFields, if a referenced
*  field is hilited, should the hilite HTML be included in the summary field value?
*  
*  Example use case: Consider a grid containing a numeric field, and a summary field
*  which contains some string value, plus the contents of the numeric field.
*  If a hilite is defined for the grid which turns the numeric field text red when
*  the value is negative, this property will govern whether the number will also be
*  rendered in red within the summary field cells. Any other text in the summary field
*  cells would not be effected by this hilite.
*  
*  Default implementation returns DataBoundComponent.includeHilitesInSummaryFields.
*  
*  To control hilites showing in group summaries, see ListGrid.showHilitesInGroupSummary.
* 
* 
* @param {string} name of the summary field
* @param {string} name of the field referenced by this summary
*/
shouldIncludeHiliteInSummaryField(summaryFieldName:string, usedFieldName:string): boolean;

/**
*  Get the index of the provided record.
*  
*  Override in subclasses to provide more specific behavior, for instance, when data holds a
*  large number of records
* 
* 
* @param {Record} the record whose index is to be retrieved
*/
getRecordIndex(record:Record): number;

/**
*  Get the value of the titleField for the passed record
*  
*  Override in subclasses 
* 
* 
* @param {Record} the record whose index is to be retrieved
*/
getTitleFieldValue(record:Record): string;

/**
*  Method to return the fieldName which represents the "title" for records in this
*  Component.
*  If this.titleField is explicitly specified it will always be used.
*  Otherwise, default implementation will check DataSource.titleField for databound
*  compounds.
*  For non databound components returns the first defined field name of "title",
*  "name", or "id" where the field is visible. If we don't find any
*  field-names that match these titles, the first field in the component will be used instead.
* 
*/
getTitleField(): string;

/**
*  Return all CSS style declarations associated with the hilites of a record's field.
* 
* @param {Record} * @param {string} if set, returned CSS will be appended to this text
* @param {string} field object identifying whose CSS is to be returned
*/
getRecordHiliteCSSText(record:Record, cssText:string, field:string): string;

/**
*  Return the SortSpecifiers representing the current sort configuration of this
*  component.
* 
*/
getSort(): Array<Partial<SortSpecifier>>;

/**
*  Sort this component by a list of SortSpecifiers. If the component's data is not a 
*  ResultSet, only the first specifier is applied.
*  
* 
* @param {Array<Partial<SortSpecifier>>} List of SortSpecifier objects, one  per sort-field and direction
*/
setSort(sortSpecifiers:Array<Partial<SortSpecifier>>): void;

/**
*  Show a dialog to configure the sorting of multiple fields on this component. Calls through
*  to MultiSortDialog.askForSort, passing this component as the fieldSource and the
*  current DataBoundComponent.getSort if there is one.
*  
*  The generated multiSortDialog can be customized via DataBoundComponent.multiSortDialogDefaults, 
*  DataBoundComponent.multiSortDialogProperties.
* 
* 
*/
askForSort(): void;

/**
*  Returns true if the component's current data model is marked as offline. This does not 
*  necessarily mean that the component has no data; it may have data that was supplied from
*  the Offline.
* 
*/
isOffline(): boolean;

/**
*  Shows a FieldPicker interface allowing end-users to edit
*  the fields currently shown by this DataBoundComponent.
* 
* 
*/
editFields(): void;

/**
*  Notification method fired when a user-generated field is added to this component via
*  DataBoundComponent.editFormulaField or DataBoundComponent.editSummaryField.
*  
*  Returning false from this method will prevent the field being added at all. Note that
*  this also provides an opportunity to modify the generated field object - any changes
*  made to the field parameter will show up when the field is displayed in the ListGrid.
* 
* 
* @param {ListGridField} User generated summary or formula field
*/
userAddedField(field:ListGridField): boolean;

/**
*  Called when the selection changes. Note that this method fires exactly once for any given
*  change to the selection unlike the ListGrid.selectionChanged event.
*  
*  This event is fired once after selection/deselection has completed. The result is
*  one event per mouse-down event. For a drag selection there will be two events fired:
*  one when the first record is selected and one when the range is completed.
*  
*  This event is also fired when selection is updated by a direct call to one of the
*  DataBoundComponent select/deselect methods. Calls on the Selection
*  object do not trigger this event.
* 
* 
* @param {object} first selected record in the selection, if any, which may or may not be the first record in sort order if the DataBoundComponent is sorted. This parameter is typically used when only one record can be selected at a time.
* @param {Array<Partial<object>>} List of records that are now selected
*/
selectionUpdated(record:object, recordList:Array<Partial<object>>): void;




}

/**
* A means of identifying the properties in an exported dataset - either the property name
*  or its title.
*/
type PropertyIdentifier = 
/**
* Identify properties by internal name
*/
"name" |

/**
* Identify properties by localized descriptive title
*/
"title";




/**
* The Offline class provides features for storing and retrieving values in persistent browser
*  storage. Because this storage is both persistent (ie, it remains available after the 
*  browser is closed and re-opened) and local to the browser, it allows applications to operate
*  when the client is not connected to a network, albeit with some obvious limitations.
*  
*  As well as providing straightforward APIs for storing, retrieving and removing values, 
*  Offline support is integrated into the normal request/response cycle. You can configure 
*  the framework so that server responses are automatically cached in Offline storage. Then,
*  at some future point when the client is offline, responses for known requests are returned 
*  from the Offline cache.
*  
*  SmartClient makes use of various underlying storage techniques, depending on what the 
*  browser supports, but the API to a SmartClient program is the same regardless of the 
*  underlying storage - simple key/value pairs. Currently, we provide Offline support for all 
*  HTML5-compliant browsers, plus earlier versions of Internet Explorer (6 and 7). The amount 
*  of storage available is dictated by the browser, and varies from approximately 500KB to 
*  approximately 5MB.
*/
class Offline extends Class {
/**
* Creates a new Offline
* 
* @param typeCheckedProperties {Partial<Offline>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Offline} The new instance
*/
static create(typeCheckedProperties? : Partial<Offline>, uncheckedProperties?: Object): Offline;






/**
*  Returns true if the current browser session is offline (ie, not connected to a network).
*  If an online/offline state has been set explicitly (see Offline.goOffline and 
*  Offline.goOnline), the explicitly-set state will be returned. Otherwise, the 
*  offline state as reported by the browser will be returned. See 
*  Offline.useNativeOfflineDetection for important notes on 
*  browser detection of offline state.
* 
* 
*/
static isOffline(): boolean;

/**
*  Explicitly sets this session into offline mode. This setting will override whatever 
*  state the browser reports. This allows users to manually set an application into 
*  offline or online state.
*  
* 
*/
static goOffline(): void;

/**
*  Explicitly sets this session into online mode. This setting will override whatever 
*  state the browser reports. This allows users to manually set an application into 
*  offline or online state.
*  
* 
*/
static goOnline(): void;

/**
*  Tells the Offline system to query the browser for the current online/offline state.
*  Calling this method switches off the explicit offline mode setting switched on by 
*  calling Offline.goOnline or Offline.goOffline.
*  
*  It is important to note that browsers vary quite considerably in their ability to 
*  detect that they are offline. Many older browsers simply can't do it; HTML5 browsers
*  expose the navigator.onLine property, but each browser's implementation 
*  is different. Some browsers have a manual "Work Offline" mode which allows the user 
*  to make the decision, and SmartClient provides an equivalent mechanism with the 
*  goOffline and goOnline methods. Generally speaking, these 
*  methods are more reliable than allowing the browser to decide whether your application
*  is offline.
*  
* 
*/
static useNativeOfflineDetection(): void;

/**
*  Stores the passed-in value in browser-local storage, mapped to the passed-in key. If
*  there is no room left to store the value, we discard values from the offline store, 
*  oldest first, until there is room to store the value. If you don't want this behavior,
*  explicitly pass false in the recycleEntries parameter.
*  
*  Note that limitations in the underlying storage engines mean that only primitive 
*  values - Strings, numbers and booleans - can be stored. If you wish to store an Array
*  or Object, you will have to serialize it to JSON first, and then eval it 
*  after retrieval to turn it back into an object.
*  
*  Note: This method throws an exception if it could not store the value (either 
*  because storage is full and recycleEntries was false, or because the value to store is
*  simply too large)
*  
* 
* @param {string} The key to use when storing the value
* @param {any} The value to store
* @param {boolean=} If false, suppresses the default behavior of repeatedly                  discarding the oldest entry if there is insufficient                  space to store the value
*/
static put(key:string, value:any, recycleEntries?:boolean): void;

/**
*  Returns the value mapped to the passed-in key from browser-local storage, or null if 
*  no such key exists.
*  
* 
* @param {string} The key to retrieve a value for
*/
static get(key:string): any;

/**
*  Removes the key/value pair mapped by the passed-in key from browser-local storage
*  
* 
* @param {string} The key to remove
*/
static remove(key:string): void;






}

/**
* A header span appears as a second level of headers in a ListGrid, spanning one or more
*  ListGrid columns and their associated headers.
*  
*  See ListGrid.headerSpans.
*  
*  In addition to the properties documented here, all other properties specified on the
*  headerSpan object will be passed to the Class.create method of the
*  ListGrid.headerSpanConstructor. This allows you to set properties such as
*  Button.baseStyle or StretchImgButton.src directly in a
*  headerSpan.
*/
class HeaderSpan {


/**
* Name for this headerSpan, for use in APIs like ListGrid.setHeaderSpanTitle.
*  
*  Name is optional, but if specified, must be unique for this ListGrid (but not globally
*  unique) as well as a valid JavaScript identifier, as specified by ECMA-262 Section 7.6
*  (the String.isValidID function can be used to test whether a name is a valid JavaScript
*  identifier).
* @type {Identifier}
* @default null
*/
name?: Identifier;

/**
* List of fields that this header spans. Fields should be identified by their value for
*  ListGridField.name.
*  
*  Developers may define multiple levels of header-spans by specifying HeaderSpan.spans
*  however a span cannot be specified with both fields and spans.
* @type {Array<Partial<string>>}
* @default null
*/
fields?: Array<Partial<string>>;

/**
* This property allows developer to "nest" header spans, grouping fields together by
*  multiple layers of granularity.
*  
*  For example a group of fields could be nested within two layers of header spans as follows:
*  
*  
*  { title:"Europe", spans:[
*    {title:"France", fields:["Paris", "Lyon"]},
*    {title:"UK", fields:["London", "Glasgow"]},
*    {title:"Spain", fields:["Barcelona"]}
*  ]
*  }
*  
*  
*  
*  Note that a span definition can not include both spans
*  and HeaderSpan.fields.
* @type {Array<Partial<HeaderSpan>>}
* @default null
*/
spans?: Array<Partial<HeaderSpan>>;

/**
* A title for this headerSpan, to display in the headerSpan button for this headerSpan
*  and in other contexts such as the ListGrid.canPickFields.
* 
*  Note: if you want to use HTML tags to affect the display of the header, you should do so
*  via HeaderSpan.headerTitle instead so that other places where the title
*  appears in the UI are not affected. Refer to discussion at ListGridField.title.
* @type {string}
* @default null
*/
title?: string;

/**
* Optional title for the headerSpan button for this headerSpan. If specified this will be
*  displayed in the headerSpan button instead of HeaderSpan.title. Set to an empty
*  string to suppress the title in the header button entirely.
* @type {string}
* @default null
*/
headerTitle?: string;

/**
* Custom base style to apply to the header button created for this span instead
*  of ListGrid.headerBaseStyle.
*  
*  Note that depending on the header button constructor, you may have to specify
*  HeaderSpan.headerTitleStyle as well.
* @type {CSSStyleName}
* @default null
*/
headerBaseStyle?: CSSStyleName;

/**
* Custom titleStyle to apply to the header button created for this span instead of
*  ListGrid.headerTitleStyle.
*  
*  Note that this will typically only have an effect if
*  ListGrid.headerButtonConstructor is set to StretchImgButton or a subclass
*  thereof.
* @type {CSSStyleName}
* @default null
*/
headerTitleStyle?: CSSStyleName;

/**
* Height of this headerSpan. Defaults to ListGrid.headerSpanHeight.
* @type {Integer}
* @default null
*/
height?: Integer;

/**
* Vertical alignment of the title of this headerSpan.
*  
*  Defaults to listGrid.headerSpanVAlign if unset.
* @type {VerticalAlignment}
* @default null
*/
valign?: VerticalAlignment;

/**
* Horizontal alignment of the title of this headerSpan.
* @type {Alignment}
* @default "center"
*/
align?: Alignment;

/**
* Should the span title wrap if there is not enough space horizontally to accommodate it.
*  If unset, default behavior is derived from ListGrid.wrapHeaderSpanTitles. (This
*  is a soft-wrap - if set the title will wrap at word boundaries.)
* @type {boolean}
* @default null
*/
wrap?: boolean;










}







/**
* One of the supported formats for data-export. If you are doing a 
*  ListGrid.exportClientData to one of the native spreadsheet
*  formats (xls or ooxml), we also export Hilite coloring. So, if
*  Hilites are causing a particular cell to be rendered as green text on a blue background, 
*  the corresponding cell in the exported spreadsheet document will also be colored that way.
*/
type ExportFormat = 
/**
* Export data as XML records
*/
"xml" |

/**
* Export data as JSON objects
*/
"json" |

/**
* Export data in comma-separated format
*/
"csv" |

/**
* Export data in native Microsoft Excel 97 format
*/
"xls" |

/**
* Export data in native Microsoft Excel 2007 format (also called XLSX)
*/
"ooxml" |

/**
* Custom server-side logic will do the export
*/
"custom";






/**
* An atomic type such as a string or number, that is generally stored, displayed and
*  manipulated as a single value.
*  
*  SimpleTypes can be created at any time, and subsequently referred to as a 
*  DataSourceField.type in DataSource and
*  DataBoundComponent. This allows you to define
*  SimpleType.validators, SimpleType.normalDisplayFormatter
*  and SimpleType.editorType behaviors for a type to be reused across all
*  DataBoundComponent.
*  
*  The SimpleType class also allows data to be stored in some opaque format but treated as
*  simple atomic values as far as SmartClient components are concerned by implementing
*  SimpleType.getAtomicValue and SimpleType.updateAtomicValue methods.
*  For example, if some record has a field value set to a javascript object with the
*  following properties:
*  
*  { stringValue:"A String", length: 9 }
*  
*  this value could be treated as a simple string by defining a SimpleType with 
*  SimpleType.inheritsFrom set to "text" and a custom 
*  getAtomicValue() method that simply extracted the "stringValue"
*  attribute from the data object. DataBoundComponents would then display
*  the string value, and use it for sorting and other standard databinding features.
*  
*  Note that the term "simpleType" is used in the same sense as in
*  http://www.w3.org/TR/xmlschema-0/, and
*  XMLTools.loadXMLSchema will create new SimpleType definitions.
*  
*  When using the SmartClient Server, SimpleTypes can be defined server-side, and should
*  be defined server-side if validators are going to be declared so that the server will
*  enforce validation. To define server-side SimpleTypes using Component XML you should create
*  file {typeName}.type.xml in the following format:
*  
*   &lt;SimpleType name="{typeName}" inheritsFrom="{otherSimpleType}" 
*          editorType="{FormItemClassName}"&gt;
*    &lt;validators&gt;
*     &lt;!-- validator definition just like DataSourceField --&gt;
*    &lt;/validators&gt;
*   &lt;/SimpleType&gt;
*  
*  .. and place this file alongside your DataSource files (.ds.xml) files - in any of folders
*  listed in project.datasources property in server_properties.
*  
*  SimpleTypes can be loaded via DataSourceLoader or loadDSTag and
*  should be loaded before the definitions of any DataSources that use them (so
*  generally put all SimpleType definitions first).
*  
*  Define validators in the server-side type definition, for example:
*  
*   &lt;SimpleType name="countryCodeType" inheritsFrom="text"&gt;
*    &lt;validators&gt;
*     &lt;validator type="lengthRange" min="2" max="2"
*      errorMessage="Length of country code should be equal to 2." /&gt;
*     &lt;validator type="regexp" expression="[A-Z][A-Z]"
*      errorMessage="CountryCode should have only uppercase letters." /&gt;
*    &lt;/validators&gt;
*   &lt;/SimpleType&gt;
*  
*  
*  For client-side formatters, add these to the type definition after loading it from the
*  server, for example:
*  
*   
*    isc.SimpleType.getType("independenceDateType").addProperties({
*      normalDisplayFormatter : function (value) {
*        if (value == null) return "";
*        return "&lt;i&gt;" + (value.getYear() + 1900) + "&lt;/i&gt;";
*      }
*    });
*   
*  
*  
*  Note that formatters must be added to the SimpleType definition before any
*  DataBoundComponent binds to a DataSource that uses the SimpleType.
*  
*  An example is ${isc.DocUtils.linkForExampleId('formsCustomSimpleType', 'here')}.
*  
*/
class SimpleType extends Class {
/**
* Creates a new SimpleType
* 
* @param typeCheckedProperties {Partial<SimpleType>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SimpleType} The new instance
*/
static create(typeCheckedProperties? : Partial<SimpleType>, uncheckedProperties?: Object): SimpleType;


/**
* Name of the type, used to refer to the type from DataSourceField.type.
* @type {Identifier}
* @default null
*/
name?: Identifier;

/**
* Name of another SimpleType from which this type should inherit.
*  
*  Validators, if any, will be combined. All other SimpleType properties default to the
*  inherited type's value.
* @type {Identifier}
* @default null
*/
inheritsFrom?: Identifier;

/**
* Validators to apply to value of this type.
* @type {Array<Partial<Validator>>}
* @default null
*/
validators?: Array<Partial<Validator>>;

/**
* List of legal values for this type, like DataSourceField.valueMap.
* @type {ValueMap}
* @default null
*/
valueMap?: ValueMap;

/**
* Classname of the FormItem that should be the default for editing values of this type (eg
*  "SelectItem").
*  
*  You can create a simple custom FormItem by adding default FormItem.icons that
*  launch custom value picking dialogs (an example is in the QuickStart
*  Guide, Chapter 9, Extending SmartClient). By setting simpleType.editorType
*  to the name of your custom FormItem, forms will automatically use the custom FormItem,
*  as will grids performing ListGrid.canEdit.
* @type {FormItemClassName}
* @default null
*/
editorType?: FormItemClassName;

/**
* Classname of the FormItem that should be used to display values of this type when a field
*  is marked as DataSourceField.canEdit and the field is displayed
*  in an editor type component like a DynamicForm.
*  
*  May be overridden by DataSourceField.readOnlyEditorType.
* @type {FormItemClassName}
* @default null
*/
readOnlyEditorType?: FormItemClassName;

/**
* Classname of the FormItem that should be used to edit values of this type if it appears
*  in a filter row.
*  
*  May be overridden by DataSourceField.filterEditorType.
* @type {FormItemClassName}
* @default null
*/
filterEditorType?: FormItemClassName;

/**
* These are properties that are essentially copied onto any DataSourceField where the
*  property is applied. The supported properties are only client-side properties.
* @type {Partial<DataSourceField>}
* @default null
*/
fieldProperties?: Partial<DataSourceField>;

/**
* FormatString for numeric or date formatting. See DataSourceField.format.
* @type {FormatString}
* @default null
*/
format?: FormatString;

/**
* FormatString used during exports for numeric or date formatting. See
*  DataSourceField.exportFormat.
* @type {FormatString}
* @default null
*/
exportFormat?: FormatString;

/**
* Set of search operators valid for this type. 
*  
*  If not specified, the SimpleType.inheritsFrom type's operators will be used, finally
*  defaulting to the default operators for the basic types (eg, integer).
* @type {Array<Partial<OperatorId>>}
* @default null
*/
validOperators?: Array<Partial<OperatorId>>;

/**
* The default search-operator for this data-type.
* @type {OperatorId}
* @default null
*/
defaultOperator?: OperatorId;

/**
* A set of key-value pairs that represent the names and titles of the grouping modes 
*  available to values of this type, for use in components that support grouping.
*  
*  Some types provide a set of builtin groupingModes, as covered 
*  builtinGroupingModes.
*  
*  Use SimpleType.getGroupValue and SimpleType.getGroupTitle to implement
*  custom grouping logic for each of the grouping modes you provide.
* @type {ValueMap}
* @default null
*/
groupingModes?: ValueMap;

/**
* In components that support grouping, the default mode from the available 
*  SimpleType.groupingModes to use when grouping values of this type.
* @type {string}
* @default null
*/
defaultGroupingMode?: string;





/**
*  Retrieve a simpleType definition by type name
* 
* @param {string} the name of the simpleType to return
*/
static getType(typeName:string): SimpleType;

/**
*  Registers a new SummaryFunction by name. After calling this method,
*  developers may specify the name passed in as a standard summaryFunction
*  (for example in ListGridField.summaryFunction).
* 
* @param {string} name for the newly registered summaryFunction
* @param {Function} New summary function. This function should take 2 parameters  
- records: an array of records for which a summary must be generated  
- field: a field definition  
- summaryConfig: summary configuration (see SummaryConfiguration)  and return a summary value for the field across the records.
*/
static registerSummaryFunction(functionName:string, method:Function): void;

/**
*  Set up a default summary function for some field type.
*  
*  Note that the following default summary functions are set up when SmartClient initializes:
*  - "integer" defaults to "sum"
*  - "float" defaults to "sum".
* 
* 
* @param {string} type name
* @param {SummaryFunction} summary function to set as the default for  this data type.
*/
static setDefaultSummaryFunction(typeName:string, summaryFunction:SummaryFunction): void;

/**
*  Retrieves the default summary function for some field type.
* 
* @param {string} type name
*/
static getDefaultSummaryFunction(typeName:string): SummaryFunction;

/**
*  Applies a SummaryFunction to an array of records
* 
* @param {Array<Partial<object>>} set of records to retrieve a summary value for
* @param {DataSourceField} field for which we're picking up a summary value
* @param {SummaryFunction} SummaryFunction to apply to the records in order to retrieve the summary value. May be specified as an explicit function  or string of script to execute, or a SummaryFunction identifier
* @param {SummaryConfiguration} config that affects summary calculation
*/
static applySummaryFunction(records:Array<Partial<object>>, field:DataSourceField, summaryFunction:SummaryFunction, summaryConfig:SummaryConfiguration): any;



/**
*  Optional method to extract an atomic value (such as a string or number)
*  from some arbitrary live data value. If defined, this method will be called
*  for every field value of the specified type in order to convert from the
*  raw data value to an atomic type to be used for standard DataBinding features
*  such as sorting and editing.
*  
*  The "reason" parameter is passed by the framework to indicate why it is asking for the 
*  atomic value. Your method can use this value to affect the atomic value that is 
*  returned - for example, if the reason is "sort" you could return the atomic value 
*  converted to upper-case, to impose case-insensitive sorting. Reason strings used 
*  by the framework are:
*  
* - "edit" Retrieving the edit value of the field in a DynamicForm or 
*         ListGrid
*  
* - "format" Retrieving the value to format it for display
*  
* - "mask" Retrieving the value to present it for masked input
*  
* - "filter" Retrieving the value for use in a filter comparison
*  
* - "sort" Retrieving the value for use in a sort comparison
*  
* - "group" Retrieving the value for use in a group comparison
*  
* - "formula" Retrieving the value for use in a formula calculation
*  
* - "vm_getValue" Retrieving the value from ValuesManager.getValue
*  
* - "validate" Retrieving the value for validation, or setting the value if validation
*           caused it to change
*  
* - "compare" Retrieving the "old" or "new" value from ListGrid.cellHasChanges
*  
* - "getRawValue" Retrieving the raw value of a ListGrid cell
*  
* - "criteria" Setting the value from DynamicForm.setValuesAsCriteria
*  
* - "updateValue" Setting the value from internal methods of DynamicForm 
*            or ValuesManager 
*  
* - "setRawValue" Setting the raw value of a ListGrid cell
*  
* - "saveLocally" Setting the value from ListGrid.saveLocally
*  
* 
* 
* @param {any} Raw data value to convert. Typically this would be a field  value for some record.
* @param {string} The reason your getAtomicValue() method is being  called
*/
getAtomicValue(value:any, reason:string): any;

/**
*  Optional method to update a live data value with an edited atomic value
*  (such as a string or number). If defined this method will be called
*  when the user edits data in a field of this type, allowing the developer
*  to convert from the atomic type to a raw data value for storage.
*  
*  Note that if the user is editing a field which did not previously have a value, the
*  'currentValue' will be null. This method should handle this (creating a new data value).
* 
* 
* @param {any} New atomic value. This should match the underlying atomic type specified by the SimpleType.inheritsFrom attribute.
* @param {any} Existing data value to be updated.
* @param {string} The reason your updateAtomicValue() method is being  called. See SimpleType.getAtomicValue for the reason strings used by the framework
*/
updateAtomicValue(atomicValue:any, currentValue:any, reason:string): any;

/**
*  Optional method to allow you to write a custom comparator for this SimpleType. If 
*  implemented, this method will be used by the framework when it needs to compare two
*  values of a field for equality - for example, when considering if an edited field 
*  value has changed. If you do not implement this method, values will be compared using
*  standard techniques, so you should only provide an implementation if you have some 
*  unusual requirement.
*  
*  Implementations of this method should return the following:
*  
* - 0 if the two values are equal
*  
* - -1 if the first value is greater than the second
*  
* - 1 if the second value is greater than the first
*  
* 
* 
* @param {any} First value for comparison
* @param {any} Second value for comparison
* @param {DataSourceField | ListGridField | DetailViewerField | FormItem} Field definition from a dataSource or dataBoundComponent.
*/
compareValues(value1:any, value2:any, field:DataSourceField | ListGridField | DetailViewerField | FormItem): Integer;

/**
*  Formatter for values of this type when compact display is required, for example, in a
*  ListGrid or TreeGrid.
*  
*  When this formatter is called, the SimpleType object is available as "this". 
*  
*  A formatter can make itself configurable on a per-component or per-field basis by
*  checking properties on the component or field. For example, a formatter for account IDs
*  may want to omit a prefix in views where it is redundant, and could check a flag
*  listGridField.omitAccountIdPrefix for this purpose.
* 
* 
* @param {any} value to be formatted
* @param {string=} field descriptor from the component calling the formatter, if           applicable. Depending on the calling component, this could be a            ListGridField, TreeGridField, etc
* @param {DataBoundComponent=} component calling this formatter, if applicable
* @param {object=} Full record, if applicable
*/
shortDisplayFormatter(value:any, field?:string, component?:DataBoundComponent, record?:object): void;

/**
*  Normal formatter for values of this type used in a StaticTextItem or
*  DetailViewer.
*  
*  When this formatter is called, the SimpleType object is available as "this". 
*  
*  A formatter can make itself configurable on a per-component or per-field basis by
*  checking properties on the component or field. For example, a formatter for account IDs
*  may want to omit a prefix in views where it is redundant, and could check a flag
*  detailViewer.omitAccountIdPrefix for this purpose.
* 
* 
* @param {any} value to be formatted
* @param {string=} field descriptor from the component calling the formatter, if           applicable. Depending on the calling component, this could be a            FormItem, DetailViewerField, etc
* @param {DataBoundComponent=} component calling this formatter, if applicable
* @param {object=} Full record, if applicable
*/
normalDisplayFormatter(value:any, field?:string, component?:DataBoundComponent, record?:object): void;

/**
*  Formatter for values of this type when displayed in a freeform text editor, such as
*  a TextItem.
*  
*  See also SimpleType.parseInput for parsing an edited text value back to
*  a data value.
* 
* @param {any} value to be formatted
* @param {FormItem=} Editor for this field
* @param {DynamicForm=} DynamicForm containing this editor
* @param {Record=} Current edit values for this record, as displayed in   the edit component.
*/
editFormatter(value:any, field?:FormItem, form?:DynamicForm, record?:Record): string;

/**
*  Parser to convert some user-edited value to an underlying data value of this type.
*  This parser is called when storing out values edited in a freeform editor such as
*  a TextItem. Typically this will convert from the format produced by 
*  SimpleType.editFormatter back to a data value.
* 
* 
* @param {string} edited value provided by the user
* @param {FormItem=} Editor for this field
* @param {DynamicForm=} DynamicForm containing this editor
* @param {Record=} Current edit values for this record, as displayed in   the edit component.
*/
parseInput(value:string, field?:FormItem, form?:DynamicForm, record?:Record): any;

/**
*  Returns the set of SimpleType.groupingModes available for values 
*  of this type in components that support grouping.
* 
*/
getGroupingModes(): ValueMap;

/**
*  Returns a group value appropriate for the passed record, field and value, in the passed 
*  component.
*  
* 
* @param {any} the record value to return a group value for
* @param {Record} the record containing the passed value
* @param {object} the field relating to the value to be processed
* @param {string} the name of the field relating to the value to be processed
* @param {Canvas} the component, usually a ListGrid, containing the              passed record
*/
getGroupValue(value:any, record:Record, field:object, fieldName:string, component:Canvas): any;

/**
*  Returns a string value appropriate for the title of the group containing the passed
*  value.
*  
* 
* @param {any} the record value to return a group title for
* @param {Record} the record containing the passed group value
* @param {object} the field relating to the value to be processed
* @param {string} the name of the field relating to the value to be processed
* @param {Canvas} the component, usually a ListGrid, containing the              passed record
*/
getGroupTitle(value:any, record:Record, field:object, fieldName:string, component:Canvas): string;




}

/**
* A record which specifies files for use with fileSource.
*/
class FileSpec {


/**
* The name of the file, without any extension to indicate FileSpec.fileType or
*  FileSpec.fileFormat.
* @type {string}
* @default null
*/
fileName?: string;

/**
* The type of the file, e.g. "ds" for datasource, or "proj" for project.
* @type {string}
* @default null
*/
fileType?: string;

/**
* The format of the file, e.g. "xml" or "js"
* @type {string}
* @default null
*/
fileFormat?: string;










}

/**
* CSS class name to apply to some HTML element on this page. This is a string that should
*  match the css class defined for the page in an external stylesheet or in inline
*  html &lt;STYLE&gt; tags.
*  
*  As a general rule, wherever it is possible to provide a CSS styleName (such as
*  Canvas.styleName or Button.baseStyle, your CSS style can specify border,
*  margins, padding, and any CSS attributes controlling background or text styling. You
*  should not specify any CSS properties related to positioning, clipping, sizing or
*  visibility (such as "overflow", "position", "display", "visibility" and "float") - use
*  SmartClient APIs for this kind of control.
*  
*  Because text wrapping cannot be consistently controlled cross-browser from CSS alone,
*  you should use SmartClient properties such as Button.wrap instead of the
*  corresponding CSS properties, when provided.
*  
*  Content contained within SmartClient components can use arbitrary CSS, with the
*  caveat that the content should be tested on all supported browsers, just as content
*  outside of SmartClient must be.
*  
*  Special note on CSS margins: wherever possible, use CSS padding and border in
*  lieu of CSS margins, or non-CSS approaches such as Layout.layoutMargin,
*  Canvas.snapTo, or absolute positioning (including specifying percentage left/top
*  coordinates). We recommend this because CSS specifies a very complicated and widely
*  criticized "margin-collapse" behavior which has surprising effects when margins exist on
*  both parents and children. Compounding the problem, margins are implemented very
*  differently on different browsers, especially when it comes to HTML margins.
*  
*  Note about CSS "box models"
*  
*  The CSS "box model" defines whether the size applied to a DOM element includes padding,
*  borders or margins, or whether such settings effectively increase the size of the
*  component beyond the size specified in CSS.
*  
*  In SmartClient, the size configured for a component includes border, padding and
*  margins if specified (in CSS terminology, the box model is "margin-box"). This allows
*  CSS borders, margins and padding to be treated as purely visual properties with no
*  effect on sizing or layout.
*/
type CSSStyleName = string;


/**
* A TileGrid is a DataBoundComponent that displays a list of objects as a set
*  of "tiles", where each tile represents one object, and the tiles are laid out in a grid with
*  multiple tiles per row. Each tile displays one or more properties of the object it
*  represents.
*/
class TileGrid extends TileLayout implements DataBoundComponent {
/**
* Creates a new TileGrid
* 
* @param typeCheckedProperties {Partial<TileGrid>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {TileGrid} The new instance
*/
static create(typeCheckedProperties? : Partial<TileGrid>, uncheckedProperties?: Object): TileGrid;


/**
* Array of field definitions to control the default rendering of tiles.
*  
*  If not specified, if the DataSource has an DataSource.iconField, only the
*  iconField and DataSource.titleField will be shown.
*  Otherwise, all non-DataSourceField.hidden
*  non-DataSourceField.detail fields will be shown, similar to the default set of
*  fields shown by a ListGrid.
*  
*  Only applicable if using the default SimpleTile class for tiles.
*  
*  For SimpleTiles, it is possible to use DetailViewerField.getCellStyle and 
*  StatefulCanvas.getStateSuffix to make a single field statefully styled:
*  
*  
*  isc.TileGrid.create({
*    fields:[
*      {name:'animalName',
*       getCellStyle : function (value, field, record, viewer) {
*         if (value == "Tiger") return "tigerStyle" + viewer.currentTile.getStateSuffix();
*         else return viewer.tileGrid.tileValueStyle + viewer.currentTile.getStateSuffix();
*       }
*      }
*    ]
*  });
*  
*  
*  
* @type {Array<Partial<DetailViewerField>>}
* @default null
*/
fields?: Array<Partial<DetailViewerField>>;

/**
* List of tiles that may be used by the TileGrid to show its current data set. Note that the
*  SmartClient framework manages this array for optimal performance, and not all tiles in the
*  array are necessarily visible or assigned a record. This is true regardless of whether
*  TileGrid.recycleTiles is set or not.
*  
*  The number of records in the TileGrid's current TileGrid.data
*  may be determined by calling List.getLength on it.
* @type {Array<Partial<Canvas>>}
* @default null
*/
tiles?: Array<Partial<Canvas>>;

/**
* By default, TileGrids will not show fields marked DataSourceField.detail
*  in the DataSource. See also TileGrid.fields.
* @type {boolean}
* @default false
*/
showDetailFields?: boolean;

/**
* When using the default SimpleTile, CSS style for each value shown within a tile.
* @type {CSSStyleName}
* @default "tileValue"
*/
tileValueStyle?: CSSStyleName;

/**
* Should tile values change state when the mouse goes over them?
* @type {boolean}
* @default false
*/
valuesShowRollOver?: boolean;

/**
* Should tile values change state when they are selected?
* @type {boolean}
* @default true
*/
valuesShowSelected?: boolean;

/**
* Should tile values change state when the mouse goes down on them?
* @type {boolean}
* @default false
*/
valuesShowDown?: boolean;

/**
* Horizontal alignment for tile values: "left", "right" or "center".
* @type {string}
* @default "center"
*/
tileValueAlign?: string;

/**
* Whether values should be allowed to wrap by default, or should be shown on one line
*  regardless of length.
* @type {boolean}
* @default false
*/
wrapValues?: boolean;

/**
* A List of TileRecord objects, specifying the data to be used to create the
*  tiles. 
*  
*  This property will typically not be explicitly specified for databound TileGrids, where
*  the data is returned from the server via databound component methods such as
*  TileGrid.fetchData. In this case the data objects will be set to a 
*  ResultSet rather than a simple array.
* @type {Array<Partial<Record>> | Array<Partial<TileRecord>> | any}
* @default null
*/
data?: Array<Partial<Record>> | Array<Partial<TileRecord>> | any;

/**
* How many tiles should be present in a line when printing?
* @type {number}
* @default null
*/
printTilesPerLine?: number;

/**
* The DataSource that this component should bind to for default fields and for performing
*  DSRequest.
*  
*  Can be specified as either a DataSource instance or the String ID of a DataSource.
* @type {DataSource | GlobalId}
* @default null
*/
dataSource?: DataSource | GlobalId;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from TileGrid.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only TileGrid.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default false
*/
autoFetchData?: boolean;

/**
* If TileGrid.autoFetchData is true, this attribute allows the developer to
*  specify a textMatchStyle for the initial ListGrid.fetchData call.
* @type {TextMatchStyle}
* @default "substring"
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Criteria to be used when TileGrid.autoFetchData is set.
*  
*  This property supports dynamicCriteria - use Criterion.valuePath
*  to refer to values in the Canvas.ruleScope.
* @type {Criteria}
* @default null
*/
initialCriteria?: Criteria;

/**
* Criteria that are never shown to or edited by the user and are cumulative with any criteria
*  provided via DataBoundComponent.initialCriteria and related methods
* @type {Criteria}
* @default null
*/
implicitCriteria?: Criteria;

/**
* How to fetch and manage records retrieve from the server. See FetchMode.
*  
*  This setting only applies to the ResultSet automatically created by calling
*  ListGrid.fetchData. If a pre-existing ResultSet is passed to setData() instead, it's
*  existing setting for ResultSet.fetchMode applies.
* @type {FetchMode}
* @default "paged"
*/
dataFetchMode?: FetchMode;

/**
* Defines a tileGrid's clickable-selection behavior.
* @type {SelectionStyle}
* @default "multiple"
*/
selectionType?: SelectionStyle;

/**
* A TileGrid automatically creates one tile per record in the dataset, via the
*  AutoChild pattern.
*  
*  By default, the SimpleTile class will be used, and will contain content as rendered
*  by a DetailViewer, based on the provided TileGrid.fields (or on the default
*  set of fields).
*  
*  
*  To create a completely different appearance, override TileGrid.tileConstructor with the name
*  of the custom SmartClient class to use for each tile. For example, subclass
*  SimpleTile and override Canvas.getInnerHTML, returning custom
*  HTML for each tile.
*  
*    isc.defineClass("MyCustomTile", "SimpleTile").addProperties({
*     getInnerHTML : function () {
*       return this.Super("getInnerHTML", arguments) +
*         this.getRecord().width + " x " + this.getRecord().height;
*     }
*    });
* 
*    isc.TileGrid.create({
*     tileConstructor:"MyCustomTile"
*    });
*  
*  
*  Note that you can also override tile behaviors on a per-record basis, via
*  TileRecord.tileConstructor and TileRecord.tileProperties.
*  
*  
* @type {SimpleTile}
* @default null
*/
tile?: SimpleTile;

/**
* Common properties to use when creating every tile.
* @type {Partial<Canvas>}
* @default null
*/
tileProperties?: Partial<Canvas>;

/**
* Classname of a SmartClient component to use for each tile rendered by this TileGrid. Tiles
*  are created by the AutoChild pattern; see TileGrid.tile.
*  
*  Any subclass of Canvas is allowed, but typically any custom class will derive from
*  SimpleTile.
*  
*  When using a custom component for tileConstructor, DataBoundComponents that display multiple
*  Records (ListGrid, DetailViewer) will have data provided via ListGrid.setData, and
*  components that display a single Record (DynamicForm) will have
*  DynamicForm.setValues called on them.
*  
*  If the component is not a recognized DataBoundComponent subclass, the Record can be
*  accessed via 
*  this.record.
*  
*  
*  If you implement particularly simple or particularly complex tile interfaces, you may wish
*  to adjust the property TileGrid.drawAllMaxTiles.
* @type {SCClassName}
* @default "SimpleTile"
*/
tileConstructor?: SCClassName;

/**
* DetailViewer instance used to render the content shown in Tiles by default.
* @type {DetailViewer}
* @default null
*/
detailViewer?: DetailViewer;

/**
* Properties for the DetailViewer that is automatically created to render the contents of tiles
*  by default.
* @type {Partial<DetailViewer>}
* @default null
*/
detailViewerProperties?: Partial<DetailViewer>;

/**
* This property determines whether tiles that are no longer visible (due to scrolling) are
*  recycled, allowing a large number of records to be displayed using a (potentially) much
*  smaller set of tiles.
*  
*  Recyling tiles may significantly reduce the number of live tile widgets needed to support
*  a particular TileGrid, but may also result in extra work when the TileGrid is scrolled, as
*  a scroll that brings off-screen tiles into view will require recycling tiles that have
*  left the view, even if the new tiles have been visited before (in previous scrolling).
*  
*  Recycling will occur when TileGrid.getTile is called, unless the supplied record
*  (or record specifed by index) is currently bound to an existing tile. Even if recycling
*  is not enabled, the record associated with a given tile may change if the TileGrid data
*  changes.
*  
*  For more control over the tile creation and recycling process, 
*  see TileGrid.createTile and TileGrid.updateTile.
* @type {boolean}
* @default true
*/
recycleTiles?: boolean;

/**
* Whether tiles are created and drawn for all records, or only for those currently visible.
*  
*  This setting is incompatible with TileGrid.dataFetchMode: "paged" as it requires all
*  records matching the criteria to be fetched from the server at once.
* @type {boolean}
* @default false
*/
showAllRecords?: boolean;

/**
* The string to display in the body of a tileGrid while data is being loaded.
*  Use "&#36;{loadingImage}" to include Canvas.loadingImageSrc.
* @type {HTMLString}
* @default "${loadingImage}&amp;nbsp;Loading data..."
*/
loadingDataMessage?: HTMLString;

/**
* The CSS style name applied to the loadingDataMessage string if displayed.
* @type {CSSStyleName}
* @default "loadingDataMessage"
*/
loadingDataMessageStyle?: CSSStyleName;

/**
* If drawing all tiles would cause no more than drawAllMaxTiles tiles to be
*  rendered, the full dataset will instead be drawn even if TileGrid.showAllRecords
*  is false and incremental rendering would have otherwise been used.
*  
*  The drawAllMaxTiles setting prevents incremental rendering from being used in
*  situations where it's really unnecessary, such as a 25 record dataset which happens to be
*  in a grid with a viewport showing only 15 or so tiles. Incremental rendering causes a brief
*  "flash" during scrolling as the visible portion of the dataset is redrawn, and a better 
*  scrolling experience can be obtained in this situation by drawing the entire dataset up
*  front, which in this example would have negligible effect on initial draw time.
*  
*  drawAllMaxTiles:0 disables this features. You may want to disable this feature
*  if performance is an issue and:
*  
*  
* - you very frequently redraw a grid
*  
* - you do a lot of computation when rendering each tile
*  
* - you are showing many grids on one screen and the user won't scroll most of them
*  
* @type {Integer}
* @default "25"
*/
drawAllMaxTiles?: Integer;

/**
* If set, when the dataset changes due to filtering, sorting or other actions, any tiles that
*  were showing before and after the change will animate from their old positions to their new
*  positions.
* @type {boolean}
* @default true
*/
animateTileChange?: boolean;

/**
* Style for the overall TileGrid component.
* @type {CSSStyleName}
* @default "tileGrid"
*/
styleName?: CSSStyleName;

/**
* Visual appearance to show when the tile is being dragged.
* @type {DragAppearance}
* @default "tracker"
*/
tileDragAppearance?: DragAppearance;

/**
* Indicates whether tiles can be reordered by dragging within this TileGrid.
*  
*  NOTE: If canReorderTiles is initially enabled or might be
*  TileGrid.setCanReorderTiles after the grid is created,
*  it may be desirable to disable Canvas.useTouchScrolling
*  so that touch-dragging a tile starts a reorder operation rather than a scroll. If
*  Canvas.disableTouchScrollingForDrag is set to true, then touch
*  scrolling will be disabled automatically. However, for accessibility
*  reasons, it is recommended to leave touch scrolling enabled and provide an alternative
*  set of controls that can be used to perform drag-reordering of tiles.
* @type {boolean}
* @default false
*/
canReorderTiles?: boolean;

/**
* Indicates whether tiles can be dragged from this TileGrid and dropped elsewhere.
*  
*  NOTE: If canDragTilesOut is initially enabled or might be
*  TileGrid.setCanDragTilesOut after the grid is created,
*  it may be desirable to disable Canvas.useTouchScrolling
*  so that touch-dragging a tile starts a drag operation rather than a scroll. If
*  Canvas.disableTouchScrollingForDrag is set to true, then touch
*  scrolling will be disabled automatically. However, for accessibility
*  reasons, it is recommended to leave touch scrolling enabled and provide an alternative
*  set of controls that can be used to perform drag and drop of tiles out of the grid.
* @type {boolean}
* @default false
*/
canDragTilesOut?: boolean;

/**
* Indicates whether records can be dropped into this TileGrid.
* @type {boolean}
* @default false
*/
canAcceptDroppedRecords?: boolean;

/**
* CSS Style to apply to the drag tracker when dragging occurs on this component.
* @type {CSSStyleName}
* @default "gridDragTracker"
*/
dragTrackerStyle?: CSSStyleName;

/**
* Indicates what to do with data dragged into another DataBoundComponent. See
*  DragDataAction type for details.
* @type {DragDataAction}
* @default "move"
*/
dragDataAction?: DragDataAction;

/**
* If you have a databound tileGrid and you scroll out of the currently loaded dataset, by
*  default you will see blank tiles until the server returns the data for those rows. The
*  loadingMessage attribute allows you to specify arbitrary html that will be shown in each
*  such "blank" tile while the data for that tile is loading.
*  (e.g. "&lt;DIV ALIGN='CENTER'&gt;LOADING&lt;/DIV&gt;")
* @type {HTMLString}
* @default null
*/
loadingMessage?: HTMLString;

/**
* Indicates whether the text of the emptyMessage property should be displayed if no data is
*  available.
* @type {boolean}
* @default true
*/
showEmptyMessage?: boolean;

/**
* The string to display in the body of a tileGrid with an empty data array, if
*  TileGrid.showEmptyMessage is true.
* @type {HTMLString}
* @default "No tiles to show."
*/
emptyMessage?: HTMLString;

/**
* The CSS style name applied to the TileGrid.emptyMessage if displayed.
* @type {CSSStyleName}
* @default "emptyMessage"
*/
emptyMessageStyle?: CSSStyleName;







/**
*  Retrieves data from the DataSource that matches the specified criteria.
*  
*  When fetchData() is first called, if data has not already been provided via
*  ListGrid.setData, this method will create a ResultSet, which will be
*  configured based on component settings such as DataBoundComponent.fetchOperation
*  and DataBoundComponent.dataPageSize, as well as the general purpose
*  ListGrid.dataProperties. The created ResultSet will automatically send a DSRequest
*  to retrieve data from ListGrid.dataSource, and from then on will 
*  automatically manage paging through large datasets, as well as performing filtering and
*  sorting operations inside the browser when possible - see the ResultSet docs for
*  details.
*  
*  NOTE: do not use both DataBoundComponent.autoFetchData and a
*  call to fetchData() - this may result in two DSRequests to fetch data. Use
*  either DataBoundComponent.autoFetchData and Criteria or a manual call to fetchData()
*  passing criteria.
*  
*  Whether a ResultSet was automatically created or provided via ListGrid.setData, subsequent
*  calls to fetchData() will simply call ResultSet.setCriteria.
*  
*  Changes to criteria may or may not result in a DSRequest to the server due to
*  ResultSet.useClientFiltering. You can call
*  DataBoundComponent.willFetchData to determine if new criteria will result in a
*  server fetch.
*  
*  If you need to force data to be re-fetched, you can call
*  ListGrid.invalidateCache and new data will automatically be fetched
*  from the server using the current criteria and sort direction. NOTE: when using
*  invalidateCache() there is no need to also call fetchData()
*  and in fact this could produce unexpected results.
*  
*  This method takes an optional callback parameter (set to a DSCallback) to fire when
*  the fetch completes. Note that this callback will not fire if no server fetch is performed.
*  In this case the data is updated synchronously, so as soon as this method completes you
*  can interact with the new data. If necessary, you can use
*  DataBoundComponent.willFetchData to determine whether or not a server
*  fetch will occur when fetchData() is called with new criteria.
*  
*  In addition to the callback parameter for this method, developers can use 
*  ListGrid.dataArrived to be notified every time data is loaded.
*  
*  By default, this method assumes a TextMatchStyle of "exact"; that can be overridden
*  by supplying a different value in the requestProperties parameter. 
*  See DataBoundComponent.willFetchData;
* 
* 
* @param {Criteria=} Search criteria. If a DynamicForm is passed                     in as this argument instead of a raw criteria                      object, will be derived by calling                      DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required
* @param {DSRequest=} additional properties to set on the DSRequest                     that will be issued
*/
fetchData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Retrieves data that matches the provided criteria and displays the matching data in this
*  component.
*  
*  This method behaves exactly like ListGrid.fetchData except that
*  DSRequest.textMatchStyle is automatically set to "substring" so that String-valued
*  fields are matched by case-insensitive substring comparison.
* 
* 
* @param {Criteria=} Search criteria.            If a DynamicForm is passed in as this argument            instead of a raw criteria object, will be derived by calling            DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required; see                      ListGrid.fetchData for details
* @param {DSRequest=} for databound components only - optional              additional properties to set on the DSRequest that will be issued
*/
filterData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Invalidate the current data cache for this databound component via a call to
*  the dataset's invalidateCache() method, for example,
*  ResultSet.invalidateCache.
*  
*  NOTE: there is no need to call invalidateCache() when a save operation
*  is performed on a DataSource. Automatic cache synchronization features will automatically
*  update caches - see ResultSet for details. If automatic cache synchronization isn't
*  working, troubleshoot the problem using the steps suggested 
*  in the FAQ rather
*  than just calling invalidateCache(). Calling invalidateCache() unnecessarily
*  causes extra server load and added code complexity.
*  
*  Calling invalidateCache() will automatically cause a new fetch to 
*  be performed with the current set of criteria if data had been previously fetched and the
*  component is currently drawn with data visible - there is no need to manually call
*  fetchData() after invalidateCache() and this could result in duplicate fetches. 
*  
*  While data is being re-loaded after a call to invalidateCache(), the widget is
*  in a state similar to initial data load - it doesn't know the total length of the dataset
*  and any APIs that act on records or row indices will necessarily fail and should not be
*  called. To detect that the widget is in this state, call ResultSet.lengthIsKnown.
*  
*  invalidateCache() only has an effect if this components dataset is a data
*  manager class that manages a cache (eg ResultSet or ResultTree). If data was provided as a
*  simple Array or List, invalidateCache() does nothing.
*  
* 
*/
invalidateCache(): void;

/**
*  Remove the currently selected records from this component.
*  If this is a databound grid, the records will be removed directly from the DataSource.
*  
*  If no records are selected, no action is taken. The grid will automatically be
*  updated if the record deletion succeeds.
* 
* 
* @param {DSCallback=} callback to fire when the data has been removed
* @param {DSRequest=} additional properties to set on the DSRequest                     that will be issued
*/
removeSelectedData(callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Returns all selected records, as an Array.
* 
* 
* @param {boolean=} When true, partially selected records will not be returned.                  Otherwise, both fully and partially selected records are                  returned.
*/
getSelection(excludePartialSelections?:boolean): Array<Partial<ListGridRecord>>;

/**
*  Returns the first selected record in this component.
*  
*  NOTE: If a record is returned, it should be treated as read-only and not
*  modified.
* 
*/
getSelectedRecord(): TileRecord;

/**
* 
*  Select/deselect a Record passed in explicitly, or by index.
* 
* 
* @param {Record | number} record (or row number) to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRecord(record:Record | number, newState?:boolean): void;

/**
* 
*  Deselect a Record passed in explicitly, or by index.
*  
*  Synonym for selectRecord(record, false)
* 
* 
* @param {Record | number} record (or row number) to deselect
*/
deselectRecord(record:Record | number): void;

/**
* 
*  Select/deselect a list of Records passed in explicitly, or by index.
*  
*  Note that developers may wish to use TileGrid.selectRange to select a single
*  contiguous range.
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRecords(records:Array<Partial<Record>> | number, newState?:boolean): void;

/**
*  Select a contiguous range of records by index
* 
* @param {Integer} start of selection range
* @param {Integer} end of selection range (non-inclusive)
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectRange(startRow:Integer, endRow:Integer, newState?:boolean): void;

/**
* 
*  Deselect a list of Records passed in explicitly, or by index.
*  
*  Synonym for selectRecords(records, false)
*  
*  Note that developers may wish to use TileGrid.deselectRange to select a single
*  contiguous range.
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to deselect
*/
deselectRecords(records:Array<Partial<Record>> | number): void;

/**
*  Deselect a contiguous range of records by index.
*  
*  This is a synonym for selectRange(startRow, endRow, false);
* 
* @param {Integer} start of selection range
* @param {Integer} end of selection range (non-inclusive)
*/
deselectRange(startRow:Integer, endRow:Integer): void;

/**
*  Select all records
* 
* 
*/
selectAllRecords(): void;

/**
* 
*  Deselect all records
* 
* 
*/
deselectAllRecords(): void;

/**
*  Whether at least one item is selected
* 
*/
anySelected(): boolean;

/**
*  If defined, this method will be called when a new tile is required. Note that
*  this method is in complete control of how the tile is constructed, so that
*  properties such as TileGrid.tileProperties and others needed by TileGrid
*  will be applied only after this method returns.
* 
* 
* @param {Record} record that will be associated with new tile
* @param {Integer} index of the record in the tileGrid
*/
createTile(record:Record, tileIndex:Integer): Canvas;

/**
*  If both this method and TileGrid.createTile are defined and
*  TileGrid.recycleTiles is true, this method will be called when the
*  framework needs to recycle a tile to be used with a new record. This
*  notification provides an opportunity to update any widget properties that
*  depend on the specifics of the record.
* 
* 
* @param {Record} record that will be associated with the recycled tile
* @param {Integer} index of the record in the tileGrid
* @param {Canvas} the tile to be recycled
*/
updateTile(record:Record, tileIndex:Integer, reclaimedTile:Canvas): void;

/**
*  Given a tile within this this tile-grid, this method will return the associated record.
* 
* 
* @param {Canvas} Tile you want to get the record for
*/
getTileRecord(tile:Canvas): TileRecord;

/**
*  Provides a new data set to the TileGrid after the grid has been created or drawn.
*  The TileGrid will redraw to show the new data automatically.
* 
* @param {Array<Partial<Record>> | Array<Partial<TileRecord>> | ResultSet} data to show in the list
*/
setData(newData:Array<Partial<Record>> | Array<Partial<TileRecord>> | ResultSet): void;

/**
*  When using the default SimpleTile class as TileGrid.tileConstructor, this
*  method provides the HTML to be displayed within each tile. See TileGrid.tile.
*  
* 
* @param {TileRecord} the tile for which HTML should be retrieved
*/
getTileHTML(tileRecord:TileRecord): HTMLString;

/**
*  Returns the tile for the passed record or record index.
*  
*  Note that this method may be overridden but developers should be aware that this
*  method may be called repeatedly for the same record each time the TileGrid refreshes
*  that row. If you override this API, you will need to cache and re-use the same
*  tile objects per record. Typically this would be achieved by storing a pool of Tile
*  objects that are re-used if a Record with the same primaryKey is passed to getTile().
*  
*  When calling this method directly, if TileGrid.showAllRecords is false, this may 
*  return null for records that are not currently visible.
* 
* 
* @param {TileRecord | Integer} record or index of record in this.data
*/
getTile(tile:TileRecord | Integer): Canvas;

/**
*  Returns the index of the specified tile.
* 
* @param {Canvas} Tile you want to get the index for
*/
getTileIndex(tile:Canvas): Integer;

/**
*  Returns the tile currently under the mouse.
* 
*/
getCurrentTile(): SimpleTile;

/**
*  Executed when the tileGrid receives a 'click' event on a
*  tile. The default implementation does nothing -- override to perform some action
*  when any record is clicked.
*  A record event handler can be specified either as
*  a function to execute, or as a string of script to evaluate. If the handler is defined
*  as a string of script, all the parameters below will be available as variables for use
*  in the script.
*  If you want to cancel the click based on the parameters, return false. Otherwise, return 
*  true so that the click event be registered with the tile.
* 
* 
* @param {TileGrid} the TileGrid itself
* @param {Canvas} the tile that was clicked on
* @param {TileRecord} the record that was clicked on
*/
recordClick(viewer:TileGrid, tile:Canvas, record:TileRecord): void;

/**
*  Executed when the tileGrid receives a 'doubleclick' event on a
*  tile. The default implementation does nothing -- override to perform some action
*  when any record is doubleclicked.
*  A record event handler can be specified either as
*  a function to execute, or as a string of script to evaluate. If the handler is defined
*  as a string of script, all the parameters below will be available as variables for use
*  in the script.
*  If you want to cancel the doubleclick based on the parameters, return false. Otherwise, return 
*  true so that the doubleclick event be registered with the tile.
* 
* 
* @param {TileGrid} the TileGrid itself
* @param {Canvas} the tile that was doubleclicked on
* @param {TileRecord} the record that was doubleclicked on
*/
recordDoubleClick(viewer:TileGrid, tile:Canvas, record:TileRecord): void;

/**
*  Executed when the tileGrid receives a context-click (right mouse button) event on a
*  tile. The default implementation does nothing -- override to perform some action
*  when any record is right-clicked.
*  Return false to cancel the native behavior (suppressing
*  the browser context menu).
*  
*  
*  A record event handler can be specified either as
*  a function to execute, or as a string of script to evaluate. If the handler is defined
*  as a string of script, all the parameters below will be available as variables for use
*  in the script.
*  If you want to cancel the click based on the parameters, return false. Otherwise, return 
*  true so that the click event be registered with the tile.
* 
* 
* @param {TileGrid} the TileGrid itself
* @param {Canvas} the tile that was clicked on
* @param {TileRecord} the record that was clicked on
*/
recordContextClick(viewer:TileGrid, tile:Canvas, record:TileRecord): boolean;

/**
*  Called when selection changes within this tileGrid. Note this method fires for
*  each record for which selection is modified - so when a user clicks inside a tileGrid this
*  method will typically fire twice (once for the old record being deselected, and once for
*  the new record being selected).
* 
* 
* @param {Record} record for which selection changed
* @param {boolean} New selection state (true for selected, false for unselected)
*/
selectionChanged(record:Record, state:boolean): void;

/**
*  This is not allowed for tileGrid. Instead, use TileGrid.addData.
* 
* 
*/
addTile(): void;

/**
*  This is not allowed for tileGrid. Instead, use TileGrid.removeData.
* 
* 
*/
removeTile(): void;

/**
*  Perform a DataSource "add" operation to add new records to this component's DataSource.
* 
* 
* @param {Record} new record
* @param {DSCallback=} method to call on operation completion
* @param {Partial<DSRequest>=} additional properties to set on the DSRequest                     that will be issued
*/
addData(newRecord:Record, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Perform a DataSource "remove" operation to remove records from this component's
*  DataSource.
* 
* 
* @param {Record} primary key values of record to delete,                      (or complete record)
* @param {DSCallback=} method to call on operation completion
* @param {Partial<DSRequest>=} additional properties to set on the DSRequest                     that will be issued
*/
removeData(data:Record, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Setter for TileGrid.canReorderTiles.
* 
*/
setCanReorderTiles(): void;

/**
*  Setter for TileGrid.canDragTilesOut.
* 
*/
setCanDragTilesOut(): void;

/**
*  Setter for TileGrid.canAcceptDroppedRecords.
* 
*/
setCanAcceptDroppedRecords(): void;

/**
*  Sets the custom tracker HTML to display next to the mouse when the user initiates a drag
*  operation on this component. Default implementation will examine ListGrid.dragTrackerMode
*  and set the custom drag tracker to display the appropriate HTML based on the selected record.
*  
*  To display custom drag tracker HTML, this method may be overridden - call 
*  EventHandler.setDragTracker to actually update the drag tracker HTML.
* 
*/
setDragTracker(): boolean;

/**
*  Method to return the fieldName which represents the "title" for records in this
*  Component.
*  If this.titleField is explicitly specified it will always be used.
*  Otherwise, default implementation will check DataSource.titleField for databound
*  compounds.
*  For non databound components returns the first defined field name of "title",
*  "name", or "id" where the field is visible. If we don't find any
*  field-names that match these titles, the first field in the component will be used instead.
* 
*/
getTitleField(): string;

/**
*  Return "title" HTML to display as a drag tracker when the user drags some record.
*  Default implementation will display the cell value for the title field (see 
*  ListGrid.getTitleField) for the record(s) being dragged (including any
*  icons / custom formatting / styling, etc).
*  
*  Note: Only called if ListGrid.dragTrackerMode is set to "title".
* 
* @param {ListGridRecord} First selected record being dragged
* @param {number} row index of first record being dragged
*/
getDragTrackerTitle(record:ListGridRecord, rowNum:number): string;

/**
* 
*  During a drag-and-drop interaction, this method is called to transfer a set of records that
*  were dropped onto some other component. This method is called after the set of records has
*  been copied to the other component. Whether or not this component's data is modified is 
*  determined by the value of DataBoundComponent.dragDataAction.
*  
*  With a dragDataAction of "move", a databound component will issue "remove"
*  dsRequests against its DataSource to actually remove the data, via
*  DataSource.removeData.
* 
* 
*/
transferDragData(): Array<any>;

/**
* 
*  During a drag-and-drop interaction, this method returns the set of records being dragged out
*  of the component. In the default implementation, this is the list of currently selected
*  records.
*  
*  This method is consulted by ListGrid.willAcceptDrop.
*  
*  NOTE: If this component is a Tree.isMultiLinkTree TreeGrid, 
*  this method returns a list of NodeLocators rather than a list of records. Each 
*  nodeLocator contains a pointer to the associated record in its node
*  property.
* 
* 
* @param {DataBoundComponent} source component from which the records will be transferred
*/
getDragData(source:DataBoundComponent): Array<Partial<Record>>;

/**
*  Simulates a drag / drop type transfer of the selected records in some other component to this
*  component, without requiring any user interaction. This method acts on the dropped records 
*  exactly as if they had been dropped in an actual drag / drop interaction, including any 
*  special databound behavior invoked by calling
*  DataBoundComponent.getDropValues for each dropped record.
*  
*  To transfer all data in, for example, a ListGrid, call 
*  ListGrid.selectAllRecords first.
*  
*  Note that drag/drop type transfers of records between components are asynchronous operations:
*  SmartClient may need to perform server turnarounds to establish whether dropped records 
*  already exist in the target component. Therefore, it is possible to issue a call to 
*  transferSelectedData() and/or the ListGrid.drop method of a databound 
*  component whilst a transfer is still active. When this happens, SmartClient adds the 
*  second and subsequent transfer requests to a queue and runs them one after the other. If 
*  you want to be notified when a transfer process has actually completed, either provide a 
*  callback to this method or implement DataBoundComponent.dropComplete.
*  
*  See the dragging documentation for an overview of list grid drag/drop data
*  transfer.
*  
* 
* @param {DataBoundComponent} source component from which the records will be transferred
* @param {Integer=} target index (drop position) of the rows within this grid.
* @param {Callback=} optional callback to be fired when the transfer process has            completed. The callback will be passed a single parameter "records",            the list of records actually transferred to this component.
*/
transferSelectedData(source:DataBoundComponent, index?:Integer, callback?:Callback): void;

/**
*  Sets some presentation properties (visibility, width, userFormula and userSummary) of the 
*  grid fields based on the ListGridFieldState object passed in.
*  Used to restore previous state retrieved from the grid by a call to TileGrid.getFieldState.
* 
* 
* @param {ListGridFieldState} state to apply to the grid's fields.
*/
setFieldState(fieldState:ListGridFieldState): void;

/**
*  Returns a snapshot of the current presentation of this grid's fields as 
*  a ListGridFieldState object.
*  
*  This object can be passed to TileGrid.setFieldState to reset this grid's fields to
*  the current state.
*  
*  Note that the information stored includes the current width and visibility of each of this 
*  grid's fields.
* 
* 
*/
getFieldState(): ListGridFieldState;

/**
*  Notification method fired when new data arrives from the server to be displayed in this
*  tileGrid, (for example in response to the user scrolling a new set of tiles into view).
*  Only applies to databound tileGrid where the TileGrid.data attribute is a
*  ResultSet.
*  This method is fired directly in
*  response to ResultSet.dataArrived firing on the data object.
* 
* @param {Integer} starting index of the newly loaded set of records
* @param {Integer} ending index of the newly loaded set of records (non inclusive).
*/
dataArrived(startRecord:Integer, endRecord:Integer): void;

/**
*  Sends the current filter criteria and sort direction to the server, then exports data in the
*  requested DSRequest.exportAs.
*  
*  A variety of DSRequest settings, such as 
*  DSRequest.exportAs and DSRequest.exportFilename, affect the 
*  exporting process: see DSRequest.exportResults for further detail.
*  
*  Note that data exported via this method does not include any client-side formatting and
*  relies on both the SmartClient server and server-side DataSources. To export client-data 
*  with formatters applied, 
*  see ListGrid.exportClientData, which still requires the
*  SmartClient server but does not rely on server-side DataSource definitions (.ds.xml files).
*  
*  For more information on exporting data, see DataSource.exportData.
* 
* 
* @param {DSRequest=} additional properties to set on the DSRequest                      that will be issued
* @param {DSCallback=} callback to invoke on completion. Note that this parameter                 only applies where DSRequest.exportToClient is                  explicitly set to false, because file downloads do not                  provide ordinary SmartClient callbacks
*/
exportData(requestProperties?:DSRequest, callback?:DSCallback): void;



// stubs from interface: DataBoundComponent
// -----------------------------------------------------------------------------------------------------------


/**
* Before we start editing values in this DataBoundComponent, should we perform a deep clone 
*  of the underlying values. See DataSource.deepCloneOnEdit for details of what this means.
*  
*  If this value is not explicitly set, it defaults to the value of DataSource.deepCloneOnEdit.
*  This value can be overridden per-field with DataSourceField.deepCloneOnEdit.
*  
*  Like the other deepCloneOnEdit settings, this flag only has an effect if you are 
*  editing a values object that contains nested objects or arrays, using 
*  Canvas.dataPaths.
* @type {boolean}
* @default null
*/
deepCloneOnEdit?: boolean;

/**
* When editing values in this DataBoundComponent, should we perform a deep clone of values 
*  that are not associated with a field (ie, attributes on the record that do not map to a 
*  component field either directly by name, or by FormItem.dataPath. If this value is
*  not explicitly set, it defaults to the value of 
*  DataSource.deepCloneNonFieldValuesOnEdit if there is a dataSource, or to the value 
*  of the static DataSource.deepCloneNonFieldValuesOnEdit if there is no
*  dataSource.
*  
*  Like the other deepCloneOnEdit settings, this flag only has an effect if you are 
*  editing a values object that contains nested objects or arrays.
* @type {boolean}
* @default null
*/
deepCloneNonFieldValuesOnEdit?: boolean;

/**
* When using DataBoundComponent.dataFetchMode, how many records to fetch at a time. If set to
*  a positive integer, dataPageSize will override the default 
*  ResultSet.resultSize for ResultSets automatically created when you call
*  ListGrid.fetchData (and similarly for the ResultTree.resultSize of
*  ResultTrees). Leaving dataPageSize at its default means to just use the default page
*  size of the data container.
*  
*  Note that regardless of the dataPageSize setting, a component will always fetch
*  all of data that it needs to draw. Settings such as
*  ListGrid.showAllRecords,
*  ListGrid.drawAllMaxCells and
*  ListGrid.drawAheadRatio can cause more rows than the configured
*  dataPageSize to be fetched.
* @type {Integer}
* @default null
*/
dataPageSize?: Integer;

/**
* Delay in milliseconds before fetching data.
*  
*  This setting only applies to the ResultSet automatically created by calling
*  ListGrid.fetchData. If a pre-existing ResultSet is passed to setData() instead, it's
*  existing setting for ResultSet.fetchDelay applies.
* @type {Integer}
* @default null
*/
dataFetchDelay?: Integer;

/**
* If true, the set of fields given by the "default binding" (see 
*  DataBoundComponent.fields) is used, with any fields specified in
*  component.fields acting as overrides that can suppress or modify the
*  display of individual fields, without having to list the entire set of fields that
*  should be shown.
*  
*  If component.fields contains fields that are not found in the DataSource,
*  they will be shown after the most recently referred to DataSource field. If the new
*  fields appear first, they will be shown first.
*  
*  ${isc.DocUtils.linkForExampleId('validationFieldBinding', 'This example')} shows a mixture of component
*  fields and DataSource fields, and how they interact for validation.
*  
*  This setting may be cleared if a FieldPicker is used to edit the component's field
*  order.
* @type {boolean}
* @default false
*/
useAllDataSourceFields?: boolean;

/**
* Whether to show fields marked hidden:true when a DataBoundComponent is given a
*  DataSource but no component.fields.
*  
*  The hidden property is used on DataSource fields to mark fields that are
*  never of meaning to an end user.
* @type {boolean}
* @default false
*/
showHiddenFields?: boolean;

/**
* Whether to show fields of non-atomic types when a DataBoundComponent is given a
*  DataSource but no component.fields.
*  
*  If true, the component will show fields that declare a complex type, for example, a
*  field 'shippingAddress' that declares type 'Address', where 'Address' is the ID of a
*  DataSource that declares the fields of a shipping address (city, street name, etc).
*  
*  Such fields may need custom formatters or editors in order to create a usable interface,
*  for example, an Address field in a ListGrid might use a custom formatter to combine the
*  relevant fields of an address into one column, and might use a pop-up dialog for
*  editing.
* @type {boolean}
* @default true
*/
showComplexFields?: boolean;

/**
* DSRequest.operationId this component should use when performing fetch operations.
* @type {string}
* @default null
*/
fetchOperation?: string;

/**
* DSRequest.operationId this component should use when performing update operations.
* @type {string}
* @default null
*/
updateOperation?: string;

/**
* DSRequest.operationId this component should use when performing add operations.
* @type {string}
* @default null
*/
addOperation?: string;

/**
* DSRequest.operationId this component should use when performing remove operations.
* @type {string}
* @default null
*/
removeOperation?: string;

/**
* The list of field-names to export. If provided, the field-list in the exported output is 
*  limited and sorted as per the list.
*  
*  If exportFields is not provided, the exported output includes all visible fields 
*  from this component, sorted as they appear.
* @type {Array<Partial<string>>}
* @default null
*/
exportFields?: Array<Partial<string>>;

/**
* Setting exportAll to true prevents the component from passing it's list of fields to the 
*  export call. The result is the export of all visible fields from DataSource.fields.
*  
*  If exportAll is false, an export operation will first consider 
*  DataBoundComponent.exportFields, if it's set, and fall back on all visible fields from
*  DataSource.fields otherwise.
* @type {boolean}
* @default false
*/
exportAll?: boolean;

/**
* If Summary rows exist for this component, whether to include them when exporting client data.
* @type {boolean}
* @default true
*/
exportIncludeSummaries?: boolean;

/**
* Flag controlling when to recategorize records being dropped on a databound
*  component from another databound component.
* @type {RecategorizeMode}
* @default "checked"
*/
dragRecategorize?: RecategorizeMode;

/**
* If set, detect and prevent duplicate records from being transferred to this component, either via
*  drag and drop or via DataBoundComponent.transferSelectedData. When a duplicate transfer is detected,
*  a dialog will appear showing the DataBoundComponent.duplicateDragMessage.
*  
*  If the component either does not have a DataSource or has a DataSource with no
*  DataSourceField.primaryKey declared, duplicate checking is off by
*  default. If duplicate checking is enabled, it looks for an existing record in the dataset
*  that has all of the properties of the dragged record, and considers that a duplicate.
*  
*  For DragDataAction:"copy" where the target DataSource is related to the source
*  DataSource by foreignKey, a duplicate means that the target list, as filtered by the current
*  criteria, already has a record whose value for the foreignKey field matches the
*  primaryKey of the record being transferred.
*  
*  For example, consider dragging "employees" to "teams", where "teams" has a field
*  "teams.employeeId" which is a foreignKey pointing to "employees.id", and the target
*  grid has search criteria causing it to show all the members of one team. A duplicate -
*  adding an employee to the same team twice - is when the target grid's dataset contains an
*  record with "employeeId" matching the "id" field of the dropped employee.
* @type {boolean}
* @default null
*/
preventDuplicates?: boolean;

/**
* Message to show when a user attempts to transfer duplicate records into this component, and
*  DataBoundComponent.preventDuplicates is enabled.
*  
*  If set to null, duplicates will not be reported and the dragged duplicates will not be
*  saved.
* @type {string}
* @default "Duplicates not allowed"
*/
duplicateDragMessage?: string;

/**
* Indicates whether the text of the offlineMessage property should be displayed if no data is
*  available because we do not have a suitable offline cache
* @type {boolean}
* @default true
*/
showOfflineMessage?: boolean;

/**
* Message to display when this DataBoundComponent attempts to load data that is not available
*  because the browser is currently offline. Depending on the component, the message is either
*  displayed in the component's body, or in a pop-up warning dialog.
* @type {string}
* @default "This data not available while offline"
*/
offlineMessage?: string;

/**
* Message to display when this DataBoundComponent attempts to save data while the application
*  is offline.
* @type {string}
* @default "Data cannot be saved because you are not online"
*/
offlineSaveMessage?: string;

/**
* Indicates whether to add "drop values" to items dropped on this component, if both 
*      the source and target widgets are databound, either to the same DataSource or 
*      to different DataSources that are related via a foreign key. "Drop values" are 
*      properties of the dropped item that you wish to change (and persist) as a 
*      result of the item being dropped on this grid.
*      
*      If this value is true and this component is databound, DataBoundComponent.getDropValues will 
*      be called for every databound item dropped on this grid, and an update performed
*      on the item
* @type {boolean}
* @default true
*/
addDropValues?: boolean;

/**
* When an item is dropped on this component, and DataBoundComponent.addDropValues is true and both 
*      the source and target widgets are databound, either to the same DataSource or 
*      to different DataSources that are related via a foreign key, this object 
*      provides the "drop values" that SmartClient will apply to the dropped object 
*      before updating it.
*      
*      If this property is not defined, SmartClient defaults to returning the selection
*      criteria currently in place for this component. Thus, any databound items (for example, 
*      rows from other grids bound to the same DataSource) dropped on the grid will,
*      by default, be subjected to an update that makes them conform to the grid's 
*      current filter criteria.
* @type {object}
* @default null
*/
dropValues?: object;

/**
* Does this component represent singular or multiple "records" objects?
*  Options are "multiple" or "single", or "either"
* @type {string}
* @default "multiple"
*/
dataArity?: string;

/**
* Indicates whether or not this component will load its data 
*  DataSource.progressiveLoading.
* @type {boolean}
* @default null
*/
progressiveLoading?: boolean;

/**
* If this component is bound to a dataSource, this attribute may be specified to customize
*  what fields from the dataSource may be edited by default. For example the SearchForm
*  class has this attribute set to "canFilter" which allows search forms to edit
*  dataSource fields marked as canEdit:false (but not those marked as
*  canFilter:false).
*  
*  Note that if canEdit is explicitly specified on a field in 
*  the DataBoundComponent.fields array, that property will be respected in preference to 
*  the canEditAttribute value. (See FormItem.canEdit, ListGridField.canEdit).
*  Also note that individual dataBoundComponents may have additional logic around whether a field
*  can be edited - for example ListGrid.canEditCell may be overridden.
* @type {Identifier}
* @default "canEdit"
*/
canEditFieldAttribute?: Identifier;

/**
* The useFlatFields flag causes all simple type fields anywhere in a nested
*  set of DataSources to be exposed as a flat list for form binding. 
*  
*  useFlatFields is typically used with imported metadata, such as 
*  XMLTools.loadXMLSchema from a 
*  XMLTools.loadWSDL, as a means of eliminating levels of XML
*  nesting that aren't meaningful in a user interface, without the cumbersome and fragile
*  process of mapping form fields to XML structures.
*  
*  For example, having called WebService.getInputDS to retrieve the input message
*  schema for a web service operation whose input message looks like this:
*  
*  &lt;FindServices&gt;
*    &lt;searchFor&gt;search text&lt;/searchFor&gt;
*    &lt;Options&gt;
*      &lt;caseSensitive&gt;false&lt;/caseSensitive&gt;
*    &lt;/Options&gt;
*    &lt;IncludeInSearch&gt;
*      &lt;serviceName&gt;true&lt;/serviceName&gt;
*      &lt;documentation&gt;true&lt;/documentation&gt;
*      &lt;keywords&gt;true&lt;/keywords&gt;
*    &lt;/IncludeInSearch&gt;
*  &lt;/FindServices&gt;
*  
*  Setting useFlatFields on a DynamicForm that is bound to this input
*  message schema would result in 5 FormItem reflecting the 5 simple type
*  fields in the message.
*  
*  For this form, the result of DynamicForm.getValues might look
*  like:
*  
*  {
*   searchFor: "search text",
*   caseSensitive: false,
*   serviceName: true,
*   documentation : true,
*   keywords : true
*  }
*  When contacting a WebService, these values can be automatically
*  mapped to the structure of the input message for a web service operation by setting
*  WSRequest.useFlatFields (for use with WebService.callOperation) or by setting
*  DSRequest.useFlatFields (for use with a DataSource that is
*  wsdlBinding via
*  OperationBinding.wsOperation). 
*  
*  Using these two facilities in conjunction (component.useFlatFields and
*  request.useFlatFields) allows gratuitous nesting to be consistently bypassed in both the user
*  presentation and when providing the data for XML messages.
*  
*  You can also set OperationBinding.useFlatFields to automatically enable 
*  "flattened" XML serialization (request.useFlatFields) for all DataSource requests of a
*  particular operationType.
*  
*  Note that useFlatFields is not generally recommended for use with structures
*  where multiple simple type fields exist with the same name, however if used with such a
*  structure, the first field to use a given name wins. "first" means the first field
*  encountered in a depth first search. "wins" means only the first field will be present as a
*  field when data binding.
* @type {boolean}
* @default null
*/
useFlatFields?: boolean;

/**
* If this attribute is set to false, any attributes in the component's values object that do
*  not map to a DataSourceField or FormItem will not 
*  be tracked when checking for changes. You should only set this flag to false if you know 
*  that your code does not store additional, non-field values in the component's data, or if 
*  you do store such values, but you don't care that they are not checked for changes. 
*  This flag is primarily provided to avoid performance issues in cases where developers are
*  storing large numbers of extra attributes in component data; generally speaking, you should
*  only consider setting it to false if you have a use case like this.
*  
*  Note, even with this flag set to false, these extra values will still be managed and stored 
*  by SmartClient; they just will not be checked when the component's values are inspected to 
*  see if they have changed. This may lead to methods like ListGrid.rowHasChanges
*  returning false when you are expecting it to return true. In this case, either switch this 
*  flag back to true (or just do not set it false), or provide a field definition for the 
*  affected attribute(s).
* @type {boolean}
* @default true
*/
canChangeNonFieldValues?: boolean;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define
*  grid hilites using the HiliteEditor.
*  
*  User-added hilites can be persisted via DataBoundComponent.getHiliteState and 
*  DataBoundComponent.setHiliteState.
* @type {boolean}
* @default false
*/
canEditHilites?: boolean;

/**
* Hilites to be applied to the data for this component. See hiliting.
* @type {Array<Partial<Hilite>>}
* @default null
*/
hilites?: Array<Partial<Hilite>>;

/**
* Specifies a list of icons that can be used in DataBoundComponent.editHilites.
*  
*  hiliteIcons should be specified as an Array of SCImgURL.
*  When present, the hilite editing interface shown when DataBoundComponent.editHilites is called 
*  will offer the user a drop down for picking one of these icons when defining either a 
*  simple or advanced hilite rule.
*  
*  If the user picks an icon, the created hiliting rule will have Hilite.icon set to 
*  the chosen icon. DataBoundComponent.hiliteIconPosition controls where the icon will 
*  appear for that field -- the default is that it appears in front of the normal cell content.
*  This can also be overridden at the field level.
* @type {Array<Partial<string>>}
* @default "["[SKINIMG]/Dialog/notify.png", "[SKINIMG]/Dialog/warn.png", "[SKINIMG]/actions/approve.png"]"
*/
hiliteIcons?: Array<Partial<string>>;

/**
* When DataBoundComponent.hiliteIcons are present, where the hilite icon will be placed 
*  relative to the field value. See HiliteIconPosition. Can be overridden at the
*  field level.
* @type {HiliteIconPosition}
* @default "before"
*/
hiliteIconPosition?: HiliteIconPosition;

/**
* Default width and height of DataBoundComponent.hiliteIcons for this component.
*  Can be overridden at the component level via explicit 
*  DataBoundComponent.hiliteIconWidth and
*  DataBoundComponent.hiliteIconHeight, or at the field level via 
*  ListGridField.hiliteIconSize,
*  ListGridField.hiliteIconWidth and 
*  ListGridField.hiliteIconHeight
* @type {number}
* @default "12"
*/
hiliteIconSize?: number;

/**
* Width for hilite icons for this component.
*  Overrides DataBoundComponent.hiliteIconSize.
*  Can be overridden at the field level.
* @type {number}
* @default null
*/
hiliteIconWidth?: number;

/**
* Height for hilite icons for this listGrid.
*  Overrides DataBoundComponent.hiliteIconSize.
*  Can be overridden at the field level
* @type {number}
* @default null
*/
hiliteIconHeight?: number;

/**
* How much padding should there be on the left of DataBoundComponent.hiliteIcons by default?
*  Can be overridden at the field level
* @type {number}
* @default "2"
*/
hiliteIconLeftPadding?: number;

/**
* How much padding should there be on the right of DataBoundComponent.hiliteIcons by default?
*  Can be overridden at the field level
* @type {number}
* @default "2"
*/
hiliteIconRightPadding?: number;

/**
* Marker that can be set on a record to flag that record as hilited. Should be set to a value
*  that matches Hilite.id for a hilite defined on this component.
* @type {string}
* @default "_hilite"
*/
hiliteProperty?: string;

/**
* Initial hilite state for the grid. 
*  
*  ListGrid.viewState can be used to initialize all view properties of the grid.
*  When doing so, hiliteState is not needed because viewState
*  includes it as well. If both are provided, hiliteState has priority for
*  hilite state.
*  
*  
*  To retrieve current state call DataBoundComponent.getHiliteState.
*  
* @type {string}
* @default null
*/
hiliteState?: string;

/**
* The strategy to use when generating names for new fields in this component. The default 
*  strategy, "simple", combines the field-type with an index maintained by field-type and 
*  component instance. For example, "formulaField1".
* @type {FieldNamingStrategy}
* @default "simple"
*/
fieldNamingStrategy?: FieldNamingStrategy;

/**
* Text for a menu item allowing users to edit grid highlights.
* @type {string}
* @default "Edit Highlights..."
*/
editHilitesText?: string;

/**
* The title for the DataBoundComponent.editHilites dialog.
* @type {string}
* @default "Edit Highlights"
*/
editHilitesDialogTitle?: string;

/**
* The title for the DataBoundComponent.fieldEditorWindow used to edit calculated
*  fields.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will
*  be evaluated as JS code whenever the message is displayed.
*  
*  Two dynamic variables are available - "builderType", either Formula or Summary, and 
*  "fieldTitle", which is the title of the calculated field being edited.
* @type {HTMLString}
* @default "${builderType} Editor [${fieldTitle}]"
*/
fieldEditorWindowTitle?: HTMLString;

/**
* The Window used to edit calculated fields for this component.
* @type {Window}
* @default null
*/
fieldEditorWindow?: Window;

/**
* This component's HiliteEditor instance used to allow the user to create, modify, or delete
*  hilites.
*  
*  The following autoChildUsage apply:
*  
*  
* - DataBoundComponent.hiliteIcons for HiliteEditor.hiliteIcons
*  
* @type {HiliteEditor}
* @default null
*/
hiliteEditor?: HiliteEditor;

/**
* If the result of a formula evaluation is invalid (specifically, if isNaN(result)==true),
*  badFormulaResultValue is displayed instead. The default value is ".".
* @type {string}
* @default "."
*/
badFormulaResultValue?: string;

/**
* If a summary format string contains an invalid field reference, replace the reference
*  with the missingSummaryFieldValue. The default value is "-".
* @type {string}
* @default "-"
*/
missingSummaryFieldValue?: string;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define a new
*  field based on values present in other fields, using the FormulaBuilder.
*  
*  User-added formula fields can be persisted via ListGrid.getFieldState and 
*  ListGrid.setFieldState.
* @type {boolean}
* @default false
*/
canAddFormulaFields?: boolean;

/**
* Text for a menu item allowing users to add a formula field
* @type {string}
* @default "Add formula column..."
*/
addFormulaFieldText?: string;

/**
* Text for a menu item allowing users to edit a formula field
* @type {string}
* @default "Edit formula..."
*/
editFormulaFieldText?: string;

/**
* Text for a menu item allowing users to remove a formula field
* @type {string}
* @default "Remove formula"
*/
removeFormulaFieldText?: string;

/**
* Adds an item to the header context menu allowing users to launch a dialog to define a new
*  text field that can contain both user-defined text and the formatted values present in other 
*  fields, using the SummaryBuilder.
*  
*  User-added summary fields can be persisted via ListGrid.getFieldState and 
*  ListGrid.setFieldState.
* @type {boolean}
* @default false
*/
canAddSummaryFields?: boolean;

/**
* Text for a menu item allowing users to add a formula field
* @type {string}
* @default "Add summary column..."
*/
addSummaryFieldText?: string;

/**
* Text for a menu item allowing users to edit the formatter for a field
* @type {string}
* @default "Edit summary format..."
*/
editSummaryFieldText?: string;

/**
* Text for a menu item allowing users to remove a summary field
* @type {string}
* @default "Remove summary format..."
*/
removeSummaryFieldText?: string;

/**
* When assembling a value for a DataBoundComponent.canAddSummaryFields, if a referenced
*  field is hilited, should the hilite HTML be included in the summary field value?
*  
*  To control hilites showing in group summaries, see ListGrid.showHilitesInGroupSummary.
* @type {boolean}
* @default true
*/
includeHilitesInSummaryFields?: boolean;

/**
* Best field to use for a user-visible title for an individual record from this
*  component.
*  
*  This attribute has the same function as DataSource.iconField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
titleField?: string;

/**
* Designates a field of FieldType:"image" as the field to use when rendering a
*  record as an image, for example, in a TileGrid. 
*  
*  This attribute has the same function as DataSource.iconField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
iconField?: string;

/**
* Name of the field that has the second most pertinent piece of textual information in the
*  record, for use when a DataBoundComponent needs to show a short summary of a record.
*  
*  This attribute has the same function as DataSource.infoField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
infoField?: string;

/**
* Name of the field that has the most pertinent numeric, date, or enum value, for use when a
*  DataBoundComponent needs to show a short summary of a record.
*  
*  This attribute has the same function as DataSource.dataField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
dataField?: string;

/**
* Name of the field that has a long description of the record, or has the primary text data
*  value for a record that represents an email message, SMS, log or similar.
*  
*  This attribute has the same function as DataSource.descriptionField but can be
*  set for a component with no dataSource, or can be used to override the dataSource setting.
* @type {string}
* @default null
*/
descriptionField?: string;

/**
* The message to display to the user if an export of a DataBoundComponent's data is attempted
*  while the DataBoundComponent's data is null or an empty list.
* @type {string}
* @default "You are attempting to export an empty dataset"
*/
emptyExportMessage?: string;

/**
* Properties to apply to the MultiSortDialog which gets automatically generated
*  when DataBoundComponent.askForSort is called.
*  
*  See also ListGrid.showHeaderSpanTitlesInSortEditor and ListGrid.sortEditorSpanTitleSeparator
* @type {Partial<MultiSortDialog>}
* @default null
*/
multiSortDialogProperties?: Partial<MultiSortDialog>;

/**
* Class level defaults to apply to the MultiSortDialog which gets automatically
*  generated when DataBoundComponent.askForSort is called.
*  
*  See also ListGrid.showHeaderSpanTitlesInSortEditor and ListGrid.sortEditorSpanTitleSeparator
* @type {Partial<MultiSortDialog>}
* @default null
*/
multiSortDialogDefaults?: Partial<MultiSortDialog>;

/**
* For databound components that support editing, the error message for a failed validator
*  that does not specify its own errorMessage.
* @type {HTMLString}
* @default "Invalid value"
*/
unknownErrorMessage?: HTMLString;

/**
* A message to display to the user if server-side validation fails with an error but the
*  server did not provide an error message
* @type {string}
* @default "Error during validation; no error details were provided"
*/
noErrorDetailsMessage?: string;





/**
*  Returns whether there are any pending DSRequests initiated by this
*  DataBoundComponent. May not include any requests sent by directly calling the
*  DataSource APIs (rather than the DataBoundComponent APIs).
*  
* 
*/
static requestsArePending(): boolean;



/**
*  This method is invoked on the source component whenever a drag operation or 
*  DataBoundComponent.transferSelectedData completes. This method is called when the entire chain of 
*  operations - including, for databound components, server-side updates and subsequent 
*  integration of the changes into the client-side cache - has completed.
*  There is no default implementation of this method; you are intended to override it if you 
*  are interested in being notified when drag operations complete.
* 
* 
*/
dragComplete(): void;

/**
*  This method is invoked whenever a drop operation or DataBoundComponent.transferSelectedData 
*  targeting this component completes. A drop is considered to be complete when all the client-
*  side transfer operations have finished. This includes any server turnarounds SmartClient 
*  needs to make to check for duplicate records in the target component; it specifically does 
*  not include any add or update operations sent to the server for databound components. If 
*  you want to be notified when the entire drag operation - including server updates and cache
*  synchronization - has completed, override DataBoundComponent.dragComplete
*  on the source component.
*  There is no default implementation of this method; you are intended to override it if you 
*  are interested in being notified when drop operations complete.
* 
* 
* @param {Array<Partial<Record>>} The list of records actually transferred to          this component (note that this is not necessarily the same thing as the           list of records dragged out of the source component because it doesn't           include records that were excluded because of collisions with existing           records)
*/
dropComplete(transferredRecords:Array<Partial<Record>>): void;

/**
*  For a component with a specified DataSource, find the associated dataSource field object
*  from a specified DataPath.
* 
* @param {DataPath} dataPath for which the field definition should be returned.
*/
getDataPathField(dataPath:DataPath): void;

/**
*  Can the field be edited? This base method always returns false, but it's
*  overridden by subclasses such as DynamicForm and ListGrid.
* 
* 
* @param {object | number | string} field object or identifier
*/
fieldIsEditable(field:object | number | string): boolean;

/**
*  Return a field by a field index or field name.
* 
* 
* @param {FieldName | number} field index or field.name
*/
getField(fieldID:FieldName | number): object;

/**
*  Find the index of a currently visible field.
* 
* 
* @param {FieldName | string} field name or field
*/
getFieldNum(fieldName:FieldName | string): Integer;

/**
*  This API is equivalent to List.find but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
find(advancedCriteria:AdvancedCriteria): object;

/**
*  This API is equivalent to List.findAll but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
findAll(advancedCriteria:AdvancedCriteria): Array<any>;

/**
*  This API is equivalent to List.findIndex but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {AdvancedCriteria} AdvancedCriteria to use with
*/
findIndex(advancedCriteria:AdvancedCriteria): Integer;

/**
*  This API is equivalent to List.findNextIndex but searches for a matching record among already-loaded data only. 
*  Use ListGrid.fetchData to load data from the server.
* 
* @param {Integer} first index to consider
* @param {AdvancedCriteria} AdvancedCriteria to use with
* @param {Integer=} last index to consider
*/
findNextIndex(startIndex:Integer, advancedCriteria:AdvancedCriteria, endIndex?:Integer): Integer;

/**
*  Bind to a new DataSource.
*  
*  Like passing the "dataSource" property on creation, binding to a DataSource means that the
*  component will use the DataSource to provide default data for its fields.
*  
*  When binding to a new DataSource, if the component has any existing "fields" or has a dataset,
*  these will be discarded by default, since it is assumed the new DataSource may represent a
*  completely unrelated set of objects. If the old "fields" are still relevant, pass them to
*  setDataSource().
*  
* 
* @param {GlobalId | DataSource} DataSource to bind to
* @param {Array<Partial<DataSourceField>>=} optional array of fields to use
*/
setDataSource(dataSource:GlobalId | DataSource, fields?:Array<Partial<DataSourceField>>): void;

/**
*  Returns an array of Alignment for this grid
* 
*/
getFieldAlignments(): Array<Partial<Alignment>>;

/**
*  Compares the specified criteria with the current criteria applied to this component's
*  data object and determines whether the new criteria could be satisfied from the currently
*  cached set of data, or if a new filter/fetch operation will be required.
*  
*  This is equivalent to calling this.data.willFetchData(...).
*  Always returns true if this component is not showing a set of data from the dataSource.
*  
*  Note that to predict correctly the decision that will be made by filter/fetch, you'll need to
*  pass the same TextMatchStyle that will be used by the future filter/fetch. Fetching
*  manually (e.g. ListGrid.fetchData) will by default use "exact" while filtering
*  (e.g. ListGrid.filterData) will by default use "substring". If the component
*  is configured for autofetch (i.e. ListGrid.autoFetchData: true), that will
*  use ListGrid.autoFetchTextMatchStyle, which defaults to "substring". If nothing/null
*  is passed for the style, this method assumes you want the style from the last filter/fetch.
*  
*  To determine what TextMatchStyle is being used, check the RPC Tab of the
*  debugging and check the relevant DSRequest. 
*  
* 
* @param {Criteria} new criteria to test.
* @param {TextMatchStyle=} New text match style. If not passed assumes    textMatchStyle will not be modified.
*/
willFetchData(newCriteria:Criteria, textMatchStyle?:TextMatchStyle): boolean;

/**
*  Attempt to find the record in the resultSet that has a primary key value that matches the 
*  passed in parameter value. Only the locally cached data will be searched. 
*  Checks only loaded rows and will not trigger a fetch. Returns null if there is no match, 
*  data is not loaded, or there is no ResultSet.dataSource.
*  
*  Note, if you pass a simple value to this method, it will be matched against the first 
*  primaryKey field. For DataSources with a composite primary key (multiple primaryKey fields),
*  pass
*  a criteria object containing just your primaryKeys, like this: 
*  { firstPkField: "value", secondPkField: 25 }.
*  
* 
* 
* @param {object} primary key value to search for
*/
findByKey(keyValue:object): Record;

/**
*  Select a single Record passed in explicitly, or by index, and deselect everything else.
*  When programmatic selection of records is a requirement and 
*  ListGrid.selectionType is "single", use this method rather than 
*  DataBoundComponent.selectRecord to 
*  enforce mutually-exclusive record-selection.
* 
* 
* @param {Record | number} record (or row number) to select
*/
selectSingleRecord(record:Record | number): void;

/**
*  Returns the number of selected records.
* 
* 
*/
getSelectionLength(): Integer;

/**
*  Return the set of hilite-objects currently applied to this DataBoundComponent. These
*  can be serialized for storage and then restored to a component later via 
*  DataBoundComponent.setHilites.
* 
* 
*/
getHilites(): Array<Partial<Hilite>>;

/**
*  Accepts an array of hilite objects and applies them to this DataBoundComponent. See also
*  DataBoundComponent.getHilites for a method of retrieving the hilite
*  array for storage, including hilites manually added by the user.
* 
* 
* @param {Array<Partial<Hilite>>} Array of hilite objects
*/
setHilites(hilites:Array<Partial<Hilite>>): void;

/**
*  Get the current hilites encoded as a String, for saving.
* 
* 
*/
getHiliteState(): string;

/**
*  Set the current hilites based on a hiliteState String previously returned from
*  DataBoundComponent.getHiliteState.
* 
* @param {string} hilites state encoded as a String
*/
setHiliteState(hiliteState:string): void;

/**
*  Enable / disable a DataBoundComponent.hilites
* 
* 
* @param {string} ID of hilite to enable
* @param {boolean=} new enabled state to apply - if null, defaults to true
*/
enableHilite(hiliteID:string, enable?:boolean): void;

/**
*  Disable a hilite
* 
* 
* @param {string} ID of hilite to disable
*/
disableHilite(hiliteID:string): void;

/**
*  Enable all hilites.
* 
* 
* @param {boolean=} new enabled state to apply - if null, defaults to true
*/
enableHiliting(enable?:boolean): void;

/**
*  Disable all hilites.
* 
* 
*/
disableHiliting(): void;

/**
*  Shows a HiliteEditor interface allowing end-users to edit
*  the data-hilites currently in use by this DataBoundComponent.
* 
* 
*/
editHilites(): void;

/**
* 
*  Transfer a list of Records from another component (does not have to be a databound
*  component) into this component. This method is only applicable to list-type components,
*  such as ListGrid, TreeGrid or TileGrid
*  
*  This method implements the automatic drag-copy and drag-move behaviors of components like
*  ListGrid, and calling it is equivalent to completing a drag and drop of the
*  dropRecords.
*  
*  Note that this method is asynchronous - it may need to perform server turnarounds to prevent
*  duplicates in the target component's data. If you wish to be notified when the transfer 
*  process has completed, you can either pass the optional callback to this method or implement
*  the DataBoundComponent.dropComplete method on this component.
*  
*  See also DataBoundComponent.transferSelectedData.
* 
* 
* @param {Array<Partial<Record>>} Records to transfer to this component
* @param {Record} The target record (eg, of a drop interaction), for context
* @param {Integer} Insert point in this component's data for the transferred records
* @param {Canvas} The databound or non-databound component from which the records              are to be transferred.
* @param {Callback=} optional callback to be fired when the transfer process has completed
*/
transferRecords(dropRecords:Array<Partial<Record>>, targetRecord:Record, index:Integer, sourceWidget:Canvas, callback?:Callback): void;

/**
*  Returns the "drop values" to apply to a record dropped on this component prior to update. Only
*  applicable to databound components - see DataBoundComponent.dropValues for more details. If multiple records 
*  are being dropped, this method is called for each of them in turn.
*  
*  The default implementation of this method returns the following:
*  
*  
* - Nothing, if DataBoundComponent.addDropValues is false
*  
* - dropValues, if that property is set. If the component's criteria object is applicable (as explained
*  in the next item), it is merged into dropValues, with properties in dropValues taking precedence.
*  
* - The component's criteria object, if the most recent textMatchStyle for the component was "exact" 
*    and it is simple criteria (ie, not an AdvancedCriteria object)
*  
* - Otherwise nothing
*  
*  
*  You can override this method if you need more complex setting of drop values than can be 
*  provided by simply supplying a dropValues object.
*  
* 
* @param {Record} record being dropped
* @param {DataSource} dataSource the record being dropped is bound to
* @param {Record} record being dropped on
* @param {Integer} index of record being dropped on
* @param {Canvas} widget where dragging began
*/
getDropValues(record:Record, sourceDS:DataSource, targetRecord:Record, index:Integer, sourceWidget:Canvas): object;

/**
*  Convenience method to display a FormulaBuilder to create a new Formula Field. This 
*  is equivalent to calling DataBoundComponent.editFormulaField with 
*  no parameter.
* 
* 
*/
addFormulaField(): void;

/**
*  Method to display a FormulaBuilder to edit a formula Field. If the function is called
*  without a parameter, a new field will be created when the formula is saved.
* 
* 
* @param {string} Field to edit or null to add a new formula field
*/
editFormulaField(field:string): void;

/**
*  Get the computed value of a DataBoundComponent.canAddFormulaFields.
* 
* @param {string} field that has a formula
* @param {Record} record to use to compute formula value
*/
getFormulaFieldValue(field:string, record:Record): Double | string;

/**
*  Convenience method to display a SummaryBuilder to create a new Summary Field. This 
*  is equivalent to calling DataBoundComponent.editSummaryField with 
*  no parameter.
* 
* 
*/
addSummaryField(): void;

/**
*  Method to display a SummaryBuilder to edit a Summary Field. If the function is called
*  without a parameter, a new field will be created when the summary is saved.
* 
* 
* @param {string} Field to edit or null to add a new summary column
*/
editSummaryField(field:string): void;

/**
*  Get the computed value of a DataBoundComponent.canAddSummaryFields.
* 
* @param {string} field that has a summary format
* @param {Record} record to use to compute formula value
*/
getSummaryFieldValue(field:string, record:Record): string;

/**
*  When assembling a value for a DataBoundComponent.canAddSummaryFields, if a referenced
*  field is hilited, should the hilite HTML be included in the summary field value?
*  
*  Example use case: Consider a grid containing a numeric field, and a summary field
*  which contains some string value, plus the contents of the numeric field.
*  If a hilite is defined for the grid which turns the numeric field text red when
*  the value is negative, this property will govern whether the number will also be
*  rendered in red within the summary field cells. Any other text in the summary field
*  cells would not be effected by this hilite.
*  
*  Default implementation returns DataBoundComponent.includeHilitesInSummaryFields.
*  
*  To control hilites showing in group summaries, see ListGrid.showHilitesInGroupSummary.
* 
* 
* @param {string} name of the summary field
* @param {string} name of the field referenced by this summary
*/
shouldIncludeHiliteInSummaryField(summaryFieldName:string, usedFieldName:string): boolean;

/**
*  Get the index of the provided record.
*  
*  Override in subclasses to provide more specific behavior, for instance, when data holds a
*  large number of records
* 
* 
* @param {Record} the record whose index is to be retrieved
*/
getRecordIndex(record:Record): number;

/**
*  Get the value of the titleField for the passed record
*  
*  Override in subclasses 
* 
* 
* @param {Record} the record whose index is to be retrieved
*/
getTitleFieldValue(record:Record): string;

/**
*  Return all CSS style declarations associated with the hilites of a record's field.
* 
* @param {Record} * @param {string} if set, returned CSS will be appended to this text
* @param {string} field object identifying whose CSS is to be returned
*/
getRecordHiliteCSSText(record:Record, cssText:string, field:string): string;

/**
*  Return the SortSpecifiers representing the current sort configuration of this
*  component.
* 
*/
getSort(): Array<Partial<SortSpecifier>>;

/**
*  Sort this component by a list of SortSpecifiers. If the component's data is not a 
*  ResultSet, only the first specifier is applied.
*  
* 
* @param {Array<Partial<SortSpecifier>>} List of SortSpecifier objects, one  per sort-field and direction
*/
setSort(sortSpecifiers:Array<Partial<SortSpecifier>>): void;

/**
*  Show a dialog to configure the sorting of multiple fields on this component. Calls through
*  to MultiSortDialog.askForSort, passing this component as the fieldSource and the
*  current DataBoundComponent.getSort if there is one.
*  
*  The generated multiSortDialog can be customized via DataBoundComponent.multiSortDialogDefaults, 
*  DataBoundComponent.multiSortDialogProperties.
* 
* 
*/
askForSort(): void;

/**
*  Returns true if the component's current data model is marked as offline. This does not 
*  necessarily mean that the component has no data; it may have data that was supplied from
*  the Offline.
* 
*/
isOffline(): boolean;

/**
*  Shows a FieldPicker interface allowing end-users to edit
*  the fields currently shown by this DataBoundComponent.
* 
* 
*/
editFields(): void;

/**
*  Notification method fired when a user-generated field is added to this component via
*  DataBoundComponent.editFormulaField or DataBoundComponent.editSummaryField.
*  
*  Returning false from this method will prevent the field being added at all. Note that
*  this also provides an opportunity to modify the generated field object - any changes
*  made to the field parameter will show up when the field is displayed in the ListGrid.
* 
* 
* @param {ListGridField} User generated summary or formula field
*/
userAddedField(field:ListGridField): boolean;

/**
*  Called when the selection changes. Note that this method fires exactly once for any given
*  change to the selection unlike the ListGrid.selectionChanged event.
*  
*  This event is fired once after selection/deselection has completed. The result is
*  one event per mouse-down event. For a drag selection there will be two events fired:
*  one when the first record is selected and one when the range is completed.
*  
*  This event is also fired when selection is updated by a direct call to one of the
*  DataBoundComponent select/deselect methods. Calls on the Selection
*  object do not trigger this event.
* 
* 
* @param {object} first selected record in the selection, if any, which may or may not be the first record in sort order if the DataBoundComponent is sorted. This parameter is typically used when only one record can be selected at a time.
* @param {Array<Partial<object>>} List of records that are now selected
*/
selectionUpdated(record:object, recordList:Array<Partial<object>>): void;




}

/**
* What the JSONEncoder should do when it encounters a circular reference in an object
*  structure.
*/
type JSONCircularReferenceMode = 
/**
* circular references in Arrays will be represented as a null entry, and objects        will have a property with a null value
*/
"omit" |

/**
* leave a string marker, the JSONEncoder.circularReferenceMarker,         wherever a circular reference is found
*/
"marker" |

/**
* leave a string marker followed by the path to the first occurrence of        the circular reference from the top of the object tree that was serialized.
*         This potentially allows the original object graph to be reconstructed.
*/
"path";


/**
* TextItem subclass for managing a DataPath
*/
class DataPathItem extends TextItem {
/**
* Creates a new DataPathItem
* 
* @param typeCheckedProperties {Partial<DataPathItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DataPathItem} The new instance
*/
static create(typeCheckedProperties? : Partial<DataPathItem>, uncheckedProperties?: Object): DataPathItem;











}

/**
* What to do if the user hits Up or Down arrow key while editing a cell.
*/
type ArrowKeyEditAction = 
/**
* The grid will take no special action when the user presses up or down  arrow keys within an editor
*/
"none" |

/**
* The grid will intercept up and down arrow keypresses and navigate  to the next or previous edit row by generating an appropriate EditCompletionEvent
*/
"editNext";


/**
* Group Summary.
*/
class GroupSummary {











}

/**
* Method of, or reason for, navigation between panes.
*/
type NavigationMethod = 
/**
* a "back" NavigationButton has been clicked
*/
"backClick" |

/**
* a side panel NavigationButton has been clicked
*/
"sideClick" |

/**
* application code called            SplitPane.setCurrentPane,
*             SplitPane.showNavigationPane,
*             SplitPane.showListPane,
*             SplitPane.showDetailPane, etc.
*/
"programmatic" |

/**
* application code directly called            SplitPane.navigatePane
*/
"navigatePane" |

/**
* a seletion change automatically called              SplitPane.navigatePane
*/
"selectionChanged" |

/**
* browser navigation triggered SplitPane navigation
*/
"historyCallback";


/**
*/
type TEXTAREA_WRAP = 
/**
* don't allow wrapping at all
*/
"OFF" |

/**
* when the entered text reaches the edge of the text area,                 wrap visibly but don't include line breaks in the textarea
*                  value
*/
"SOFT" |

/**
* when the entered text reaches the edge of the text area,                  insert a line break
*/
"HARD";


/**
* An autoChild is an automatically generated subcomponent that a component creates to
*  handle part of its presentation or functionality. An example is the Window component and
*  its subcomponent the "header".
*  
*  See autoChildUsage for more information.
*/
type AutoChild = object;


/**
* Operators that can evaluate a set of criteria and produce a combined result.
*/
type LogicalOperator = 
/**
* true if all criteria are true
*/
"and" |

/**
* true if any criteria are true
*/
"or" |

/**
* true if all criteria are false
*/
"not";




/**
* Specially styled Button subclass used by the NavigationBar class.
*/
class NavigationButton extends Button {
/**
* Creates a new NavigationButton
* 
* @param typeCheckedProperties {Partial<NavigationButton>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {NavigationButton} The new instance
*/
static create(typeCheckedProperties? : Partial<NavigationButton>, uncheckedProperties?: Object): NavigationButton;


/**
* Default baseStyle for navigation buttons. Note that the special NavigationButton.backBaseStyle and
*  NavigationButton.forwardBaseStyle are applied if NavigationButton.direction is set.
* @type {CSSStyleName}
* @default "navButton"
*/
baseStyle?: CSSStyleName;

/**
* Base style for navigation buttons where Direction is set to "back"
* @type {CSSStyleName}
* @default "navBackButton"
*/
backBaseStyle?: CSSStyleName;

/**
* Base style for navigation buttons where Direction is set to "forward"
* @type {CSSStyleName}
* @default "navForwardButton"
*/
forwardBaseStyle?: CSSStyleName;

/**
* Navigation direction for this button. If set to "forward" or
*  "back" the special NavigationButton.forwardBaseStyle or NavigationButton.backBaseStyle
*  will be applied.
* @type {NavigationDirection}
* @default "none"
*/
direction?: NavigationDirection;










}

/**
* A decimal (or "floating point") number, for example, 5.5. May not be null.
*/
type float = Float;






/**
*/
type ImportFormat = 
/**
* XML format: same as that expected by the adminConsole for         DataSource testData
*/
"xml" |

/**
* JSON format: a JSON Array of JSON Objects
*/
"json" |

/**
* Comma-separated values, or in general delimiter-separated values based         on a provided delimiter.
*/
"csv" |

/**
* Auto-detect format
*/
"auto";


/**
* A dialog for picking fields to display from among the available fields.
*  
*  This is typically useful in scenarios where there are many more fields than can reasonably
*  fit on screen. The application can start off displaying a few of the fields by default (such
*  as the most commonly-needed fields), and show a FieldPickerWindow to allow the user to
*  customize which fields to display as well as the order in which to display them.
*/
class FieldPickerWindow extends Window {
/**
* Creates a new FieldPickerWindow
* 
* @param typeCheckedProperties {Partial<FieldPickerWindow>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {FieldPickerWindow} The new instance
*/
static create(typeCheckedProperties? : Partial<FieldPickerWindow>, uncheckedProperties?: Object): FieldPickerWindow;


/**
* @type {string}
* @default "Field Picker"
*/
title?: string;

/**
* By default, a FieldPickerWindow will close automatically if the mouse is clicked outside
*  of it. To have the window shown with true modality, set this attribute to false.
* @type {boolean}
* @default true
*/
autoDismiss?: boolean;

/**
* A FieldPicker for altering the working field-set in a 
*  DataBoundComponent.
* @type {FieldPicker}
* @default null
*/
fieldPicker?: FieldPicker;










}











/**
* Base class for graphical elements drawn in a DrawPane. All properties and methods
*  documented here are available on all DrawItems unless otherwise specified.
*  
*  Each DrawItem has its own local transform that maps its 
*  DrawItem to the drawing coordinate system that is shared by
*  all DrawItems in the same DrawPane (explained DrawPane). The local
*  transform is a combination of rotation, scaling, and other affine transformations. The
*  DrawItem is first DrawItem.translate, then DrawItem.scale,
*  then DrawItem.xShearFactor in the direction of the x-axis, then
*  DrawItem.yShearFactor in the directiton of the y-axis, and then finally
*  DrawItem.rotation.
*  
*  Note that DrawItems as such should never be created, only concrete subclasses such as
*  DrawGroup and DrawLine.
*  
*  See DrawPane for the different approaches to create DrawItems.
*/
class DrawItem extends BaseWidget {
/**
* Creates a new DrawItem
* 
* @param typeCheckedProperties {Partial<DrawItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DrawItem} The new instance
*/
static create(typeCheckedProperties? : Partial<DrawItem>, uncheckedProperties?: Object): DrawItem;


/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "DrawItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* Relative stacking order of this draw item with respect to other items in the same DrawPane
*  or DrawGroup.
*  
*  null means that the zIndex has not been resolved. Upon adding this draw item to a
*  DrawPane or DrawGroup, this item's zIndex will be resolved to
*  the next higher auto-assigned zIndex. Note that this may still be less than another item's
*  zIndex if DrawItem.bringToFront was called on that item.
*  
*  If two items within the same DrawPane or DrawGroup have the
*  same zIndex, then they are stacked in the order in which they were added to the
*  DrawPane or DrawGroup.
*  
*  When the DrawPane's DrawPane.drawingType is
*  "bitmap",
*  
*  zIndex, DrawItem.bringToFront, and DrawItem.sendToBack are not supported
*  for DrawLabels on iOS due to platform limitations.
* @type {Integer}
* @default null
*/
zIndex?: Integer;

/**
* If set, specifies the cursor to display when the mouse pointer is over this DrawItem.
* @type {Cursor}
* @default null
*/
cursor?: Cursor;

/**
* Will this DrawItem fire hover events when the user hovers over it?
* @type {boolean}
* @default null
*/
canHover?: boolean;

/**
* If DrawItem.canHover is true, should we show the global hover canvas by default
*  when the user hovers over this DrawItem?
* @type {boolean}
* @default true
*/
showHover?: boolean;

/**
* If this.canHover is true, how long should the mouse be kept over this
*  widget before the hover event is fired
* @type {number}
* @default "300"
*/
hoverDelay?: number;

/**
* Default DrawItem.getHoverHTML that is displayed in the global hover canvas if
*  the user hovers over this DrawItem and DrawItem.showHover is true.
* @type {HTMLString}
* @default null
*/
prompt?: HTMLString;

/**
* DrawPane this drawItem should draw in.
*  
*  If this item has a DrawGroup, the drawGroup's drawPane is automatically used.
* @type {DrawPane}
* @default null
*/
drawPane?: DrawPane;

/**
* DrawGroup this drawItem is a member of.
* @type {DrawGroup}
* @default null
*/
drawGroup?: DrawGroup;

/**
* Pixel width for lines.
* @type {Integer}
* @default "3"
*/
lineWidth?: Integer;

/**
* Line color
* @type {CSSColor}
* @default "#808080"
*/
lineColor?: CSSColor;

/**
* Opacity for lines, as a number between 0 (transparent) and 1 (opaque).
* @type {Float}
* @default "1.0"
*/
lineOpacity?: Float;

/**
* Pattern for lines, eg "solid" or "dash".
*  
*  Note that support in old browsers, such as Internet Explorer versions before IE11, is
*  limited for DrawPane.drawingType "bitmap" to items with straight
*  edges - DrawLines, DrawPaths, and DrawRects with no 
*  DrawRect.rounding.
* @type {LinePattern}
* @default "solid"
*/
linePattern?: LinePattern;

/**
* Style of drawing the endpoints of a line.
*  
*  Note that for dashed and dotted lines, the lineCap style affects each dash or dot.
* @type {LineCap}
* @default "round"
*/
lineCap?: LineCap;

/**
* Fill color to use for shapes. The default of 'null' is transparent.
* @type {CSSColor}
* @default null
*/
fillColor?: CSSColor;

/**
* Fill gradient to use for shapes. If a string it uses the gradient identifier parameter provided in 
*  DrawPane.addGradient. Otherwise it expects one of SimpleGradient,
*  LinearGradient or RadialGradient.
* @type {Gradient | string}
* @default null
*/
fillGradient?: Gradient | string;

/**
* Opacity of the fillColor, as a number between 0 (transparent) and 1 (opaque).
* @type {Float}
* @default "1.0"
*/
fillOpacity?: Float;

/**
* Shadow used for all DrawItem subtypes.
* @type {Shadow}
* @default null
*/
shadow?: Shadow;

/**
* Rotation in degrees about the DrawItem.getCenter.
*  The positive direction is clockwise.
* @type {Float}
* @default "0.0"
*/
rotation?: Float;

/**
* The slope of an x-shearing transformation applied to this DrawItem. Each point in the
*  shape is moved along the x-axis a distance that is proportional to the initial
*  y-coordinate of the point.
* @type {Float}
* @default "0.0"
*/
xShearFactor?: Float;

/**
* The slope of a y-shearing transformation applied to this DrawItem. Each point in the
*  shape is moved along the y-axis a distance that is proportional to the initial
*  x-coordinate of the point.
* @type {Float}
* @default "0.0"
*/
yShearFactor?: Float;

/**
* Array holds 2 values representing scaling along x and y dimensions.
* @type {Array<Partial<Float>>}
* @default null
*/
scale?: Array<Partial<Float>>;

/**
* Array holds two values representing translation along the x and y dimensions.
* @type {Array<Partial<Float>>}
* @default null
*/
translate?: Array<Partial<Float>>;

/**
* Style of arrow head to draw at the beginning of the line or path.
* @type {ArrowStyle}
* @default null
*/
startArrow?: ArrowStyle;

/**
* Style of arrow head to draw at the end of the line or path.
* @type {ArrowStyle}
* @default null
*/
endArrow?: ArrowStyle;

/**
* Array of control knobs to display for this item. Each KnobType specified in this array
*  will turn on UI element(s) allowing the user to manipulate this drawItem. To update the
*  set of knobs at runtime use DrawItem.showKnobs and DrawItem.hideKnobs.
*  
*  NOTE: Unless otherwise documented, DrawItem types only support
*  "resize" and "move"
*  
*  knobs.
* @type {Array<Partial<KnobType>>}
* @default null
*/
knobs?: Array<Partial<KnobType>>;

/**
* A string to show at the DrawItem.getCenter of this DrawItem.
*  
*  When set to a non-null value (including an empty string), the DrawItem.titleLabel
*  DrawLabel AutoChild will be created automatically and positioned at the center of
*  this DrawItem.
* @type {string}
* @default null
*/
title?: string;

/**
* When a non-null DrawItem.title is set, this AutoChild is created
*  automatically and positioned at the DrawItem.getCenter of this DrawItem
*  . The titleLabel moves with this DrawItem and, depending
*  on TitleRotationMode, can rotate with this DrawItem
*  as well.
*  
*  The following autoChildUsage applies:
*  DrawItem.title for DrawLabel.contents.
*  
*  Related to the titleLabel is the DrawItem.titleLabelBackground which allows a border to be placed around the titleLabel
*  and/or a background added. By default, shapes that are commonly filled such as
*  DrawTriangles, with the exception of DrawSectors, do not show the 
*  titleLabelBackground (see DrawItem.showTitleLabelBackground).
* @type {DrawLabel}
* @default null
*/
titleLabel?: DrawLabel;

/**
* Whether the DrawItem.titleLabel should be scaled to the maximum possible size that fits
*  inside the bounds of this item. Currently only DrawRects and DrawPolygons
*  with 90 degree angles are supported.
*  
*  Note that DrawItem.titleAutoFit isn't supported for rotated, sheared, or scaled
*  DrawItems, and that therefore the value of TitleRotationMode, which relates
*  to rotation of the item, is ignored when this property is set. However, we do support
*  having the label automatically rotate to run vertically if there's more space - see
*  DrawItem.titleAutoFitRotationMode.
* @type {boolean}
* @default false
*/
titleAutoFit?: boolean;

/**
* Whether to rotate the DrawItem.titleLabel 90 degrees clockwise while trying to maximize its
*  size in accordance with DrawItem.titleAutoFit. If automatic rotation is specified, the
*  default, the label will be rotated if and only if it allows the label to become larger.
* @type {TitleAutoFitRotationMode}
* @default "auto"
*/
titleAutoFitRotationMode?: TitleAutoFitRotationMode;

/**
* Specifies margin between label and item edges when using DrawItem.titleAutoFit.
* @type {number}
* @default "2"
*/
titleAutoFitMargin?: number;

/**
* The mode in which the DrawItem.titleLabel (if shown) is rotated with
*  this draw item.
* @type {TitleRotationMode}
* @default "neverRotate"
*/
titleRotationMode?: TitleRotationMode;

/**
* If the DrawItem.titleLabel is showing, should the DrawItem.titleLabelBackground
*  be created and placed behind the titleLabel?
*  
*  This defaults to true for DrawSectors and shapes that are not commonly filled
*  (e.g. DrawLines).
* @type {boolean}
* @default false
*/
showTitleLabelBackground?: boolean;

/**
* When the DrawItem.titleLabel is showing and DrawItem.showTitleLabelBackground
*  is true, this DrawRect AutoChild is created and placed behind the
*  titleLabel.
* @type {DrawRect}
* @default null
*/
titleLabelBackground?: DrawRect;

/**
* If the DrawItem.titleLabelBackground is visible, how much padding
*  should be left between the bounds of the DrawItem.titleLabel and the edges of
*  the titleLabelBackground?
* @type {Integer}
* @default "2"
*/
titleLabelPadding?: Integer;

/**
* Constrains drag-resizing and drag-repositioning of this draw item to either the current
*  visible area of the DrawPane or an arbitrary bounding box (if set to
*  an array of the form [left, top, left + width, top + height]). When using
*  a bounding box-type argument the left/top values can be negative, or the width/height
*  values can be greater than the dimensions of the viewable area, to allow positioning
*  or resizing the draw item beyond the confines of the draw pane.
*  
*  Note: keepInParentRect affects only user drag interactions, not programmatic moves or
*  resizes.
* @type {boolean | Array<Partial<Float>>}
* @default null
*/
keepInParentRect?: boolean | Array<Partial<Float>>;

/**
* If this DrawItem is showing "resize"
*  
*  DrawItem.knobs, should resizing the shape solely update the local
*  transform (for example, the DrawItem's DrawItem.scale or
*  DrawItem.translate)?.
*  
*  The default is false, which means that the DrawItem is allowed to modify
*  its shape properties in order to fit within a given width and height. Some examples:
*  
*  
* - A DrawOval might decrease its DrawOval.radius
*  when resized to a smaller size.
*  
* - A DrawPath might change its DrawPath.points
*  to lengthen all line segments in the path by some proportion so that it fits into a
*  larger size.
*  
*  This approach allows a DrawItem to maintain the same
*  DrawItem.lineWidth even as it is being resized.
*  
*  If this property is set to true then all visual aspects of the DrawItem,
*  including the line width and the fill, will be magnified or reduced during resizes
*  as if the DrawItem were placed under a lens.
* @type {boolean}
* @default false
*/
resizeViaLocalTransformOnly?: boolean;

/**
* This property specifies the conditions for when proportional resizing is used.
*  
*  By default the DrawItem is forced to only resize proportionally while any modifier key
*  specified in DrawItem.proportionalResizeModifiers is pressed. For example, the
*  DrawItem will change its width and height by the same percentage as long as the "Shift"
*  key is held down.
*  
*  Note that this property only has an effect if the DrawItem is showing
*  "resize"
*  
*  DrawItem.knobs.
* @type {ProportionalResizeMode}
* @default "modifier"
*/
proportionalResizing?: ProportionalResizeMode;

/**
* If DrawItem.proportionalResizing is set to "modifier"
*  
*  or "modifierOff"
*  
*  then proportional resizing of the DrawItem is activated or deactivated, respectively,
*  whenever at least one key in this set of modifier keys is pressed.
*  
*  The keys allowed in this set are: "Alt", "Ctrl", and "Shift". If this set
*  of keys is empty then proportional resizing is always used if
*  proportionalResizing is "modifier"
*  and is never used if
*  proportionalResizing is "modifierOff"
*  .
* @type {Array<Partial<KeyName>>}
* @default "["Shift"]"
*/
proportionalResizeModifiers?: Array<Partial<KeyName>>;

/**
* An opaque object specifying the local transformation that should be applied to this
*  DrawItem, obtained through a call to DrawItem.getShapeData.
*  Note: if this property is specified, you should avoid also specifying a
*  DrawItem.translate, DrawItem.scale, DrawItem.xShearFactor, DrawItem.yShearFactor, or DrawItem.rotation.
* @type {object}
* @default null
*/
shapeData?: object;

/**
* Number of pixels the cursor needs to move before the EventHandler starts a drag operation.
* @type {number}
* @default "5"
*/
dragStartDistance?: number;

/**
* Is this DrawItem draggable? If true, then the DrawItem can be drag-repositioned by the user.
* @type {boolean}
* @default false
*/
canDrag?: boolean;

/**
* Context menu to show for this object, an instance of the Menu widget.
*  
*  Note: if Canvas.destroy is called on a canvas, any specified context menu is
*  not automatically destroyed as well. This is in contrast to MenuButtons which
*  automatically destroy their specified MenuButton.menu by default. The behavior
*  is intentional as context menus are commonly reused across components.
* @type {Menu}
* @default null
*/
contextMenu?: Menu;

/**
* Should events inside this DrawItem be attributed to it regardless of which pixels are
*  actually set, if no fill is specified? If set for DrawItems that aren't closed, will
*  capture events occurring in the region that would filled if a fill were specified.
*  This property is true by default for closed shapes, and false for paths, lines, etc.
*  
*  Note: this property must be true if you're writing to an HTML5 &lt;canvas&gt; element
*  directly in your code (only applies to DrawingType "bitmap"
*  ).
* @type {boolean}
* @default "varies"
*/
eventOpaque?: boolean;

/**
* If this item is showing a
*  "move"
*  
*  DrawItem.knobs, this attribute specifies where the knob should appear
*  with respect to the draw item.
*  
*  The resize and move knobs show at the same position by default. However, when
*  both knobs are shown the move knob is offset slightly to allow access to both.
*  This position can be adjusted manually with DrawItem.moveKnobOffset.
* @type {MoveKnobPoint}
* @default "TL"
*/
moveKnobPoint?: MoveKnobPoint;

/**
* If this item is showing a "move" DrawItem.knobs, this attribute
*  allows you to specify an offset in pixels from the DrawItem.moveKnobPoint for the
*  move knob. Offset should be specified as a 2-element array of [left offset, top offset].
*  
*  This offset overrides the built-in offset used when showing both resize and move knobs.
* @type {Array<Partial<Integer>>}
* @default null
*/
moveKnobOffset?: Array<Partial<Integer>>;

/**
* If this item is showing "move"
*  
*  DrawItem.knobs, this attribute specifies the AutoChild for the
*  DrawKnob that allows a user to move the DrawItem with help of a knob located at
*  DrawItem.moveKnobPoint. Default move knob shape is green circle.
* @type {DrawKnob}
* @default null
*/
moveKnob?: DrawKnob;

/**
* If this item is showing "rotate"
*  
*  DrawItem.knobs, this attribute specifies the AutoChild for the
*  DrawKnob that allows a user to rotate the DrawItem with help of a knob located
*  above. Default rotate knob shape is green circle.
* @type {DrawKnob}
* @default null
*/
rotateKnob?: DrawKnob;

/**
* If this item is showing "resize"
*  
*  DrawItem.knobs, this attribute
*  specifies the points with respect to the draw item where resize knobs should appear.
* @type {Array<Partial<ResizeKnobPoint>>}
* @default "["TL","TR","BL","BR","T","R","B","L"]"
*/
resizeKnobPoints?: Array<Partial<ResizeKnobPoint>>;

/**
* If this item is showing "resize"
*  
*  DrawItem.knobs, this attribute specifies the MultiAutoChild for the
*  DrawKnob that allows a user to resize the DrawItem with help of knobs located at
*  corners of a bounding rectangle of current DrawItem. The default shape is a light teal
*  circle.
* @type {DrawKnob}
* @default null
*/
cornerResizeKnob?: DrawKnob;

/**
* If this item is showing "resize"
*  
*  DrawItem.knobs, this attribute specifies the MultiAutoChild for the
*  DrawKnob that allows a user to resize the DrawItem with help of knobs located at
*  centers of edges of a bounding rectangle of current DrawItem. The default shape is a light
*  teal square.
* @type {DrawKnob}
* @default null
*/
sideResizeKnob?: DrawKnob;

/**
* If this item is showing "resize"
*  
*  DrawItem.knobs, this attribute specifies the AutoChild for the
*  DrawRect that draws a rectangle frame which connects all resize knobs of current
*  DrawItem.
* @type {DrawRect}
* @default null
*/
resizeOutline?: DrawRect;

/**
* If this item is showing "resize"
*  
*  DrawItem.knobs will the resize outline be shown or not.
* @type {boolean}
* @default true
*/
showResizeOutline?: boolean;

/**
* If true, when a DrawItem is DrawItem.moveTo or DrawItem.resizeTo, the transform
*  is applied by manipulating the shape coordinates, if possible, rather than by introducing
*  scaling, shearing, rotation, or translation. This is only supported currently for 
*  DrawRect, DrawOval, DrawDiamond, DrawImage, and DrawLabel,
*  and only if no shearing is already present. Further, it's only possible to keep the
*  transform simple if both axes are scaled by the same amount during the resize (or end up at
*  the same scale if the DrawItem is already scaled unevenly), unless the rotation angle is a
*  multiple of 90 degrees.
*  
*  For DrawPolygon and other shapes not based on a box (top/left/width/height), we can't
*  safely just modify coordinates to effect a resize as we can do for DrawRect (and 
*  similar), so resizing will normally introduce or modify the transform, potentially 
*  introducing scaling or shearing, rather than modifying coordinates. For such 
*  DrawItems, we avoid trying to manipulate the coordinates, in part, because there's a
*  danger that the floating point error may accumulate over time and warp the shape.
* @type {boolean}
* @default true
*/
useSimpleTransform?: boolean;

/**
* If this item is showing "startPoint"
*  
*  DrawItem.knobs, this attribute specifies the AutoChild for the
*  DrawKnob for start point of current drawItem.
* @type {DrawKnob}
* @default null
*/
startKnob?: DrawKnob;

/**
* If this item is showing "endPoint"
*  
*  DrawItem.knobs, this attribute specifies the AutoChild for the
*  DrawKnob for end point of current drawItem.
* @type {DrawKnob}
* @default null
*/
endKnob?: DrawKnob;

/**
* Flag indicating a drawItem has been destroyed, similar to Canvas.destroyed.
* @type {boolean}
* @default null
*/
destroyed?: boolean;

/**
* Flag indicating a drawItem is mid-destruction, similar to Canvas.destroying.
* @type {boolean}
* @default null
*/
destroying?: boolean;





/**
*  Computes the angle in degrees from the positive X axis to the difference vector
*  v2 - v1 between the two given vectors.
* 
* @param {Double} X coordinate of v1
* @param {Double} Y coordinate of v1
* @param {Double} X coordinate of v2
* @param {Double} Y coordinate of v2
*/
static computeAngle(px1:Double, py1:Double, px2:Double, py2:Double): Double;



/**
*  Sets a property on this DrawItem, calling the appropriate setter method if one is found and
*  is Class.isMethodSupported.
* 
* @param {string} name of the property to set
* @param {any} new value for the property
*/
setPropertyValue(propertyName:string, newValue:any): void;

/**
*  Returns the DrawItem.zIndex of this draw item.
* 
* @param {boolean=} if true and this item's zIndex is null, resolve to the next higher zIndex.
*/
getZIndex(resolveToNumber?:boolean): Integer;

/**
*  Setter for DrawItem.zIndex.
* 
* @param {Integer} new zIndex. If null and this draw item is already in a DrawPane or DrawGroup, then this item's zIndex will be set to the next higher auto-assigned zIndex.  Note that when setting draw items' zIndexes via this advanced API, the application should take over management of all draw items' zIndexes, and DrawItem.bringToFront / DrawItem.sendToBack should not be used, as those APIs assume automatic management of zIndexes.
*/
setZIndex(newZIndex:Integer): void;

/**
*  Places this draw item at the top of the stacking order so that it appears in front of other
*  draw items in the same DrawPane or DrawGroup.
*  
*  When the DrawPane's DrawPane.drawingType is
*  "bitmap",
*  
*  DrawItem.zIndex, bringToFront(), and DrawItem.sendToBack are not supported
*  for DrawLabels on iOS due to platform limitations.
* 
*/
bringToFront(): void;

/**
*  Places this draw item at the bottom of the stacking order so that it appears behind other
*  draw items in the same DrawPane or DrawGroup.
*  
*  When the DrawPane's DrawPane.drawingType is
*  "bitmap",
*  
*  DrawItem.zIndex, DrawItem.bringToFront, and sendToBack() are not supported
*  for DrawLabels on iOS due to platform limitations.
* 
*/
sendToBack(): void;

/**
* 
* 
* @param {Cursor} new cursor.
*/
setCursor(newCursor:Cursor): void;

/**
*  Returns the center point of this draw item in local coordinates. Generally this is the
*  center of the DrawItem.getBoundingBox, but some item types may use
*  a different point. For example, DrawTriangle uses the
*  incenter
*  of the triangle.
* 
*/
getCenter(): Point;

/**
*  Calculates the bounding box of the shape in the DrawPane.
*  
*  Note that the bounding box of the shape when transformed into the global coordinate system
*  is available from the method DrawItem.getResizeBoundingBox.
* 
* 
*/
getBoundingBox(): Array<Partial<Double>>;

/**
*  Calculates the bounding box of the shape reflected by the
*  DrawItem.showResizeOutline shown when dragging the
*  DrawItem.resizeKnobPoints. This method is similar to DrawItem.getBoundingBox
*  except that the coordinates returned by this method are in the global coordinate system
*  (described DrawPane) rather than the local coordinate system.
* 
*/
getResizeBoundingBox(): Array<any>;

/**
*  Returns true if the given point in the drawing coordinate system, when converted to
*  coordinates in this DrawItem's local coordinate system, is within the
*  DrawItem.getBoundingBox of this DrawItem's shape.
*  
*  This method can be used to quickly check whether the given point is definitely not within
*  the DrawItem shape. To check whether the point is within the DrawItem shape, use the slower
*  but exact DrawItem.isPointInPath method.
* 
* @param {Integer} X coordinate of the point in the drawing coordinate system.
* @param {Integer} Y coordinate of the point in the drawing coordinate system.
*/
isInBounds(x:Integer, y:Integer): boolean;

/**
*  Returns an opaque JavaScript object representing the current local transformation applied to
*  the DrawItem's local coordinates, as defined by DrawItem.translate, DrawItem.scale, 
*  DrawItem.xShearFactor, DrawItem.yShearFactor, and DrawItem.rotation. The object may be serialized
*  and deserialized as JSON, and passed into the constructor block as DrawItem.shapeData to
*  restore the local transformation.
*  Note: this doesn't include any sepatate configuration, such as for a DrawRect
*  the current values of DrawRect.left, DrawRect.top,
*  DrawRect.width, or DrawRect.height.
* 
* 
*/
getShapeData(): object;

/**
*  Returns true if the given point in the drawing coordinate system is within this DrawItem's shape,
*  taking into account local transforms.
* 
* @param {Integer} X coordinate of the test point.
* @param {Integer} Y coordinate of the test point.
*/
isPointInPath(x:Integer, y:Integer): boolean;

/**
*  Computes the angle in degrees from the positive X axis to the difference vector
*  v2 - v1 between the two given vectors.
* 
* @param {Double} X coordinate of v1
* @param {Double} Y coordinate of v1
* @param {Double} X coordinate of v2
* @param {Double} Y coordinate of v2
*/
computeAngle(px1:Double, py1:Double, px2:Double, py2:Double): Double;

/**
*  Returns the page-relative left coordinate of the widget on the page, in pixels.
* 
*/
getPageLeft(): number;

/**
*  Returns the page-relative top coordinate of the widget on the page, in pixels
* 
*/
getPageTop(): number;

/**
*  Setter for DrawItem.canDrag.
* 
* 
* @param {boolean} new value for this.canDrag.
*/
setCanDrag(canDrag:boolean): void;

/**
*  Notification fired when the user starts to drag this DrawItem. Will only fire if DrawItem.canDrag
*  is true.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
dragStart(): boolean;

/**
*  Notification fired for every mouseMove event triggered while the user is dragging this
*  DrawItem. Will only fire if DrawItem.canDrag is true.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
dragMove(): boolean;

/**
*  Notification fired when the user stops dragging this DrawItem. Will only fire if DrawItem.canDrag
*  is true.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
dragStop(): boolean;

/**
*  Notification fired when the user presses the left mouse button on this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
mouseDown(): boolean;

/**
*  Notification fired when the user releases the left mouse button on this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
mouseUp(): boolean;

/**
*  Notification fired when the user clicks on this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
click(): boolean;

/**
*  Notification fired when the mouse enters this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
mouseOver(): boolean;

/**
*  Notification fired when the user moves the mouse over this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
mouseMove(): boolean;

/**
*  Notification fired when the mouse leaves this DrawItem.
*  
*  Note that if this item is part of a DrawGroup, then the group's DrawGroup.useGroupRect
*  setting affects whether this item receives the notification. If useGroupRect is true, then
*  this item will not receive the notification. Otherwise, the item receives the
*  notification and notification bubbles up to the group.
* 
*/
mouseOut(): boolean;

/**
*  Executed when the right mouse button is clicked. The default implementation of
*  this method auto-creates a Menu from the Canvas.contextMenu property on
*  this component and then calls Menu.showContextMenu on it to show it.
*  
*  If you want to show a standard context menu, you can simply define your Menu and set it as the
*  contextMenu property on your component - you do not need to override this method.
*  
*  If you want to do some other processing before showing a menu or do something else entirely, then
*  you should override this method. Note that the return value from this method controls whether or
*  not the native browser context menu is shown.
* 
* 
*/
showContextMenu(): boolean;

/**
*  If DrawItem.canHover is true for this DrawItem, the hover() string method will
*  be fired when the user hovers over this DrawItem. If this method returns false, it will
*  suppress the default behavior of showing a hover canvas if DrawItem.showHover
*  is true.
* 
*/
hover(): boolean;

/**
*  If DrawItem.showHover is true, when the user holds the mouse over this DrawItem for
*  long enough to trigger a hover event, a hover canvas is shown by default. This method returns
*  the contents of that hover canvas. Default implementation returns DrawItem.prompt -
*  override for custom hover HTML. Note that returning null or an empty string will
*  suppress the hover canvas altogether.
* 
*/
getHoverHTML(): HTMLString;

/**
*  Draws this item into its current DrawItem.drawPane.
*  
*  NOTE: For performance reasons, the DrawPane may draw this item on a
*  delay to allow multiple items to be added and drawn at one time. The DrawPane.refreshNow
*  API will force this item to be drawn immediately.
* 
*/
draw(): void;

/**
*  Called when we start drawing for this DrawItem to the DrawItem.drawPane's underlying
*  HTML5 &lt;canvas&gt; element. Only called if the DrawingType is
*  "bitmap".
*  
*  There is no default implementation of this method.
* 
*/
drawStart(): void;

/**
*  Called when we finish drawing for this DrawItem to the DrawItem.drawPane's underlying
*  HTML5 &lt;canvas&gt; element. Only called if the DrawingType is
*  "bitmap".
*  
*  There is no default implementation of this method.
* 
*/
drawEnd(): void;

/**
*  Notification method fired when this component is explicitly moved.
*  Note that a component's position on the screen may also be changed due to an ancestor being
*  moved. The Canvas.parentMoved method provides a notification entry point to catch
*  that case as well.
* 
* 
* @param {Integer} horizontal difference between current and previous position
* @param {Integer} vertical difference between current and previous position
*/
moved(deltaX:Integer, deltaY:Integer): void;

/**
*  Observable method called whenever 
*  a DrawItem changes size.
* 
*/
resized(): void;

/**
*  Shows a set of control knobs for this drawItem. Updates DrawItem.knobs to include the
*  specified knobType, and if necessary draws out the appropriate control knobs.
* 
* @param {KnobType | Array<Partial<KnobType>>} knobs to show
*/
showKnobs(knobType:KnobType | Array<Partial<KnobType>>): void;

/**
*  Shows all supported control knobs for this drawItem. Updates DrawItem.knobs to include the
*  supported knobTypes and if necessary draws out the appropriate control knobs.
* 
*/
showAllKnobs(): void;

/**
*  Hides a set of control knobs for this drawItem. Updates DrawItem.knobs to remove the
*  specified knobType, and clears any drawn knobs for this knobType. 
* 
* @param {KnobType | Array<Partial<KnobType>>} knobs to hide
*/
hideKnobs(knobType:KnobType | Array<Partial<KnobType>>): void;

/**
*  Hides all control knobs for this drawItem. Updates DrawItem.knobs to remove
*  all knobTypes and clears any drawn knobs. 
* 
*/
hideAllKnobs(): void;

/**
*  Setter for DrawItem.moveKnobOffset.
* 
* @param {Array<Partial<Integer>>=} the new move knob offset. This is a 2-element array of [left offset, top offset]. If null, then [0,0]  is assumed.
*/
setMoveKnobOffset(newMoveKnobOffset?:Array<Partial<Integer>>): void;

/**
*  Setter for the DrawItem.title of this DrawItem.
* 
* @param {string=} new title.
*/
setTitle(newTitle?:string): void;

/**
*  If DrawItem.canDrag is true and the DrawItem.knobs include "resize" knobs,
*  then this notification method will be fired when the user drag-resizes the draw item.
* 
* @param {string} provides which knob of the DrawItem.resizeKnobPoints was dragged
* @param {Integer} new x-coordinate of the knob
* @param {Integer} new y-coordinate of the knob
* @param {Integer} horizontal distance moved
* @param {Integer} vertical distance moved
*/
dragResizeMove(position:string, x:Integer, y:Integer, dX:Integer, dY:Integer): void;

/**
*  Setter for DrawItem.drawPane.
* 
* @param {DrawPane} new value for this.drawPane.
*/
setDrawPane(drawPane:DrawPane): void;

/**
*  Erase this drawItem's visual representation and remove it from its DrawGroup (if any) and
*  DrawPane.
*  
*  To re-draw the item within the DrawPane, call DrawItem.draw again, or use
*  DrawPane.addDrawItem to move to another DrawGroup.
*  
* 
*/
erase(): void;

/**
*  Permanently destroys this DrawItem, similar to Canvas.destroy.
* 
* 
*/
destroy(): void;

/**
*  Generates a string containing the SVG source of this DrawItem.
* 
*  NOTE: The generated SVG source assumes that the default namespace is http://www.w3.org/2000/svg
*  and that namespace prefix xlink refers to namespace name http://www.w3.org/1999/xlink.
* 
* 
*/
getSvgString(): void;

/**
*  Make this drawItem visible.
* 
*/
show(): void;

/**
*  Hide this drawItem.
* 
*/
hide(): void;

/**
*  Update lineWidth for this drawItem.
* 
* @param {Integer} new pixel lineWidth
*/
setLineWidth(width:Integer): void;

/**
*  Update lineColor for this drawItem.
* 
* @param {CSSColor} new line color. Pass null for transparent.
*/
setLineColor(color:CSSColor): void;

/**
*  Update lineOpacity for this drawItem.
* 
* @param {Float} new opacity, as a number between 0 (transparent) and 1 (opaque).
*/
setLineOpacity(opacity:Float): void;

/**
*  Update linePattern for this drawItem.
* 
* @param {LinePattern} new linePattern to use
*/
setLinePattern(pattern:LinePattern): void;

/**
*  Update lineCap for this drawItem.
* 
* @param {LineCap} new lineCap to use
*/
setLineCap(cap:LineCap): void;

/**
*  Update shadow for this drawItem.
* 
* @param {Shadow} new shadow
*/
setShadow(shadow:Shadow): void;

/**
*  Does this DrawItem Class.isMethodSupported DrawItem.setStartArrow?
*  For example, this is false for DrawRect and DrawOval, and true for DrawLine.
* 
*/
supportsStartArrow(): boolean;

/**
*  Set the arrowhead at the beginning of this path.
*  
*  NOTE: Not all DrawItem classes support arrowheads. You can use DrawItem.supportsStartArrow
*  to dynamically check whether a DrawItem instance supports this method.
* 
* @param {ArrowStyle} style of arrow to use
*/
setStartArrow(arrowStyle:ArrowStyle): void;

/**
*  Does this DrawItem Class.isMethodSupported DrawItem.setEndArrow?
*  For example, this is false for DrawRect and DrawOval, and true for DrawLine.
* 
*/
supportsEndArrow(): boolean;

/**
*  Set the arrowhead at the end of this path.
*  
*  NOTE: Not all DrawItem classes support arrowheads. You can use DrawItem.supportsEndArrow
*  to dynamically check whether a DrawItem instance supports this method.
* 
* @param {ArrowStyle} style of arrow to use
*/
setEndArrow(arrowStyle:ArrowStyle): void;

/**
*  Move the shape by the specified deltas for the left and top coordinate.
* 
* 
* @param {Integer} change to left coordinate in pixels
* @param {Integer} change to top coordinate in pixels
*/
moveBy(dX:Integer, dY:Integer): void;

/**
*  Move the DrawItem to the specified coordinates in the global coordinate system. The
*  specified coordinates will become the top-left point of the
*  DrawItem.getResizeBoundingBox.
* 
* @param {Integer} new left coordinate in pixels
* @param {Integer} new top coordinate in pixels
*/
moveTo(left:Integer, top:Integer): void;

/**
*  Change the center point for this DrawItem.
* 
* @param {Integer} X coordinate of the center point (in the global coordinate system).
* @param {Integer} Y coordinate of the center point (in the global coordinate system).
*/
setCenterPoint(left:Integer, top:Integer): void;

/**
*  Resize the shape by the specified deltas.
* 
* @param {Integer} number of pixels to resize by horizontally
* @param {Integer} number of pixels to resize by vertically
*/
resizeBy(dX:Integer, dY:Integer): void;

/**
*  Resize to the specified size
* 
* @param {Integer} new width
* @param {Integer} new height
*/
resizeTo(width:Integer, height:Integer): void;

/**
*  Rotate the shape by the relative rotation in degrees
* 
* @param {Float} number of degrees to rotate from current orientation.
*/
rotateBy(degrees:Float): void;

/**
*  Rotate the shape by the absolute rotation in degrees
* 
* @param {Float} number of degrees to rotate
*/
rotateTo(degrees:Float): void;

/**
*  Scale the shape by the x, y multipliers
* 
* @param {Float} scale in the x direction
* @param {Float} scale in the y direction
*/
scaleBy(x:Float, y:Float): void;

/**
*  Scale the shape by the x, y multipliers
* 
* @param {Float} scale in the x direction
* @param {Float} scale in the y direction
*/
scaleTo(x:Float, y:Float): void;

/**
*  Update fillColor for this drawItem.
* 
* @param {CSSColor} new fillColor to use. Pass null for transparent.
*/
setFillColor(color:CSSColor): void;

/**
*  Update fillGradient for this drawItem.
* 
* @param {Gradient} new gradient to use. Pass null for transparent.
*/
setFillGradient(gradient:Gradient): void;

/**
*  Update fillOpacity for this drawItem.
* 
* @param {Float} new opacity, as a number between 0 (transparent) and 1 (opaque).
*/
setFillOpacity(opacity:Float): void;




}

/**
* Sets up a real inheritance structure for Javascript objects.
* 	We separate out class objects from prototypes, so each gets its own inheritance chain. 
* 	This allows us to set up superclass calls, maintain class vs. instance variables and more!
* 
* 	The ClassFactory is a singleton object that holds the miscellaneous pieces of our inheritance
* 	mechanism.
* 
* 	Your main interaction with the ClassFactory is to create new classes:
*  ClassFactory.defineClass("MyClass", "mySuperClass");
*/
class ClassFactory {






/**
* 
*  Create a new SmartClient class, which can then be used to create instances of this
*  object type, via Class.create.
*  
*  The new Class is returned by defineClass, is available as
*  isc.ClassName and is also available in global scope if not in
*  isc. Typically, Class.addProperties is then
*  called to establish different defaults in the new class, or to add methods. For
*  example:
*  
*   isc.defineClass("MyListGrid", "ListGrid").addProperties({
*     headerHeight : 40, // change default for listGrid.headerHeight
* 
*     // override listGrid.recordClick
*     recordClick : function (viewer, record) { 
*       isc.say(record.description);
*     }
*   })
*   isc.MyListGrid.create(); // create an instance of the new class
*  
*  
*  See also Class.Super for calling superclass methods.
*  
*  NOTE: isc.defineClass() also creates a new function
*  isA.ClassName() object for identifying instances of
*  this Class.
* 
* 
* @param {string} Name for the new class.
* @param {Class=} Optional SuperClass Class object or name
*/
static defineClass(className:string, superClass?:Class): Class;

/**
* 
*  Intentionally clobber an existing SmartClient Class, if it already exists. Works 
*  identically to ClassFactory.defineClass, except that no warning is logged to the
*  console.
* 
* 
*/
static overwriteClass(): void;

/**
* 
* 	Given a class name, return a pointer to the Class object for that class
* 
* 
* @param {string} name of a class
*/
static getClass(className:string): Class;

/**
* 
*  Given the name of a class, create an instance of that class.
* 	
* 
* @param {string} Name of a class.   	(ClassObject)	Actual class object to use.
* @param {object=} Properties to apply to the instance.
* @param {object=} More properties to apply to the instance.
* @param {object=} Yet more properties to apply to the instance.
*/
static newInstance(className:string, props?:object, props2?:object, props3?:object): Class;






}

/**
* Possible values for the state of ListGrid groups when groupBy is called
*/
type GroupStartOpen = 
/**
* open all groups
*/
"all" |

/**
* open the first group
*/
"first" |

/**
* start with all groups closed
*/
"none";








/**
* Do we display an arrow for the sorted field ?
*/
type SortArrow = 
/**
* Don't show a sort arrow at all.
*/
"none" |

/**
* Display sort arrow in the upper-right corner (above the scrollbar) only.
*/
"corner" |

/**
* Display sort arrow above each field header only.
*/
"field" |

/**
* Display sort arrow above each field header AND in corner above scrollbar.BOTH:"both", // NOTE: Canvas establishes this constant
*/
"both";


/**
* Header item for a collapsible section in a DynamicForm. Each SectionItem
*  is associated with a number of other items in the form, which will be shown or
*  hidden as a group when the section is expanded or collapsed. Clicking on a
*  SectionItem will expand or collapse the section.
*  
*  To make a form where only one section is expanded at a time, set
*  DynamicForm.sectionVisibilityMode to "mutex".
*/
class SectionItem extends CanvasItem {
/**
* Creates a new SectionItem
* 
* @param typeCheckedProperties {Partial<SectionItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {SectionItem} The new instance
*/
static create(typeCheckedProperties? : Partial<SectionItem>, uncheckedProperties?: Object): SectionItem;


/**
* Section items show their value as title text for the section.
*  Therefore the simplest way to specify this text on the form item directly is via the
*  defaultValue attribute.
* @type {string}
* @default "Section Header"
*/
defaultValue?: string;

/**
* IDs of the items that should be considered a member of this section.
* @type {Array<Partial<GlobalId>>}
* @default null
*/
itemIds?: Array<Partial<GlobalId>>;

/**
* Whether this form section should initially be visible.
* @deprecated * As of SmartClient version 5.5, use SectionItem.sectionExpanded       instead.

* @type {boolean}
* @default true
*/
sectionVisible?: boolean;

/**
* Whether this form section should be initially collapsed. Can be set programmatically
*  via SectionItem.expandSection and SectionItem.collapseSection.
* @type {boolean}
* @default true
*/
sectionExpanded?: boolean;

/**
* Name of the Canvas subclass to use as a header that labels the section and allows
*  showing and hiding. The default class be skinned, or trivial subclasses created to
*  allow different appearances for SectionItems in different forms.
*  Very advanced developers can use the following information to create custom header
*  classes.
* @type {string}
* @default "SectionHeader"
*/
sectionHeaderClass?: string;

/**
* Whether this section header can be collapsed. If set false, suppresses open/close state
*  icon
* @type {boolean}
* @default true
*/
canCollapse?: boolean;

/**
* If true, the header for this Section will be included in the page's tab
*  order for accessibility.
*  May also be set at the DynamicForm level via 
*  DynamicForm.canTabToSectionHeaders.
*  
*  See accessibility.
* @type {boolean}
* @default null
*/
canTabToHeader?: boolean;







/**
*  Returns a boolean indicating whether this SectionItem is expanded.
* 
* 
*/
isExpanded(): boolean;

/**
*  Expands a section, showing all the items contained within the section.
* 
*/
expandSection(): void;

/**
*  Collapse a sectionItem, and hide all the items within the section (not including
*  the header).
* 
*/
collapseSection(): void;




}



/**
*/
type LinkDataFetchMode = 
/**
* In this mode, link data is fetched from the ResultTree.linkDataSource and nodes are separately fetched from the 
*  ResultTree.dataSource. The two fetches are sent together in a 
*  RPCManager.startQueue, with the link data fetch first and the separate 
*  node fetch second. This makes it possible for your server-side code to use the results
*  of the link data fetch to constrain the node fetch (ie, only fetch node information 
*  for nodes that appear in a link)
*/
"separate" |

/**
* In this mode, nodes and link data are fetched together from the main ResultTree.dataSource, and any duplicated node IDs are handled by creating multiple
*  links to a single node. In this mode, the ResultTree.linkDataSource is only used
*  for update operations.
*  
*  Note that the end result of a "single" fetch is exactly the same as fetching link data 
*  and nodes separately using "separate" mode; "separate" mode is also conceptually clearer
*  since it emphasises the fact that nodes and link data are separate things. We provide
*  "single" mode because, in some cases, it may be more efficient to fetch the two types of
*  data together in a single database fetch, using DataSourceField.includeFrom or 
*  some other kind of join technique on the server.
*/
"single";


/**
* The strategy to use when generating field names - for example, for new formula or summary 
*  fields created using the built-in editors.
*/
type FieldNamingStrategy = 
/**
* generate names in the format fieldTypeX, where field type might be     "formulaField" and X is an index specific to the field-type and component instance
*/
"simple" |

/**
* generates a UUID for all generated field names
*/
"uuid";




/**
* Class for editable multi-line text areas (uses HTML &lt;TEXTAREA&gt; object)
*  automatically expands to accomodate its content
*/
class AutoFitTextAreaItem extends TextAreaItem {
/**
* Creates a new AutoFitTextAreaItem
* 
* @param typeCheckedProperties {Partial<AutoFitTextAreaItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AutoFitTextAreaItem} The new instance
*/
static create(typeCheckedProperties? : Partial<AutoFitTextAreaItem>, uncheckedProperties?: Object): AutoFitTextAreaItem;


/**
* If specified, the autoFitTextArea will not grow taller than this height
* @type {Integer}
* @default null
*/
maxHeight?: Integer;










}

/**
*/
type FieldAuditMode = 
/**
* the field value will be saved for all audited adds and updates, regardless of         whether it's changed or present in the update
*/
"always" |

/**
* the field value will be saved for all audited adds, and for updates in which         the value changed
*/
"change" |

/**
* the field value will never be saved for any DataSource audit operation
*/
"never";


/**
* Response sent by the server in response to a DSRequest. Contains
*  all the properties available on the basic RPCResponse, in addition to the
*  properties listed here.
*/
class DSResponse extends RPCResponse {
/**
* Creates a new DSResponse
* 
* @param typeCheckedProperties {Partial<DSResponse>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSResponse} The new instance
*/
static create(typeCheckedProperties? : Partial<DSResponse>, uncheckedProperties?: Object): DSResponse;


/**
* The DSRequest.clientContext object as set on the DSRequest.
* @type {object}
* @default null
*/
clientContext?: object;

/**
* The DataSource of this DSResponse.
* @type {string}
* @default null
*/
dataSource?: string;

/**
* The operation type of the request corresponding to this DSResponse.
* @type {DSOperationType}
* @default null
*/
operationType?: DSOperationType;

/**
* The operation ID of the request corresponding to this DSResponse.
* @type {string}
* @default null
*/
operationId?: string;

/**
* For "fetch" operations, this is the array of Records fetched. For "update", "add", and
*  "remove" operations, this is typically an array containing a single Record representing
*  the record that was updated, added, or removed.
* @type {Array<Partial<Record>> | any}
* @default null
*/
data?: Array<Partial<Record>> | any;

/**
* Same meaning as RPCResponse.status, except DSResponses have additional error codes,
*  such as RPCResponse.STATUS_VALIDATION_ERROR.
* @type {Integer}
* @default "see below"
*/
status?: Integer;

/**
* An extra property of each DSResponse to a queued request that indicates whether the queue
*  as a whole succeeded. A queueStatus of RPCResponse.STATUS_SUCCESS, or 0, indicates
*  that the queue succeeded whereas a queueStatus of RPCResponse.STATUS_FAILURE, or -1,
*  indicates that the queue failed.
* 
*  For example, if two "update" requests are sent in a queue and the first succeeded, but
*  the second failed validation, then both DSResponses' queueStatus would be -1, but the
*  DSResponse.status of the first would be RPCResponse.STATUS_SUCCESS and
*  the status of the second would be an error code such as RPCResponse.STATUS_VALIDATION_ERROR.
* @type {Integer}
* @default "see below"
*/
queueStatus?: Integer;

/**
* Server-side validation errors for an attempted "update" or "add" operation, as a 
*  JS Object where each property name is a field name from the record and each property
*  value contains error information.
*  
*  To extract just the simple error strings for each field we recommend passing this
*  object to DataSource.getSimpleErrors
*  
*  The Java API DSResponse.addError(fieldName, errorMessage) is used to send server-side
*  errors to the client. See the Java Server Reference for